/*
 * GStreamer
 * Copyright 2007 Danilo Freire <danilo@embedded.ufcg.edu.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <assert.h>

#include <linux/soundcard.h>

#include "radio_calls.h"

gint
gst_radio_open(GstRadioBin *radiobin)
{
	  GST_DEBUG_OBJECT (GST_ELEMENT(radiobin), "Trying to open device %s",
			            radiobin->radio_dev);
	  //g_print("open device %s \n", radiobin->radio_dev);
	  
	  if ((radiobin->radio_fd = open(radiobin->radio_dev, O_RDONLY))< 0)
	  {
		  goto error;
	  }
	  
	  if (ioctl(radiobin->radio_fd, VIDIOC_QUERYCAP, &(radiobin->vc))<0)
	  { 
	      goto error;
	  }
	 
	  (radiobin->vt).index = 0;
	  ioctl(radiobin->radio_fd, VIDIOC_S_TUNER, &(radiobin->vt));
	  
	  if (!((radiobin->vc).capabilities & V4L2_CAP_TUNER))
	  {
	      goto error;
	  }
	  
	  radiobin->mixer_fd = open(radiobin->mixer_dev, O_RDWR);
	  if (radiobin->mixer_fd < 0)
	  {
		  //g_print("Error on mixer open\n");
		  goto error;
	  }
	  //gst_radio_unmute(radiobin);

	  return TRUE;

error:
	 if (radiobin->radio_fd >= 0)
	 {
		 close(radiobin->radio_fd);
		 radiobin->radio_fd = -1;
	 }
	 return FALSE;
}

void
gst_radio_close(GstRadioBin *radiobin)
{
	GST_DEBUG_OBJECT (GST_ELEMENT(radiobin), "Trying to close device %s",
			            radiobin->radio_dev);
	
	gst_radio_mute(radiobin);
	if (radiobin->mixer_fd >=0)
	{
	  //g_print("close device %s \n", radiobin->mixer_dev);
	  close(radiobin->mixer_fd);
	  radiobin->mixer_fd = -1;
	}
	if (radiobin->radio_fd >= 0)
	{
	  //g_print("close device %s \n", radiobin->radio_dev);
	  close(radiobin->radio_fd);
	  radiobin->radio_fd = -1;
	}
}

gint
gst_radio_setfreq(GstRadioBin *radiobin, guint freq)
{	
	gint res;
	
	if (radiobin->radio_fd < 0)
		return -1;
	(radiobin->vf).tuner = 0;
	(radiobin->vf).frequency = (freq*FREQ_FRAC);
 
	res = ioctl(radiobin->radio_fd, VIDIOC_S_FREQUENCY, &(radiobin->vf));
	
	if(res > 0)
		radiobin->freq = freq;
	
	return res;
}

guint
gst_radio_getfreq(GstRadioBin *radiobin)
{
	gint res;
	gulong freq;
	
	if (radiobin->radio_fd < 0)
		return -1;
	
	res = ioctl(radiobin->radio_fd, VIDIOC_G_FREQUENCY, &(radiobin->vf));
	
	if(res < 0)
		return -1;
	
	freq = (radiobin->vf).frequency;
	freq /= FREQ_FRAC;
	
	radiobin->freq = freq;
	
	return freq;
}

void
gst_radio_mute(GstRadioBin *radiobin)
{
	gint res;
	(radiobin->vctrl).id = V4L2_CID_AUDIO_MUTE;
	(radiobin->vctrl).value = 1;
	res = ioctl(radiobin->radio_fd, VIDIOC_S_CTRL, &(radiobin->vctrl));
	if( res > 0 )
	{
		radiobin->silent = TRUE;
	}
}

void
gst_radio_unmute(GstRadioBin *radiobin)
{
	//g_print("umute device %s \n", radiobin->radio_dev);
	gint res;
	(radiobin->vctrl).id = V4L2_CID_AUDIO_MUTE;
	(radiobin->vctrl).value = 0;
	res = ioctl(radiobin->radio_fd, VIDIOC_S_CTRL, &(radiobin->vctrl));
	if( res > 0 )
	{
		radiobin->silent = FALSE;
	}
}

void
gst_radio_setvolume(GstRadioBin *radiobin, guint vol)
{
	gint res;
	//g_print("set volume %i \n", vol);

	vol |= (vol << 8);
	res = ioctl(radiobin->mixer_fd, SOUND_MIXER_WRITE_LINE, &vol);
	if(res > 0)
	{
		radiobin->vol = vol;
	}
}

guint
gst_radio_getvolume(GstRadioBin *radiobin)
{
	guint vol;
	gint res;
	
	if (radiobin->mixer_fd < 0)
	{	
		goto error;
	}
	
	res = ioctl(radiobin->mixer_fd, SOUND_MIXER_READ_LINE, &vol);
	
	if(res > 0)
	{
		radiobin->vol = vol;
		return vol;
	}

error:
	return -1;
}

void
gst_radio_get_capabilities(void)
{
	
}

gint
gst_radio_get_signal_strength(GstRadioBin *radiobin)
{
        if (ioctl(radiobin->radio_fd, VIDIOC_G_TUNER, &(radiobin->vt)) < 0)
        {
                return -1;
        }
        usleep(SAMPLEDELAY);
        return radiobin->vt.signal;
}

void
gst_radio_set_mixer(char* soundmixer)
{
	
}
