/*
   GStreamer based motion JPEG server
   Copyright (C) 2008  Seppo Yliklaavu <seppo.yliklaavu@gmail.com>

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <glib.h>
#include <glib/gprintf.h>
#include <unistd.h>
#include <string.h>
#include <strings.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <arpa/inet.h>
#include <getopt.h>
#include <gst/gst.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib-lowlevel.h>
#include "ctmf.h"

/* widths and heights for different video formats */
#define WIDTH_SQCIF   128
#define WIDTH_QQVGA   160
#define WIDTH_QCIF    176
#define WIDTH_QVGA    320
#define WIDTH_VGA     640
#define HEIGHT_SQCIF  96
#define HEIGHT_QQVGA  120
#define HEIGHT_QCIF   144
#define HEIGHT_QVGA   240
#define HEIGHT_VGA    480

/* some limits */
#define MIN_FPS                1
#define MAX_FPS               15
#define MIN_QUALITY            0
#define MAX_QUALITY          100
#define MAX_NUM_CLIENTS        5
#define MAX_FILE_NAME_LENGTH 100
#define NUM_OF_IGNORED_FRAMES  5

/* dbus related defines */
#define DBUS_INTERFACE_NAME "com.nokia.gstmjpg"
#define DBUS_OBJECT_PATH    "/com/nokia/gstmjpg"

enum {
    SQCIF,
    QQVGA,
    QCIF,
    QVGA,
    VGA
};

typedef struct
{
    guint numLumaPixels;
    guint width;
    guint height;
    guint radius;
    guint memsize;
    guint threshold;
} MotionDetParams;

typedef struct
{
    guint fps;
    gint size;
    guint quality;
    double thresholdDouble;
    gchar *fileName;
    gint port;
    gint radius;
} CmdLineParams;

static DBusHandlerResult msgHandler(DBusConnection *connection,
                                    DBusMessage *msg,
                                    gpointer user_data);

/* motion branch picture scaling factor */
static const guint scalefactor = 2;

static CmdLineParams cmdLineParams[1];
static MotionDetParams motionDetParams[1];

static gint socketFd = -1;

static GMutex *jpegDataMutex = NULL;
static GCond *jpegDataCV = NULL;

static guint8 *prevLumaImage = NULL;
static guint8 *tmpLumaImage = NULL;
static gboolean motionDetected = FALSE;
static GMutex *motionDetectedMutex = NULL;
static gint fileFd = -1;

static guint8 *jpegBuffer = NULL;
static guint numBytesInJpegBuffer = 0;

static gint numClients = 0;
static GMutex *numClientsMutex = NULL;

static GstElement *pipeline = NULL;

static DBusConnection* connection = NULL;
static DBusObjectPathVTable vtable = {NULL, msgHandler, NULL};
static GMainLoop *loop = NULL;

/* Helper to calculate fps */
static guint GetFps(const char *in)
{

    gint fps = strtol(in, NULL, 10);

    if (fps < MIN_FPS)
    {
        fps = MIN_FPS;
    }
    else if (fps > MAX_FPS)
    {
        fps = MAX_FPS;
    }

    return (fps);

}

/* Helper to calculate jpeg quality */
static guint GetQuality(const char *in)
{

    gint quality = strtol(in, NULL, 10);

    if (quality < MIN_QUALITY)
    {
        quality = MIN_QUALITY;
    }
    else if (quality > MAX_QUALITY)
    {
        quality = MAX_QUALITY;
    }

    return (quality);

}

/* Helper to calculate threshold */
static double GetThreshold(const char *in)
{

    double thr = strtod(in, NULL);

    if (thr < 0.0)
    {
        thr = 0.0;
    }

    return (thr);

}

/* Helper to calculate frame height */
static guint GetHeight(const gint size)
{

    guint height;

    switch (size)
    {
        case SQCIF:
            height = HEIGHT_SQCIF;
            break;

        case QQVGA:
            height = HEIGHT_QQVGA;
            break;

        case QCIF:
            height = HEIGHT_QCIF;
            break;

        case QVGA:
            height = HEIGHT_QVGA;
            break;

        case VGA:
            height = HEIGHT_VGA;
            break;

        default:
            height = HEIGHT_QVGA;
            break;
    }

    return (height);

}

/* Helper to calculate frame width */
static guint GetWidth(const gint size)
{

    guint width;

    switch (size)
    {
        case SQCIF:
            width = WIDTH_SQCIF;
            break;

        case QQVGA:
            width = WIDTH_QQVGA;
            break;

        case QCIF:
            width = WIDTH_QCIF;
            break;

        case QVGA:
            width = WIDTH_QVGA;
            break;

        case VGA:
            width = WIDTH_VGA;
            break;

        default:
            width = WIDTH_QVGA;
            break;
    }

    return (width);

}

/* Helper to calculate video format */
static gint GetSize(const char *in)
{

    gint size;

    if (!strcasecmp(in, "SQCIF"))
    {
        size = SQCIF;
    }
    else if (!strcasecmp(in, "QCIF"))
    {
        size = QCIF;
    }
    else if (!strcasecmp(in, "QVGA"))
    {
        size = QVGA;
    }
    else if (!strcasecmp(in, "QQVGA"))
    {
        size = QQVGA;
    }
    else if (!strcasecmp(in, "VGA"))
    {
        size = VGA;
    }
    else
    {
        size = QVGA;
    }

    return (size);

}

/* GSreamer bus message handler */
static gboolean BusMessageHandler(GstBus *bus,
                                  GstMessage *message,
                                  gpointer data)
{

    /*GstState state;*/

    g_assert(bus);
    g_assert(message);

    switch (GST_MESSAGE_TYPE (message))
    {
        case GST_MESSAGE_WARNING:
            {
                GError *err;
                gchar *debug;

                gst_message_parse_error(message, &err, &debug);
                g_warning("%s(): GST_MESSAGE_WARNING: %s", G_STRFUNC,
                          err->message);
                g_error_free(err);
                g_free(debug);
                break;
            }

        case GST_MESSAGE_ERROR:
            {
                GError *err;
                gchar *debug;

                gst_message_parse_error(message, &err, &debug);
                g_warning("%s(): GST_MESSAGE_ERROR: %s",G_STRFUNC,err->message);
                g_error_free(err);
                g_free(debug);
                break;
            }

        case GST_MESSAGE_EOS:
            break;

        case GST_MESSAGE_STATE_CHANGED: 
            break;

        default:
            break;
    }

    /* remove message from the queue */
    return (TRUE);
}

/* Callback function to store JPEG data to buffer and to signal cond var */
static void StoreJPEGData(GstElement *object,
                          GstBuffer * buffer,
                          GstPad *pad,
                          gpointer user_data)
{

    guint8 *data = (guint8*)GST_BUFFER_DATA(buffer);
    guint size = GST_BUFFER_SIZE(buffer);

    /* store new jpeg data and signal the threads about it */
    g_mutex_lock(jpegDataMutex);
    memcpy(jpegBuffer, data, size);
    numBytesInJpegBuffer = size;
    g_cond_broadcast(jpegDataCV);
    g_mutex_unlock(jpegDataMutex);

}

/* Function to set motion detected flag */
static void SetMotionDetectedFlag(const gboolean value)
{
    g_mutex_lock(motionDetectedMutex);
    motionDetected = value;
    g_mutex_unlock(motionDetectedMutex);
}

/* Function to get motion detected flag */
static gboolean GetMotionDetectedFlag(void)
{
    gboolean value;

    g_mutex_lock(motionDetectedMutex);
    value = motionDetected;
    g_mutex_unlock(motionDetectedMutex);

    return (value);
}

/* Function to increment number of clients */
static void IncrementNumClients(void)
{
    g_mutex_lock(numClientsMutex);
    numClients++;
    g_mutex_unlock(numClientsMutex);
}

/* Function to decrement number of clients */
static void DecrementNumClients(void)
{
    g_mutex_lock(numClientsMutex);
    g_assert(numClients > 0);
    numClients--;
    g_mutex_unlock(numClientsMutex);
}

/* Function to get number of clients */
static guint GetNumClients(void)
{
    guint value;

    g_mutex_lock(numClientsMutex);
    value = numClients;
    g_mutex_unlock(numClientsMutex);

    return (value);
}

/* Function to calculate sum of absolute differences */
static guint CalculcateSAD(const guint8 *img1,
                           const guint8 *img2,
                           const guint length)
{

    guint i;
    guint sad = 0;

    for(i = 0; i < length; i++)
    {
        sad += ABS(img1[i] - img2[i]);
    }

    return (sad);

}

/* Callback function to store luma component and to detect possible motion */
static void CheckForMotion(GstElement *object,
                           GstBuffer * buffer,
                           GstPad *pad,
                           gpointer userData)

{

    guint sad;
    guint8 *data = (guint8*)GST_BUFFER_DATA(buffer);
    MotionDetParams *params =  (MotionDetParams*)userData;
    static guint invocation = 0;

    /* filter luma component using median filter */
    ctmf(data,             /* src */
         tmpLumaImage,     /* dst */
         params->width,    /* width */
         params->height,   /* height */
         params->width,    /* src_step */
         params->width,    /* dst_step */
         params->radius,   /* r */
         1,                /* cn */
         params->memsize); /* memsize */

    if (invocation > NUM_OF_IGNORED_FRAMES) /* dump the first frames */
    {
        sad = CalculcateSAD(prevLumaImage, tmpLumaImage, params->numLumaPixels);
        if (sad > params->threshold)
        {
            g_message("motion detected, sad=%u, sad thr=%u", sad, params->threshold);
            SetMotionDetectedFlag(TRUE);
        }
        else
        {
            SetMotionDetectedFlag(FALSE);
        }
    }
    else
    {
        /* do some sanity check during the first invocations */
        invocation++;
        g_assert(params);
        g_assert((params->width * params->height) == params->numLumaPixels);
        g_assert((params->numLumaPixels + params->numLumaPixels / 2) ==
                 GST_BUFFER_SIZE(buffer));
    }

    memcpy(prevLumaImage, tmpLumaImage, params->numLumaPixels);

}

/* Function to create GStreamer motion detection bin/branch of the pipeline */
static GstElement* CreateMotionDetectionBin(const gint size,
                                            const guint sf)
{

    guint numMotionLumaPixels;
    GstElement *bin = NULL; 
    GstElement *mvrate = NULL; 
    GstElement *mqueue = NULL; 
    GstElement *msink = NULL; 
    GstElement *mscale = NULL; 
    GstCaps *mcaps1 = NULL;
    GstCaps *mcaps2 = NULL;
    GstPad *pad = NULL;

    g_assert(sf % 2 == 0);

    numMotionLumaPixels = (GetWidth(size) * GetHeight(size)) / (sf * sf);

    bin = gst_bin_new("mdbin");
    msink = gst_element_factory_make("fakesink", "msink");
    mscale = gst_element_factory_make("videoscale", "mvideoscale");
    mvrate = gst_element_factory_make("videorate", "mvrate");
    mqueue = gst_element_factory_make("queue", "mqueue");
    g_assert(bin && mqueue && mvrate && mscale && msink);

    mcaps1 = gst_caps_new_simple("video/x-raw-yuv",
                                 "framerate", GST_TYPE_FRACTION, 1, 1,
                                 NULL);
    mcaps2 = gst_caps_new_simple("video/x-raw-yuv",
                                 "width",  G_TYPE_INT, GetWidth(size) / sf,
                                 "height", G_TYPE_INT, GetHeight(size) / sf,
                                 NULL);

    gst_bin_add_many(GST_BIN(bin), mqueue, mvrate, mscale, msink, NULL);
    if (gst_element_link_many(mqueue, mvrate, NULL) == FALSE ||
        gst_element_link_filtered(mvrate, mscale, mcaps1) == FALSE ||
        gst_element_link_filtered(mscale, msink, mcaps2) == FALSE)
    {
        g_error("Gstreamer linking error");
    }
    gst_caps_unref(mcaps1);
    gst_caps_unref(mcaps2);

    pad = gst_element_get_pad(mqueue, "sink");
    g_assert(pad);
    if (gst_element_add_pad(bin, gst_ghost_pad_new("sink", pad)) == FALSE)
    {
        g_error("Gstreamer error when adding a pad");
    }
    gst_object_unref(pad);

    /* 0 = nearest-neighbour */
    /* 1 = bilinear */
    /* 2 = 4-tap */
    g_object_set(G_OBJECT(mscale), "method", 0, NULL);

    motionDetParams->width = GetWidth(size) / sf;
    motionDetParams->height = GetHeight(size) / sf;
    motionDetParams->numLumaPixels = numMotionLumaPixels;
    motionDetParams->radius = cmdLineParams->radius;
    motionDetParams->memsize = 32 * 1024;
    motionDetParams->threshold = 
        (guint)((double)numMotionLumaPixels*cmdLineParams->thresholdDouble);

    g_object_set(G_OBJECT(msink), "signal-handoffs", TRUE, NULL);
    g_signal_connect(G_OBJECT(msink),"handoff",G_CALLBACK(CheckForMotion),
                     (gpointer)motionDetParams);

    return (bin);

}

/* Function to create GStreamer pipeline */
static void CreatePipeline(const gint size,
                           const guint fps,
                           const guint quality,
                           const guint sf,
                           const gboolean enableMotionDet)
{

    GstElement *src = NULL; 
    GstElement *vrate = NULL; 
    GstElement *tee = NULL; 
    GstElement *conv = NULL; 
    GstElement *overlay = NULL; 
    GstElement *enc = NULL; 
    GstElement *queue = NULL; 
    GstElement *sink = NULL; 
    GstElement *mdbin = NULL;
    GstBus *bus = NULL;
    GstCaps *caps1 = NULL;
    GstCaps *caps2 = NULL;
    GstCaps *caps3 = NULL;

    pipeline = gst_pipeline_new("pipeline");
    src = gst_element_factory_make("v4l2src", "src");
    /*src = gst_element_factory_make("videotestsrc", "src");*/
    vrate = gst_element_factory_make("videorate", "vrate");
    conv = gst_element_factory_make("ffmpegcolorspace", "conv");
    overlay = gst_element_factory_make("clockoverlay", "clockoverlay");
    tee = gst_element_factory_make("tee", "tee");
    enc = gst_element_factory_make("jpegenc", "enc");
    queue = gst_element_factory_make("queue", "queue");
    sink = gst_element_factory_make("fakesink", "sink");
    g_assert(pipeline);
    g_assert(src);
    g_assert(vrate);
    g_assert(conv);
    g_assert(overlay);
    g_assert(tee);
    g_assert(enc);
    g_assert(queue);
    g_assert(sink);

    caps1 = gst_caps_new_simple("video/x-raw-yuv",
                                "width",     G_TYPE_INT, GetWidth(size),
                                "height",    G_TYPE_INT, GetHeight(size),
                                "framerate", GST_TYPE_FRACTION, MAX_FPS, 1,
                                NULL);
    g_assert(caps1);
    caps2 = gst_caps_new_simple("video/x-raw-yuv",
                                "framerate", GST_TYPE_FRACTION, fps, 1,
                                NULL);
    g_assert(caps2);
    caps3 = gst_caps_new_simple("video/x-raw-yuv",
                                "format",GST_TYPE_FOURCC,GST_STR_FOURCC("I420"),
                                NULL);
    g_assert(caps3);

    gst_bin_add_many(GST_BIN(pipeline), src, vrate, conv, overlay, tee, enc, 
                     queue, sink, NULL);
    if (gst_element_link_filtered(src, vrate, caps1) == FALSE ||
        gst_element_link_filtered(vrate, conv, caps2) == FALSE ||
        gst_element_link_filtered(conv, tee, caps3) == FALSE ||
        gst_element_link_many(tee, queue, overlay, enc, sink, NULL) == FALSE)
    {
        g_error("Gstreamer linking error");
    }

    g_object_set(G_OBJECT(queue), "leaky", 2, "max-size-buffers", fps, NULL);
    g_object_set(G_OBJECT(overlay), "valignment", 1, "halignment", 2, "xpad", 5,
                 "ypad", 5, NULL);
    /* decrease font size with smaller image sizes */
    if (size != QVGA && size != VGA)
    {
        g_object_set(G_OBJECT(overlay), "font-desc", "15", NULL);
    }
    g_object_set(G_OBJECT(enc), "quality", quality,  NULL);
    g_object_set(G_OBJECT(sink), "signal-handoffs", TRUE, NULL);
    g_signal_connect(G_OBJECT(sink),"handoff",G_CALLBACK(StoreJPEGData), NULL);

    gst_caps_unref(caps1);
    gst_caps_unref(caps2);
    gst_caps_unref(caps3);

    /*g_object_set(G_OBJECT(src), "is-live", TRUE, "pattern", 1, NULL);*/

    /* motion detection branch */
    if (enableMotionDet == TRUE)
    {
        mdbin = CreateMotionDetectionBin(size, sf);
        if (gst_bin_add(GST_BIN(pipeline), mdbin) == FALSE ||
            gst_element_link(tee, mdbin) == FALSE)
        {
            g_error("Gstreamer bin add or linking error");
        }
    }
    else
    {
        g_message("Skipping GStreamer motion detection branch setup");
    }

    bus = gst_pipeline_get_bus(GST_PIPELINE(pipeline));
    g_assert(bus);
    gst_bus_add_watch(bus, BusMessageHandler, NULL);

    /* if motion detection is on set pipeline to playing, otherwise to paused */
    if (gst_element_set_state(pipeline,
                              enableMotionDet == TRUE ?
                              GST_STATE_PLAYING : GST_STATE_PAUSED) ==
        GST_STATE_CHANGE_FAILURE)
    {
        g_error("Error when trying to set pipeline playing");
    }

}

/* Motion JPEG file writer thread */
static gpointer FileThread(gpointer ptr)
{

    guint8 *buffer = NULL;
    guint length;
    gchar header[200];
    gint fd = (gint)ptr;

    g_message("FileThread starting... fd = %d", fd);

    /* allocate local data buffer */
    buffer = g_malloc(GetWidth(cmdLineParams->size) *
                      GetHeight(cmdLineParams->size));
    if (buffer == NULL)
    {
        g_critical("Could not allocate memory");
        return (NULL);
    }

    while (1)
    {
        g_mutex_lock(jpegDataMutex);
        g_cond_wait(jpegDataCV, jpegDataMutex);

        if (GetMotionDetectedFlag() == FALSE)
        {
            g_mutex_unlock(jpegDataMutex);
            continue;
        }
        memcpy(buffer, jpegBuffer, numBytesInJpegBuffer);
        length = numBytesInJpegBuffer;
        g_mutex_unlock(jpegDataMutex);

        g_sprintf(header,
                  "\r\n"
                  "--MyBoundary\r\nContent-Type: image/jpeg\r\n"
                  "Content-Length: %u\r\n\r\n", numBytesInJpegBuffer);

        write(fd, header, strlen(header));
        write(fd, buffer, length);
    }

    g_free(buffer);

    return (NULL);

}

/* Client serving thread */
static gpointer ClientThread(gpointer ptr)
{

    guint8 *buffer = NULL;
    guint length;
    gint fd = (int)ptr;
    const char header1[] = 
        "Pragma: no-cache\r\n"
        "Cache-Control: no-store, no-cache\r\n"
        "Connection: close\r\n"
        "Content-Type: multipart/x-mixed-replace; boundary=--MyBoundary\r\n";
    char header2[200];

    g_message("client thread starting... fd = %d", fd);

    if (fileFd == -1 &&
        (pipeline == NULL || gst_element_set_state(pipeline, GST_STATE_PLAYING) ==
         GST_STATE_CHANGE_FAILURE))
    {
        g_error("Error when trying to set pipeline playing");
    }

    IncrementNumClients();

    /* allocate local data buffer */
    buffer = g_malloc(GetWidth(cmdLineParams->size) *
                      GetHeight(cmdLineParams->size));
    if (buffer == NULL)
    {
        g_critical("Could not allocate memory");
        goto end;
    }

    if (send(fd, header1, strlen(header1), MSG_NOSIGNAL) < 0)
    {
        g_warning("send failed (header1): %s", strerror(errno));
        goto end;
    }

    while (1)
    {
        g_mutex_lock(jpegDataMutex);
        g_cond_wait(jpegDataCV, jpegDataMutex);
        memcpy(buffer, jpegBuffer, numBytesInJpegBuffer);
        length = numBytesInJpegBuffer;
        g_mutex_unlock(jpegDataMutex);

        g_sprintf(header2,
                  "\r\n"
                  "--MyBoundary\r\nContent-Type: image/jpeg\r\n"
                  "Content-Length: %u\r\n\r\n", numBytesInJpegBuffer);

        if (send(fd, header2, strlen(header2), MSG_NOSIGNAL) < 0 ||
            send(fd, buffer, length, MSG_NOSIGNAL) < 0)
        {
            g_warning("send failed (header2 or data): %s", strerror(errno));
            break;
        }
    }

end:
    close(fd);

    g_free(buffer);

    DecrementNumClients();

    /* if motion detection is off and there are no more clients ==> pause the
     * pipeline */
    if (fileFd == -1)
    {
        g_mutex_lock(numClientsMutex);
        if (numClients == 0)
        {
            if (gst_element_set_state(pipeline, GST_STATE_PAUSED) ==
                GST_STATE_CHANGE_FAILURE)
            {
                g_error("Error when trying to set pipeline playing");
            }
        }
        g_mutex_unlock(numClientsMutex);
    }

    g_message("client thread exiting..., fd = %d", fd);

    return (NULL);

}

/*  Main server thread */
static gpointer ServerThread(gpointer ptr)
{

    struct sockaddr_in sa;
    struct sockaddr_in saClient;
    socklen_t addrLen;
    gint clientSocketFd;
    gint optval;

    if ((socketFd = socket(PF_INET, SOCK_STREAM, 0)) < 0)
    {
        g_error("socket failed: %s", strerror(errno));
    }

    /* set SO_REUSEADDR on a socket to true (1) */
    optval = 1;
    if (setsockopt(socketFd,SOL_SOCKET,SO_REUSEADDR,&optval,sizeof optval) < 0)
    {
        g_error("setsockopt failed: %s", strerror(errno));
    }

    memset(&sa, 0, sizeof(sa));
    sa.sin_family = AF_INET;
    sa.sin_port = htons(cmdLineParams->port);
    sa.sin_addr.s_addr = htonl(INADDR_ANY);

    if (bind(socketFd, (struct sockaddr *)&sa, sizeof(sa)) < 0)
    {
        g_error("bind failed: %s", strerror(errno));
    }

    if (listen(socketFd, 5) < 0)
    {
        g_error("listen failed: %s", strerror(errno));
    }
    while (1)
    {
        addrLen = sizeof(struct sockaddr_in);
        clientSocketFd = accept(socketFd,
                                (struct sockaddr *)&saClient,
                                &addrLen);
        if (clientSocketFd < 0)
        {
            g_warning("accept failed: %s", strerror(errno));
            continue;
        }
        else
        {
            if (GetNumClients() < MAX_NUM_CLIENTS)
            {
                g_message("new connection, spawning a new thread...");
                if (g_thread_create(&ClientThread,
                                    (gpointer)clientSocketFd,
                                    FALSE,
                                    NULL) == NULL)
                {
                    g_message("g_thread_create failed");
                    close(clientSocketFd);
                }
            }
            else
            {
                g_message("maximum number of clients reached (%u)",
                          MAX_NUM_CLIENTS);
                close(clientSocketFd);
            }
        }
    }

    close(socketFd);

}

/* Function to initialize DBus */
static gboolean InitDBus(void)
{
    DBusError err;
    gint ret;

    dbus_g_thread_init();

    dbus_error_init(&err);

    connection = dbus_bus_get(DBUS_BUS_SESSION, &err);
    if (dbus_error_is_set(&err))
    { 
        g_critical("Connection Error (%s)", err.message); 
        dbus_error_free(&err); 
        return(FALSE);
    }
    if (NULL == connection)
    { 
        g_critical("Connection Error (%s)", err.message); 
        return(FALSE);
    }
    dbus_connection_setup_with_g_main(connection, g_main_context_default());

    ret = dbus_bus_request_name(connection,
                                DBUS_INTERFACE_NAME, 
                                DBUS_NAME_FLAG_REPLACE_EXISTING,
                                &err);
    if (dbus_error_is_set(&err))
    { 
        g_critical("Name Error (%s)", err.message); 
        dbus_error_free(&err); 
        return(FALSE);
    }
    if (DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER != ret)
    { 
        g_critical("Not primary name owner (%s)", err.message); 
        return(FALSE);
    }

    if (!dbus_connection_register_object_path (connection,
                                               DBUS_OBJECT_PATH,
                                               &vtable,
                                               NULL))

    {
        g_critical("Could not register DBus object path");
        return(FALSE);
    }

    g_message("DBus initialized");

    return(TRUE);

}

/* Function to deinitialize DBus */
static void DeinitDBus(void)
{

    if (connection)
    {
        dbus_connection_unregister_object_path(connection, DBUS_OBJECT_PATH);
        dbus_bus_release_name(connection, DBUS_INTERFACE_NAME, NULL);
        dbus_connection_unref(connection);
        connection = NULL;
    }
    g_message("DBus deinitialized");

}

/* Function to execute when start message is received */
static gboolean Start(const gboolean enableMotionDet)
{

    guint numLumaPixels;

    /* don't start this machine if one is already running */
    if (pipeline)
    {
        return (TRUE);
    }

    jpegDataMutex = g_mutex_new();
    g_assert(jpegDataMutex);
    jpegDataCV = g_cond_new();
    g_assert(jpegDataCV);

    numClientsMutex = g_mutex_new();
    g_assert(numClientsMutex);

    numLumaPixels =GetWidth(cmdLineParams->size)*GetHeight(cmdLineParams->size);
    jpegBuffer = g_malloc(numLumaPixels);
    g_assert(jpegBuffer);

    if (enableMotionDet == TRUE &&
        cmdLineParams->fileName != NULL)
    {
        fileFd = open(cmdLineParams->fileName,
                      O_WRONLY | O_CREAT | O_APPEND,
                      S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
        if (fileFd == -1)
        {
            g_error("Could not open file: %s", cmdLineParams->fileName);
        }
        g_message("Motion detection enabled");
    }
    else if (enableMotionDet == TRUE)
    {
        g_warning("Motion detection was asked for but no file name was given");
    }
    else
    {
        g_message("Motion detection disabled");
    }

    CreatePipeline(cmdLineParams->size,
                   cmdLineParams->fps,
                   cmdLineParams->quality,
                   scalefactor,
                   fileFd != -1 ? TRUE : FALSE);

    g_message("Pipeline created");

    if (g_thread_create(&ServerThread, NULL, FALSE, NULL) == NULL)
    {
        g_error("Could not start server thread");
    }
    g_message("Server thread spawned");

    if (fileFd != -1)
    {
        g_assert(scalefactor % 2 == 0);

        motionDetectedMutex = g_mutex_new();
        g_assert(motionDetectedMutex);

        prevLumaImage = g_malloc(numLumaPixels / (scalefactor * scalefactor));
        g_assert(prevLumaImage);
        tmpLumaImage = g_malloc(numLumaPixels / (scalefactor * scalefactor));
        g_assert(prevLumaImage);

        if (g_thread_create(&FileThread, (gpointer)fileFd, FALSE, NULL) == NULL)
        {
            g_error("Could not start file thread");
        }
        g_message("File thread spawned");
    }


    return (TRUE);

}

/* Function to execute when stop message is received */
static void Stop(void)
{

    if (pipeline)
    {
        if (gst_element_set_state(pipeline, GST_STATE_NULL) ==
            GST_STATE_CHANGE_FAILURE)
            g_warning("Error when stopping and releasing pipeline");
    }

    DeinitDBus();

    if (loop) g_main_loop_quit(loop);

    if (fileFd != -1 ) close(fileFd);

    if (socketFd != -1 ) close(socketFd);

    g_free(jpegBuffer);

    g_free(tmpLumaImage);

    g_free(prevLumaImage);

    fflush(NULL);

}

/* DBus message handler */
static DBusHandlerResult msgHandler(DBusConnection *connection,
                                    DBusMessage *message,
                                    gpointer user_data)
{

    DBusMessage *reply;
 
    if (dbus_message_is_method_call(message, DBUS_INTERFACE_NAME, "start"))
    {
        g_message("Got start msg");
        if (Start(FALSE) == TRUE)
        {
            reply = dbus_message_new_method_return(message);
        }
        else
        {
            reply = dbus_message_new_error(message, DBUS_ERROR_FAILED,
                                           "Could not start mjpg server");
        }
        dbus_connection_send(connection, reply, NULL);
        dbus_message_unref(reply);
    }
    else if (dbus_message_is_method_call(message, DBUS_INTERFACE_NAME,
                                         "start_with_motion_detection"))
    {
        g_message("Got start_with_motion_detection msg");
        if (Start(TRUE) == TRUE)
        {
            reply = dbus_message_new_method_return(message);
        }
        else
        {
            reply = dbus_message_new_error(message, DBUS_ERROR_FAILED,
                                           "Could not start mjpg server");
        }
        dbus_connection_send(connection, reply, NULL);
        dbus_message_unref(reply);
    }
    else if (dbus_message_is_method_call(message, DBUS_INTERFACE_NAME, "stop"))
    {
        g_message("Got stop msg");
        reply = dbus_message_new_method_return(message);
        dbus_connection_send(connection, reply, NULL);
        dbus_message_unref(reply);
        exit(EXIT_SUCCESS);
    }
    else {
        g_warning("unknown dbus msg");
        return (DBUS_HANDLER_RESULT_NOT_YET_HANDLED);
    }

    return (DBUS_HANDLER_RESULT_HANDLED);

}

/* Function to set the default parameter values */
static void FillDefaultValues(CmdLineParams *const params)
{

    g_assert(params);

    /* default values */
    params->fps             = 8;
    params->size            = QVGA;
    params->quality         = 70;
    params->thresholdDouble = 3.0;
    params->fileName        = NULL;
    params->port            = 8081;
    params->radius          = 2;

}

/* Function to parse command line paramaters */
static void ParseCommandLineParams(const int argc,
                                   char *const argv[],
                                   CmdLineParams *const params)
{

    int c;
    int option_index = 0;
    static struct option long_options[] =
    {
        {"size",      required_argument, 0, 's'},
        {"fps",       required_argument, 0, 'f'},
        {"quality",   required_argument, 0, 'q'},
        {"threshold", required_argument, 0, 't'},
        {"out",       required_argument, 0, 'o'},
        {"port",      required_argument, 0, 'p'},
        {"radius",    required_argument, 0, 'r'},
        {0, 0, 0, 0}
    };

    g_assert(argv);
    g_assert(params);

    while ((c = getopt_long(argc,
                            argv,
                            "s:f:q:t:o:p:r:",
                            long_options,
                            &option_index)) != -1)
    {
        switch (c)
        {
            case 's':
                params->size = GetSize(optarg);
                g_message("size set to '%d'", params->size);
                break;
            case 'f':
                params->fps = GetFps(optarg);
                g_message("fps set to '%d'", params->fps);
                break;
            case 'q':
                params->quality = GetQuality(optarg);
                g_message("quality set to '%d'", params->quality);
                break;
            case 't':
                params->thresholdDouble = GetThreshold(optarg);
                g_message("threshold set to '%lf'", params->thresholdDouble);
                break;
            case 'o':
                params->fileName = g_strdup(optarg);
                g_message("fileName set to '%s'", params->fileName);
                break;
            case 'p':
                params->port = atoi(optarg); 
                g_message("port set to '%d'", params->port);
                break;
            case 'r':
                params->radius = atoi(optarg); 
                g_message("radius set to '%d'", params->radius);
                break;
            case '?':
                break;
            default:
                abort();
        }
    }
}

/* main */
int main(int argc, char *argv[])
{

    /* close stdin since we do not need it */
    close(STDIN_FILENO);

    g_message("starting up...");

    g_thread_init(NULL);

    /* this makes GStreamer to find our plugins, for some reason giving
     * --gst-plugin-path=/usr/lib/gstmjpg from the command line did not work */
    setenv("GST_PLUGIN_PATH", "/usr/lib/gstmjpg", 1);

    /* initialize gstreamer */
    gst_init(&argc, &argv);

    /* fill default values */
    FillDefaultValues(cmdLineParams);

    /* then parse cmd line params */
    ParseCommandLineParams(argc, argv, cmdLineParams);

    loop = g_main_loop_new(NULL, FALSE);
    g_assert(loop);

    if (InitDBus() == FALSE)
    {
        g_critical("Could not initialize DBus, exiting...");
        exit(EXIT_FAILURE);
    }

    g_atexit(Stop);

    g_main_loop_run(loop);

    return (0);

}
