/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gpxview.h"
#include <math.h>     // for isnan()

struct icon_data {
  GdkPixbuf **data;
  int count;
} icons[] = {
  { NULL, CACHE_TYPE_MAX+1 },  /* ICON_CACHE_TYPE */
  { NULL, CACHE_CONT_MAX+1 },  /* ICON_CACHE_SIZE */
  { NULL, 9 },                 /* ICON_STARS      */
  { NULL, LOG_TYPE_MAX+1 },    /* ICON_LOG        */
  { NULL, 9 },                 /* ICON_HEADING    */
  { NULL, WPT_SYM_MAX+1 },     /* ICON_WPT        */
  { NULL, 1 },                 /* ICON_TB         */
  { NULL, 13 },                /* ICON_MISC       */
  { NULL, 3 },                 /* ICON_FILE       */
  { NULL, -1 }
};

/* ICON_CACHE_TYPE */
const char *cache_type_icon_name[] = {
  "traditional", "multi",      "mystery", "virtual", "webcam", "event",
  "letterbox",   "earthcache", "wherigo", "megaevent", "cito"
};

/* ICON_CACHE_SIZE */
const char *cache_size_icon_name[] = {
  "regular", "small",      "micro", "other", "not_chosen", "large",
  "virtual" 
};

/* ICON_STARS      */
const char *stars_icon_name[] = {
  "1", "1_5", "2", "2_5", "3", "3_5", "4", "4_5", "5" 
};

/* ICON_LOG        */
const char *log_icon_name[] = {
  "smile", "sad", "maint", "note", "big_smile", "enabled",
  "greenlight", "rsvp", "attended", "camera", "disabled",
  "needsmaint", "coord_update",
  "traffic_cone",
  "traffic_cone", /* LOG_TYPE_NEEDS_ARCHIVED */
  "traffic_cone",
};  

/* ICON_HEADING    */
const char *heading_icon_name[] = {
  "n", "ne", "e", "se", "s", "sw", "w", "nw", "none"
};

/* ICON_WPT        */
const char *wpt_sym_icon_name[] = {
  "multistage", "parking", "final", "question",
  "trailhead", "refpoint",
};

/* ICON_TB         */
const char *tb_icon_name[] = {
  "tb_coin"
};

/* ICON_MISC         */
const char *misc_icon_name[] = {
  "maemo-mapper-out", "override", "locked", "unlocked", "found", 
  "maemo-mapper-in", "note", "delete", "paypal", 
  "user", "zoomin", "zoomout", "info"
};

/* ICON_FILE         */
const char *file_icon_name[] = {
  "gc", "folder", "zip"
};

static void icons_load(int type, char *format, const char *names[]) {
  int i;

  if(!icons[type].count) {
    icons[type].data = NULL;
    return;
  }

  icons[type].data = malloc(sizeof(GdkPixbuf *) * icons[type].count);
  
  for(i=0;i<icons[type].count;i++) {
    if(names[i]) {
      GError *error = NULL;
      char filename[128];
      strcpy(filename, ICONPATH);
      snprintf(filename+strlen(filename),
	      sizeof(filename)-strlen(filename), format, names[i], "png");
      icons[type].data[i] = gdk_pixbuf_new_from_file(filename, &error);

      if(error) {
	/* try gif */
	error = NULL;
	strcpy(filename, ICONPATH);
	snprintf(filename+strlen(filename),
		 sizeof(filename)-strlen(filename), format, names[i], "gif");
	icons[type].data[i] = gdk_pixbuf_new_from_file(filename, &error);

	if(error) {
	  error = NULL;
	  /* try again in local dir */
	  strcpy(filename, "./data/icons/");
	  snprintf(filename+strlen(filename), 
		   sizeof(filename)-strlen(filename), format, names[i], "png");
	  icons[type].data[i] = gdk_pixbuf_new_from_file(filename, &error);
	  
	  if(error) {
	    error = NULL;
	    /* try gif */
	    strcpy(filename, "./data/icons/");
	    snprintf(filename+strlen(filename), 
		   sizeof(filename)-strlen(filename), format, names[i], "gif");
	    icons[type].data[i] = gdk_pixbuf_new_from_file(filename, &error);
	    
	    
	    if(error) {
	      icons[type].data[i] = NULL;
	      g_warning("Could not load cache type icon %s: %s\n", 
			names[i], error->message);
	      g_error_free(error);
	      error = NULL;
	    }
	  }
	}   
      }
    } else
      icons[type].data[i] = NULL;
  }
}

void icons_init(void) {

  /* load cache type icons */
  icons_load(ICON_CACHE_TYPE, "cache_type_%s.%s", cache_type_icon_name);

  /* load cache container/size icons */
  icons_load(ICON_CACHE_SIZE, "cache_size_%s.%s", cache_size_icon_name);

  /* load cache difficulty/terrain icons */
  icons_load(ICON_STARS, "stars%s.%s", stars_icon_name);

  /* load cache difficulty/terrain icons */
  icons_load(ICON_LOG, "log_icon_%s.%s", log_icon_name);

  /* load icons to visualize heading */
  icons_load(ICON_HEADING, "heading_%s.%s", heading_icon_name);

  /* load icons to visualize heading */
  icons_load(ICON_WPT, "wpt_%s.%s", wpt_sym_icon_name);

  /* load travelbug icon */
  icons_load(ICON_TB, "%s.%s", tb_icon_name);

  /* load misc icons */
  icons_load(ICON_MISC, "%s.%s", misc_icon_name);

  /* load file icons */
  icons_load(ICON_FILE, "file_%s.%s", file_icon_name);
}

void icons_free(void) {
  int i;
  struct icon_data *icon = icons;

  while(icon->count >= 0) {
    if(icon->count) {
      for(i=0;i<icon->count;i++)
	if(icon->data[i])
	  gdk_pixbuf_unref(icon->data[i]);

      free(icon->data);
    }

    icon++;
  }
}

GdkPixbuf *icon_get(int type, int num) {
  if(num < 0)                  return NULL;
  if(num >= icons[type].count) return NULL;

  return icons[type].data[num];
}

GtkWidget *icon_get_widget(int type, int num) {
  GdkPixbuf *pbuf = icon_get(type, num);
  if(!pbuf) return NULL;

  return gtk_image_new_from_pixbuf(pbuf);
}

GdkPixbuf *icon_bearing(pos_t from, pos_t to) {
  if(isnan(from.lat) || isnan(from.lon) || 
     isnan(to.lat) || isnan(to.lon))
    return icon_get(ICON_HEADING, 8);

  int idx = (gpx_pos_get_bearing(from, to)+22.5)/45.0;

  /* make sure we stay in icon bounds */
  if(idx < 0) idx += 8;
  if(idx > 7) idx -= 8;
  return icon_get(ICON_HEADING, idx);
}

