/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <string.h>
#include <math.h>

#include "gpxview.h"


#ifdef USE_MAEMO
#include <gpsbt.h>
#include <gpsmgr.h>
#include <errno.h>
#endif

/* maybe user configurable later on ... */
#define GPSD_HOST "127.0.0.1"
#define GPSD_PORT 2947

gps_sat_t *gps_get_sats(appdata_t *appdata) {
  gps_sat_t *retval = NULL;

  g_mutex_lock(appdata->gps_state->mutex);
  if(appdata->gps_state->gpsdata.set & SATELLITE_SET) {
    int i;
    retval = malloc(sizeof(gps_sat_t));
    retval->num = appdata->gps_state->gpsdata.satellites;

    retval->PRN  = malloc(sizeof(int)*retval->num);
    retval->used = malloc(sizeof(int)*retval->num);
    retval->ss   = malloc(sizeof(int)*retval->num);

    if(retval->num) {
      for(i=0;i<retval->num;i++) {
	retval->PRN[i]  = appdata->gps_state->gpsdata.PRN[i];
	retval->ss[i]   = appdata->gps_state->gpsdata.ss[i];
	retval->used[i] = appdata->gps_state->gpsdata.used[i];
      }
    }
  }

  g_mutex_unlock(appdata->gps_state->mutex);

  return retval;
}

pos_t *gps_get_pos(appdata_t *appdata) {
  static pos_t retval;

  retval.lat = NAN;

  g_mutex_lock(appdata->gps_state->mutex);
  if(appdata->gps_state->gpsdata.set & STATUS_SET)
    if(appdata->gps_state->gpsdata.status != STATUS_NO_FIX)
      if(appdata->gps_state->gpsdata.set & LATLON_SET) 
	retval = appdata->gps_state->gpsdata.fix.pos;

  g_mutex_unlock(appdata->gps_state->mutex);

  if(isnan(retval.lat))
    return NULL;

  return &retval;
}

float gps_get_heading(appdata_t *appdata) {
  float retval = NAN;

  g_mutex_lock(appdata->gps_state->mutex);
  if(appdata->gps_state->gpsdata.set & STATUS_SET)
    if(appdata->gps_state->gpsdata.status != STATUS_NO_FIX) 
      if(appdata->gps_state->gpsdata.set & TRACK_SET) 
	retval = appdata->gps_state->gpsdata.fix.track;

  g_mutex_unlock(appdata->gps_state->mutex);
  return retval;
}

float gps_get_epe(appdata_t *appdata) {
  float retval = NAN;

  g_mutex_lock(appdata->gps_state->mutex);
  if(appdata->gps_state->gpsdata.set & STATUS_SET)
    if(appdata->gps_state->gpsdata.status != STATUS_NO_FIX) 
      retval = appdata->gps_state->gpsdata.fix.eph;

  g_mutex_unlock(appdata->gps_state->mutex);
  return retval;
}

static int gps_connect(gps_state_t *gps_state) {
  GnomeVFSResult vfs_result;
#ifdef USE_MAEMO
  char errstr[256] = "";

  /* We need to start gpsd (via gpsbt) first. */
  memset(&gps_state->context, 0, sizeof(gpsbt_t));
  errno = 0;

  if(gpsbt_start(NULL, 0, 0, 0, errstr, sizeof(errstr),
		 0, &gps_state->context) < 0) {
    printf("Error connecting to GPS receiver: (%d) %s (%s)\n",
	   errno, strerror(errno), errstr);
  }
#endif

  /************** from here down pure gnome/gtk/gpsd ********************/

  /* try to connect to gpsd */
  /* Create a socket to interact with GPSD. */

  int retries = 5;
  while(retries && 
	(GNOME_VFS_OK != (vfs_result = gnome_vfs_inet_connection_create(
		&gps_state->iconn, GPSD_HOST, GPSD_PORT, NULL)))) {
    printf("Error creating connection to GPSD, retrying ...\n");

    retries--;
    sleep(1);
  }

  if(!retries) {
    printf("Finally failed ...\n");
    return -1;
  }

  retries = 5;
  while(retries && ((gps_state->socket = 
     gnome_vfs_inet_connection_to_socket(gps_state->iconn)) == NULL)) {
    printf("Error creating connecting GPSD socket, retrying ...\n");

    retries--;
    sleep(1);
  }

  if(!retries) {
    printf("Finally failed ...\n");
    return -1;
  }

  GTimeVal timeout = { 10, 0 };
  if(GNOME_VFS_OK != (vfs_result = gnome_vfs_socket_set_timeout(
	gps_state->socket, &timeout, NULL))) {
    printf("Error setting GPSD timeout\n");
    return -1;
  } 

  printf("GPSD connected ...\n");

  return 0;
}	

void gps_clear_fix(struct gps_fix_t *fixp) {
  fixp->mode = MODE_NOT_SEEN;
  fixp->pos.lat = fixp->pos.lon = NAN;
  fixp->track = NAN;
  fixp->speed = NAN;
  fixp->climb = NAN;
  fixp->altitude = NAN;
  fixp->ept = NAN;
  fixp->eph = NAN;
  fixp->epv = NAN;
  fixp->epd = NAN;
  fixp->eps = NAN;
  fixp->epc = NAN;
}

/* unpack a daemon response into a status structure */
static void gps_unpack(char *buf, struct gps_data_t *gpsdata) {
  char *ns, *sp, *tp;
  int i;

  for(ns = buf; ns; ns = strstr(ns+1, "GPSD")) {
    if(strncmp(ns, "GPSD", 4) == 0) {
      /* the following should execute each time we have a good next sp */
      for (sp = ns + 5; *sp != '\0'; sp = tp+1) {
	tp = sp + strcspn(sp, ",\r\n");
	if (*tp == '\0') tp--;
	else *tp = '\0';
	
	switch (*sp) {
	case 'A':
	  if (sp[2] == '?') {
	    gpsdata->fix.altitude = NAN;
	  } else {
	    (void)sscanf(sp, "A=%lf", &gpsdata->fix.altitude);
	    gpsdata->set |= ALTITUDE_SET;
	  }
	  break;
	  /* B - baudrate isn't supported by gpxview */
	  /* C - cycle isn't supported by gpxview */
	  /* D - utc time isn't supported by gpxview */
	case 'E':
	  gpsdata->epe = gpsdata->fix.eph = gpsdata->fix.epv = NAN;
	  /* epe should always be present if eph or epv is */
	  if (sp[2] != '?') {
	    char epe[20], eph[20], epv[20];
	    (void)sscanf(sp, "E=%s %s %s", epe, eph, epv);
#define DEFAULT(val) (val[0] == '?') ? NAN : g_ascii_strtod(val, NULL)
	    gpsdata->epe = DEFAULT(epe);
	    gpsdata->fix.eph = DEFAULT(eph);
	    gpsdata->fix.epv = DEFAULT(epv);
#undef DEFAULT
	  }
	  break;
	  /* F - device name isn't supported by gpxview */
	  /* I - gps id isn't supported by gpxview */
	  /* K - known devices list isn't supported by gpxview */
	case 'M':
	  if (sp[2] == '?') {
	    gpsdata->fix.mode = MODE_NOT_SEEN;
	  } else {
	    gpsdata->fix.mode = atoi(sp+2);
	    gpsdata->set |= MODE_SET;
	  }
	  break;
	  /* N - driver mode reporting isn't supported by gpxview */
	case 'O':
	  if (sp[2] == '?') {
	    gpsdata->set = 
	      (gpsdata->set & SATELLITE_SET) | // fix for below
	      MODE_SET | STATUS_SET;  // this clears sat info??
	    gpsdata->status = STATUS_NO_FIX;
	    gps_clear_fix(&gpsdata->fix);
	  } else {
	    struct gps_fix_t nf;
	    char tag[MAXTAGLEN+1], alt[20];
	    char eph[20], epv[20], track[20],speed[20], climb[20];
	    char epd[20], eps[20], epc[20], mode[2];
	    char timestr[20], ept[20], lat[20], lon[20];
	    int st = sscanf(sp+2, 
			    "%8s %19s %19s %19s %19s %19s %19s %19s %19s %19s %19s %19s %19s %19s %1s",
			    tag, timestr, ept, lat, lon,
			    alt, eph, epv, track, speed, climb,
			    epd, eps, epc, mode);
	    if (st >= 14) {
#define DEFAULT(val) (val[0] == '?') ? NAN : g_ascii_strtod(val, NULL)
	      nf.pos.lat = DEFAULT(lat);
	      nf.pos.lon = DEFAULT(lon);
	      nf.ept = DEFAULT(ept);
	      nf.altitude = DEFAULT(alt);
	      nf.eph = DEFAULT(eph);
	      nf.epv = DEFAULT(epv);
	      nf.track = DEFAULT(track);
	      nf.speed = DEFAULT(speed);
	      nf.climb = DEFAULT(climb);
	      nf.epd = DEFAULT(epd);
	      nf.eps = DEFAULT(eps);
	      nf.epc = DEFAULT(epc);
#undef DEFAULT
	      if (st >= 15)
		nf.mode = (mode[0] == '?') ? MODE_NOT_SEEN : atoi(mode);
	      else
		nf.mode = (alt[0] == '?') ? MODE_2D : MODE_3D;
	      if (alt[0] != '?')
		gpsdata->set |= ALTITUDE_SET | CLIMB_SET;
	      if (isnan(nf.eph)==0)
		gpsdata->set |= HERR_SET;
	      if (isnan(nf.epv)==0)
		gpsdata->set |= VERR_SET;
	      if (isnan(nf.track)==0)
		gpsdata->set |= TRACK_SET | SPEED_SET;
	      if (isnan(nf.eps)==0)
		gpsdata->set |= SPEEDERR_SET;
	      if (isnan(nf.epc)==0)
		gpsdata->set |= CLIMBERR_SET;
	      gpsdata->fix = nf;
	      gpsdata->set |= TIME_SET|TIMERR_SET|LATLON_SET|MODE_SET;
	      gpsdata->status = STATUS_FIX;
	      gpsdata->set |= STATUS_SET;
	    }
	  }
	  break;
	case 'P':
	  if (sp[2] == '?') {
	    gpsdata->fix.pos.lat = NAN;
	    gpsdata->fix.pos.lon = NAN;
	  } else {
	    char lat[20], lon[20];
	    (void)sscanf(sp, "P=%19s %19s", lat, lon);
	    gpsdata->fix.pos.lat = g_ascii_strtod(lat, NULL);
	    gpsdata->fix.pos.lon = g_ascii_strtod(lon, NULL);
	    gpsdata->set |= LATLON_SET;
	  }
	  break;
	case 'Q':
	  if (sp[2] == '?') {
	    gpsdata->satellites_used = 0;
	    gpsdata->pdop = 0;
	    gpsdata->hdop = 0;
	    gpsdata->vdop = 0;
	  } else {
	    (void)sscanf(sp, "Q=%d %lf %lf %lf %lf %lf",
			 &gpsdata->satellites_used,
			 &gpsdata->pdop,
			 &gpsdata->hdop,
			 &gpsdata->vdop,
			 &gpsdata->tdop,
			 &gpsdata->gdop);
	    gpsdata->set |= HDOP_SET | VDOP_SET | PDOP_SET;
	  }
	  break;
	case 'S':
	  if (sp[2] == '?') {
	    gpsdata->status = -1;
	  } else {
	    gpsdata->status = atoi(sp+2);
	    gpsdata->set |= STATUS_SET;
	  }
	  break;
	case 'T':
	  if (sp[2] == '?') {
	    gpsdata->fix.track = NAN;
	  } else {
	    (void)sscanf(sp, "T=%lf", &gpsdata->fix.track);
	    gpsdata->set |= TRACK_SET;
	  }
	  break;
	case 'U':
	  if (sp[2] == '?') {
	    gpsdata->fix.climb = NAN;
	  } else {
	    (void)sscanf(sp, "U=%lf", &gpsdata->fix.climb);
	    gpsdata->set |= CLIMB_SET;
	  }
	  break;
	case 'V':
	  if (sp[2] == '?') {
	    gpsdata->fix.speed = NAN;
	  } else {
	    (void)sscanf(sp, "V=%lf", &gpsdata->fix.speed);
	    /* V reply is in kt, fix.speed is in metres/sec */
	    gpsdata->fix.speed = gpsdata->fix.speed / MPS_TO_KNOTS;
	    gpsdata->set |= SPEED_SET;
	  }
	  break;
	case 'X':
	  if (sp[2] == '?') 
	    gpsdata->online = -1;
	  else {
	    (void)sscanf(sp, "X=%lf", &gpsdata->online);
	    gpsdata->set |= ONLINE_SET;
	  }
	  break;
	case 'Y':
	  if (sp[2] == '?') {
	    gpsdata->satellites = 0;
	  } else {
	    int j, i1, i2, i3, i4, i5;
	    int PRN[MAXCHANNELS];
	    int elevation[MAXCHANNELS], azimuth[MAXCHANNELS];
	    int ss[MAXCHANNELS], used[MAXCHANNELS];
	    char tag[MAXTAGLEN+1], timestamp[21];
	    
	    (void)sscanf(sp, "Y=%8s %20s %d ", 
			 tag, timestamp, &gpsdata->satellites);
	    for (j = 0; j < gpsdata->satellites; j++) {
	      PRN[j]=elevation[j]=azimuth[j]=ss[j]=used[j]=0;
	    }
	    //	    printf("sats = %d\n", gpsdata->satellites);
	    for (j = 0, gpsdata->satellites_used = 0; j < gpsdata->satellites; j++) {
	      if ((sp != NULL) && ((sp = strchr(sp, ':')) != NULL)) {
		sp++;
		(void)sscanf(sp, "%d %d %d %d %d", &i1, &i2, &i3, &i4, &i5);
		PRN[j] = i1;
		elevation[j] = i2; azimuth[j] = i3;
		ss[j] = i4; used[j] = i5;
		if (i5 == 1)
		  gpsdata->satellites_used++;
	      }
	    }
	    memcpy(gpsdata->PRN, PRN, sizeof(PRN));
	    memcpy(gpsdata->elevation, elevation, sizeof(elevation));
	    memcpy(gpsdata->azimuth, azimuth,sizeof(azimuth));
			memcpy(gpsdata->ss, ss, sizeof(ss));
			memcpy(gpsdata->used, used, sizeof(used));
			/*@ +compdef @*/
	  }
	  gpsdata->set |= SATELLITE_SET;
	  break;
	  /* Z and $ - profiling isn't supported by gpxview */
	}
      }
    }
  }
}

gpointer gps_thread(gpointer data) {
  GnomeVFSFileSize bytes_read;
  GnomeVFSResult vfs_result;
  char str[512];
  appdata_t *appdata = (appdata_t*)data;
  int cnt=1000;

  const char *msg_pos = "o\r\n";   /* pos request */
  const char *msg_sat = "y\r\n";   /* sat request */

  appdata->gps_state->gpsdata.set = 0;

  gboolean connected = FALSE;

  while(1) {
    if(appdata->use_gps) {
      if(!connected) {
	printf("trying to connect\n");

	if(gps_connect(appdata->gps_state) < 0)
	  sleep(10);
	else 
	  connected = TRUE;
      } else {
	const char *msg;
	if(!cnt) msg = msg_sat;
	else     msg = msg_pos;
      
	if(GNOME_VFS_OK == 
	   (vfs_result = gnome_vfs_socket_write(appdata->gps_state->socket,
		      msg, strlen(msg)+1, &bytes_read, NULL))) {

	  /* update every second, wait here to make sure a complete */
	  /* reply is received */
	  if(cnt <= 1) usleep(500000);
	  else 	       sleep(1);
	  
	  if(bytes_read == (strlen(msg)+1)) {
	    vfs_result = gnome_vfs_socket_read(appdata->gps_state->socket,
			       str, sizeof(str)-1, &bytes_read, NULL);
	    if(vfs_result == GNOME_VFS_OK) {
	      str[bytes_read] = 0; 
	    
	      //	  printf("msg: %s (%d)\n", str, strlen(str));
	      
	      g_mutex_lock(appdata->gps_state->mutex);
	      
	      if(!cnt) appdata->gps_state->gpsdata.set &= ~SATELLITE_SET;
	      else     appdata->gps_state->gpsdata.set &= 
			 ~(TIME_SET|TIMERR_SET|LATLON_SET|MODE_SET|STATUS_SET);
	    
	      gps_unpack(str, &appdata->gps_state->gpsdata);
	      g_mutex_unlock(appdata->gps_state->mutex);
	    }
	  }
	}
	if(cnt++ >= 5) cnt = 0;
      }
    } else {
      if(connected) {
	printf("stopping GPS connection due to user request\n");
	gnome_vfs_inet_connection_destroy(appdata->gps_state->iconn, NULL);

#ifdef USE_MAEMO
	gpsbt_stop(&appdata->gps_state->context);
#endif
	connected = FALSE;
      } else
	sleep(1);
    }
  }

  printf("GPS thread ended???\n");
  return NULL;
}

void gps_init(appdata_t *appdata) {
  appdata->gps_state = malloc(sizeof(gps_state_t));
  memset(appdata->gps_state, 0, sizeof(gps_state_t));

  /* start a new thread to listen to gpsd */
  appdata->gps_state->mutex = g_mutex_new();
  appdata->gps_state->thread_p = 
    g_thread_create(gps_thread, appdata, FALSE, NULL);
}

void gps_release(appdata_t *appdata) { 
#ifdef USE_MAEMO
  gpsbt_stop(&appdata->gps_state->context);
#endif
  free(appdata->gps_state);
}
