/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <string.h>
#include <math.h>

#include "gpxview.h"

#ifdef USE_MAEMO
#include "dbus.h"
#endif

#define COMPASS_SIZE    230
#define LETTER_SPACE    0.2
#define DIAMOND_WIDTH   0.05
#define DIAMOND_HEIGHT  0.2
#define ARROW_WIDTH     0.3
#define ARROW_LENGTH    0.7
#define UPDATE_MS       1000   /* gps updates are sent once a second */

#define SAT_WIDTH 330
#define SAT_HEIGHT 60

/* http://www.gtk.org/tutorial1.2/gtk_tut-30.html */
/* http://developer.gimp.org/api/2.0/gdk/gdk-Drawing-Primitives.html */


static float rad2deg(float rad) {
  return fmodf(360.0 + (180.0/M_PI) * rad, 360.0);
}

static void compass_draw(GtkWidget *widget, cache_context_t *context) {
  float f;
  int i;

  gint width = widget->allocation.width;
  gint height = widget->allocation.height;
  gint diameter = (height < width)?height:width;

  gint xcenter = width/2;
  gint ycenter = height/2;
  gint radius = diameter/2;

  /* erase background */
  gdk_draw_rectangle(context->gotoc.compass_pixmap, 
		     widget->style->bg_gc[GTK_STATE_NORMAL], TRUE,
		     0, 0, width, height);

  /* use white rosetta background unless the real background */
  /* is very bright. Use bright grey then */
  
  if(widget->style->bg[GTK_STATE_NORMAL].red + 
     widget->style->bg[GTK_STATE_NORMAL].green +
     widget->style->bg[GTK_STATE_NORMAL].blue < 3*60000) {
    /* background is very bright, use white */
    gdk_draw_arc(context->gotoc.compass_pixmap, widget->style->white_gc, TRUE,
		 (width-diameter)/2, (height-diameter)/2, diameter, diameter,
		 0, 360*64);
  } else {
    GdkGC *lgrey_gc = gdk_gc_new(context->gotoc.compass_pixmap);
    gdk_gc_copy(lgrey_gc, widget->style->black_gc);
    GdkColor lgrey_color;
    gdk_color_parse("#DDDDDD", &lgrey_color);
    gdk_gc_set_rgb_fg_color(lgrey_gc,  &lgrey_color);

    gdk_draw_arc(context->gotoc.compass_pixmap, lgrey_gc, TRUE,
		 (width-diameter)/2, (height-diameter)/2, diameter, diameter,
		 0, 360*64);
  }

  /* draw the locked/unlocked icon */
  gdk_draw_pixbuf(context->gotoc.compass_pixmap, 
		  widget->style->fg_gc[GTK_STATE_NORMAL],
		  icon_get(ICON_MISC, context->appdata->compass_locked?2:3), 
		  0, 0, (width-diameter)/2 + diameter/32, 
		  (height+diameter)/2 - 16 - diameter/32 , 16, 16,
		  GDK_RGB_DITHER_NONE,0,0);
		  
  /* don't update heading if the compass is locked */
  if(!context->appdata->compass_locked) {
    int i, valid = 0, cnt=0;
    double x_sum = 0.0, y_sum = 0.0;

    /* shift heading buffer up one entry and add new value at entry 0 */
    for(i=MAX_AVERAGE-1;i>0;i--) 
      context->gotoc.head_avg[i] = context->gotoc.head_avg[i-1];
    context->gotoc.head_avg[0] = gps_get_heading(context->appdata)*M_PI/180.0;

    //    printf("Damping = %d\n", context->appdata->compass_damping);
    //    printf("add heading %f\n", rad2deg(context->gotoc.head_avg[0]));

    /* determine number of valid entries */
    for(i=0;i<MAX_AVERAGE && valid<context->appdata->compass_damping;i++) 
      if(!isnan(context->gotoc.head_avg[i])) 
	valid++;

    /* map back to angle if at least one value has been added */
    if(valid) {
      /* map all valid antries onto a circle */
      for(i=0;i<MAX_AVERAGE && cnt<context->appdata->compass_damping;i++) {
	if(!isnan(context->gotoc.head_avg[i])) {
	  float weight = 1.0 - ((float)(cnt++)/(float)valid);
	  printf("weight = %f * %f\n", 
		 weight, rad2deg(context->gotoc.head_avg[i]));
	  x_sum += weight * sin(context->gotoc.head_avg[i]);
	  y_sum += weight * cos(context->gotoc.head_avg[i]);
	}
      }

      printf("%d valid heading entries\n", valid);
      context->gotoc.heading = atan2(x_sum, y_sum);
      printf("averaged heading: %f\n", 
	     rad2deg(context->gotoc.heading));
    } else { 
      //      printf("no valid heading, keeping old value heading\n");
    }
  }

  if(!isnan(context->gotoc.heading)) {

    for(i=0,f=0;f<2*M_PI-M_PI/8;f+=M_PI/4,i++) {
      float ang = f - context->gotoc.heading;

      if(!(i&1)) {
	/* draw diamonds */
	GdkPoint diamond[4];
	
#define OUT  (1.0-LETTER_SPACE)

	diamond[0].x = xcenter + radius * OUT *  sin(ang);
	diamond[0].y = ycenter + radius * OUT * -cos(ang);
	
	diamond[1].x = xcenter + radius * (OUT-DIAMOND_HEIGHT/2) * 
	  sin(ang+DIAMOND_WIDTH);
	diamond[1].y = ycenter + radius * (OUT-DIAMOND_HEIGHT/2)*
	  -cos(ang+DIAMOND_WIDTH);
	
	diamond[2].x = xcenter + radius * (OUT-DIAMOND_HEIGHT) *  sin(ang);
	diamond[2].y = ycenter + radius * (OUT-DIAMOND_HEIGHT) * -cos(ang);
	
	diamond[3].x = xcenter + radius * (OUT-DIAMOND_HEIGHT/2) * 
	  sin(ang-DIAMOND_WIDTH);
	diamond[3].y = ycenter + radius * (OUT-DIAMOND_HEIGHT/2) *
	  -cos(ang-DIAMOND_WIDTH);
	
	gdk_draw_polygon(context->gotoc.compass_pixmap, 
			 widget->style->black_gc, TRUE,
			 diamond, sizeof(diamond)/sizeof(GdkPoint));
	
	const char *str[] = { "N", "E", "S", "W" };
	PangoLayout *layout = gtk_widget_create_pango_layout(widget, _(str[i/2]));
	int tw, th;
	pango_layout_get_pixel_size(layout, &tw, &th);  
	
	gdk_draw_layout(context->gotoc.compass_pixmap, widget->style->black_gc,
		xcenter + radius * (1.0-LETTER_SPACE/2)*sin(ang) - tw/2, 
		ycenter + radius * (1.0-LETTER_SPACE/2)*-cos(ang) - th/2,
		layout);
	
	g_object_unref(layout);
      } else
	gdk_draw_line(context->gotoc.compass_pixmap, widget->style->black_gc,
		      xcenter + radius * 0.9 *  sin(ang), 
		      ycenter + radius * 0.9 * -cos(ang),
		      xcenter + radius * 1.0 *  sin(ang), 
		      ycenter + radius * 1.0 * -cos(ang));
    }
    
    /* draw arrow */
    
    /* setup required colors */
    GdkGC *arrow_gc = gdk_gc_new(context->gotoc.compass_pixmap);
    gdk_gc_copy(arrow_gc, widget->style->black_gc);
    GdkColor arrow_color;
    gdk_color_parse("#800000", &arrow_color);
    gdk_gc_set_rgb_fg_color(arrow_gc,  &arrow_color);
    
    GdkPoint arrow[4];

    pos_t *pos = gps_get_pos(context->appdata);
    if(pos) {
      context->appdata->gps = *pos;   /* save position */

      float arot =
	gpx_pos_get_bearing(*pos, context->gotoc.pos) * 
	M_PI/180 - context->gotoc.heading;

      arrow[0].x = xcenter + radius *  ARROW_LENGTH *  sin(arot);
      arrow[0].y = ycenter + radius *  ARROW_LENGTH * -cos(arot);
      
      arrow[1].x = xcenter + radius * -ARROW_LENGTH *  sin(arot+ARROW_WIDTH);
      arrow[1].y = ycenter + radius * -ARROW_LENGTH * -cos(arot+ARROW_WIDTH);
      
      arrow[2].x = xcenter + radius * -0.5 * ARROW_LENGTH *  sin(arot);
      arrow[2].y = ycenter + radius * -0.5 * ARROW_LENGTH * -cos(arot);
      
      arrow[3].x = xcenter + radius * -ARROW_LENGTH *  sin(arot-ARROW_WIDTH);
      arrow[3].y = ycenter + radius * -ARROW_LENGTH * -cos(arot-ARROW_WIDTH);
      
      gdk_draw_polygon(context->gotoc.compass_pixmap, arrow_gc, TRUE,
		       arrow, sizeof(arrow)/sizeof(GdkPoint));    
    } else {
      PangoLayout *layout;

      if(context->appdata->use_gps)
	layout = gtk_widget_create_pango_layout(widget, _("No fix"));
      else
	layout = gtk_widget_create_pango_layout(widget, _("GPS disabled"));

      int tw, th;
      pango_layout_get_pixel_size(layout, &tw, &th);  
      
      gdk_draw_layout(context->gotoc.compass_pixmap, widget->style->black_gc,
		      xcenter - tw/2, ycenter - th/2, layout);
	
      g_object_unref(layout);
    }
  }
}

/* Create a new backing pixmap of the appropriate size */
static gint compass_configure_event(GtkWidget *widget, 
				    GdkEventConfigure *event, gpointer data) {
  cache_context_t *context = (cache_context_t*)data;

  if(context->gotoc.compass_pixmap)
    gdk_pixmap_unref(context->gotoc.compass_pixmap);

  context->gotoc.compass_pixmap = gdk_pixmap_new(widget->window,
				     widget->allocation.width,
				     widget->allocation.height,
				     -1);
  compass_draw(widget, context);
  //  goto_update(context);

  return TRUE;
}

/* Redraw the screen from the backing pixmap */
static gint compass_expose_event(GtkWidget *widget, GdkEventExpose *event, 
			 gpointer data) {
  cache_context_t *context = (cache_context_t*)data;

  gdk_draw_pixmap(widget->window,
                  widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                  context->gotoc.compass_pixmap,
                  event->area.x, event->area.y,
                  event->area.x, event->area.y,
                  event->area.width, event->area.height);

  return FALSE;
}

static void sat_draw(GtkWidget *widget, cache_context_t *context) {
  gint width = widget->allocation.width;
  gint height = widget->allocation.height;

  /* erase background */
  gdk_draw_rectangle(context->gotoc.sat_pixmap, 
		     widget->style->bg_gc[GTK_STATE_NORMAL], TRUE,
		     0, 0, width, height);

  gps_sat_t *sat = gps_get_sats(context->appdata);
  if(sat && sat->num) {
    /* setup required colors */
    GdkGC *used_gc = gdk_gc_new(context->gotoc.sat_pixmap);
    gdk_gc_copy(used_gc, widget->style->black_gc);
    GdkColor used_color;
    gdk_color_parse("#008000", &used_color);   // green
    gdk_gc_set_rgb_fg_color(used_gc,  &used_color);

#define SAT_SPACING 3
    int i, x;
    int swid = (width-SAT_SPACING*(sat->num-1))/sat->num;

#if 0
    int max_ss = 1;
    for(i=0;i<sat->num;i++)
      if(sat->ss[i] > max_ss) max_ss = sat->ss[i];

    printf("max_ss = %d dB\n", max_ss);
#else
    /* as of xgps, a ss of 40 and more is "plenty" */
    int max_ss = 40;
    for(i=0;i<sat->num;i++)
      if(sat->ss[i] > max_ss) sat->ss[i] = max_ss;
#endif    

    if(swid > 40) {
      swid = 40;
      x = (width-sat->num*swid)/2;
    } else
      x = 0;

    for(i=0;i<sat->num;i++) {
      char str[32];
#ifdef USE_MAEMO
      snprintf(str, sizeof(str), "<span size=\"small\">%d</span>", sat->PRN[i]);
      PangoLayout *layout = gtk_widget_create_pango_layout(widget, NULL);
      pango_layout_set_markup(layout, str, strlen(str));
#else    
      snprintf(str, sizeof(str), "%d", sat->PRN[i]);
      PangoLayout *layout = gtk_widget_create_pango_layout(widget, str);
#endif

      int tw, th;
      pango_layout_get_pixel_size(layout, &tw, &th);  
      gdk_draw_layout(context->gotoc.sat_pixmap, widget->style->black_gc,
		      x + swid/2 - tw/2, height - th, layout);
	
      g_object_unref(layout);

      int h = (height-th) * sat->ss[i] / max_ss; 

      gdk_draw_rectangle(context->gotoc.sat_pixmap, 
			 sat->used[i]?used_gc:widget->style->black_gc, 
			 TRUE,
			 x, height-h-th, swid, h);

      x += SAT_SPACING+swid;
    }

    /* free sat infos */
    free(sat->PRN);
    free(sat->ss);
    free(sat->used);
  } else {
    PangoLayout *layout = gtk_widget_create_pango_layout(widget, _("No SAT info"));
    int tw, th;
    pango_layout_get_pixel_size(layout, &tw, &th);  
    gdk_draw_layout(context->gotoc.sat_pixmap, widget->style->black_gc,
		    (width - tw)/2, (height - th)/2, layout);
    
    g_object_unref(layout);
  }

  /* free the sat structure */
  if(sat)
    free(sat);
}

/* Create a new backing pixmap of the appropriate size */
static gint sat_configure_event(GtkWidget *widget, GdkEventConfigure *event,
			    gpointer data) {
  cache_context_t *context = (cache_context_t*)data;

  if(context->gotoc.sat_pixmap)
    gdk_pixmap_unref(context->gotoc.sat_pixmap);

  context->gotoc.sat_pixmap = gdk_pixmap_new(widget->window,
					 widget->allocation.width,
					 widget->allocation.height,
					 -1);
  sat_draw(widget, context);

  return TRUE;
}

/* Redraw the screen from the backing pixmap */
static gint sat_expose_event(GtkWidget *widget, GdkEventExpose *event, 
			 gpointer data) {
  cache_context_t *context = (cache_context_t*)data;

  gdk_draw_pixmap(widget->window,
                  widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                  context->gotoc.sat_pixmap,
                  event->area.x, event->area.y,
                  event->area.x, event->area.y,
                  event->area.width, event->area.height);

  return FALSE;
}

gint goto_destroy_event(GtkWidget *widget, gpointer data ) {
  cache_context_t *context = (cache_context_t*)data;

  printf("destroying goto view\n");

  /* stop timer */
  if(context->gotoc.handler_id)
    gtk_timeout_remove(context->gotoc.handler_id);

  return FALSE;
}

static gboolean goto_update(gpointer data) {
  cache_context_t *context = (cache_context_t*)data;

  if(context->gotoc.sat_pixmap) {
    static int sub = 0;

    if(!sub) {
      /* draw sat view */
      sat_draw(context->gotoc.sat_area, context);
      gtk_widget_queue_draw_area(context->gotoc.sat_area, 0,0,
				 context->gotoc.sat_area->allocation.width, 
				 context->gotoc.sat_area->allocation.height);
    }

    if(sub++ == 5) sub = 0;
  }

  if(context->gotoc.compass_pixmap) {
    /* draw compass */
    compass_draw(context->gotoc.compass_area, context);
    gtk_widget_queue_draw_area(context->gotoc.compass_area, 0,0,
			       context->gotoc.compass_area->allocation.width, 
			       context->gotoc.compass_area->allocation.height);
  }

  pos_t *pos = gps_get_pos(context->appdata);
  if(pos) {
    char str[16];
    gpx_pos_get_distance_str(str, sizeof(str), 
			     *pos, context->gotoc.pos, 
			     context->appdata->imperial);
    gtk_label_set_text(GTK_LABEL(context->gotoc.distance_label), str);

    snprintf(str, sizeof(str), _("%.1f°"), 
	     gpx_pos_get_bearing(*pos,  context->gotoc.pos));
    gtk_label_set_text(GTK_LABEL(context->gotoc.bearing_label), str);
  } else {
    gtk_label_set_text(GTK_LABEL(context->gotoc.distance_label), "-----");
    gtk_label_set_text(GTK_LABEL(context->gotoc.bearing_label), "-----");
  }

  float epe = gps_get_epe(context->appdata);
  if(isnan(epe))
    gtk_label_set_text(GTK_LABEL(context->gotoc.epe_label), "-----");
  else {
    char str[16];
    if(context->appdata->imperial) {
      epe *= 3.2808;
      snprintf(str, sizeof(str), "%.1f ft", epe);
    } else
      snprintf(str, sizeof(str), "%.1f m", epe);

    gtk_label_set_text(GTK_LABEL(context->gotoc.epe_label), str);
  }

  return TRUE;   // fire again
}

static gint waypoint_changed_event(GtkWidget *widget, gpointer data ) {
  cache_context_t *context = (cache_context_t*)data;
  int wpt_idx = gtk_combo_box_get_active(GTK_COMBO_BOX(widget));

  /* get position of selected waypoint */
  if(wpt_idx == 0) 
    context->gotoc.pos = context->appdata->geomath;
  else if(wpt_idx == 1)
    //    context->gotoc.pos = gpx_cache_pos(context->cache);
    context->gotoc.pos = notes_get_pos(context);
  else {
    wpt_t *wpt = context->cache->wpt;
    while(wpt_idx > 2) {
      g_assert(wpt != NULL);
      wpt = wpt->next;
      wpt_idx--;
    }
    context->gotoc.pos = wpt->pos;
  }

  char str[32];
  pos_lat_str(str, sizeof(str), context->gotoc.pos.lat);
  gtk_label_set_text(GTK_LABEL(context->gotoc.lat_lbl), str);
  pos_lon_str(str, sizeof(str), context->gotoc.pos.lon);
  gtk_label_set_text(GTK_LABEL(context->gotoc.lon_lbl), str);

  goto_update(context);

  return FALSE;
}

static void manual_pos_update(cache_context_t *context) {
  context->gotoc.pos.lat = context->appdata->manual_goto.lat;
  context->gotoc.pos.lon = context->appdata->manual_goto.lon;

  char str[32];
  pos_lat_str(str, sizeof(str), context->gotoc.pos.lat);
  gtk_label_set_text(GTK_LABEL(context->gotoc.lat_lbl), str);
  pos_lon_str(str, sizeof(str), context->gotoc.pos.lon);
  gtk_label_set_text(GTK_LABEL(context->gotoc.lon_lbl), str);

  goto_update(context);
}

static void on_posedit_clicked(GtkWidget *button, gpointer data) {
  cache_context_t *context = (cache_context_t*)data;

  GtkWidget *dialog = gtk_dialog_new_with_buttons(_("Edit Position"),
	  GTK_WINDOW(context->appdata->window), GTK_DIALOG_MODAL,
          GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
	  GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);

  GtkWidget *lat, *lon, *label;
  GtkWidget *table = gtk_table_new(2, 2, FALSE);

  gtk_table_attach_defaults(GTK_TABLE(table),
		   label = gtk_label_new(_("Latitude:")), 0, 1, 0, 1);
  gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
  gtk_table_attach_defaults(GTK_TABLE(table), 
		    lat = lat_entry_new(context->appdata->manual_goto.lat),
		    1, 2, 0, 1);

  gtk_table_attach_defaults(GTK_TABLE(table),
		    label = gtk_label_new(_("Longitude:")), 0, 1, 1, 2);
  gtk_misc_set_alignment(GTK_MISC(label), 1.f, 0.5f);
  gtk_table_attach_defaults(GTK_TABLE(table),
		   lon = lon_entry_new(context->appdata->manual_goto.lon), 
		   1, 2, 1, 2);

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), table);

  gtk_widget_show_all(dialog);

  if(GTK_RESPONSE_ACCEPT == gtk_dialog_run(GTK_DIALOG(dialog))) {
    if(isnan(lat_get(lat)) || isnan(lon_get(lon))) 
      errorf(_("Ignoring invalid position"));
    else {
      context->appdata->manual_goto.lat = lat_get(lat);
      context->appdata->manual_goto.lon = lon_get(lon);
      
      manual_pos_update(context);
    }
  }
  gtk_widget_destroy(dialog);
}

static gint radio_changed_event(GtkWidget *widget, gpointer data ) {
  cache_context_t *context = (cache_context_t*)data;

  if(!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))) {
    manual_pos_update(context);
    gtk_widget_set_sensitive(context->gotoc.edit_but, TRUE);
  } else {
    waypoint_changed_event(context->gotoc.cbox, data);
    gtk_widget_set_sensitive(context->gotoc.edit_but, FALSE);
  }

  return FALSE;
}

#ifdef USE_MAEMO
static void on_mm_button_clicked(GtkButton *button, gpointer data) {
  cache_context_t *context = (cache_context_t*)data;

  dbus_mm_set_position(context->appdata, &context->gotoc.pos);
}
#endif

static gboolean compass_clicked_event(GtkWidget *widget, GdkEventButton *event,
			    gpointer user_data) {
  cache_context_t *context = (cache_context_t*)user_data;

  context->appdata->compass_locked = !context->appdata->compass_locked;

  printf("compass is now %slocked\n", 
	 context->appdata->compass_locked?"":"un");

  if(context->gotoc.compass_pixmap) {
    /* draw compass */
    compass_draw(context->gotoc.compass_area, context);
    gtk_widget_queue_draw_area(context->gotoc.compass_area, 0,0,
			       context->gotoc.compass_area->allocation.width, 
			       context->gotoc.compass_area->allocation.height);
  }
  return FALSE;
}

GtkWidget *goto_cache(cache_context_t *context) {
  int i;

  /* clear list used for averaging */
  for(i=0;i<MAX_AVERAGE;i++)
    context->gotoc.head_avg[i] = NAN;

  context->gotoc.pos = gpx_cache_pos(context->cache);

  GtkWidget *hbox = gtk_hbox_new(FALSE, 0);

  context->gotoc.compass_area = gtk_drawing_area_new();
  gtk_drawing_area_size(GTK_DRAWING_AREA(context->gotoc.compass_area), 
			COMPASS_SIZE, COMPASS_SIZE);

  gtk_signal_connect(GTK_OBJECT(context->gotoc.compass_area), "expose_event",
		     (GtkSignalFunc)compass_expose_event, context);
  gtk_signal_connect(GTK_OBJECT(context->gotoc.compass_area),"configure_event",
		     (GtkSignalFunc)compass_configure_event, context);
  gtk_signal_connect(GTK_OBJECT(context->gotoc.compass_area),
		     "button_press_event",
		     (GtkSignalFunc)compass_clicked_event, context);

  gtk_widget_set_events(context->gotoc.compass_area, GDK_EXPOSURE_MASK);
  gtk_widget_add_events(context->gotoc.compass_area, GDK_BUTTON_PRESS_MASK);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), context->gotoc.compass_area);

  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);
  GtkWidget *table = gtk_table_new(8, 2, FALSE);

  /* ---------- combo box containing waypoint names ------- */
  GtkWidget *radio =  gtk_radio_button_new_with_label(NULL, _("Waypoint:"));
  gtk_table_attach_defaults(GTK_TABLE(table), radio, 0,1,0,1);

  context->gotoc.cbox = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX(context->gotoc.cbox), _("Geomath"));
  gtk_combo_box_append_text(GTK_COMBO_BOX(context->gotoc.cbox), 
			    context->cache->id);

  wpt_t *wpt = context->cache->wpt;
  while(wpt) {
    gtk_combo_box_append_text(GTK_COMBO_BOX(context->gotoc.cbox), wpt->id);
    wpt = wpt->next;
  }
  gtk_combo_box_set_active(GTK_COMBO_BOX(context->gotoc.cbox), 1);
  gtk_signal_connect(GTK_OBJECT(context->gotoc.cbox), "changed",
		     (GtkSignalFunc)waypoint_changed_event, context);
  gtk_table_attach_defaults(GTK_TABLE(table), context->gotoc.cbox, 1,2,0,1);

  /* -------------- manual entry field ------------------------- */
  char str[32];
  gtk_table_attach_defaults(GTK_TABLE(table), 
	  gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(radio), 
						      _("Manual:")), 0,1,1,2);
  gtk_signal_connect(GTK_OBJECT(radio), "clicked",
		     (GtkSignalFunc)radio_changed_event, context);

  context->gotoc.edit_but = gtk_button_new_with_label(_("Edit..."));
  gtk_table_attach_defaults(GTK_TABLE(table), context->gotoc.edit_but, 1,2,1,2);
  gtk_signal_connect(GTK_OBJECT(context->gotoc.edit_but), "clicked",
		     (GtkSignalFunc)on_posedit_clicked, context);
  gtk_widget_set_sensitive(context->gotoc.edit_but, FALSE);

#if 0
  context->gotoc.man_lat = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), context->gotoc.man_lat, 1,2,1,2);
  pos_lat_str(str, sizeof(str), context->appdata->manual_goto.lat);
  gtk_entry_set_text(GTK_ENTRY(context->gotoc.man_lat), str);
  textbox_disable(context->gotoc.man_lat);
  context->gotoc.man_lon = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), context->gotoc.man_lon, 1,2,2,3);
  pos_lon_str(str, sizeof(str), context->appdata->manual_goto.lon);
  gtk_entry_set_text(GTK_ENTRY(context->gotoc.man_lon), str);  
  textbox_disable(context->gotoc.man_lon);
  
  gtk_signal_connect(GTK_OBJECT(context->gotoc.man_lat), "changed",
		     (GtkSignalFunc)entry_changed_event, context);
  gtk_signal_connect(GTK_OBJECT(context->gotoc.man_lon), "changed",
		     (GtkSignalFunc)entry_changed_event, context);
#endif

  gtk_table_set_row_spacing(GTK_TABLE(table), 2, 16);

  /* -------------- waypoint coordinates ------------------------- */
  /* SIZE_SMALL doesn't work here as setting the label returns to normal */
  context->gotoc.lat_lbl = pos_lat(context->gotoc.pos.lat, SIZE_NORMAL, 
			       STRIKETHROUGH_NONE);
  gtk_table_attach_defaults(GTK_TABLE(table), context->gotoc.lat_lbl,0,1,3,4);
  context->gotoc.lon_lbl = pos_lon(context->gotoc.pos.lon, SIZE_NORMAL, 
			       STRIKETHROUGH_NONE);
  gtk_table_attach_defaults(GTK_TABLE(table), context->gotoc.lon_lbl,1,2,3,4);

  /* -------------- distance label ------------------------- */
  gtk_table_attach_defaults(GTK_TABLE(table), 
	    gtk_label_new(_("Distance:")), 0,1,4,5);
  gtk_table_attach_defaults(GTK_TABLE(table),
	    (context->gotoc.distance_label = gtk_label_new("-----")), 1,2,4,5);

  /* -------------- bearing label ------------------------- */
  gtk_table_attach_defaults(GTK_TABLE(table), 
	    gtk_label_new(_("Bearing:")), 0,1,5,6);
  gtk_table_attach_defaults(GTK_TABLE(table),
	    (context->gotoc.bearing_label = gtk_label_new("-----")), 1,2,5,6);

  /* -------------- error label ------------------------- */
  gtk_table_attach_defaults(GTK_TABLE(table), 
	    gtk_label_new(_("Est. error:")), 0,1,6,7);
  gtk_table_attach_defaults(GTK_TABLE(table),
	    (context->gotoc.epe_label = gtk_label_new("-----")), 1,2,6,7);

  gtk_table_set_row_spacing(GTK_TABLE(table), 6, 16);

  /* -------------- sat view box ------------------------- */
  GtkWidget *ihbox = gtk_hbox_new(FALSE,0);

  context->gotoc.sat_area = gtk_drawing_area_new();
  gtk_drawing_area_size(GTK_DRAWING_AREA(context->gotoc.sat_area), 
 			SAT_WIDTH, SAT_HEIGHT);

  gtk_signal_connect(GTK_OBJECT(context->gotoc.sat_area), "expose_event",
		     (GtkSignalFunc)sat_expose_event, context);
  gtk_signal_connect(GTK_OBJECT(context->gotoc.sat_area),"configure_event",
		     (GtkSignalFunc)sat_configure_event, context);

  gtk_widget_set_events(context->gotoc.sat_area, GDK_EXPOSURE_MASK);

  gtk_box_pack_start(GTK_BOX(ihbox), context->gotoc.sat_area, 1,0,0);

#ifdef USE_MAEMO
  GtkWidget *mm_button = gtk_button_new();
  gtk_button_set_image(GTK_BUTTON(mm_button), icon_get_widget(ICON_MISC, 0));
  gtk_signal_connect(GTK_OBJECT(mm_button), "clicked",
		     (GtkSignalFunc)on_mm_button_clicked, context);
  gtk_box_pack_start(GTK_BOX(ihbox), mm_button, 1,0,0);
#endif
  gtk_table_attach_defaults(GTK_TABLE(table), ihbox, 0,2,7,8);

  /* ------------------------------------------------------- */

  gtk_box_pack_start(GTK_BOX(vbox), table, 1,0,0);
  gtk_box_pack_start(GTK_BOX(hbox), vbox, 1,0,0);
 
  context->gotoc.handler_id = gtk_timeout_add(UPDATE_MS, 
					     goto_update, context);

  return hbox;
}

void goto_coordinate_update(cache_context_t *context) {
  if(!context->notes.modified)
    return;
  
  waypoint_changed_event(context->gotoc.cbox, context);
}
