/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stddef.h>
#include <stdlib.h>
#include "gpxview.h"

#define GCONF_PATH         "/apps/gpxview/"
#define GCONF_KEY_GPX      GCONF_PATH "gpx%d"
#define GCONF_KEY_CNT      GCONF_PATH "entries"

#define GCONF_KEY_LOC_NAME GCONF_PATH "location%d/name"
#define GCONF_KEY_LOC_LAT  GCONF_PATH "location%d/latitude"
#define GCONF_KEY_LOC_LON  GCONF_PATH "location%d/longitude"
#define GCONF_KEY_LOC_CNT  GCONF_PATH "location_entries"

#include <string.h>

enum {
  STORE_STRING, STORE_FLOAT, STORE_INT, STORE_BOOL,
};

typedef struct {
  char *key;
  int type;
  int offset;
} store_t;

#define OFFSET(a) offsetof(appdata_t, a)

static store_t store[] = {
  { "image_path",       STORE_STRING, OFFSET(image_path) },
  { "path",             STORE_STRING, OFFSET(path) },
  { "geotext/text",     STORE_STRING, OFFSET(geotext_text) },
  { "geotext/shift",    STORE_INT,    OFFSET(geotext_shift) },
  { "mmpoi_path",       STORE_STRING, OFFSET(mmpoi_path) },
  { "garmin_path",      STORE_STRING, OFFSET(garmin_path) },
  { "garmin_ign_found", STORE_BOOL,   OFFSET(garmin_ign_found) },
  { "active_location",  STORE_INT,    OFFSET(active_location) },
  { "mmpoi_use_radius", STORE_BOOL,   OFFSET(mmpoi_use_radius) },
  { "mmpoi_radius",     STORE_FLOAT,  OFFSET(mmpoi_radius) },
  { "use_gps",          STORE_BOOL,   OFFSET(use_gps) },
  { "imperial",         STORE_BOOL,   OFFSET(imperial) },
  { "compass_locked",   STORE_BOOL,   OFFSET(compass_locked) },
  { "latitude",         STORE_FLOAT,  OFFSET(home.lat) },
  { "longitude",        STORE_FLOAT,  OFFSET(home.lon) },
  { "manual_goto_lat",  STORE_FLOAT,  OFFSET(manual_goto.lat) },
  { "manual_goto_lon",  STORE_FLOAT,  OFFSET(manual_goto.lon) },
  { "gps_lat",          STORE_FLOAT,  OFFSET(gps.lat) },
  { "gps_lon",          STORE_FLOAT,  OFFSET(gps.lon) },
  { "load_images",      STORE_BOOL,   OFFSET(load_images) },
  { "search_in",        STORE_INT,    OFFSET(search) },
  { "search_days",      STORE_INT,    OFFSET(search_days) },
  { "search_str",       STORE_STRING, OFFSET(search_str) },
  { "gpxlist_items",    STORE_INT,    OFFSET(gpxlist_items) },
  { "cachelist_items",  STORE_INT,    OFFSET(cachelist_items) },
  { "compass_damping",  STORE_INT,    OFFSET(compass_damping) },
  { "cachelist_hide_found", STORE_BOOL, OFFSET(cachelist_hide_found) },
#ifdef USE_MAEMO
  { "mmpoi_dontlaunch", STORE_BOOL,   OFFSET(mmpoi_dontlaunch) },
  { "cachelist_dss",    STORE_BOOL,   OFFSET(cachelist_disable_screensaver) },
  { "goto_dss",         STORE_BOOL,   OFFSET(goto_disable_screensaver) },
  { "cachelist_update", STORE_BOOL,   OFFSET(cachelist_update) },
#endif

  { NULL, -1, -1 }
};

void gconf_save_state(appdata_t *appdata) {
  int entries = 0;

  gpx_t *gpx = appdata->gpx;
  while(gpx) {
    char str[128];
    snprintf(str, sizeof(str), GCONF_KEY_GPX, entries++);
    gconf_client_set_string(appdata->gconf_client, str, gpx->filename, NULL);
    gpx = gpx->next;
  }

  gconf_client_set_int(appdata->gconf_client, GCONF_KEY_CNT, entries, NULL);

  /* -------------- save locations (excl. home location) --------------- */
  entries = 0;
  location_t *loc = appdata->location;
  while(loc) {
    char str[128];
    snprintf(str, sizeof(str), GCONF_KEY_LOC_NAME, entries);
    gconf_client_set_string(appdata->gconf_client, str, loc->name, NULL);
    snprintf(str, sizeof(str), GCONF_KEY_LOC_LAT, entries);
    gconf_client_set_float(appdata->gconf_client, str, loc->pos.lat, NULL);
    snprintf(str, sizeof(str), GCONF_KEY_LOC_LON, entries);
    gconf_client_set_float(appdata->gconf_client, str, loc->pos.lon, NULL);
    entries++;
    loc = loc->next;
  }

  gconf_client_set_int(appdata->gconf_client, GCONF_KEY_LOC_CNT, entries, NULL);

  /* store everything listed in the store table */
  store_t *st = store;
  while(st->key) {
    char key[256];
    void **ptr = ((void*)appdata) + st->offset;
    snprintf(key, sizeof(key), GCONF_PATH "%s", st->key);

    switch(st->type) {
    case STORE_STRING: 
      if((char*)(*ptr)) {
	gconf_client_set_string(appdata->gconf_client, key, (char*)(*ptr), NULL);
      }
      break;

    case STORE_BOOL: 
      gconf_client_set_bool(appdata->gconf_client, key, *((int*)ptr), NULL);
      break;

    case STORE_INT: 
      gconf_client_set_int(appdata->gconf_client, key, *((int*)ptr), NULL);
      break;

    case STORE_FLOAT: 
      gconf_client_set_float(appdata->gconf_client, key, *((float*)ptr), NULL);
      break;

    default:
      printf("Unsupported type %d\n", st->type);
      break;
    }

    st++;
  }
}

void gconf_load_state(appdata_t *appdata) {
  gpx_t **gpx = &appdata->gpx;

  while(*gpx) gpx = &((*gpx)->next);

  int i, entries = gconf_client_get_int(appdata->gconf_client, 
				 GCONF_KEY_CNT, NULL);

  for(i=0;i<entries;i++) {
    char str[128];
    snprintf(str, sizeof(str), GCONF_KEY_GPX, i);
    char *fname = gconf_client_get_string(appdata->gconf_client, str, NULL);

    if(fname) {
      if(g_file_test(fname, G_FILE_TEST_IS_DIR)) 
	*gpx = gpx_parse_dir(fname);
      else
	*gpx = gpx_parse(fname);
      
      free(fname);
    }

    /* use next gpx entry of this was loaded successfully */
    if(*gpx)
      gpx = &((*gpx)->next);
  }

  /* ------------- load locations --------------------- */
  entries = gconf_client_get_int(appdata->gconf_client, 
				 GCONF_KEY_LOC_CNT, NULL);

  location_t **loc = &appdata->location;
  for(i=0;i<entries;i++) {
    *loc = g_new0(location_t, 1);
    if(*loc) {
      char str[128];
      snprintf(str, sizeof(str), GCONF_KEY_LOC_NAME, i);
      (*loc)->name = gconf_client_get_string(appdata->gconf_client, str, NULL);
      snprintf(str, sizeof(str), GCONF_KEY_LOC_LAT, i);
      (*loc)->pos.lat  = gconf_client_get_float(appdata->gconf_client, str, NULL);
      snprintf(str, sizeof(str), GCONF_KEY_LOC_LON, i);
      (*loc)->pos.lon  = gconf_client_get_float(appdata->gconf_client, str, NULL);
      
      loc = &((*loc)->next);
    }
  }

  /* restore everything listed in the store table */
  store_t *st = store;
  while(st->key) {
    char key[256];
    void **ptr = ((void*)appdata) + st->offset;
    snprintf(key, sizeof(key), GCONF_PATH "%s", st->key);

    switch(st->type) {
    case STORE_STRING: {
      char **str = (char**)ptr;
      *str = gconf_client_get_string(appdata->gconf_client, key, NULL);
      } break;

    case STORE_BOOL: 
      *((int*)ptr) = gconf_client_get_bool(appdata->gconf_client, key, NULL);
      break;

    case STORE_INT: 
      *((int*)ptr) = gconf_client_get_int(appdata->gconf_client, key, NULL);
      break;

    case STORE_FLOAT: 
      *((float*)ptr) = gconf_client_get_float(appdata->gconf_client, key, NULL);
      break;

    default:
      printf("Unsupported type %d\n", st->type);
      break;
    }

    st++;
  }

  /* ----- set all kinds of defaults ------- */

  if(!appdata->home.lon || !appdata->home.lat) {
    appdata->home.lat = DEFAULT_LAT;
    appdata->home.lon = DEFAULT_LON;
  }

  if(!appdata->manual_goto.lon || !appdata->manual_goto.lat) {
    appdata->manual_goto.lat = DEFAULT_LAT;
    appdata->manual_goto.lon = DEFAULT_LON;
  }

  if(!appdata->compass_damping) appdata->compass_damping = 1;
  
  if(!appdata->gps.lon || !appdata->gps.lat) {
    appdata->gps.lat = DEFAULT_LAT;
    appdata->gps.lon = DEFAULT_LON;
  }

  if(!appdata->mmpoi_radius) 
    appdata->mmpoi_radius = 100.0;  // 100 km

  if(!appdata->search)
    appdata->search = SEARCH_NAME | SEARCH_ID;

  if(!appdata->image_path) {
#ifndef USE_MAEMO
    char *p = getenv("HOME");
    if(p) {
      /* build image path in home directory */
      appdata->image_path = 
	malloc(strlen(p)+strlen(DEFAULT_IMAGE_PATH_HOME)+2);
      strcpy(appdata->image_path, p);
      if(appdata->image_path[strlen(appdata->image_path)-1] != '/')
	strcat(appdata->image_path, "/");
      strcat(appdata->image_path, DEFAULT_IMAGE_PATH_HOME);
    } else
#endif
    appdata->image_path = strdup(DEFAULT_IMAGE_PATH);
  }

  if(!appdata->mmpoi_path) {
    char *p = getenv("HOME");
    if(p) {
      /* build image path in home directory */
      appdata->mmpoi_path = 
	malloc(strlen(p)+strlen(DEFAULT_MMPOI_PATH)+2);
      strcpy(appdata->mmpoi_path, p);
      if(appdata->image_path[strlen(appdata->mmpoi_path)-1] != '/')
	strcat(appdata->mmpoi_path, "/");
      strcat(appdata->mmpoi_path, DEFAULT_MMPOI_PATH);
    } else
      appdata->mmpoi_path = strdup(DEFAULT_MMPOI_PATH);
  }

  if(!appdata->garmin_path) {
    char *p = getenv("HOME");
    if(p) {
      /* build image path in home directory */
      appdata->garmin_path = 
	malloc(strlen(p)+strlen(DEFAULT_GARMIN_PATH)+2);
      strcpy(appdata->garmin_path, p);
      if(appdata->image_path[strlen(appdata->garmin_path)-1] != '/')
	strcat(appdata->garmin_path, "/");
      strcat(appdata->garmin_path, DEFAULT_GARMIN_PATH);
    } else
      appdata->garmin_path = strdup(DEFAULT_GARMIN_PATH);
  }

  /* make sure image path actually exists */
  checkdir(appdata->image_path);

  if(!appdata->gpxlist_items)
    appdata->gpxlist_items = GPXLIST_ITEM_DEFAULT;

  if(!appdata->cachelist_items)
    appdata->cachelist_items = CACHELIST_ITEM_DEFAULT;

}

