/**
 * gps-camera-gallery-view.c
 * Copyright (C) 2007 Sanna Salmijarvi (ssalmija@gmail.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */ 
#include <gtk/gtk.h>
#include <glib.h>
#include <libosso.h>
#include <ossoemailinterface.h>

#include "gps-camera-gallery-view.h"
#include "gps-camera-common.h"
#include "gps-camera-define.h"
#include "gps-camera-states.h"

/**
  Function to show the image on image display.
  @param camerawindow is the pointer to the CameraView structure.
  @param file_name pointer to the image path.
  @return none
 */
static void gpscamera_show_image(CameraView * camerawindow, gchar * file_name);

/**
  Function to set default selection in view.
  @param camerawindow is the pointer to the CameraView structure.
  @return TRUE on success FALSE if no row exists.
 */
static gboolean gpscamera_set_selection_default(CameraView * camerawindow);

static void 
gpscamera_show_image(CameraView * camerawindow, gchar * file_name)
{
	GdkPixbuf *image_orig = NULL;
	GdkPixbuf *image_resize = NULL;
	
	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);
	g_return_if_fail(camerawindow != NULL);

	if ((file_name == NULL) || (gpscamera_check_file_exists(file_name) 
								== FALSE)) {
		osso_log(LOG_WARNING, "File does not exist");
		gpscamera_show_info_note(GTK_WINDOW(camerawindow->uiwindow),
			      DGETTEXT_FM("file not found"));
		camerawindow->counter = camerawindow->counter-1;
		osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
		return;
	}
	image_orig = gdk_pixbuf_new_from_file(file_name, NULL);
	if (image_orig == NULL) {
		gtk_widget_hide(camerawindow->imagedisplay);
		gtk_widget_hide(camerawindow->image_event_box);
		osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
		return;
	}
	
	image_resize = gdk_pixbuf_scale_simple(image_orig, NORMAL_SCREEN_WIDTH,
				           NORMAL_SCREEN_HEIGHT, 
					   GDK_INTERP_BILINEAR);

	if (!camerawindow->imagedisplay) {
		camerawindow->imagedisplay = gtk_image_new_from_pixbuf
							(image_resize);
	} else {
		gtk_image_set_from_pixbuf(GTK_IMAGE(camerawindow->imagedisplay), 
							image_resize);
	}
	if ((GTK_WIDGET_VISIBLE(camerawindow->cameradisplay) == FALSE)) {
		gtk_widget_show(camerawindow->imagedisplay);
		gtk_widget_show(camerawindow->image_event_box);
	}
	g_object_unref(image_orig);
	g_object_unref(image_resize);
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
}
static void set_photo_number_label_text(CameraView *camerawindow, gchar *file_path)
{
	gchar *label_text = NULL;
	GList *images =NULL;

	label_text = g_strdup_printf("%d / %d",camerawindow->counter, 
						camerawindow->photo_amount);
	gtk_label_set_text(GTK_LABEL(camerawindow->label_photo_number), label_text);
                                                        
	images = gpscamera_read_gpslocation_file(file_path);
	if((images ==NULL)||(g_list_length(images) ==0)){
		gtk_widget_hide(camerawindow->gps_icon_on);
		gtk_widget_show(camerawindow->gps_icon_off);
		gtk_widget_show(camerawindow->label_photo_number);
		if(images ==NULL){
			return;
		}
		
	} else{
		gtk_widget_hide(camerawindow->gps_icon_off);
		gtk_widget_show(camerawindow->gps_icon_on);
		gtk_widget_show(camerawindow->label_photo_number);
	}

	g_list_free(images);
	g_free(label_text);
	return;

}

static gboolean gpscamera_set_selection_default(CameraView * camerawindow)
{
	
	gchar *file_path = NULL;
	
	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);
	g_return_val_if_fail(camerawindow != NULL,FALSE);
	
	camerawindow->counter =1;

	file_path = g_list_nth_data(camerawindow->photos, camerawindow->counter-1);

		
	gpscamera_show_image(camerawindow, file_path);
	set_photo_number_label_text(camerawindow, file_path);
	
	gtk_widget_show(camerawindow->image_event_box);
	gtk_widget_show(camerawindow->imagedisplay);
	

	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	return TRUE;
}

static gint
gpscamera_sort_function(gchar * afile_path,
			   gchar * bfile_path)
{
	
	osso_log(LOG_DEBUG, "Start Function %s \n", __FUNCTION__);
	
	if (strcmp(afile_path, bfile_path) < 0)
		return 1;
	if (strcmp(afile_path, bfile_path) > 0)
		return -1;
	return 0;
}

void gpscamera_view_append_model(CameraView * camerawindow, gchar * photo_path)
{
	GnomeVFSDirectoryHandle *handle = NULL;
	GnomeVFSFileInfo *info = NULL;
	GList *pathuri = NULL;
	gchar *fpath = NULL;
	gchar *photo = NULL;

	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);
	
	g_return_if_fail(camerawindow != NULL);
	g_return_if_fail(photo_path != NULL);
	
	pathuri = g_list_append(pathuri, photo_path);
	
	while (pathuri != NULL) {
		
		gnome_vfs_directory_open_from_uri(&handle, gnome_vfs_uri_new
				(gnome_vfs_get_uri_from_local_path
				(pathuri->data)), GNOME_VFS_FILE_INFO_DEFAULT);
		if (handle != NULL) {
			info = gnome_vfs_file_info_new();
			gnome_vfs_file_info_clear(info);
			while (gnome_vfs_directory_read_next(handle, info) 
				== GNOME_VFS_OK) {
				
				while (gtk_events_pending()) {
				           gtk_main_iteration();
				}
				if (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_TYPE) {
					if (info->name != NULL) {
						if ((info->name[0] == '.' && info->name[1] == '\0')
						    || (info->name[0] == '.' && info->name[1] == '.'
							&& info->name[2] == '\0')) {
							continue;
						}
						fpath =
						    g_strdup_printf("%s%s", (gchar *) pathuri->data,
								    info->name);
						camerawindow->photos = g_list_append(camerawindow->photos, fpath);
					}
				}
				gnome_vfs_file_info_clear(info);
				
			}
			gnome_vfs_file_info_unref(info);
			gnome_vfs_directory_close(handle);
			handle = NULL;			
		}
		
		pathuri = pathuri->next;
	}
	camerawindow->photo_amount = g_list_length(camerawindow->photos);
	camerawindow->photos = g_list_sort(camerawindow->photos,
                                                         gpscamera_sort_function);
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);

}	

gboolean gpscamera_gallery_view_launch(CameraView * camerawindow)
{
	gchar *photo_path = NULL;
	gchar *label_text = NULL;
		
	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);
	g_return_val_if_fail(camerawindow != NULL,FALSE);
	
	
	photo_path = g_strdup_printf("%s%s", HOME_DIR, HOME_CAMERA);
		
	gpscamera_view_append_model(camerawindow, photo_path);

	/*if directory hasn't any photos*/
	if(g_list_length(camerawindow->photos)==0){
		gpscamera_show_info_note(GTK_WINDOW(camerawindow->uiwindow), 
					_("No photos!"));
		label_text = g_strdup_printf("0 / 0");
		gtk_label_set_text(GTK_LABEL(camerawindow->label_photo_number), 
								label_text);
		gtk_widget_show(camerawindow->label_photo_number);
			
	}else{
	
		if (camerawindow->model_append == FALSE) {
			gpscamera_set_selection_default(camerawindow);
		}
		camerawindow->model_append = FALSE;
	}
	
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	return TRUE;
}

void gpscamera_gallery_view_image_next(CameraView * camerawindow)
{
	gchar *file_path = NULL;
		
	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);
	g_return_if_fail(camerawindow != NULL);

		
	if(camerawindow->delete_file!=1){
		camerawindow->counter=camerawindow->counter+1;
		file_path = g_list_nth_data(camerawindow->photos, 
							camerawindow->counter-1);
		
	}else{
		file_path = g_list_nth_data(camerawindow->photos, 
							camerawindow->counter-1);

	}
	if((camerawindow->counter>camerawindow->photo_amount)||
						(camerawindow->photo_amount==0)){
		gpscamera_show_info_note(GTK_WINDOW(camerawindow->uiwindow), 
							_("No more photos"));
		camerawindow->counter=camerawindow->counter-1;
		osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
		return FALSE;
	} else{
	
		set_photo_number_label_text(camerawindow, file_path);
		gpscamera_show_image(camerawindow, file_path);
	}
	
	camerawindow->delete_file=0;
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	return TRUE;
}

void gpscamera_gallery_view_image_prev(CameraView * camerawindow)
{
	gchar *file_path = NULL;
		
	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);
	g_return_if_fail(camerawindow != NULL);

	camerawindow->counter=camerawindow->counter-1;
	file_path = g_list_nth_data(camerawindow->photos, camerawindow->counter-1);

	if( camerawindow->counter<=0){
		gpscamera_show_info_note(GTK_WINDOW(camerawindow->uiwindow),
							 _("No more photos"));
		camerawindow->counter=camerawindow->counter+1;
		osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
		return FALSE;
	} else{
				
		set_photo_number_label_text(camerawindow, file_path);
		gpscamera_show_image(camerawindow, file_path);
	}

	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	return;
}

void gpscamera_gallery_view_delete_file(CameraView * camerawindow)
{
	gboolean ret_val = FALSE;
	gchar *file_path = NULL;
	gint response = 0;
	gchar *info_string = NULL;
	GtkWidget *note = NULL;
	gchar *label_text = NULL;

	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);
	g_return_if_fail(camerawindow != NULL);
	
	file_path = g_list_nth_data(camerawindow->photos, camerawindow->counter-1);
	note = gpscamera_delete_dialog_create(camerawindow);

	response = gtk_dialog_run(GTK_DIALOG(note));
	
	if (response == GTK_RESPONSE_OK) {
		ret_val = gpscamera_delete_file(file_path);
		switch (ret_val) {
		case GPSCAMERA_DELETE_SUCCESS:
			camerawindow->delete_file=1;
			camerawindow->photos = g_list_remove(camerawindow->photos,
                                                         file_path);
			camerawindow->photo_amount = g_list_length
						(camerawindow->photos);
			gpscamera_gallery_view_image_next(camerawindow);
			/*if last photo has been deleted*/
			if(camerawindow->photo_amount == 0){
				label_text = g_strdup_printf("0 / 0");
				gtk_label_set_text(GTK_LABEL
				(camerawindow->label_photo_number), label_text);
				gtk_widget_hide(camerawindow->gps_icon_off);
				gtk_widget_hide(camerawindow->gps_icon_on);
				gtk_image_set_from_file(GTK_IMAGE(camerawindow->imagedisplay),
                                "/usr/share/gpscamera/pictures/gpscam_display_background.png");
			}
			break;
		case GPSCAMERA_DELETE_NO_PERM:
			gpscamera_show_info_note(GTK_WINDOW(camerawindow->uiwindow), 
						_("You haven't perm to delete file"));
			break;
		case GPSCAMERA_FILE_NA:
			osso_log(LOG_WARNING, "File path does not exist\n");
			gpscamera_show_info_note(GTK_WINDOW(camerawindow->uiwindow),
				      DGETTEXT_FM("File not found"));
			break;
		default:
			gpscamera_show_info_note(GTK_WINDOW(camerawindow->uiwindow), 
					_("Error operation"));
			break;
		}
	}
	gtk_widget_destroy(GTK_WIDGET(note));
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	return;
}

void gpscamera_send_mail(CameraView * camerawindow)
{
	GSList *pathuri = NULL;
	gchar *file_path = NULL;
	gboolean ret_val = FALSE;
	osso_return_t return_val;

	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);
	g_return_if_fail(camerawindow != NULL);
  
	file_path = g_list_nth_data(camerawindow->photos, camerawindow->counter-1);

	ret_val = gpscamera_check_file_exists(file_path);
	if (ret_val == FALSE) {
		osso_log(LOG_WARNING, "File path does not exist\n");
		
		osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
		return;
	}

	pathuri = g_slist_append(pathuri, gnome_vfs_get_uri_from_local_path
								(file_path));
	
	return_val =
		osso_email_files_email(camerawindow->context, pathuri);
		
	if (return_val == OSSO_OK) {
			
		osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
		return;
	}

	if (return_val == OSSO_ERROR) {
			osso_log(LOG_WARNING, "File path does not exist\n");
	}
	

	if (return_val == OSSO_INVALID){
		osso_log(LOG_WARNING, "Invalid parameter\n");
	}

	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
}
