/* 
 * gps-camera-flickr.h 
 * Copyright (C) 2007 Sanna Salmijarvi (ssalmija@gmail.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */


#ifndef GPS_CAMERA_FLICKR_H
#define GPS_CAMERA_FLICKR_H

#ifdef __cplusplus
extern "C" {
#endif

/* needed for xmlDocPtr */
#include <libxml/tree.h>


/* Use gcc 3.1+ feature to allow marking of deprecated API calls.
 * This gives a warning during compiling.
 */
#if ( __GNUC__ == 3 && __GNUC_MINOR__ > 0 ) || __GNUC__ > 3
#ifdef __APPLE_CC__
/* OSX gcc cpp-precomp is broken */
#define FLICKCURL_DEPRECATED
#else
#define FLICKCURL_DEPRECATED __attribute__((deprecated))
#endif
#else
#define FLICKCURL_DEPRECATED
#endif

/* The main object type */
typedef struct flickcurl_s flickcurl;
  

/* Forward structure references */
struct flickcurl_s;
struct gpscamera_flickr_photo_s;


/**
 * gpscamera_flickr_location:
 * @latitude: The latitude from -90 to 90
 * @longitude: The longitude from -180 to 180
 * @accuracy: Recorded accuracy level of the location.
 *   World level is 1, Country is ~3, Region ~6, City ~11, Street
 *   ~16. Current range is 1-16. Defaults to 16 if not specified. (<0 for none)
 */
typedef struct 
{
  double latitude;
  double longitude;
  int accuracy;
} gpscamera_flickr_location;

/**
 * flickcurl_upload_params:
 * @photo_file: photo filename
 * @title: title or NULL
 * @description: description of photo (HTML) (or NULL)
 * @tags: space-separated list of tags (or NULL)
 * @is_public: is public photo boolean (non-0 true)
 * @is_friend: is friend photo boolean (non-0 true)
 * @is_family: is family photo boolean (non-0 true)
 * @safety_level: 1=safe, 2=moderate, 3=restricted
 * @content_type: 1=photo, 2=screenshot, 3=other/artwork
 *
 * Photo upload parameters
 */
typedef struct {
  const char* photo_file;
  const char *title;
  const char *description;
  const char *tags;  
  int is_public;
  int is_friend;
  int is_family;  
  int safety_level;
  int content_type;
} flickcurl_upload_params;


/**
 * gpscamera_flickr_upload_status:
 * @photoid: photo ID that was uploaded/replaced (upload only)
 * @secret: secret of photo uploaded (replace only)
 * @originalsecret: secret of original photo (replace only)
 * @ticketid: ticket ID for asynchronous upload (replace only)
 *
 * An upload response.
 *
 */
typedef struct {
  char *photoid;
  char *secret;
  char *originalsecret;
  char *ticketid;
} gpscamera_flickr_upload_status;


/* callback handlers */
typedef void (*flickcurl_message_handler)(void *user_data, const char *message);

/* library init - call once before creating anything */
void gpscamera_flickr_init(void);
/* library cleanup - call once before exit */
void gpscamera_flickr_finish(void);


/* flickcurl* object constructor */
flickcurl* gpscamera_flickr_new(void);

/* flickcurl* object destructor */
void gpscamera_flickr_free(flickcurl *fc);

/* flickcurl* object set methods */
void gpscamera_flickr_set_api_key(flickcurl* fc, const char *api_key);
void gpscamera_flickr_set_auth_token(flickcurl *fc, const char* auth_token);
void gpscamera_flickr_set_shared_secret(flickcurl* fc, const char *secret);
void gpscamera_flickr_set_write(flickcurl *fc, int is_write);

/* flickcurl* object set methods */
const char* gpscamera_flickr_get_api_key(flickcurl *fc);
const char* gpscamera_flickr_get_shared_secret(flickcurl *fc);
const char* gpscamera_flickr_get_auth_token(flickcurl *fc);

/* Flickr API calls */

/* flickr.auth */
char* gpscamera_flickr_auth_checkToken(flickcurl* fc, const char* token);
char* gpscamera_flickr_auth_getFrob(flickcurl* fc);
char* gpscamera_flickr_auth_getFullToken(flickcurl* fc, const char* frob);

/* flickr.photos.geo */
int gpscamera_flickr_photos_geo_setLocation(flickcurl* fc, const char* photo_id,
					 gpscamera_flickr_location* location);

/* Upload API */
FLICKCURL_DEPRECATED gpscamera_flickr_upload_status* gpscamera_flickr_photos_upload
						(flickcurl* fc, 
						const char* photo_file, 
						const char *title, 
						const char *description, 
						const char *tags, 
						int is_public, 
						int is_friend, 
						int is_family);
gpscamera_flickr_upload_status* flickcurl_photos_upload_params
				(flickcurl* fc, flickcurl_upload_params* params);
void gpscamera_flickr_free_upload_status(gpscamera_flickr_upload_status* status);
FLICKCURL_DEPRECATED void gpscamera_flickr_upload_status_free
					(gpscamera_flickr_upload_status* status);

#ifdef __cplusplus
}
#endif

#endif
