/* 
 *
 * gps-camera-flickr-common.c - Flickr common functions
 * Copyright (C) 2007 Sanna Salmijarvi (ssalmija@gmail.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 * 
 */

#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/* for access() and R_OK */
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#undef HAVE_STDLIB_H
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#include "gps-camera-flickr.h"
#include "gps-camera-flickr_internal.h"


static void
gpscamera_flickr_error_varargs(flickcurl* fc, const char *message, 
                        va_list arguments)
{
  if(fc->error_handler) {
    char *buffer=my_vsnprintf(message, arguments);
    if(!buffer) {
      fprintf(stderr, "flickcurl: Out of memory\n");
      return;
    }
    fc->error_handler(fc->error_data, buffer);
    free(buffer);
  } else {
    fprintf(stderr, "flickcurl error - ");
    vfprintf(stderr, message, arguments);
    fputc('\n', stderr);
  }
}

  
void
gpscamera_flickr_error(flickcurl* fc, const char *message, ...)
{
  va_list arguments;

  va_start(arguments, message);
  gpscamera_flickr_error_varargs(fc, message, arguments);
  va_end(arguments);
}

  
static size_t
gpscamera_flickr_write_callback(void *ptr, size_t size, size_t nmemb, 
                         void *userdata) 
{
  flickcurl* fc=(flickcurl*)userdata;
  int len=size*nmemb;
  int rc=0;
  
  if(fc->failed)
    return 0;

  fc->total_bytes += len;
  
  if(!fc->xc) {
    xmlParserCtxtPtr xc;

    xc = xmlCreatePushParserCtxt(NULL, NULL,
                                 (const char*)ptr, len,
                                 (const char*)fc->uri);
    if(!xc)
      rc=1;
    else {
      xc->replaceEntities = 1;
      xc->loadsubset = 1;
    }
    fc->xc=xc;
  } else
    rc=xmlParseChunk(fc->xc, (const char*)ptr, len, 0);

  if(rc)
    gpscamera_flickr_error(fc, "XML Parsing failed");

  return len;
}


flickcurl*
gpscamera_flickr_new(void)
{
  flickcurl* fc;

  fc=(flickcurl*)calloc(1, sizeof(flickcurl));
  if(!fc)
    return NULL;

  /* DEFAULT delay between requests is 1000ms i.e 1 request/second max */
  fc->request_delay=1000;
  
  if(!fc->curl_handle) {
    fc->curl_handle=curl_easy_init();
    fc->curl_init_here=1;
  }

#ifndef CURLOPT_WRITEDATA
#define CURLOPT_WRITEDATA CURLOPT_FILE
#endif

  /* send all data to this function  */
  curl_easy_setopt(fc->curl_handle, CURLOPT_WRITEFUNCTION, 
                   gpscamera_flickr_write_callback);
  /* ... using this data pointer */
  curl_easy_setopt(fc->curl_handle, CURLOPT_WRITEDATA, fc);


  /* Make it follow Location: headers */
  curl_easy_setopt(fc->curl_handle, CURLOPT_FOLLOWLOCATION, 1);

  curl_easy_setopt(fc->curl_handle, CURLOPT_ERRORBUFFER, fc->error_buffer);

  return fc;
}


void
gpscamera_flickr_free(flickcurl *fc)
{
  if(fc->xc) {
    if(fc->xc->myDoc) {
      xmlFreeDoc(fc->xc->myDoc);
      fc->xc->myDoc=NULL;
    }
    xmlFreeParserCtxt(fc->xc); 
  }

  if(fc->api_key)
    free(fc->api_key);
  if(fc->secret)
    free(fc->secret);
  if(fc->auth_token)
    free(fc->auth_token);
  if(fc->method)
    free(fc->method);

  /* only tidy up if we did all the work */
  if(fc->curl_init_here && fc->curl_handle) {
    curl_easy_cleanup(fc->curl_handle);
    fc->curl_handle=NULL;
  }

  if(fc->error_msg)
    free(fc->error_msg);

  if(fc->param_fields) {
    int i;
    
    for(i=0; fc->param_fields[i]; i++) {
      free(fc->param_fields[i]);
      free(fc->param_values[i]);
    }
    free(fc->param_fields);
    free(fc->param_values);
    fc->param_fields=NULL;
    fc->param_values=NULL;
    fc->parameter_count=0;
  }
  if(fc->upload_field)
    free(fc->upload_field);
  if(fc->upload_value)
    free(fc->upload_value);

  free(fc);
}


void
gpscamera_flickr_init(void)
{
  curl_global_init(CURL_GLOBAL_ALL);
  xmlInitParser();
}


void
gpscamera_flickr_finish(void)
{
  xmlCleanupParser();
  curl_global_cleanup();
}

void
gpscamera_flickr_set_api_key(flickcurl* fc, const char *api_key)
{

  if(fc->api_key)
    free(fc->api_key);
  fc->api_key=strdup(api_key);
}


const char*
gpscamera_flickr_get_api_key(flickcurl* fc)
{
  return fc->api_key;
}


void
gpscamera_flickr_set_shared_secret(flickcurl* fc, const char *secret)
{
  if(fc->secret)
    free(fc->secret);
  fc->secret=strdup(secret);
}


const char*
gpscamera_flickr_get_shared_secret(flickcurl* fc)
{
  return fc->secret;
}


void
gpscamera_flickr_set_auth_token(flickcurl *fc, const char* auth_token)
{
  if(fc->auth_token)
    free(fc->auth_token);
  fc->auth_token=strdup(auth_token);
}


const char*
gpscamera_flickr_get_auth_token(flickcurl *fc)
{
  return fc->auth_token;
}


void
gpscamera_flickr_set_sign(flickcurl *fc)
{
  fc->sign=1;
}


static int
compare_args(const void *a, const void *b) 
{
  return strcmp(*(char**)a, *(char**)b);
}


static void
gpscamera_flickr_sort_args(flickcurl *fc, const char *parameters[][2], int count)
{
  qsort(parameters, count, sizeof(char*[2]), compare_args);
}


static int
gpscamera_flickr_prepare_common(flickcurl *fc, 
                         const char* url,
                         const char* method,
                         const char* upload_field, const char* upload_value,
                         const char* parameters[][2], int count,
                         int parameters_in_url)
{

  int i;
  char *md5_string=NULL;
  size_t* values_len=NULL;

  if(!url || !parameters)
    return 1;
  
  /* If one is given, both are required */
  if((upload_field || upload_value) && (!upload_field || !upload_value))
    return 1;

  fc->failed=0;
  fc->error_code=0;
  if(fc->error_msg) {
    free(fc->error_msg);
    fc->error_msg=NULL;
  }

  /* Default to read */
  fc->is_write=0;
 
  if(fc->param_fields) {
    for(i=0; fc->param_fields[i]; i++) {
      free(fc->param_fields[i]);
      free(fc->param_values[i]);
    }

    free(fc->param_fields);
    free(fc->param_values);
    fc->param_fields=NULL;
    fc->param_values=NULL;
    fc->parameter_count=0;
  }
  if(fc->upload_field) {
    free(fc->upload_field);
    fc->upload_field=NULL;
  }

  if(fc->upload_value) {
    free(fc->upload_value);
    fc->upload_value=NULL;
  }
  
  if(!fc->secret) {
    gpscamera_flickr_error(fc, "No shared secret");
    return 1;
  }
  if(!fc->api_key) {
    gpscamera_flickr_error(fc, "No API key");
    return 1;
  }

  if(fc->method)
    free(fc->method);
  if(method)
    fc->method=strdup(method);
  else
    fc->method=NULL;

  if(fc->method) {
    parameters[count][0]  = "method";
    parameters[count++][1]= fc->method;
	
 }

  parameters[count][0]  = "api_key";
  parameters[count++][1]= fc->api_key;
	

  if(fc->auth_token) {
    parameters[count][0]  = "auth_token";
    parameters[count++][1]= fc->auth_token;
  }

  parameters[count][0]  = NULL;

  /* +1 for api_sig +1 for NULL terminating pointer */
  fc->param_fields=(char**)calloc(count+2, sizeof(char*));
  fc->param_values=(char**)calloc(count+2, sizeof(char*));
  values_len=(size_t*)calloc(count+2, sizeof(size_t));

  if(fc->auth_token || fc->sign)
    gpscamera_flickr_sort_args(fc, parameters, count);

  /* Save away the parameters and calculate the value lengths */
  for(i=0; parameters[i][0]; i++) {
    size_t param_len=strlen(parameters[i][0]);

    values_len[i]=strlen(parameters[i][1]);

    fc->param_fields[i]=(char*)malloc(param_len+1);
    strcpy(fc->param_fields[i], parameters[i][0]);
    fc->param_values[i]=(char*)malloc(values_len[i]+1);
    strcpy(fc->param_values[i], parameters[i][1]);
  }

  if(upload_field) {
    fc->upload_field=(char*)malloc(strlen(upload_field)+1);
    strcpy(fc->upload_field, upload_field);

    fc->upload_value=(char*)malloc(strlen(upload_value)+1);
    strcpy(fc->upload_value, upload_value);
  }

  if(fc->auth_token || fc->sign) {
    size_t buf_len=0;
    char *buf;
    
    buf_len=strlen(fc->secret);
	
    for(i=0; parameters[i][0]; i++)
      buf_len += strlen(parameters[i][0]) + values_len[i];
	

    buf=(char*)malloc(buf_len+1);
    strcpy(buf, fc->secret);
    for(i=0; parameters[i][0]; i++) {
      strcat(buf, parameters[i][0]);
      strcat(buf, parameters[i][1]);
    }


    md5_string=MD5_string(buf);
	    
    parameters[count][0]  = "api_sig";
    parameters[count][1]= md5_string;

    /* Add a new parameter pair */
    values_len[count]=32; /* MD5 is always 32 */
    fc->param_fields[count]=(char*)malloc(7+1); /* 7=strlen(api_sig) */
    strcpy(fc->param_fields[count], parameters[count][0]);
    fc->param_values[count]=(char*)malloc(32+1); /* 32=MD5 */
    strcpy(fc->param_values[count], parameters[count][1]);

    count++;

    free(buf);
    
    parameters[count][0] = NULL;
  }

  strcpy(fc->uri, url);

  if(parameters_in_url) {
    for(i=0; parameters[i][0]; i++) {
      char *value=(char*)parameters[i][1];
      char *escaped_value=NULL;

      if(!parameters[i][1])
        continue;

      strcat(fc->uri, parameters[i][0]);
      strcat(fc->uri, "=");
      if(!strcmp(parameters[i][0], "method")) {
        /* do not touch method name */
      } else
        escaped_value=curl_escape(value, values_len[i]);

      if(escaped_value) {
        strcat(fc->uri, escaped_value);
        curl_free(escaped_value);
      } else
        strcat(fc->uri, value);
      strcat(fc->uri, "&");
    }

    /* zap last & */
    fc->uri[strlen(fc->uri)-1]= '\0';
  }


  if(md5_string)
    free(md5_string);

  if(values_len)
    free(values_len);

  return 0;
}


int
gpscamera_flickr_prepare(flickcurl *fc, const char* method,
                  const char* parameters[][2], int count)
{
  if(!method) {
    gpscamera_flickr_error(fc, "No method to prepare");
    return 1;
  }
  
  return gpscamera_flickr_prepare_common(fc,
                                  "http://www.flickr.com/services/rest/?",
                                  method,
                                  NULL, NULL,
                                  parameters, count,
                                  1);
}


int
gpscamera_flickr_prepare_upload(flickcurl *fc, 
                         const char* url,
                         const char* upload_field, const char* upload_value,
                         const char* parameters[][2], int count)

{
  return gpscamera_flickr_prepare_common(fc,
                                  url,
                                  NULL,
                                  upload_field, upload_value,
                                  parameters, count,
                                  0);
}


xmlDocPtr
gpscamera_flickr_invoke(flickcurl *fc)
{
  struct curl_slist *slist=NULL;
  xmlDocPtr doc=NULL;
  struct timeval now;

  if(!fc->uri) {
    gpscamera_flickr_error(fc, "No Flickr URI prepared to invoke");
    return NULL;
  }
  
  gettimeofday(&now, NULL);

  memcpy(&fc->last_request_time, &now, sizeof(struct timeval));

  if(fc->xc) {
    if(fc->xc->myDoc) {
      xmlFreeDoc(fc->xc->myDoc);
      fc->xc->myDoc=NULL;
    }
    xmlFreeParserCtxt(fc->xc); 
    fc->xc=NULL;
  }

  if(fc->proxy)
    curl_easy_setopt(fc->curl_handle, CURLOPT_PROXY, fc->proxy);

  if(fc->user_agent)
    curl_easy_setopt(fc->curl_handle, CURLOPT_USERAGENT, fc->user_agent);

  /* Insert HTTP Accept: header */
  if(fc->http_accept)
    slist=curl_slist_append(slist, (const char*)fc->http_accept);

  /* specify URL to call */
  curl_easy_setopt(fc->curl_handle, CURLOPT_URL, fc->uri);

  fc->total_bytes=0;

  if(fc->is_write){
    curl_easy_setopt(fc->curl_handle, CURLOPT_POST, 1); /* Set POST */
	
}
  else
    curl_easy_setopt(fc->curl_handle, CURLOPT_POST, 0);  /* Set GET */

  if(slist)
    curl_easy_setopt(fc->curl_handle, CURLOPT_HTTPHEADER, slist);

  if(fc->upload_field) {
    struct curl_httppost* post = NULL;
    struct curl_httppost* last = NULL;
    int i;
  
    /* Main parameters */
    for(i=0; fc->param_fields[i]; i++) {
      curl_formadd(&post, &last, CURLFORM_PTRNAME, fc->param_fields[i],
                   CURLFORM_PTRCONTENTS, fc->param_values[i],
                   CURLFORM_END);
    }
   
    /* Upload parameter */
    curl_formadd(&post, &last, CURLFORM_PTRNAME, fc->upload_field,
                 CURLFORM_FILE, fc->upload_value, CURLFORM_END);

    /* Set the form info */
    curl_easy_setopt(fc->curl_handle, CURLOPT_HTTPPOST, post);
  }

  if(curl_easy_perform(fc->curl_handle)) {
    /* failed */
    fc->failed=1;
    gpscamera_flickr_error(fc, fc->error_buffer);
	 
  } else {
    long lstatus;

#ifndef CURLINFO_RESPONSE_CODE
#define CURLINFO_RESPONSE_CODE CURLINFO_HTTP_CODE
#endif

    /* Requires pointer to a long */
    if(CURLE_OK == 
       curl_easy_getinfo(fc->curl_handle, CURLINFO_RESPONSE_CODE, &lstatus) )
      fc->status_code=lstatus;

  }
 
  if(slist)
    curl_slist_free_all(slist);

  if(!fc->failed) {
    xmlNodePtr xnp;
    xmlAttr* attr;
    int failed=0;
    
    xmlParseChunk(fc->xc, NULL, 0, 1);

    doc=fc->xc->myDoc;
    if(!doc) {
      gpscamera_flickr_error(fc, "Failed to create XML DOM for document");
      fc->failed=1;
      goto tidy;
    }

    xnp = xmlDocGetRootElement(doc);
    if(!xnp) {
      gpscamera_flickr_error(fc, "Failed to parse XML");
      fc->failed=1;
      goto tidy;
    }
 
    for(attr=xnp->properties; attr; attr=attr->next) {
      if(!strcmp((const char*)attr->name, "stat")) {
        const char *attr_value=(const char*)attr->children->content;

        if(strcmp(attr_value, "ok"))
          failed=1;
        break;
      }
    }

    if(failed) {
      xmlNodePtr err=xnp->children->next;
      for(attr=err->properties; attr; attr=attr->next) {
        const char *attr_name=(const char*)attr->name;
        const char *attr_value=(const char*)attr->children->content;
        if(!strcmp(attr_name, "code"))
          fc->error_code=atoi(attr_value);
        else if(!strcmp(attr_name, "msg"))
          fc->error_msg=strdup(attr_value);
      }

      if(fc->method){
        gpscamera_flickr_error(fc, "Method %s failed with error %d - %s", 
                        fc->method, fc->error_code, fc->error_msg);
	}
      else{
        gpscamera_flickr_error(fc, "Call failed with error %d - %s", 
                     fc->error_code, fc->error_msg);
	}  
      fc->failed=1;
    }
  }

  tidy:
  if(fc->failed)
    doc=NULL;
   
  /* reset special flags */
  fc->sign=0;
 
  return doc;
}


char*
gpscamera_flickr_xpath_eval(flickcurl *fc, xmlXPathContextPtr xpathCtx,
                     const xmlChar* xpathExpr) 
{
  xmlXPathObjectPtr xpathObj=NULL;
  xmlNodeSetPtr nodes;
  int i;
  char* value=NULL;
  
  xpathObj = xmlXPathEvalExpression(xpathExpr, xpathCtx);
  if(!xpathObj) {
    gpscamera_flickr_error(fc, "Unable to evaluate XPath expression \"%s\"", 
                    xpathExpr);
    fc->failed=1;
    goto tidy;
  }
    
  nodes=xpathObj->nodesetval;
  for(i=0; i < xmlXPathNodeSetGetLength(nodes); i++) {
    xmlNodePtr node=nodes->nodeTab[i];
    
    if(node->type != XML_ATTRIBUTE_NODE &&
       node->type != XML_ELEMENT_NODE) {
      gpscamera_flickr_error(fc, "Got unexpected node type %d", node->type);
      fc->failed=1;
      break;
    }
    if(node->children)
      value=strdup((char*)node->children->content);
    break;
  }

  tidy:
  if(xpathObj)
    xmlXPathFreeObject(xpathObj);

  return value;
}


void
gpscamera_flickr_set_write(flickcurl *fc, int is_write)
{
  fc->is_write=is_write;
}


