/* 
 * gps-camera-flickr-apis.c 
 * Copyright (C) 2007 Sanna Salmijarvi (ssalmija@gmail.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#undef HAVE_STDLIB_H
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#include "gps-camera-flickr.h"
#include "gps-camera-flickr_internal.h"



/**
 * gpscamera_flickr_auth_checkToken:
 * @fc: context
 * @frob: frob string
 * Get the credentials attached to an authentication token.
 * Implements flickr.auth.checkToken (0.9)
 * Must be signed.
 * FIXME: Cannot confirm this works, get intermittent results.
 * Return value: permissions string or NULL on failure
 **/
char*
gpscamera_flickr_auth_checkToken(flickcurl* fc, const char* token)
{
  const char * parameters[6][2];
  int count=0;
  char *perms=NULL;
  xmlDocPtr doc=NULL;
  xmlXPathContextPtr xpathCtx=NULL; 

  if(!token)
    return NULL;
  
  parameters[count][0]   = "auth_token";
  parameters[count++][1] = (char*)token;

  parameters[count][0]   = NULL;

  gpscamera_flickr_set_sign(fc);
  
  if(gpscamera_flickr_prepare(fc, "flickr.auth.checkToken", parameters, count))
    goto tidy;

  doc=gpscamera_flickr_invoke(fc);
  if(!doc)
    goto tidy;
  
  xpathCtx = xmlXPathNewContext(doc);
  if(xpathCtx) {
    perms=gpscamera_flickr_xpath_eval(fc, xpathCtx,
                                    (const xmlChar*)"/rsp/auth/perms");
    xmlXPathFreeContext(xpathCtx);
  }

  tidy:

  return perms;
}


/**
 * gpscamera_flickr_auth_getFrob:
 * @fc: context
 * Get a frob to be used during authentication
 * Implements flickr.auth.getFrob (0.9)
 * Must be signed.  Does not require authentication.
 * Return value: frob string or NULL on failure
 **/
char*
gpscamera_flickr_auth_getFrob(flickcurl* fc)
{
  const char * parameters[5][2];
  int count=0;
  char *frob=NULL;
  xmlDocPtr doc=NULL;
  xmlXPathContextPtr xpathCtx=NULL; 
  
  parameters[count][0]   = NULL;

  gpscamera_flickr_set_sign(fc);
  
  if(gpscamera_flickr_prepare(fc, "flickr.auth.getFrob", parameters, count))
    goto tidy;

  doc=gpscamera_flickr_invoke(fc);

  if(!doc)
    goto tidy;
  
  xpathCtx = xmlXPathNewContext(doc);

  if(xpathCtx) {
    frob=gpscamera_flickr_xpath_eval(fc, xpathCtx, (const xmlChar*)"/rsp/frob");
    xmlXPathFreeContext(xpathCtx);
  }

  tidy:

  return frob;
}


/**
 * gpscamera_flickr_auth_getFullToken:
 * @fc: context
 * @frob: frob string
 * Turn a frob into an auth_token
 * Implements flickr.auth.getFullToken (0.5)
 * Must be signed.
 * Return value: token string or NULL on failure
 **/
char* gpscamera_flickr_auth_getFullToken(flickcurl* fc, const char* frob)
{
  const char * parameters[6][2];
  int count=0;
  char *auth_token=NULL;
  xmlDocPtr doc=NULL;
  xmlXPathContextPtr xpathCtx=NULL; 
  
  parameters[count][0]   = "mini_token";
  parameters[count++][1] = (char*)frob;

  parameters[count][0]   = NULL;

  gpscamera_flickr_set_sign(fc);

  if(gpscamera_flickr_prepare(fc, "flickr.auth.getFullToken", parameters, count))
    goto tidy;

  doc=gpscamera_flickr_invoke(fc);
  if(!doc)
    goto tidy;
  
  xpathCtx = xmlXPathNewContext(doc);
  if(xpathCtx) {
    auth_token=gpscamera_flickr_xpath_eval(fc, xpathCtx,
                                    (const xmlChar*)"/rsp/auth/token");
    xmlXPathFreeContext(xpathCtx);
  }

  tidy:

  return auth_token;
}

/**
 * gpscamera_flickr_photos_upload_params:
 * @fc: context
 * Uploads a photo with safety level and content type
 * Implements Uploading Photos (0.11)
 * Return value: #gpscamera_flickr_upload_status or NULL on failure
 **/
gpscamera_flickr_upload_status*
gpscamera_flickr_photos_upload_params(flickcurl* fc, flickcurl_upload_params* params)
{
  const char* parameters[12][2];
  int count=0;
  xmlDocPtr doc=NULL;
  xmlXPathContextPtr xpathCtx=NULL; 
  gpscamera_flickr_upload_status* status=NULL;
  char is_public_s[2];
  char is_friend_s[2];
  char is_family_s[2];
  char safety_level_s[2];
  char content_type_s[2];
  
  if(!params->photo_file)
    return NULL;

  if(access((const char*)params->photo_file, R_OK)) {
    	printf("Photo file %s cannot be read: %s", params->photo_file);
    return NULL;
  }

  is_public_s[0]=params->is_public ? '1' : '0';
  is_public_s[1]='\0';
  is_friend_s[0]=params->is_friend ? '1' : '0';
  is_friend_s[1]='\0';
  is_family_s[0]=params->is_family ? '1' : '0';
  is_family_s[1]='\0';

  if(params->safety_level >= 1 && params->safety_level <= 3) {
    safety_level_s[0]='0' + params->safety_level;
    safety_level_s[1]='\0';
  } else
    params->safety_level= -1;
  
  if(params->content_type >= 1 && params->content_type <= 3) {
    content_type_s[0]='0' + params->content_type;
    content_type_s[1]='\0';
  } else
    params->content_type= -1;
  
  if(params->title) {
    parameters[count][0]  = "title";
    parameters[count++][1]= params->title;
  }
  if(params->description) {
    parameters[count][0]  = "description";
    parameters[count++][1]= params->description;
  }
  if(params->tags) {
    parameters[count][0]  = "tags";
    parameters[count++][1]= params->tags;
  }
  if(params->safety_level >= 0) {
    parameters[count][0]  = "safety_level";
    parameters[count++][1]= safety_level_s;
  }
  if(params->content_type >= 0) {
    parameters[count][0]  = "content_type";
    parameters[count++][1]= content_type_s;
  }
  parameters[count][0]  = "is_public";
  parameters[count++][1]= is_public_s;
  parameters[count][0]  = "is_friend";
  parameters[count++][1]= is_friend_s;
  parameters[count][0]  = "is_family";
  parameters[count++][1]= is_family_s;

  parameters[count][0]  = NULL;
 
  if(gpscamera_flickr_prepare_upload(fc,
                            "http://api.flickr.com/services/upload/",
                            "photo", params->photo_file,
                            parameters, count))
	goto tidy;

  doc=gpscamera_flickr_invoke(fc);
  if(!doc)
    goto tidy;

  xpathCtx = xmlXPathNewContext(doc);
  if(!xpathCtx) {
    gpscamera_flickr_error(fc, "Failed to create XPath context for document");
    fc->failed=1;
    goto tidy;
  }

  status=(gpscamera_flickr_upload_status*)calloc(1, sizeof(gpscamera_flickr_upload_status));
  status->photoid=gpscamera_flickr_xpath_eval(fc, xpathCtx, (const xmlChar*)"/rsp/photoid");
  /* when async is true */
  tidy:
  if(xpathCtx)
    xmlXPathFreeContext(xpathCtx);

  if(fc->failed)
    status=NULL;

  return status;
}

/**
 * gpscamera_flickr_photos_upload:
 * @fc:  context
 * @photo_file: photo filename
 * @title: title or NULL
 * @description: description of photo (HTML) (or NULL)
 * @tags: space-separated list of tags (or NULL)
 * @is_public: is public photo boolean (non-0 true)
 * @is_friend: is friend photo boolean (non-0 true)
 * @is_family: is family photo boolean (non-0 true)
 * Uploads a photo
 * Implements Uploading Photos (0.10)
 * Return value: #gpscamera_flickr_upload_status or NULL on failure
 **/
gpscamera_flickr_upload_status*
gpscamera_flickr_photos_upload(flickcurl* fc, const char* photo_file,
                        const char *title,
                        const char *description,
                        const char *tags,
                        int is_public, int is_friend, int is_family)
{
  flickcurl_upload_params params;

  memset(&params, '\0', sizeof(flickcurl_upload_params));

  params.photo_file=photo_file;
  params.title=title;
  params.description=description;
  params.tags=tags;  
  params.is_public=is_public;
  params.is_friend=is_friend;
  params.is_family=is_family;  
  params.safety_level= -1;
  params.content_type= -1;
  
  return gpscamera_flickr_photos_upload_params(fc, &params);
}

/**
 * gpscamera_flickr_free_upload_status:
 * @status: status object
 * Destructor - free a #gpscamera_flickr_upload_status
 **/
void
gpscamera_flickr_free_upload_status(gpscamera_flickr_upload_status* status)
{
  if(status->photoid)
    free(status->photoid);
  if(status->secret)
    free(status->secret);
  if(status->originalsecret)
    free(status->originalsecret);
  if(status->ticketid)
    free(status->ticketid);
}

void
gpscamera_flickr_upload_status_free(gpscamera_flickr_upload_status* status)
{
  gpscamera_flickr_free_upload_status(status);
}

/**
 * gpscamera_flickr_photos_geo_setLocation:
 * @fc: context
 * @photo_id: The id of the photo to set location data for.
 * @location: The location
 * Sets the geo data (latitude and longitude and, optionally, the
 * accuracy level) for a photo.
 * Implements flickr.photos.geo.setLocation (0.12)
 * Return value: non-0 on failure
 **/
int
gpscamera_flickr_photos_geo_setLocation(flickcurl* fc, const char* photo_id,
                                 gpscamera_flickr_location* location)
{
  const char* parameters[11][2];
  int count=0;
  xmlDocPtr doc=NULL;
  char latitude_s[50];
  char longitude_s[50];
  char accuracy_s[50];
  int result=1;
  
  if(!photo_id)
    return 1;

  if(location->latitude < -90.0)
    location->latitude= -90.0;
  if(location->latitude > 90.0)
    location->latitude= 90.0;
  if(location->longitude < -180.0)
    location->longitude= -180.0;
  if(location->longitude > 180.0)
    location->longitude= 180.0;
  if(location->accuracy < 1 || location->accuracy > 16)
    location->accuracy = 0;

  parameters[count][0]  = "photo_id";
  parameters[count++][1]= photo_id;
  parameters[count][0]  = "lat";
  sprintf(latitude_s, "%f", location->latitude);
  parameters[count++][1]= latitude_s;
  parameters[count][0]  = "lon";
  sprintf(longitude_s, "%f", location->longitude);
  parameters[count++][1]= longitude_s;
  if(location->accuracy >= 1) {
    parameters[count][0]  = "accuracy";
    sprintf(accuracy_s, "%d", location->accuracy);
    parameters[count++][1]= accuracy_s;
  }

  parameters[count][0]  = NULL;
 
  if(gpscamera_flickr_prepare(fc, "flickr.photos.geo.setLocation", parameters, count))
    goto tidy;
 
  doc=gpscamera_flickr_invoke(fc);
  if(!doc)
    goto tidy;

  result=0;

  tidy:
  if(fc->failed)
    result=1;

  return result;
}
