/**
 * @file gps-camera-engine.h
 * Copyright (C) 2007 Sanna Salmijärvi (ssalmija@gmail.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 **/

#ifndef GPS_CAMERA_ENGINE_H
#define GPS_CAMERA_ENGINE_H


#include <glib.h>
#include <gdk/gdkx.h>
#include <gst/interfaces/xoverlay.h>


#define CAMERA_TYPE_ENGINE            (gpscamera_engine_get_type ())
#define CAMERA_ENGINE(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), CAMERA_TYPE_ENGINE, GpsCameraEngine))
#define CAMERA_ENGINE_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass), CAMERA_TYPE_ENGINE, GpsCameraEngineClass))
#define CAMERA_IS_ENGINE(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), CAMERA_TYPE_ENGINE))
#define CAMERA_IS_ENGINE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), CAMERA_TYPE_ENGINE))
#define CAMERA_ENGINE_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS((obj), CAMERA_TYPE_ENGINE, GpsCameraEngineClass))
#define CAMERA_ENGINE_GET_PRIVATE(o)  (G_TYPE_INSTANCE_GET_PRIVATE ((o), CAMERA_TYPE_ENGINE, GpsCameraEnginePrivate))

enum {
	PIPELINE_CREATION_SUCCESS,
	PIPELINE_CREATION_ERROR,
	PIPELINE_VIDEO_SRC_ERROR,
	PIPELINE_ELEMENT_CREATION_ERROR,
	PIPELINE_BUS_ERROR,
	
};


typedef struct _GpsCameraEngine       GpsCameraEngine;
typedef struct _GpsCameraEngineClass  GpsCameraEngineClass;

/**
 * GpsCameraEngine
 **/
struct _GpsCameraEngine {
        GObject  __parent;
};

/**
  GpsCameraEngineClass
*/

struct _GpsCameraEngineClass {
	GObjectClass __parent_class;

	/* -- Signals -- */

	void (*start)     (GpsCameraEngine   *engine,
			   gpointer data); 

	void (*done)      (GpsCameraEngine   *engine,
			   gpointer data);

        void (*captured)  (GpsCameraEngine   *engine,
			   gpointer data);
        
	void (*errorcapture)  (GpsCameraEngine   *engine,
			   gpointer data);

	void (*error) (GpsCameraEngine *engine,
			   gpointer data);
      
};

/**
  GpsCameraEngineAction
*/
typedef enum {
        GPSCAMERA_ENGINE_IDLE_VIEW,
       	GPSCAMERA_ENGINE_SNAPSHOT
       
} GpsCameraEngineAction;

/**
  GpsCameraEngineState
*/

typedef enum {
	ENGINE_IDLEVIEW,
	ENGINE_TAKE_PHOTO
} GpsCameraEngineState;

/**     
  This function creates the engine object 
  @param void 
  @return engine object  
*/
GpsCameraEngine* gpscamera_engine_new(void);

/**
  camera_engine_initialize:
  This function creates the necccessary elements for recording/playback 
  @param self pointer of type CameraEngine
  @param action of type CameraEngineAction
  @return gboolean TRUE on success FALSE on failure 
*/
gboolean 
gpscamera_engine_initialize(GpsCameraEngine *engine,GpsCameraEngineAction action);


/**
  This function sets the camera engine to different states. The second parameter
  CameraEngineAction value specifies the state Idle/Record/Play/Caputre.  
  Depending on it different activate functions are called.
  The third parameter specifies the location for Record/Play/Capture.
  @param self pointer to engine object
  @param action of type CameraEngineAction 
  @param location pointer of type gchar
  @return gboolean TRUE on success FALSE on failure 
*/
gboolean
gpscamera_engine_set_state(GpsCameraEngine *engine, GpsCameraEngineAction action, gchar *location);

/**
  This function returns the current error set in the engine. The error is freed by engine
  @param engine object
  @return GError 
*/
const GError* gpscamera_engine_get_error(GpsCameraEngine *engine);

/**
  This function returns the current state of the engine. 
  @param engine object
  @return CameraEngineState 
*/
GpsCameraEngineState
gpscamera_engine_get_state(GpsCameraEngine *engine);

/**
  This function sets the overlay window id on which the video stream has to be displayed
  @param engine object
  @param xid of type XID 
  @return void 
*/
void gpscamera_engine_overlay_window(GpsCameraEngine *engine , XID xid);

/**
  This function creates the engine object sets the xid and creates the pipeline
  depending on the action 
  @param void 
  @return engine object  
*/
GpsCameraEngine *
gpscamera_engine_init_with_xid_action (XID xid,GpsCameraEngineAction action);

/**
  This function unrefs the engine object 
  @param pointer to CameraEngine 
  @return void 
*/
void gpscamera_engine_unref(GpsCameraEngine *engine);

#endif 
