/**
 * gps-camera-engine.c
 ** Copyright (C) 2007 Sanna Salmijarvi (ssalmija@gmail.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 **/


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gdk/gdkx.h>
#include <glib.h>
#include <gst/gst.h>
#include <gst/interfaces/xoverlay.h>
#include <string.h>
#include <unistd.h>
#include <jpeglib.h>
#include <libgnomevfs/gnome-vfs-ops.h>

#include "gps-camera-engine-marshal.h"
#include "gps-camera-engine-debug.h"
#include "gps-camera-define.h"
#include "gps-camera-engine.h"
#include "gps-camera-controller.h"


#define g_signal_handlers_disconnect_by_data(instance, data) \
    g_signal_handlers_disconnect_matched ((instance), G_SIGNAL_MATCH_DATA, \
                                          0, 0, NULL, NULL, (data))


enum {
	START,
	DONE,
	CAPTURED,
	ERRORCAPTURE,
	ERROR,
	LAST_SIGNAL
};

/* Private Variables */
typedef struct _GpsCameraEnginePrivate GpsCameraEnginePrivate;

struct _GpsCameraEnginePrivate{

        GstElement *pipeline;
        GstElement *video_source;
        GstElement *video_converter;
        GstElement *video_sink;
        GstElement *filesink;
        GstElement *tee;
        GstBus *bus;
        XID overlay_window_id;
        gint pipeline_error;
        GError *error;
        gint action;
      	GpsCameraEngineState state;
	gchar *filename;
	gboolean takesnap;
};


static gboolean snapshot(GpsCameraEngine *self);
/**
  This function is part of GObject implementation to determine the type 
  @param void 
  @return GType 
*/
static void gpscamera_engine_init(GpsCameraEngine *self);

/**
  This function is part of GObject implementation to initialize the 
  CameraEngineClass 
  @param class pointer of type CameraEngineClass 
  @return void 
*/
static void gpscamera_engine_class_init(GpsCameraEngineClass *class);

/**
  This function is part of GObject implementation to deinitialize the 
  CameraEngine object. It is called whenever the CameraEngine object is unref'ed
  @param class pointer of type CameraEngine
  @return void 
*/
static void gpscamera_engine_shutdown(GObject *object);


/**
  This function sets the error whever there is error message on the bus
  @param self pointer to engine object
  @param error pointer of type GError 
  @return void 
*/
static void gpscamera_engine_set_error(GpsCameraEngine *self,  GError *error);

/**
  This function is called by camera_engine_set_state whenever play request comes up.
  It creates play bin and sets the pipeline to playing. 
  @param pointer to CameraEngine
  @param gchar *location of the fileto be played  
  @return gboolean TRUE on success FALSE on failure
*/
static gboolean activate_engine_play_video(GpsCameraEngine *self,gchar *file_location);

/**
  This function is callback for all messages/warning/errors placed on the bus
  @param bus pointer of type GstBus 
  @param message pointer of type GstMessage 
  @param self pointer to engine object 
  @return GstBusSyncReply 
*/
static GstBusSyncReply bus_sync_cb (GstBus * b, GstMessage * message, 
						GpsCameraEngine *self);

/**
  This function sets the pipeline to NULL and waits until the state
  change and emits error in capture. 
  @param self pointer to engine object
  @return gboolean 
*/
static gboolean error_snapshot(GpsCameraEngine *self);

/* GObject parent class */
static GObjectClass *parent_class;

/* Signals */
static guint gpscamera_engine_signals[LAST_SIGNAL] = { 0 };

/**
  This function is part of GObject implementation to determine the type 
  @param void 
  @return GType 
*/
GType gpscamera_engine_get_type (void)
{
        static GType type = 0;

        if (! type) {
                	static const GTypeInfo type_info = {
			sizeof (GpsCameraEngineClass),
			NULL,
			NULL,
			(GClassInitFunc) gpscamera_engine_class_init,
			NULL,
			NULL,
			sizeof (GpsCameraEngine),
			0,
			(GInstanceInitFunc) gpscamera_engine_init
		};

		type = g_type_register_static (G_TYPE_OBJECT,
					       "GpsCameraEngine",
					       &type_info,
					       0);
        }

        return type;
}

/**
  This function is part of GObject implementation to initialize the CameraEngineClass 
  @param class pointer of type CameraEngineClass 
  @return void 
*/
static void
gpscamera_engine_class_init (GpsCameraEngineClass *class)
{
        GObjectClass *gobject_class = G_OBJECT_CLASS (class);

        parent_class = G_OBJECT_CLASS(g_type_class_peek_parent (class));

	g_type_class_add_private(class, sizeof(GpsCameraEnginePrivate));

	gpscamera_engine_signals[START] =
                g_signal_new ("start",
			      G_TYPE_FROM_CLASS (class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GpsCameraEngineClass, start),
			      NULL, NULL,
			      gpscamera_engine_marshal_VOID__VOID,
			      G_TYPE_NONE, 
			      0);
	gpscamera_engine_signals[DONE] =
                g_signal_new ("done",
			      G_TYPE_FROM_CLASS (class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GpsCameraEngineClass, done),
			      NULL, NULL,
			      gpscamera_engine_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
	gpscamera_engine_signals[CAPTURED] =
		g_signal_new ("captured",
			      G_TYPE_FROM_CLASS (class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GpsCameraEngineClass, captured),
			      NULL, NULL,
			      gpscamera_engine_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
	gpscamera_engine_signals[ERRORCAPTURE] =
		g_signal_new ("errorcapture",
			      G_TYPE_FROM_CLASS (class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GpsCameraEngineClass, errorcapture),
			      NULL, NULL,
			      gpscamera_engine_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
	     
	gobject_class->finalize = gpscamera_engine_shutdown;
	class->start = NULL;
	class->done = NULL;
	class->captured = NULL;
	
}

/**
  This function is part of GObject implementation to initialize the 
  GpsCameraEngine Object. It is called whenever a new engine object is created 
  @param class pointer of type GpsCameraEngine
  @return void 
*/
static void gpscamera_engine_init (GpsCameraEngine *self)
{

	GpsCameraEnginePrivate *priv = NULL;

	CE_ULOG_DEBUG("%s\n",__FUNCTION__);

	g_return_if_fail (CAMERA_IS_ENGINE (self));

	priv = CAMERA_ENGINE_GET_PRIVATE(self);

	priv->pipeline 	= NULL;
	priv->video_source 	= NULL;
	priv->video_converter = NULL;
	priv->video_sink 	= NULL;
        priv->filesink= NULL;
        priv->tee= NULL;
	priv->bus= NULL;
        priv->pipeline_error	= PIPELINE_CREATION_ERROR;
	priv->overlay_window_id  = 0;
	priv->error		= NULL;
	priv->state		= -1;
	priv->takesnap = FALSE;
	priv->filename = NULL;

}

/**
  This function is part of GObject implementation to deinitialize the 
  CameraEngine object. It is called whenever the CameraEngine object is unref'ed
  @param class pointer of type CameraEngine
  @return void 
*/
static void gpscamera_engine_shutdown (GObject *object)
{
	GpsCameraEngine *self=NULL;
	GpsCameraEnginePrivate *priv = NULL; 

	CE_ULOG_DEBUG("Starts %s\n",__FUNCTION__);

        g_return_if_fail (CAMERA_IS_ENGINE (object));
  
	self = CAMERA_ENGINE (object);

	priv = CAMERA_ENGINE_GET_PRIVATE(self);

	if(priv && priv->pipeline)
	{
		gst_element_set_state(priv->pipeline,GST_STATE_NULL);
              	gst_object_unref(priv->pipeline);
        	priv->pipeline        = NULL;
	}

        priv->video_source    = NULL;
        priv->video_sink      = NULL;
        priv->filesink= NULL;
        priv->tee= NULL;
        priv->bus= NULL;
        priv->pipeline_error  = PIPELINE_CREATION_ERROR;
        priv->overlay_window_id  = 0;
        priv->state  = -1;

	gpscamera_engine_set_error(self,NULL);

        if (G_OBJECT_CLASS (parent_class)->finalize)
                G_OBJECT_CLASS (parent_class)->finalize (object);

	CE_ULOG_DEBUG("Ends %s\n",__FUNCTION__);

}

/**
  This function creates the necccessary elements for recording/playback 
  @param self pointer of type CameraEngine
  @param action of type CameraEngineAction
  @return gboolean TRUE on success FALSE on failure 
*/
gboolean gpscamera_engine_initialize(GpsCameraEngine *self,GpsCameraEngineAction action)
{
	/* Is this right way to handle x errors ?*/
	GpsCameraEnginePrivate *priv = NULL;

	CE_ULOG_DEBUG("Starts %s\n",__FUNCTION__);

	priv = CAMERA_ENGINE_GET_PRIVATE(self);
	
	if(priv->pipeline)
	{
		CE_ULOG_DEBUG("Pipeline is already present.  Pls unref the engine and create fresh\n");
		return TRUE;
	}

        priv->pipeline = gst_pipeline_new(ENGINEPIPELINE);

        if(!priv->pipeline)
        {
                CE_ULOG_ERR("Unable to create pipeline\n");
                priv->pipeline_error = PIPELINE_CREATION_ERROR;
                return FALSE;
        }
	
	
	if(action == GPSCAMERA_ENGINE_IDLE_VIEW)
	{

	        priv->video_source = gst_element_factory_make(GCONFV4L2SRC,VIDEO_SOURCE);
			
	        if(!priv->video_source){
	        	priv->video_source = gst_element_factory_make(V4L2SRC,VIDEO_SOURCE);
		}
			
		if(!priv->video_source)
	        {
	           	CE_ULOG_ERR("Unable to create src\n");
		        priv->pipeline_error = PIPELINE_VIDEO_SRC_ERROR;
	        	        return FALSE;
	        }
	        if(!gst_bin_add(GST_BIN(priv->pipeline),priv->video_source))
		{
	                CE_ULOG_ERR("Unable to add video source to pipeline\n");
	                priv->pipeline_error = PIPELINE_CREATION_ERROR;
			gst_object_unref(priv->video_source);
			return FALSE;
		}
	        priv->tee = gst_element_factory_make(TEE,TEE);
		if(priv->tee == NULL)
	        {
	                CE_ULOG_ERR("Unable to create the tee element\n");
		        priv->pipeline_error = PIPELINE_ELEMENT_CREATION_ERROR;
	        	return FALSE;
	        }
		if(!gst_bin_add(GST_BIN(priv->pipeline),priv->tee))
		{
	                CE_ULOG_ERR("Unable to add tee to pipeline\n");
	                priv->pipeline_error = PIPELINE_CREATION_ERROR;
			gst_object_unref(priv->tee);
			return FALSE;
		}
	        
	}
        priv->video_converter = gst_element_factory_make(FFMPEGCOLORSPACE,VIDEO_CONVERTER);
        if(!priv->video_converter)
        {
                CE_ULOG_ERR("Unable to create video_converter\n");
                priv->pipeline_error = PIPELINE_ELEMENT_CREATION_ERROR;
                return FALSE;
        }
        if(!gst_bin_add(GST_BIN(priv->pipeline),priv->video_converter))
	{
               	CE_ULOG_ERR("Unable to add video converter to pipeline\n");
               	priv->pipeline_error = PIPELINE_CREATION_ERROR;
		gst_object_unref(priv->video_converter);
		return FALSE;
	}

        priv->video_sink = gst_element_factory_make(XVIMAGESINK,VIDEO_SINK);
        if(!priv->video_sink)
        {
                CE_ULOG_ERR("Unable to create video_sink\n");
                priv->pipeline_error = PIPELINE_ELEMENT_CREATION_ERROR;
                return FALSE;
        }
	g_object_set(G_OBJECT(priv->video_sink),"sync",FALSE,NULL);
        if(!gst_bin_add(GST_BIN(priv->pipeline),priv->video_sink))
	{
               	CE_ULOG_ERR("Unable to add video sink to pipeline\n");
               	priv->pipeline_error = PIPELINE_CREATION_ERROR;
		gst_object_unref(priv->video_sink);
		return FALSE;
	}
	gst_x_overlay_prepare_xwindow_id (GST_X_OVERLAY(priv->video_sink));

       	priv->bus = gst_pipeline_get_bus (GST_PIPELINE (priv->pipeline));
        if(!priv->bus)
        {
                CE_ULOG_ERR("Unable to get bus\n");
		priv->pipeline_error = PIPELINE_BUS_ERROR;
		return FALSE;
        }
        gst_bus_set_sync_handler (priv->bus, (GstBusSyncHandler)bus_sync_cb, self);

	return TRUE;
}      
	
/**
  This function creates the engine object 
  @param void 
  @return engine object  
*/
GpsCameraEngine *gpscamera_engine_new (void)
{
	CE_ULOG_DEBUG("%s\n",__FUNCTION__);

	return  CAMERA_ENGINE (g_object_new (CAMERA_TYPE_ENGINE, NULL));
}

/**
  This function creates the engine object sets the xid and creates the pipeline
  depending on the action 
  @param void 
  @return engine object  
*/
GpsCameraEngine *gpscamera_engine_init_with_xid_action (XID xid,GpsCameraEngineAction action)
{

	GpsCameraEngine *engine=NULL;
	GpsCameraEnginePrivate *priv = NULL;

	CE_ULOG_DEBUG("Starts %s\n",__FUNCTION__);

	engine = CAMERA_ENGINE (g_object_new (CAMERA_TYPE_ENGINE, NULL));
	if(!engine)
	{
		ULOG_CRIT_L("Unable to create the engine object\n");
		return NULL;
	}

	if(!gpscamera_engine_initialize(engine,action))
	{
		ULOG_CRIT_L("Unable to initialize the engine object\n");
		gpscamera_engine_unref(engine);
		engine=NULL;
		return NULL;
	}

	priv = CAMERA_ENGINE_GET_PRIVATE(engine);

	if(!priv)
	{
		CE_ULOG_ERR("Unable to get priv\n");
		return;
	}

	priv->overlay_window_id=xid;

	CE_ULOG_DEBUG("Ends %s\n",__FUNCTION__);
	return engine;
}

/**
  This function is callback for all messages/warning/errors placed on the bus
  @param bus pointer of type GstBus 
  @param message pointer of type GstMessage 
  @param self pointer to engine object 
  @return GstBusSyncReply 
*/
static GstBusSyncReply
bus_sync_cb (GstBus * bus, GstMessage * message, GpsCameraEngine *self)
{
   GError *error = NULL;
   gchar *name = NULL;
   GpsCameraEnginePrivate *priv = NULL;
   const GstStructure *mess_str = NULL;

   g_return_val_if_fail (CAMERA_IS_ENGINE (self), -1);

   priv = CAMERA_ENGINE_GET_PRIVATE(self);

   switch(GST_MESSAGE_TYPE(message))
   {
      case GST_MESSAGE_STATE_CHANGED : 
		name = g_strdup(gst_element_get_name(message->src));
		CE_ULOG_DEBUG("%s-%s\n", name, 
		gst_element_state_get_name(GST_STATE(message->src)));
		if(!strcmp(name,ENGINEPIPELINE) && 
			(GST_STATE(message->src) == GST_STATE_PLAYING))
	 	{
		    CE_ULOG_DEBUG("%s State has changed\n",name);
		    g_signal_emit(G_OBJECT(self), gpscamera_engine_signals[DONE], 
				    0,NULL);
	 	}
	 	g_free(name);
		break;
	 
	 case GST_MESSAGE_ERROR	: 	
		gst_message_parse_error (message, &error, NULL);
		if(error)
	        {
	        	CE_ULOG_ERR("%d-%s-%d\n", error->code, error->message, 
					error->domain);
		       gpscamera_engine_set_error(self, error);
		       g_signal_emit(G_OBJECT(self), 
				       gpscamera_engine_signals[ERROR],0,NULL);
		       g_error_free(error);
		       error=NULL;
	    	}
	    	break;
		
	default		:	
		CE_ULOG_DEBUG("Not handling this message %d\n",
		GST_MESSAGE_TYPE(message));
	    break;
   }

   mess_str = gst_message_get_structure (message);

   if(mess_str) {
      if (!gst_structure_has_name (mess_str, "prepare-xwindow-id")) {
         return GST_BUS_PASS;
      }
   }
   
   if(priv->video_sink) {
      if (!GST_IS_X_OVERLAY (priv->video_sink)) {
         CE_ULOG_ERR("Sink is not set to overlaying\n");
      } else {
         CE_ULOG_DEBUG("Overlaying\n");
         gst_x_overlay_set_xwindow_id(GST_X_OVERLAY(priv->video_sink), 
			 priv->overlay_window_id);
	 gst_x_overlay_expose (GST_X_OVERLAY(priv->video_sink));
	 gst_message_unref(message);
	 return GST_BUS_DROP;
      }        
   }

   return GST_BUS_PASS;
}

/**
  This function sets the error whever there is error message on the bus
  @param self pointer to engine object
  @param error pointer of type GError 
  @return void 
*/
static void gpscamera_engine_set_error(GpsCameraEngine *self,  GError *error)
{
	GpsCameraEnginePrivate *priv = NULL; 

	CE_ULOG_DEBUG("Starts %s\n",__FUNCTION__);

	g_return_if_fail(CAMERA_IS_ENGINE(self));

	priv = CAMERA_ENGINE_GET_PRIVATE(self);

	if(error == NULL) {
		if(priv->error!=NULL) {
			g_error_free(priv->error);
		}
		return;
	}
	if(priv->error) {
		g_error_free(priv->error);
	}
	priv->error=g_error_copy(error);
	CE_ULOG_DEBUG("Ends %s\n",__FUNCTION__);
}

static gboolean snapshot_data(GstPad * pad, GstBuffer * buffer, gpointer data)
{
	 GpsCameraEnginePrivate *priv = NULL;
	 struct jpeg_compress_struct cinfo;
         struct jpeg_error_mgr jerr;
         int i, width, height;
         unsigned char *line;
         int byte_pixel;
         FILE *outfile;
         char tempimg[640*480*3], *temppix, *tmpin, y1, y2, u, v;
		 
	 priv = CAMERA_ENGINE_GET_PRIVATE((GpsCameraEngine *)data);
	
	 if(priv->takesnap == FALSE)
		 return TRUE;
	 priv->state = ENGINE_TAKE_PHOTO;
         
         /*dirty hack for playing the capture sound
          * still not consistent need to find a better way */
         gst_element_set_state(priv->pipeline,GST_STATE_PAUSED);
							 
	 priv->takesnap = FALSE;
	  width = 640;
          height = 480;
          cinfo.err = jpeg_std_error(&jerr);
          jpeg_create_compress(&cinfo);
		                                                                                                                               
          if ((outfile = fopen(priv->filename, "wb")) == NULL) {
                  CE_ULOG_ERR("can't open %s\n", priv->filename);
	 	  g_idle_add((GSourceFunc)error_snapshot,data);
                  return TRUE;
          }

         jpeg_stdio_dest(&cinfo, outfile);
         cinfo.image_width = width;
         cinfo.image_height = height;
         cinfo.input_components = 3;
         cinfo.in_color_space = JCS_YCbCr;
         jpeg_set_defaults(&cinfo);
         jpeg_set_quality(&cinfo, 100, TRUE);
								                                                                                                                               
         temppix=tempimg;
         tmpin=(unsigned char *) GST_BUFFER_DATA(buffer);;
         for(i=0; i < (640*480/2) ; i++)
         {
		  u  = *tmpin++;
                  y1 = *tmpin++;
                  v  = *tmpin++;
                  y2= *tmpin++;
                  *temppix++=y1;
                  *temppix++=u;
                  *temppix++=v;
                  *temppix++=y2;
                  *temppix++=u;
                  *temppix++=v;
         }
                                                                                                               
         jpeg_start_compress(&cinfo, TRUE);
         byte_pixel = 3;
         for (i = 0, line = tempimg; i < height;
              i++, line += width * byte_pixel) {
                jpeg_write_scanlines(&cinfo, &line, 1);
         }
         jpeg_finish_compress(&(cinfo));
         fclose(outfile);
	 jpeg_destroy_compress(&(cinfo));
	 g_idle_add((GSourceFunc)snapshot,data);
		 
	 return TRUE;														  
}

/**
  This function links all the elements which are required for idleview 
  @param self pointer to engine object
  @return gboolean TRUE on success FALSE on failure 
*/
static gboolean gpsengine_idle_view(GpsCameraEngine *self)
{
	GpsCameraEnginePrivate *priv = NULL;
	GstPad *pad = NULL;
	GstCaps *caps = NULL;

	CE_ULOG_DEBUG("Starts %s\n",__FUNCTION__);

	g_return_val_if_fail (CAMERA_IS_ENGINE (self),FALSE);

	priv = CAMERA_ENGINE_GET_PRIVATE(self);

	if(!priv)
	{
		CE_ULOG_ERR("Unable to get priv \n"); 
		return FALSE;
	}

	caps = gst_caps_new_simple("video/x-raw-yuv",
                                   "framerate", GST_TYPE_FRACTION, 10, 1,
				   "width", G_TYPE_INT, 640,
				   "height", G_TYPE_INT, 480,
                                    NULL);
	if(!gst_element_link_filtered(priv->video_source,
				  priv->tee,caps))
	{
		CE_ULOG_ERR("Unable to link video_source,tee caps\n");
		return FALSE;
	}
	if(!gst_element_link_many(priv->tee,priv->video_sink,NULL))
	{
		CE_ULOG_ERR("Unable to link tee, video_sink_queue,video_sink\n");
		return FALSE;
	}
	
	pad = gst_element_get_pad(GST_ELEMENT(priv->video_source), "src");
	gst_pad_add_buffer_probe(pad, G_CALLBACK(snapshot_data),self);
	gst_object_unref(pad);
	
	CE_ULOG_DEBUG("Ends %s\n",__FUNCTION__);
	return TRUE;
}


/**
  This function sets the pipeline to NULL and waits until the state
  change and emits error in capture. 
  @param self pointer to engine object
  @return gboolean 
*/
static gboolean error_snapshot(GpsCameraEngine *self)
{
	GpsCameraEnginePrivate *priv = NULL;

	CE_ULOG_DEBUG("Starts %s\n",__FUNCTION__);

	g_return_val_if_fail (CAMERA_IS_ENGINE (self),FALSE);
	
	priv = CAMERA_ENGINE_GET_PRIVATE(self);

	if(!priv)
	{
		CE_ULOG_ERR("Unable to get priv\n");
		return FALSE;
	}

	g_signal_emit(G_OBJECT(self),gpscamera_engine_signals[ERRORCAPTURE],0,NULL);

	CE_ULOG_DEBUG("Error capturing trying to set idleview\n");
	gst_element_set_state(priv->pipeline,GST_STATE_PLAYING); 
	gst_element_get_state(priv->pipeline,NULL,NULL,GST_CLOCK_TIME_NONE);
	priv->state = ENGINE_IDLEVIEW;
	g_free(priv->filename);
	CE_ULOG_DEBUG("Idleview is set\n");

	CE_ULOG_DEBUG("Ends %s\n",__FUNCTION__);
	return FALSE;
}

/**
  This function sets the pipeline to NULL and waits until the state. 
  @param self pointer to engine object
  @return gboolean 
*/
static gboolean snapshot(GpsCameraEngine *self)
{
	GpsCameraEnginePrivate *priv = NULL;

	CE_ULOG_DEBUG("Starts %s\n",__FUNCTION__);

	g_return_val_if_fail (CAMERA_IS_ENGINE (self),FALSE);
	
	priv = CAMERA_ENGINE_GET_PRIVATE(self);

	if(!priv)
	{
		CE_ULOG_ERR("Unable to get priv\n");
		return FALSE;
	}

	g_signal_emit(G_OBJECT(self),gpscamera_engine_signals[CAPTURED],0,NULL);

	CE_ULOG_DEBUG("After capturing trying to set idleview\n");
	gst_element_set_state(priv->pipeline,GST_STATE_PLAYING); 
	gst_element_get_state(priv->pipeline,NULL,NULL,GST_CLOCK_TIME_NONE);
	priv->state = ENGINE_IDLEVIEW;
	g_free(priv->filename);
	CE_ULOG_DEBUG("Idleview is set\n");

	CE_ULOG_DEBUG("Ends %s\n",__FUNCTION__);
	return FALSE;
}

/**
  This function sets the camera engine to different states. The second parameter 
  CameraEngineAction value specifies the state Idle/snapshot.  Depending on it 
  different activate functions are called.The third parameter specifies the location.
  @param self pointer to engine object
  @param action of type CameraEngineAction 
  @param location pointer of type gchar 
  @return gboolean TRUE on success FALSE on failure 
*/
gboolean gpscamera_engine_set_state(GpsCameraEngine *self, GpsCameraEngineAction action, 
							gchar *location)
{
	GpsCameraEnginePrivate *priv = NULL;
	GstStateChangeReturn state_change;

	CE_ULOG_DEBUG("Starts %s\n",__FUNCTION__);

	g_return_val_if_fail (CAMERA_IS_ENGINE (self),FALSE);

	priv = CAMERA_ENGINE_GET_PRIVATE(self);

	if(!priv)
	{
		CE_ULOG_ERR("Unable to get priv\n");
		return FALSE;
	}

	priv->action=action;

	g_signal_emit(G_OBJECT(self),gpscamera_engine_signals[START],0,NULL);
	switch(action)
	{
	  	case GPSCAMERA_ENGINE_IDLE_VIEW 	:    	
			
			if(!gpsengine_idle_view(self))
			{
				CE_ULOG_ERR("Unable to link the idle view pipeline\n");
				return FALSE;
			}

			state_change = gst_element_set_state(priv->pipeline,GST_STATE_PLAYING);

			//This is required!
			gst_element_get_state(priv->pipeline, NULL, NULL, GST_CLOCK_TIME_NONE);
			CE_ULOG_DEBUG("The state change is %d\n",state_change);

			if(state_change == GST_STATE_CHANGE_FAILURE)
			{
				CE_ULOG_ERR("Unable to show idleview\n");
				CE_ULOG_ERR("Ends %s by returning false\n",__FUNCTION__);
				return FALSE;
			}
			CE_ULOG_DEBUG("Ends %s by returning true\n",__FUNCTION__);
			priv->pipeline_error = PIPELINE_CREATION_SUCCESS;
			return TRUE;
			break;
		
	  	case GPSCAMERA_ENGINE_SNAPSHOT:    	
			
			priv->state = ENGINE_IDLEVIEW;
			priv->filename = g_strdup(location);
			priv->takesnap = TRUE;
			return TRUE;
			break;

		
	  	default			:
					CE_ULOG_ERR("Action not supported\n");
					break;
	}
	CE_ULOG_DEBUG("Ends %s\n",__FUNCTION__);	
}

/**
  This function unrefs the engine object 
  @param pointer to CameraEngine  
  @return void 
*/
void gpscamera_engine_unref(GpsCameraEngine *engine)
{
	CE_ULOG_DEBUG("Starts %s\n",__FUNCTION__);

	g_return_if_fail (CAMERA_IS_ENGINE (engine));
	if(engine != NULL) {
	        g_object_unref(engine);
		engine=NULL;
	}
	CE_ULOG_DEBUG("Ends %s\n",__FUNCTION__);
}
