/**
 * gps-camera-controller.c
 * Copyright (C) 2007 Sanna Salmijarvi (ssalmija@gmail.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */

/*Standard header files */
#include <gdk/gdkkeysyms.h>
#include <gst/gst.h>
#include <gtk/gtkcombobox.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-window.h>
#include <libosso.h>
#include <string.h>
#include <pthread.h>
#include <location/location-gpsd-control.h>

/*User defined header files */
#include "gps-camera-common.h"
#include "gps-camera-controller.h"
#include "gps-camera-define.h"
#include "gps-camera-engine.h"
#include "gps-camera-ui.h"
#include "gps-camera-gallery-view.h"
#include "gps-camera-states.h"
#include "gps-camera-flickr.h"
#include "gps-camera-gps.h"
#include "gps-camera-widget.h"

#include <curl/curl.h>

/* Static Functions defs */

/**
  Function called to show appropriate
  error string to the user and make required
  ui changes.
  @param camerawindow is the pointer to the CameraView structure.
  @return None. 
 */
static void gpscamera_engine_error(CameraView * camerawindow);


static gboolean gpscamera_action_close(GtkWidget * widget, 
		GdkEvent * event, gpointer data);
/**
  Function called to run the login dialog.
  @param settings is the pointer to the settings
  dialog  structure.
  @param setting is a pointer to saved settings structure.
  @param uiwindow is a pointer to application window.
  @return TRUE on success else FALSE.
 */
static gboolean
gpscamera_login_dialog_run(CameraLoginSettings * settings,
			   CameraView * camerawindow)
{	
	gint response = -1;
	gchar *full_token = NULL;
	gchar *minitoken = NULL;
	gchar *perms = NULL;

	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);
	g_return_val_if_fail(camerawindow != NULL,TRUE);

	response = gtk_dialog_run(GTK_DIALOG(settings->login_dialog));
	if (response == GTK_RESPONSE_CANCEL) {
		gtk_widget_hide(settings->login_dialog);
		return TRUE;

	}
	
	if (response == GTK_RESPONSE_OK) {

		gtk_widget_hide(settings->login_dialog);
				
		minitoken = 
			g_strdup(gtk_entry_get_text(GTK_ENTRY(settings->minitoken)));  
		if(minitoken!=NULL){
			/* Initialise the Flickcurl library */
			gpscamera_flickr_init();	 	
  	  		camerawindow->gpscamera_flickr=gpscamera_flickr_new();
				
		  	gpscamera_flickr_set_shared_secret
					(camerawindow->gpscamera_flickr, SECRET);
			gpscamera_flickr_set_api_key
					(camerawindow->gpscamera_flickr, API_KEY);
		  	
		 	full_token = gpscamera_flickr_auth_getFullToken
					(camerawindow->gpscamera_flickr, minitoken);
			if(!full_token)
				return FALSE;
			
		  	perms = gpscamera_flickr_auth_checkToken
					(camerawindow->gpscamera_flickr, full_token);

			/*check if perms are write*/
		  	if(strcmp( perms ,"write")==0){
		   		gpscamera_flickr_set_auth_token
					(camerawindow->gpscamera_flickr, full_token);	  
		   		gpscamera_flickr_set_write
					(camerawindow->gpscamera_flickr, 1);
				return TRUE;
										
			} else {
				return FALSE;
			}
		} else{
			return FALSE;
		}
			
	}
	
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	
}

void gpscamera_show_info_note(GtkWindow *window, const gchar *text)
{
	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);
	g_return_if_fail(window != NULL);
	g_return_if_fail(text != NULL);
	hildon_banner_show_information(GTK_WIDGET(window),NULL,text);
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
}

gboolean gpscamera_handler_check(CameraView * camerawindow)
{

	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);

	if (camerawindow == NULL) {
		return FALSE;
	}
	if(FALSE == gpscamera_handle_accquired(camerawindow)) {
		gpscamera_engine_error(camerawindow);
		return FALSE;
	}

	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	
	return TRUE;
	
}

static void gpscamera_engine_error(CameraView * camerawindow)
{
	gchar *string = NULL;

	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);

	g_return_if_fail(camerawindow != NULL);
	
	gpscamera_engine_clear_engine(camerawindow->engine);
	camerawindow->engine = NULL;
	camerawindow->last_state = camerawindow->current_state;
	camerawindow->current_state = GPSCAMERA_STATE_CAMERA;
	gpscamera_states_check(camerawindow);
	gpscamera_handler_check(camerawindow);
	string = (_("gpscamera error take photo generic"));

	if (string != NULL) {
		gpscamera_show_info_note(GTK_WINDOW(camerawindow->uiwindow), string);
	}
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
}

gboolean gpscamera_handle_accquired(CameraView * camerawindow)
{
	XID xid = 0;

	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);

	g_return_val_if_fail(camerawindow != NULL,FALSE);

	xid = gpscamera_widget_get_xid(GPSCAMERA_WIDGET(camerawindow->cameradisplay));

	camerawindow->engine = gpscamera_engine_init_with_xid_action(xid,
							GPSCAMERA_ENGINE_IDLE_VIEW);
	if(camerawindow->engine == NULL) {
		osso_log(LOG_ERR, "Camera engine init fail");
		return FALSE;
	}

	if(gpscamera_engine_set_state(camerawindow->engine, 
				GPSCAMERA_ENGINE_IDLE_VIEW, NULL) == FALSE) {
		osso_log(LOG_ERR, "Unable to set state to streaming");
		gpscamera_engine_clear_engine(camerawindow->engine);
		camerawindow->engine = NULL;
		return FALSE;
	}	

	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	return TRUE;
}

gboolean gpscamera_activate_state_camera(GtkWidget * widget, 
		GdkEvent * event, gpointer data)
{
	CameraView *camerawindow = NULL;
	camerawindow = (CameraView *) data;

	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);
	
	g_return_val_if_fail(camerawindow != NULL,FALSE);

	camerawindow->last_state = camerawindow->current_state;
	camerawindow->current_state = GPSCAMERA_STATE_CAMERA;

	gpscamera_states_check(camerawindow);
	
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	return FALSE;
}
static gboolean gpscamera_state_browsed_set(CameraView * camerawindow)
{
	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);
	g_return_val_if_fail(camerawindow != NULL,FALSE);
	
	if(camerawindow->current_state == GPSCAMERA_STATE_BROWSE_LEFT){
		gtk_widget_hide(camerawindow->radial_browse_left_eventbox);	
	}
	if(camerawindow->current_state == GPSCAMERA_STATE_BROWSE_RIGHT){
		gtk_widget_hide(camerawindow->radial_browse_right_eventbox);	
	}

	if(camerawindow->current_state = GPSCAMERA_STATE_SNAPSHOT_TAKEN){
		gtk_widget_hide(camerawindow->radial_snapshot_eventbox2);
	}
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);

	return FALSE;
}

gboolean gpscamera_activate_state_snapshot(GtkWidget * widget, 
		GdkEvent * event, gpointer data)
{
	CameraView *camerawindow = NULL;
	gchar *file_path = NULL;
	gint ret_val = -1;
	GpsCameraCoordinates coords = {0}; 
	gchar *info = NULL;
	
	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);

	camerawindow = (CameraView *) data;
	g_return_val_if_fail(camerawindow != NULL,FALSE);

	gtk_widget_show(camerawindow->radial_snapshot_eventbox2);
	

	file_path = (gchar *) gpscamera_get_file_path(camerawindow);

	if (file_path == NULL) {
		return FALSE;
	}

	if(camerawindow->gps==TRUE){
		coords = gpscamera_ask_position(camerawindow);
		if((coords.lat != 0.0) && (coords.lng != 0.0)){
			if(gpscamera_write_gpslocation_file(file_path, 
						coords.lat, coords.lng)==TRUE){
				printf("GPS saved succesful\n");	
			}				
		}
	}			

	if (file_path != NULL) {
	ret_val= (gpscamera_engine_set_state
			(camerawindow->engine, GPSCAMERA_ENGINE_SNAPSHOT, file_path));
	}

	if (ret_val == FALSE) {
		gpscamera_engine_error(camerawindow);
		g_free(file_path);
		return FALSE;
	}
	info = g_strdup_printf("Photo saved as %s", file_path);
	gpscamera_show_info_note(GTK_WINDOW(camerawindow->uiwindow), 
					_(info));	
		
	camerawindow->current_state = GPSCAMERA_STATE_SNAPSHOT_TAKEN;
	g_timeout_add(100,gpscamera_state_browsed_set, camerawindow);	
		
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	g_free(file_path);
	return FALSE;
}



gboolean gpscamera_activate_login(GtkWidget * widget, gpointer data)
{
	CameraView *camerawindow = NULL;
	gboolean ret_val = FALSE;

	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);

	camerawindow = (CameraView *) data;
	g_return_if_fail(camerawindow != NULL);
	
	if ((camerawindow->settings == NULL)) {
		camerawindow->settings = (CameraLoginSettings *) 
						g_new0(CameraLoginSettings, 1);
		gpscamera_login_dialog_create(camerawindow->settings,
			                	camerawindow->uiwindow);

	}
	
	ret_val =
		    gpscamera_login_dialog_run(camerawindow->settings,
					      camerawindow);
	
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	return ret_val;
}

gboolean gpscamera_action_upload(CameraView * camerawindow)
{
	
	gint response = 0;
	GtkWidget *dialog = NULL;
	gchar *image_path = NULL;
	gboolean login = FALSE;
	
	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);

	g_return_val_if_fail(camerawindow != NULL,FALSE);

	dialog = gpscamera_select_uploading_way_dialog(camerawindow);
	response = gtk_dialog_run(GTK_DIALOG(dialog));
	if (response == GTK_RESPONSE_OK) {
		
		/*get image path which is wanted to upload*/
		image_path = g_list_nth_data(camerawindow->photos, 
						camerawindow->counter-1);
		if (!image_path) {
			return FALSE;
		}
	
		if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON
					(camerawindow->radio[0]))==TRUE) {
		
			gtk_widget_destroy(GTK_WIDGET(dialog));
			/*if user hasn't yet login*/
			if(camerawindow->gpscamera_flickr == NULL){
			login = gpscamera_activate_login(NULL, camerawindow);
	
				if(login==TRUE){
					gpscamera_initiate_upload(camerawindow, 
								image_path);
				}else{
					gpscamera_show_info_note(GTK_WINDOW
						(camerawindow->uiwindow), 
				_("Login fail!\n Minitoken is wrong or missing."));
					if(camerawindow->gpscamera_flickr !=NULL){
						gpscamera_flickr_free(
						  camerawindow->gpscamera_flickr);
						camerawindow->gpscamera_flickr=NULL;
					}	
				}
			}else{
				gpscamera_initiate_upload(camerawindow, image_path);
			}
			
		}
		if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON
						(camerawindow->radio[1]))==TRUE) {
			gtk_widget_destroy(GTK_WIDGET(dialog));
			gpscamera_send_mail(camerawindow);
					
		}
		
	} 
	if (response == GTK_RESPONSE_CANCEL) {
		gtk_widget_destroy(GTK_WIDGET(dialog));
	}
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	return FALSE;

}

gboolean gpscamera_activate_action_upload(GtkWidget * widget, 
		GdkEvent * event, gpointer data)
{

	CameraView *camerawindow = NULL;
	
	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);

	camerawindow = (CameraView *) data;
	g_return_val_if_fail(camerawindow != NULL,FALSE);

	gtk_widget_show(camerawindow->hbox_upload2);
	if(camerawindow->photo_amount==0){
		gpscamera_show_info_note(GTK_WINDOW(camerawindow->uiwindow), 
					_("No photos, so you can't upload"));
	}else{
		camerawindow->current_state = GPSCAMERA_STATE_UPLOAD;
		gpscamera_states_check(camerawindow);	
	}
	gtk_widget_hide(camerawindow->hbox_upload2);

	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	return FALSE;
	
}

gboolean gpscamera_activate_action_delete(GtkWidget * widget, 
		GdkEvent * event, gpointer data)
{
	CameraView *camerawindow = NULL;
	gboolean ret_val = TRUE;

	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);

	camerawindow = (CameraView *) data;
	g_return_val_if_fail(camerawindow != NULL,FALSE);
	if(camerawindow->photo_amount==0){
		gpscamera_show_info_note(GTK_WINDOW(camerawindow->uiwindow), 
					_("No photos, so you can't delete"));
	}else{
		camerawindow->current_state = GPSCAMERA_STATE_DELETE;
		gpscamera_states_check(camerawindow);
	}
	
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	return FALSE;
}

/* Function to create Gps-Camera UI */
gboolean gpscamera_launch(osso_context_t * context)
{
	GtkWidget *gpscamera_view = NULL;
	CameraView *camerawindow = NULL;
	GtkMenu *menu = NULL;
	LocationGPSDControl *control;
			
	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);

	
	/*Allocating memory ,will dealloc when application is closed/exit . */
	camerawindow = (CameraView *) g_new0(CameraView, 1);

	camerawindow->program = hildon_program_get_instance();
	camerawindow->uiwindow = hildon_window_new();
	hildon_program_add_window(camerawindow->program, 
			camerawindow->uiwindow);
	
	g_set_application_name("Gps-Camera");
	//camerawindow->gps = FALSE;

	gpscamera_view = (GtkWidget *) gpscamera_views_create(camerawindow);
	if (!gpscamera_view) {
		osso_log(LOG_WARNING, "View creation aborted \n");
		return FALSE;
	}
	
	gtk_container_add(GTK_CONTAINER(camerawindow->uiwindow), gpscamera_view);

	/*Allocating memory ,will dealloc when application
	   is closed/exit . */
	camerawindow->menuitem = (CameraMenu *) g_new0(CameraMenu, 1);

	menu = (GtkMenu *) camera_ui_create_menu(camerawindow->menuitem);
	if (!menu) {
		osso_log(LOG_WARNING, "Menu creation aborted \n");
		return FALSE;
	}
	hildon_window_set_menu(HILDON_WINDOW(camerawindow->uiwindow), menu);
	gtk_widget_show_all(GTK_WIDGET(menu));

	gtk_widget_show_all(GTK_WIDGET(camerawindow->uiwindow));
		
	/* Connect signal to X in the upper corner */
	g_signal_connect(G_OBJECT(camerawindow->uiwindow), "delete_event",
			 G_CALLBACK(gpscamera_action_close), camerawindow);
		
	
	
	g_type_init ();
	control = g_object_new (LOCATION_TYPE_GPSD_CONTROL, NULL);
	location_gpsd_control_start (control);
	camerawindow->device = g_object_new (LOCATION_TYPE_GPS_DEVICE, NULL);
	camerawindow->device->fix->mode=0;
	
	g_timeout_add(2000, set_update_location, camerawindow);

	camerawindow->context = context;
	
	camerawindow->current_state = GPSCAMERA_STATE_CAMERA;
	if (FALSE == gpscamera_states_check(camerawindow)) {
		return FALSE;
	}
	
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	return TRUE;	
}


static gboolean gpscamera_action_close(GtkWidget * widget, 
		GdkEvent * event, gpointer data)
{
	CameraView *camerawindow = NULL;

	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);

	camerawindow = (CameraView *) data;
	g_return_val_if_fail(camerawindow != NULL,FALSE);	
	
	if(camerawindow->gpscamera_flickr !=NULL){
		gpscamera_flickr_free(camerawindow->gpscamera_flickr);
  		gpscamera_flickr_finish(); 
	}
	camerawindow->gps = FALSE;
	gst_deinit();
	gtk_main_quit();
	
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	return TRUE;
}

/**
 This is the function where image and its gps data is uploaded to flickr
 @param camerawindow is the pointer to the CameraView structure.
 @param image_path is the gchar pointer to the image path to be uploaded.
 @param title is the char pointer to the images title.
 @param desc is the char pointer to the images description.
*/
void gpscamera_upload(CameraView * camerawindow, const gchar * image_path, 
	char *title, char *desc){

	GList *images =NULL;
	const gchar *lat = NULL;
	const gchar *lng = NULL;
	gpscamera_flickr_upload_status* photo = NULL;
	gpscamera_flickr_location* location = NULL;

	images = gpscamera_read_gpslocation_file(image_path);
		
				
	photo = gpscamera_flickr_photos_upload(camerawindow->gpscamera_flickr, 
					image_path, title, desc, NULL, 0, 0, 0);
	if(photo!=NULL){
		gpscamera_show_info_note(GTK_WINDOW(camerawindow->uiwindow), 
				_("Uploading of the photo to flickr succeed"));	
	} else {
		gpscamera_show_info_note(GTK_WINDOW(camerawindow->uiwindow), 
				_("Uploading  of the photo to flickr Fail"));
	}
	g_free(title);
	g_free(desc);

	if((g_list_length(images)!=0) && (photo !=NULL)){

		lat = g_list_nth_data(images, g_list_length(images)-2);
		lng = g_list_nth_data(images, g_list_length(images)-1);
		
		location = malloc(sizeof(gpscamera_flickr_location));	
		location->latitude = g_ascii_strtod(lat, NULL);
		location->longitude = g_ascii_strtod(lng, NULL);
		location->accuracy =0;
				
		if(gpscamera_flickr_photos_geo_setLocation(camerawindow->gpscamera_flickr, 
						photo->photoid, location)==0){
			gpscamera_show_info_note(GTK_WINDOW(camerawindow->uiwindow),
				_("Uploading of the GPS photo to flickr succeed"));
			
		} else{
			gpscamera_show_info_note(GTK_WINDOW(camerawindow->uiwindow), 
				_("Uploading of the GPS photo to flickr fail"));
			
		}
		gpscamera_flickr_free_location(location);
		location =NULL;
		g_list_free(images);		
	}	
}

void gpscamera_engine_clear_engine(GpsCameraEngine *engine)
{
	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);
	if(engine != NULL) {
		gpscamera_engine_unref(engine);
		engine = NULL;
	}
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
}

gboolean gpscamera_change_browse_mode(GtkWidget *event_box, 
                         GdkEventButton *event, gpointer data)
{

	CameraView *camerawindow = NULL;
	camerawindow = (CameraView *) data;

	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);	
	g_return_val_if_fail(camerawindow != NULL,FALSE);	

	if(event->x<76){
		gpscamera_engine_clear_engine(camerawindow->engine);
		camerawindow->engine = NULL;
		camerawindow->last_state = camerawindow->current_state;
		camerawindow->current_state = GPSCAMERA_STATE_GALLERY;
		gpscamera_states_check(camerawindow);
	}

	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	return TRUE;
}

gboolean gpscamera_change_snapshot_mode(GtkWidget *event_box, 
                         GdkEventButton *event, gpointer data)
{

	CameraView *camerawindow = NULL;

	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);

	camerawindow = (CameraView *) data;
	g_return_val_if_fail(camerawindow != NULL,FALSE);	

	if(event->x>76){
		camerawindow->current_state = GPSCAMERA_STATE_CAMERA;
	}

	if (FALSE == gpscamera_states_check(camerawindow)) {
		return FALSE;
	}
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	return TRUE;
}

gboolean gpscamera_activate_state_browse(GtkWidget *event_box, 
                         GdkEventButton *event, gpointer data)
{

	CameraView *camerawindow = NULL;
	
	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);

	camerawindow = (CameraView *) data;
	g_return_val_if_fail(camerawindow != NULL,FALSE);	

	if(event->x<76){
		camerawindow->current_state = GPSCAMERA_STATE_BROWSE_LEFT;
		gtk_widget_show(camerawindow->radial_browse_left_eventbox);
					
		gpscamera_gallery_view_image_prev(camerawindow);
		g_timeout_add(100,gpscamera_state_browsed_set,
                                                         camerawindow);		
	}				

	if(event->x>76){
		camerawindow->current_state = GPSCAMERA_STATE_BROWSE_RIGHT;
		gtk_widget_show(camerawindow->radial_browse_right_eventbox);
		
		gpscamera_gallery_view_image_next(camerawindow);
		
		g_timeout_add(100,gpscamera_state_browsed_set,
                                                         camerawindow);	
		
	}
		
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	return TRUE;
}

void cancel_selected(GtkWidget * widget, 
		GdkEvent * event, gpointer dialog){
	
	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);

	gtk_dialog_response(GTK_DIALOG( dialog), GTK_RESPONSE_CANCEL);

	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
}

void ok_selected(GtkWidget * widget, 
		GdkEvent * event, gpointer dialog){
	
	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);

	gtk_dialog_response(GTK_DIALOG( dialog), GTK_RESPONSE_OK);

	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
}
