/**
 * gps-camera-common.c
 * Copyright (C) 2007 Sanna Salmijarvi (ssalmija@gmail.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 **/

#include <glib.h>
#include <math.h>
#include <time.h>
#include <sys/stat.h>
# include "gps-camera-define.h"
# include "gps-camera-common.h"
# include "gps-camera-controller.h"


static gboolean gpscamera_check_directory(CameraView * camerawindow, const gchar * path);

gboolean gpscamera_check_file_exists(const gchar * file)
{
	GnomeVFSResult vfs_result = GNOME_VFS_OK;
	GnomeVFSFileInfo *file_info = NULL;
	GnomeVFSURI *dir_uri_str = NULL;

	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);
	if (file == NULL) {
		return FALSE;
	}

	dir_uri_str = gnome_vfs_uri_new(file);
	file_info = gnome_vfs_file_info_new();

	vfs_result = gnome_vfs_get_file_info_uri(dir_uri_str, file_info,
						 GNOME_VFS_FILE_INFO_DEFAULT);

	if (vfs_result != GNOME_VFS_OK) {
		gnome_vfs_uri_unref(dir_uri_str);
		gnome_vfs_file_info_unref(file_info);
		return FALSE;
	}

	gnome_vfs_uri_unref(dir_uri_str);
	gnome_vfs_file_info_unref(file_info);
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	return TRUE;
}


gchar *gpscamera_get_file_path(CameraView * camerawindow)
{
	gchar *location = NULL;
	gchar *apps_dir_path = NULL;
	gchar *apps_path = NULL;
	gchar *file_path = NULL;
	gchar *name = NULL;

	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);
		
	location = g_strdup(HOME_DIR);
	
	apps_dir_path = g_strdup_printf("%s%s%s", location, SLASH, ".images");
	if (gpscamera_check_directory(camerawindow, apps_dir_path) == FALSE) {
		osso_log(LOG_ERR, "Cannot create path in device\n");
		g_free(apps_dir_path);
		return NULL;
	}
		
	apps_path = g_strdup_printf("%s%s%s", apps_dir_path, SLASH, "gpscamera");
	if (gpscamera_check_directory(camerawindow, apps_path) == FALSE) {
		g_free(apps_dir_path);
		g_free(apps_path);
		return NULL;
	}
	
	while (1) {
		name = gpscamera_get_unique_name();
		file_path = g_strdup_printf("%s%s%s", apps_path, SLASH, name);
		g_free(name);
		if (gpscamera_check_file_exists(file_path) == FALSE) {
			break;
		}
		g_free(file_path);
	}
	g_free(apps_dir_path);
	g_free(apps_path);

	osso_log(LOG_DEBUG, "File path is:%s\n", file_path);
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);
	return file_path;
}

static
gboolean gpscamera_check_directory(CameraView * camerawindow, const gchar * path)
{
	GnomeVFSResult vfs_result = GNOME_VFS_OK;
	GnomeVFSFileInfo *file_info = NULL;
	GnomeVFSURI *dir_uri_str = NULL;

	osso_log(LOG_DEBUG, "Start Function-- %s --\n", __FUNCTION__);
	if (path == NULL) {
		return FALSE;
	}

	dir_uri_str = gnome_vfs_uri_new(path);
	file_info = gnome_vfs_file_info_new();

	vfs_result = gnome_vfs_get_file_info_uri(dir_uri_str, file_info,
						 GNOME_VFS_FILE_INFO_DEFAULT);
	if (vfs_result == GNOME_VFS_ERROR_NOT_FOUND) {
		vfs_result = gnome_vfs_make_directory_for_uri(dir_uri_str, S_IRWXU);
		if (vfs_result != GNOME_VFS_OK) {
			gnome_vfs_uri_unref(dir_uri_str);
			gnome_vfs_file_info_unref(file_info);
			return FALSE;
		}
	}

	gnome_vfs_uri_unref(dir_uri_str);
	gnome_vfs_file_info_unref(file_info);
	osso_log(LOG_DEBUG, "End Function-- %s --\n", __FUNCTION__);

	return TRUE;
}

gchar *gpscamera_get_unique_name()
{
	struct tm *time_ptr = NULL;
	gchar *convert = NULL;
	time_t t = 0;
	gchar *name_string = NULL;
	gchar *time_string = NULL;	
	char date_string[100];
	char sec_string[20];
	char min_string[20];
	char hour_string[20];

	time(&t);
	time_ptr = gmtime((const time_t *) &(t));
	strftime(date_string, 100, "%y%m%d", time_ptr);
	strftime(sec_string, 20, "%S", time_ptr);
	strftime(min_string, 20, "%M", time_ptr);
	strftime(hour_string, 20, "%H", time_ptr);
	
	time_string = g_strdup_printf("%s%s%s%s%s",hour_string,"_",
						   min_string,"_",
						   sec_string);
        name_string = g_strdup_printf("%s%s%s",date_string, "-",
						time_string);
	g_free(time_string);

	convert = g_strdup_printf("%s%s", name_string, I_EXTN);     
	
	g_free(name_string);
	osso_log(LOG_DEBUG, "File name is:%s\n", convert);
	return convert;

}

gint gpscamera_delete_file(const gchar * path)
{
	
	GnomeVFSResult vfs_result = GNOME_VFS_ERROR_GENERIC;
	if (path == NULL) {
		return ERROR;
	}

	if ((vfs_result = gnome_vfs_unlink(path)) == GNOME_VFS_OK) {
		osso_log(LOG_ERR, "Unable to delete file:%s\n",
			 gnome_vfs_result_to_string(vfs_result));
		return GPSCAMERA_DELETE_SUCCESS;
	} else if (vfs_result == GNOME_VFS_ERROR_NOT_FOUND) {
		return GPSCAMERA_FILE_NA;
	} else if (vfs_result == GNOME_VFS_ERROR_READ_ONLY) {
		return GPSCAMERA_DELETE_NO_PERM;
	} else {
		return ERROR;
	}
}

int mkdir_recursive()
{
	gchar *path = NULL;
	gchar *path1 = NULL;
	struct stat s;

	path = g_strdup_printf("%s%s", g_get_home_dir(), APP_PATH);
	if (path == NULL) {
		return FALSE;
	}

	if (stat(path, &s) < 0) {
		if (mkdir(path, S_IRWXU) < 0) {
			g_free(path);
			return FALSE;
		}
		if (chmod(path, S_IRWXU) < 0) {
			g_free(path);
			return FALSE;
		}
	}
	g_free(path);

	path1 = g_strdup_printf("%s%s", g_get_home_dir(), SETTINGS_PATH);
	if (stat(path1, &s) < 0) {
		if (mkdir(path1, S_IRWXU) < 0) {
			g_free(path1);
			return FALSE;
		}
		if (chmod(path1, S_IRWXU) < 0) {
			g_free(path1);
			return FALSE;
		}
	}

	g_free(path1);
	return TRUE;
}


gboolean gpscamera_write_gpslocation_file(gchar *file_path, gdouble lat, gdouble lng)
{
	
	FILE *fp = NULL;
	gchar *file = NULL;
	struct stat s;
		
	file = g_strdup_printf("%s%s", g_get_home_dir(), FILE_PATH);
	if (file == NULL) {
		return FALSE;
	}
	
	if (stat(file, &s) < 0) {
		if (mkdir_recursive() < 0) {
			g_free(file);
			return FALSE;
		}
	}

	if ((fp = fopen(file, "a+")) == NULL) {
		return FALSE;
	}

	fprintf(fp, "%s;%f;%f;", file_path, lat, lng);
	
	fclose(fp);

	return TRUE;
}


GList* gpscamera_read_gpslocation_file(const gchar * image_path) {
	FILE *fp = NULL;
	gchar *file = NULL;
	char** images;
	ssize_t left;	
	GList* list =NULL;
	char buffer[BUFFSIZE];
	int i = 0;
	
	file = g_strdup_printf("%s%s", g_get_home_dir(), FILE_PATH);
	if (file == NULL) {
		return NULL;
	}
	
	/*open to read*/
	if ((fp = fopen(file, "r")) == NULL) {
		return NULL;
	}
	memset(buffer, '\0', sizeof(buffer));
	while (fgets(buffer, sizeof(buffer), fp) != NULL) {
		
		images = g_strsplit(buffer, ";", 0);
		memset(buffer, '\0', sizeof(buffer));
	}
	for (i = 0; images[i] != NULL; i++) {
			
		if(strcmp(images[i], image_path)==0){
			list = g_list_append(list, images[i]);
			list = g_list_append(list, images[i+1]);
			list = g_list_append(list, images[i+2]);
		}	
			
	}

	return list;
	
}

