/*****************************************************************************
 * Free42 -- an HP-42S calculator simulator
 * Copyright (C) 2004-2007  Thomas Okken
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *****************************************************************************/

#ifndef SHELL_H
#define SHELL_H 1

#include "free42.h"

/* shell_blitter()
 *
 * Callback invoked by the emulator core to cause the display, or some portion
 * of it, to be repainted.
 *
 * 'bits' is a pointer to a 1 bpp (monochrome) bitmap. The bits within a byte
 * are laid out with left corresponding to least significant, right
 * corresponding to most significant; this corresponds to the convention for
 * X11 images, but it is the reverse of the convention for MacOS and its
 * derivatives (Microsoft Windows and PalmOS).
 * The bytes are laid out sequentially, that is, bits[0] is at the top
 * left corner, bits[1] is to the right of bits[0], bits[2] is to the right of
 * bits[1], and so on; this corresponds to X11, MacOS, Windows, and PalmOS
 * usage.
 * 'bytesperline' is the number of bytes per line of the bitmap; this means
 * that the bits just below bits[0] are at bits[bytesperline].
 * 'x', 'y', 'width', and 'height' define the part of the bitmap that needs to
 * be repainted. 'x' and 'y' are 0-based coordinates, with (0, 0) being the top
 * left corner of the bitmap, and x coordinates increasing to the right, and y
 * coordinates increasing downwards. 'width' and 'height' are the width and
 * height of the area to be repainted.
 */
void shell_blitter(const char *bits, int bytesperline, int x, int y,
			     int width, int height) SHELL1_SECT;

/* shell_beeper()
 * Callback invoked by the emulator core to play a sound.
 * The first parameter is the frequency in Hz; the second is the
 * duration in ms. The sound volume is up to the GUI to control.
 * Sound playback should be synchronous (the beeper function should
 * not return until the sound has finished), if possible.
 */
void shell_beeper(int frequency, int duration) SHELL1_SECT;

/* shell_annunciators()
 * Callback invoked by the emulator core to change the state of the display
 * annunciators (up/down, shift, print, run, battery, (g)rad).
 * Every parameter can have values 0 (turn off), 1 (turn on), or -1 (leave
 * unchanged).
 * The battery annunciator is missing from the list; this is the only one of
 * the lot that the emulator core does not actually have any control over, and
 * so the shell is expected to handle that one by itself.
 */
void shell_annunciators(int updn, int shf, int prt, int run, int g, int rad) SHELL1_SECT;

/* shell_wants_cpu()
 *
 * Callback used by the emulator core to check for pending events.
 * It calls this periodically during long operations, such as running a
 * user program, or the solver, etc. The shell should not handle any events
 * in this call! If there are pending events, it should return 1; the currently
 * active invocation of core_keydown() or core_keyup() will then return
 * immediately (with a return value of 1, to indicate that it would like to get
 * the CPU back as soon as possible).
 */
int shell_wants_cpu() SHELL1_SECT;

/* Callback to suspend execution for the given number of milliseconds. No event
 * processing will take place during the wait, so the core can call this
 * without having to worry about core_keydown() etc. being re-entered.
 */
void shell_delay(int duration) SHELL1_SECT;

/* Callback to ask the shell to call core_timeout3() after the given number of
 * milliseconds. If there are keystroke events during that time, the timeout is
 * cancelled. (Pressing 'shift' does not cancel the timeout.)
 * This function supports the delay after SHOW, MEM, and shift-VARMENU.
 */
void shell_request_timeout3(int delay) SHELL1_SECT;

/* shell_read_saved_state()
 *
 * Callback to read from the saved state. The function will read up to n
 * bytes into the buffer pointed to by buf, and return the number of bytes
 * actually read. The function returns -1 if an error was encountered; a return
 * value of 0 signifies the end of input.
 * The emulator core should only call this function from core_init(), and only
 * if core_init() was called with an argument of 1. (Nothing horrible will
 * happen if you try to call this function during other contexts, but you will
 * always get an error then.)
 */
int4 shell_read_saved_state(void *buf, int4 bufsize) SHELL1_SECT;

/* shell_write_saved_state()
 * Callback to dump the saved state to persistent storage.
 * Returns 1 on success, 0 on error.
 * The emulator core should only call this function from core_quit(). (Nothing
 * horrible will happen if you try to call this function during other contexts,
 * but you will always get an error then.)
 */
bool shell_write_saved_state(const void *buf, int4 nbytes) SHELL1_SECT;

/* shell_get_mem()
 * Callback to get the amount of free memory in bytes.
 */
uint4 shell_get_mem() SHELL1_SECT;

/* shell_low_battery()
 * Callback to find out if the battery is low. Used to emulate flag 49 and the
 * battery annunciator, and also taken into account when deciding whether or
 * not to allow a power-down -- so as long as the shell provides a functional
 * implementation of shell_low_battery(), it can leave the decision on how to
 * respond to sysNotifySleepRequestEvent to core_allows_powerdown().
 */
int shell_low_battery() SHELL1_SECT;

/* shell_powerdown()
 * Callback to tell the shell that the emulator wants to power down.
 * Only called in response to OFF (shift-EXIT or the OFF command); automatic
 * power-off is left to the OS and/or shell.
 */
void shell_powerdown() SHELL1_SECT;

/* shell_random_seed()
 * When SEED is invoked with X = 0, the random number generator should be
 * seeded to a random value; the emulator core calls this function to obtain
 * it. The shell should construct a double in the range [0, 1) in a random
 * manner, using the real-time clock or some other source of randomness.
 * Note that distribution is not very important; the value will only be used to
 * seed the RNG. What's important that using shell_random_seed() guarantees
 * that the RNG will be initialized to a different sequence. This matters for
 * applications like games where you don't want the same sequence of cards
 * dealt each time.
 */
double shell_random_seed() SHELL1_SECT;

/* shell_milliseconds()
 * Returns an elapsed-time value in milliseconds. The caller should make no
 * assumptions as to what this value is relative to; it is only intended to
 * allow the emulator core make short-term elapsed-time measurements.
 */
uint4 shell_milliseconds() SHELL1_SECT;

/* shell_print()
 * Printer emulation. The first 2 parameters are the plain text version of the
 * data to be printed; the remaining 6 parameters are the bitmap version. The
 * former is used for text-mode copying and for spooling to text files; the
 * latter is used for graphics-mode coopying, spooling to image files, and
 * on-screen display.
 */
void shell_print(const char *text, int length,
		 const char *bits, int bytesperline,
		 int x, int y, int width, int height) SHELL1_SECT;

/* shell_write()
 *
 * Callback for core_export_programs(). Returns 0 if a problem occurred;
 * core_export_programs() should abort in that case.
 */
int shell_write(const char *buf, int4 buflen) SHELL1_SECT;

/* shell_read()
 *
 * Callback for core_import_programs(). Returns the number of bytes actually
 * read. Returns -1 if an error occurred; a return value of 0 signifies end of
 * input.
 */
int4 shell_read(char *buf, int4 buflen) SHELL1_SECT;

/* shell_get_bcd_table()
 * shell_put_bcd_table()
 * shell_release_bcd_table()
 * shell_bcd_table_struct
 *
 * On platforms where computing the BCD conversion table is slow, these
 * functions should be implemented so they save the table in persistent
 * storage. On other platforms, they can be no-ops.
 * shell_get_bcd_table() returns NULL if no BCD table image was found.
 */
typedef struct {
    double pos_huge_double;
    double neg_huge_double;
    double pos_tiny_double;
    double neg_tiny_double;
    int2 max_pow2;
    int2 min_pow2;
    uint4 pos_pow2exp_offset; /* Offsets are from the end of this struct */
    uint4 neg_pow2mant_offset;/* pos_pow2mant_offset is implicitly 0 */
    uint4 neg_pow2exp_offset;
} shell_bcd_table_struct;

shell_bcd_table_struct *shell_get_bcd_table() SHELL1_SECT;
shell_bcd_table_struct *shell_put_bcd_table(shell_bcd_table_struct *bcdtab,
					    uint4 size) SHELL1_SECT;
void shell_release_bcd_table(shell_bcd_table_struct *bcdtab) SHELL1_SECT;

#ifdef DEBUG
void logtofile(const char *message) SHELL1_SECT;
void lognumber(int4 num) SHELL1_SECT;
void logdouble(double num) SHELL1_SECT;
#endif

#endif
