#
# _audio.py
# Easy API - audio module.
#
# Copyright (C) 2007-2008 UFCG - Federal University of Campina Grande
# Embedded Systems and Pervasive Computing Lab.
#
# Contact: Mario Hozano Lucas de Souza <hozano@embedded.ufcg.edu.br>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#

"""This module encapsulates a set of functions to record and reproduce
audio files, and functions to help the development of audio manipulation
inside applications. Some examples of features include seek, file time,
current time position and volume control.
"""
from _playback_recorder import PlaybackRecorder
import gst
import platform
 

class Audio(PlaybackRecorder):
    """This module encapsulates a set of functions to record and reproduce
    audio files, and functions to help the development of audio manipulation
    inside applications. Some examples of features include seek, file time,
    current time position and volume control.

    The audio module has two pipelines: the former deals with reproduction of
    audio files and the latter works with audio file recordings. These two
    functions can be used in a synchronous or asynchronous way, depending on
    the application. These pipelines encapsulate Gstreamer multimedia framework
    and help avoiding codec manipulation.
    """
    
    def __init__(self):
        """Constructor for Audio class.
        """
        PlaybackRecorder.__init__(self)
    
    def _init_recorder_pipeline(self):
        """Initializes the Recorder pipeline
        """ 
        if "arm" in platform.uname()[4]:
            self._recorder_pipeline = gst.parse_launch("dsppcmsrc name=src ! wavenc !\
                                                  gnomevfssink name=sink")
        else:
            self._recorder_pipeline = gst.parse_launch("alsasrc name=src ! wavenc !\
                                                  gnomevfssink name=sink")

        self._recorder_pipeline.get_by_name("sink").connect("allow-overwrite",
                                                          lambda *x: True)
    
    def _set_recorder_file(self, filename):
        """Set the name of the recorded file.

        @type  filename: string
        @param filename: name of the recorded file
        """
        if not filename.lower().endswith('.wav'):
            filename += '.wav'
        PlaybackRecorder._set_recorder_file(self,filename)
    
    def record(self, filename="file.wav", record_time=-1, callback=None):
        """Records the audio captured by microphone.

        @type     filename: string
        @param    filename: name of the recorded file
        @type  record_time: number
        @param record_time: total time, in seconds, of the recorded file. If '-1' the 'stop' method must be called to break recording.
        @type     callback: function
        @param    callback: function that will be called after recording
        """
        self.stop()
        PlaybackRecorder.record(self, filename, record_time, callback)

_audio = Audio()

is_playing = _audio.is_playing
is_paused = _audio.is_paused
is_recording = _audio.is_recording
is_stopped = _audio.is_stopped
play = _audio.play
pause = _audio.pause
stop = _audio.stop
record = _audio.record
get_length = _audio.get_length
get_current_position = _audio.get_current_position
seek = _audio.seek
get_volume = _audio.get_volume
set_volume = _audio.set_volume