#
# audio.py 
# Easy API - audio module.
#
# Copyright (C) 2005-2006 INdT - Instituto Nokia de Tecnologia
#
# Contact: Luciano Miguel Wolf <luciano.wolf@indt.org.br>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#

import pygst
pygst.require("0.10")
import gst
import os
import time
import thread

import easy_exceptions

__all__ = ["is_playing", "is_paused", "is_recording", "is_stopped", "play",
           "pause", "stop", "record", "record_async", "get_length",
           "get_current_position", "seek", "get_volume", "set_volume"]

class FileNotExistError(easy_exceptions.EasyError): pass
class FileNotSpecifiedError(easy_exceptions.EasyError): pass
class VolumeNotSpecifiedError(easy_exceptions.EasyError): pass
class OutOfRangeError(easy_exceptions.EasyError): pass

class Audio(object):
    """This module encapsulates a set of functions to record and reproduce
    audio files, and functions to help the development of audio manipulation
    inside applications. Some examples of features include seek, file time,
    current time position and volume control.

    The audio module has two pipelines: the former deals with reproduction of
    audio files and the latter works with audio file recordings. These two
    functions can be used in a synchronous or asynchronous way, depending on
    the application. These pipelines encapsulate Gstreamer multimedia framework
    and help avoiding codec manipulation.
    """

    def __init__(self):
        """Constructor for Audio class.
        """
        self._PLAYING = False
        self._PAUSED = False
        self._RECORDING = False

        self._current_pipeline = None
        self._playback_pipeline = gst.parse_launch("playbin name=sink")
        self._record_pipeline = gst.parse_launch("dsppcmsrc ! wavenc !\
                                                  gnomevfssink name=sink")

        self._record_pipeline.get_by_name("sink").connect("allow-overwrite",
                                                          lambda *x: True)

    def _set_playback_pipeline(self):
        """Build a GStreamer pipeline to play an audio file.
        """
        if self._current_pipeline:
            self.stop()

        self._current_pipeline = self._playback_pipeline

    def _set_record_pipeline(self):
        """Build a GStreamer pipeline to record an audio file.
        """
        if self._current_pipeline:
            self.stop()

        self._current_pipeline = self._record_pipeline

    def is_playing(self):
        """Check if device is playing.

        @rtype:  boolean
        @return: True, if device is playing
        """
        return self._PLAYING

    def is_paused(self):
        """Check if device is paused.

        @rtype:  boolean
        @return: True, if device is paused
        """
        return self._PAUSED

    def is_recording(self):
        """Check if device is recording.

        @rtype:  boolean
        @return: True, if device is recording
        """
        return self._RECORDING

    def is_stopped(self):
        """Check if device is stopped.

        @rtype:  boolean
        @return: True, if device is stopped
        """
        return not (self.is_playing() or self.is_paused() or \
                    self.is_recording())

    def play(self, filename=None):
        """Start playing audio file.

        @type  filename: string
        @param filename: file name
        """
        if os.path.exists(filename):
            self._set_playback_pipeline()
            self._set_playback_file(filename)
            self._current_pipeline.set_state(gst.STATE_PLAYING)
            self._PLAYING = True

            _bus = self._current_pipeline.get_bus()
            _bus.add_signal_watch()
            self._bus_watch_id = _bus.connect("message", self._catch_EOS)
        else:
            raise FileNotExistError("File does not exist")

    def pause(self):
        """Pause or resume the current playing audio file.
        """
        if self.is_playing():
            self._current_pipeline.set_state(gst.STATE_PAUSED)
            self._PAUSED = True
            self._PLAYING = False
        elif self.is_paused():
            self._current_pipeline.set_state(gst.STATE_PLAYING)
            self._PLAYING = True
            self._PAUSED = False

    def stop(self):
        """Stop the current audio stream.
        """
        if self._current_pipeline:
            self._current_pipeline.set_state(gst.STATE_NULL)
            self._PLAYING = False
            self._PAUSED = False
            self._RECORDING = False

    def record(self, filename="file.wav", record_time=None, delay=0,
               callback=None):
        """Records the sound captured in a syncronous way.

        @type     filename: string
        @param    filename: name of the recorded file
        @type  record_time: number
        @param record_time: total time, in seconds, of the recorded file.
        @type        delay: number
        @param       delay: delay time, in seconds, that will be waited until
                            start recording
        @type     callback: function
        @param    callback: function that will be called after recording
        """
        self._set_record_pipeline()
        self._set_recorder_file(filename)

        time.sleep(delay)
        self._current_pipeline.set_state(gst.STATE_PLAYING)
        self._RECORDING = True

        if record_time:
            time.sleep(record_time)
            self.stop()
            if callable(callback):
                callback()

    def record_async(self, filename="file.wav", record_time=None, delay=0,
                     callback=None):
        """Record sound captured in an asyncronous way.

        @type     filename: string
        @param    filename: name of the recorded file
        @type  record_time: number
        @param record_time: total time, in seconds, of the recorded file
        @type        delay: number
        @param       delay: delay time, in seconds, that will be waited until
                            start recording
        @type     callback: function
        @param    callback: function that will be called after recording
        """
        thread.start_new_thread(self.record, (filename, record_time, delay,
                                              callback))

    def get_length(self):
        """Returns the audio file's length in nano-seconds.

        @rtype:  long integer
        @return: audio file's length in nano-seconds
        """
        if self.is_playing() or self.is_paused():
            query = self._current_pipeline.query_duration(gst.FORMAT_TIME,
                                                         None)

            if query:
                return query[0]

    def get_current_position(self):
        """Returns the current position (in nano-seconds) of the playing file.

        @rtype:  long integer
        @return: current audio position in nano-seconds
        """
        if self.is_playing() or self.is_paused():
            query = self._current_pipeline.query_position(gst.FORMAT_TIME,
                                                         None)

            if query:
                return query[0]

    def seek(self, value=None):
        """Seek for the 'value' position in the audio file.

        @type  value: number
        @param value: position number that will be seeked in the audio file
        """
        if self.is_playing() or self.is_paused():
            self._current_pipeline.seek_simple(gst.FORMAT_TIME,
                                        gst.SEEK_FLAG_FLUSH, value)

    def get_volume(self):
        """Return the volume level of the played audio file.

        @rtype:  long integer
        @return: volume level of the played audio file
        """
        element = self._get_current_sink_element()
        return element.get_property("volume")

    def set_volume(self, volume=None):
        """Set the volume level of the played audio file. The values range is
        from 0.0 to 10.0.

        @type  volume: double
        @param volume: number that set volume level, ranged from 0.0 to 10.0
        """
        if volume == None:
            raise VolumeNotSpecifiedError("Volume not specified")

        element = self._get_current_sink_element()
        element.set_property("volume", volume)

    def _get_current_sink_element(self):
        """Gets the sink element from pipeline.
        """
        # For pipelines containing only one element. (playback_pipeline)
        if self._current_pipeline.get_name() == "sink":
            element = self._current_pipeline
        else:
            element = self._current_pipeline.get_by_name("sink")
        return element

    def _set_recorder_file(self, filename):
        """Set the name of the recorded file.

        @type  filename: string
        @param filename: name of the recorded file
        """
        self._get_current_sink_element().set_property("location", filename)

    def _set_playback_file(self, filename):
        """Set the name of the playback file.

        @type  filename: string
        @param filename: name of the playback file
        """
        if filename:
            self._current_pipeline.set_property("uri", "file://" + filename)
        else:
            raise FileNotSpecifiedError("Filename not specified")

    def _catch_EOS(self, bus, message):
        """Get bus EOS(end of streamer) message and and stop pipeline. This
        method only works when the application runs a GLib/Gtk+ main loop.

        @type      bus: GSreamer element
        @param     bus: file playing pipeline's bus
        @type  message: string
        @param message: bus message
        """
        if message.type == gst.MESSAGE_EOS:
            bus.disconnect(self._bus_watch_id)
            self.stop()

_audio = Audio()

is_playing = _audio.is_playing
is_paused = _audio.is_paused
is_recording = _audio.is_recording
is_stopped = _audio.is_stopped
play = _audio.play
pause = _audio.pause
stop = _audio.stop
record = _audio.record
record_async = _audio.record_async
get_length = _audio.get_length
get_current_position = _audio.get_current_position
seek = _audio.seek
get_volume = _audio.get_volume
set_volume = _audio.set_volume