/*
  DiskUsage - shows the disk usage under a number of operating systems.
  Copyright (C) 2004  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <math.h>

#include <iomanip>
#include <iostream>
#include <list>
#include <locale>
#include <sstream>

#include <Lum/OS/Main.h>

#include <Lum/Base/Object.h>
#include <Lum/Base/String.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/DataStream.h>
#include <Lum/Model/Integer.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Table.h>

#include <Lum/ButtonRow.h>
#include <Lum/Dialog.h>
#include <Lum/Grid.h>
#include <Lum/Label.h>
#include <Lum/PercentBar.h>
#include <Lum/Space.h>
#include <Lum/Table.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>
#include <Lum/WindowGroup.h>

#if defined(linux)
#include <string.h>
#include <sys/vfs.h>
#include <mntent.h>
#elif defined(sun)
#include <stdio.h>
#include <string.h>
#include <sys/mnttab.h>
#include <sys/types.h>
#include <sys/statvfs.h>
#elif defined(__APPLE__)
#include <sys/param.h>
#include <sys/ucred.h>
#include <sys/mount.h>
#elif defined(__WIN32__) || defined(WIN32)
#include <windows.h>
#include <winnt.h>
#undef GetObject
#endif

class DiskInfo
{
public:
  std::wstring  name;
  std::wstring  dir;
  double        blocks;
  double        free;
  double        avail;
  double        unit; // Multiplicator to get the size in kilobytes
};

bool GetDisks(std::list<DiskInfo>& list)
{
#if defined(linux)
  FILE          *file;
  struct mntent *inf;
  struct statfs stats;
  DiskInfo      info;

  file=setmntent("/etc/mtab","r");

  if (file==NULL) {
    return false;
  }

  while (true) {
    memset(&stats,0,sizeof(stats));
    inf=getmntent(file);
    while ((inf!=NULL) && ((statfs(inf->mnt_dir,&stats)!=0) || (stats.f_blocks==0))) {
      inf=getmntent(file);
    }
    if ((inf!=NULL) && (statfs(inf->mnt_dir,&stats)==0)) {
      info.name=Lum::Base::StringToWString(inf->mnt_fsname);
      info.dir=Lum::Base::StringToWString(inf->mnt_dir);
      info.blocks=stats.f_blocks;
      info.free=stats.f_bfree;
      info.avail=stats.f_bavail;
      info.unit=stats.f_bsize/1024;

      list.push_back(info);
    }
    else {
      break;
    }
  }

  endmntent(file);

  return true;
#elif defined(sun)
  FILE           *file;
  struct mnttab  tab;
  struct statvfs stats;
  DiskInfo       info;

  file=fopen("/etc/mnttab","r");

  if (file==NULL) {
    return false;
  }

  while (getmntent(((Private*)handle)->handle,&tab)==0) {
    memset(&stats,0,sizeof(stats));
    if ((statvfs(tab.mnt_mountp,&stats)==0)) {
      info.name=Lum::Base::StringToWString(tab.mnt_special);
      info.dir=Lum::Base::StringToWString(list[x].f_mntonname);
      info.blocks=stats.f_blocks;
      info.free=stats.f_bfree;
      info.avail=stats.f_bavail;
      info.unit=1; // TODO

      list.push_back(info);
    }
    else {
      break;
    }
  }

  fclose(file);

  return true;
#elif defined(__APPLE__)
  struct statfs* devlist;
  size_t         size;
  DiskInfo       info;

  size=getmntinfo(&devlist,MNT_NOWAIT);

  if (size==0) {
    return false;
  }
  for (size_t x=0; x<size; x++) {
    info.name=Lum::Base::StringToWString(devlist[x].f_fstypename);
    info.dir=Lum::Base::StringToWString(devlist[x].f_mntonname);
    info.blocks=devlist[x].f_blocks;
    info.free=devlist[x].f_bfree;
    info.avail=devlist[x].f_bavail;
    info.unit=1; // TODO

    list.push_back(info);
  }

  return true;
#elif defined(__WIN32__) || defined(WIN32)
  UINT    old;
  wchar_t buffer[10240];
  wchar_t buffer2[MAX_PATH+1];
  size_t  offset=0;
  DWORD   compLength,flags;
  //UINT    type;

  old=SetErrorMode(SEM_FAILCRITICALERRORS);
  GetLogicalDriveStringsW(1024,buffer);

  while (buffer[offset]!='\0') {
    DiskInfo       info;
    ULARGE_INTEGER available,bytes,free;

    size_t x=0;
    while (buffer[offset]!='\0') {
      info.name.append(1,buffer[offset]);
      info.dir.append(1,buffer[offset]);
      ++x;
      ++offset;
    }
    ++offset;

    /*
    type=GetDriveTypeW(info.name.c_str());
    if (type==DRIVE_UNKNOWN || type==DRIVE_NO_ROOT_DIR || type==DRIVE_REMOVABLE) {
      continue;
    }*/

    if (GetVolumeInformationW(info.name.c_str(),buffer2,MAX_PATH+1,NULL,&compLength,&flags,NULL,0)!=0) {
      info.dir=buffer2;
    }

    if (GetDiskFreeSpaceExW(info.name.c_str(),&available,&bytes,&free)) {
      info.blocks=bytes.QuadPart/1024;
      info.free=free.QuadPart/1024;
      info.avail=available.QuadPart/1024;
      info.unit=1;
      list.push_back(info);
    }
  }

  SetErrorMode(old);

  return true;
#else
  return false;
#endif
}

std::wstring DoubleToWString(double value, double unit)
{
  std::stringstream buffer;

  buffer.imbue(std::locale(""));
  buffer.setf(std::ios::fixed);

  buffer<< std::setprecision(0);

  buffer << ceil(value*unit) << " KB";

  return Lum::Base::StringToWString(buffer.str());
}

Lum::TextValue* GetTextValueLeft(const Lum::Model::StringRef& value)
{
  Lum::TextValue   *text;
  Lum::OS::FontRef font(Lum::OS::display->GetFont(Lum::OS::Display::fontTypeProportional,
                                                  Lum::OS::Display::fontScaleCaption1));

  text=new Lum::TextValue();
  text->SetFlex(true,false);
  text->SetModel(value.Get());
  // TODO: font

  return text;
}

Lum::TextValue* GetTextValueRight(const Lum::Model::StringRef& value)
{
  Lum::TextValue   *text;
  Lum::OS::FontRef font(Lum::OS::display->GetFont(Lum::OS::Display::fontTypeProportional,
                                                  Lum::OS::Display::fontScaleCaption1));

  text=new Lum::TextValue();
  text->SetFlex(true,false);
  text->SetModel(value.Get());
  text->SetAlignment(Lum::TextValue::right);
  // TODO: font

  return text;
}

class MainWindow : public Lum::Dialog
{
private:
  Lum::Model::ActionRef              showDetail;

  Lum::Model::IntRef                 refreshTime;
  Lum::Model::ActionRef              timer;

  Lum::Table                         *table;
  Lum::Model::SingleLineSelectionRef selection;
  Lum::Model::ListTableRef           tableModel;
  Lum::Model::StringRef              device;
  Lum::Model::StringRef              directory;
  Lum::Model::StringRef              used;
  Lum::Model::StringRef              free;
  Lum::Model::StringRef              reserved;
  Lum::Model::StringRef              available;
  Lum::Model::StringRef              total;
  Lum::Model::DoubleDataStreamRef    data;

  std::list<DiskInfo>                list;

public:
  MainWindow()
  : showDetail(new Lum::Model::Action()),
    refreshTime(new Lum::Model::Int(10)),
    timer(new Lum::Model::Action()),
    selection(new Lum::Model::SingleLineSelection),
    tableModel(new Lum::Model::ListTable(3)),
    device(new Lum::Model::String()),
    directory(new Lum::Model::String()),
    used(new Lum::Model::String()),
    free(new Lum::Model::String()),
    reserved(new Lum::Model::String()),
    available(new Lum::Model::String()),
    total(new Lum::Model::String()),
    data(new Lum::Model::DoubleDataStream())
  {
    data->SetNotificationMode(Lum::Model::DoubleDataStream::notifyExplicit);
    data->SetChannels(3);

    AttachModel(showDetail);
    AttachModel(refreshTime);
    AttachModel(timer);
    AttachModel(GetOpenedAction());
  }

  ~MainWindow()
  {
    UnattachModel(GetOpenedAction());
    UnattachModel(timer);
    UnattachModel(refreshTime);
    UnattachModel(showDetail);
  }

  virtual void PreInit()
  {
    Lum::Grid             *grid;
    Lum::Label            *label;
    Lum::Panel            *panel;
    Lum::PercentBar       *percent;
    Lum::WindowGroup      *wGroup;
    Lum::Model::HeaderRef headerModel;

    wGroup=new Lum::WindowGroup();
    wGroup->SetFlex(true,true);
    wGroup->SetWidth(Lum::Base::Size::screenHRel,40);
    wGroup->SetHeight(Lum::Base::Size::screenVRel,40);

    panel=new Lum::VPanel();
    panel->SetFlex(true,true);

    label=new Lum::Label();
    label->SetFlex(true,false);
    label->AddLabel(L"Device:",GetTextValueLeft(device));
    label->AddLabel(L"Directory:",GetTextValueLeft(directory));
    panel->Add(label);

    panel->Add(new Lum::VSpace());

    grid=new Lum::Grid();
    grid->SetFlex(true,false);
    grid->SetSpace(true,false);
    grid->SetSize(2,1);

    label=new Lum::Label();
    label->SetFlex(true,false);
    label->AddLabel(L"Used:",GetTextValueRight(used));
    label->AddLabel(L"Free:",GetTextValueRight(free));
    label->AddLabel(L"Total:",GetTextValueRight(total));
    grid->SetObject(0,0,label);

    label=new Lum::Label();
    label->SetFlex(true,false);
    label->AddLabel(L"Reserved:",GetTextValueRight(reserved));
    label->AddLabel(L"Available:",GetTextValueRight(available));
    grid->SetObject(1,0,label);

    panel->Add(grid);

    panel->Add(new Lum::VSpace());

    percent=new Lum::PercentBar();
    percent->SetFlex(true,false);
    percent->SetMinWidth(Lum::Base::Size::stdCharWidth,30);
    percent->SetModel(data);
    panel->Add(percent);

    panel->Add(new Lum::VSpace());

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Dir",Lum::Base::Size::stdCharWidth,10);
    headerModel->AddColumn(L"Dev",Lum::Base::Size::stdCharWidth,10);
    headerModel->AddColumn(L"Size",Lum::Base::Size::stdCharWidth,31,true);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetModel(tableModel);
    table->SetHeaderModel(headerModel);
    table->SetSelection(selection);
    table->SetSelectionAction(showDetail);
    //table->SetDoubleClickAction(showDetail);
    table->SetShowHeader(true);
    table->GetTableView()->SetAutoFitColumns(true);
    panel->Add(table);

    wGroup->SetMain(panel);

    SetTop(wGroup);

    Dialog::PreInit();
  }

  void UpdateDisplay()
  {
    list.clear();
    if (GetDisks(list)) {
      std::list<DiskInfo>::const_iterator iter;

      iter=list.begin();

      // First try to update existing entries in the current table
      while (iter!=list.end()) {
        size_t i;

        for (i=1; i<=tableModel->GetRows(); i++) {
          if (iter->dir==tableModel->GetEntry(i)->GetString(1)) {
            break;
          }
        }

        if (i<=tableModel->GetRows()) {
          Lum::Model::ListTable::Entry    *entry=tableModel->GetEntry(i);

          Lum::PercentBar                 *percent;
          Lum::Model::DoubleDataStreamRef data;

          percent=dynamic_cast<Lum::PercentBar*>(entry->GetObject(3));
          data=dynamic_cast<Lum::Model::DoubleDataStream*>(percent->GetModel());

          data->Set(0,(iter->blocks-iter->free)/iter->blocks);
          data->Set(1,iter->avail/iter->blocks);
          data->Set(2,(iter->free-iter->avail)/iter->blocks);
          data->Notify();
        }
        else {
          Lum::PercentBar                 *percent;
          Lum::Model::DoubleDataStreamRef data;
          Lum::Model::ListTable::StdEntry *entry;

          percent=new Lum::PercentBar();
          percent->SetFlex(true,true);
          percent->SetMinWidth(Lum::Base::Size::stdCharWidth,30);

          data=new Lum::Model::DoubleDataStream();
          data->SetNotificationMode(Lum::Model::DoubleDataStream::notifyExplicit);
          data->SetChannels(3);
          data->Set(0,(iter->blocks-iter->free)/iter->blocks);
          data->Set(1,iter->avail/iter->blocks);
          data->Set(2,(iter->free-iter->avail)/iter->blocks);
          percent->SetModel(data);

          entry=new Lum::Model::ListTable::StdEntry(tableModel);
          entry->SetString(1,iter->dir);
          entry->SetString(2,iter->name);
          entry->SetObject(3,percent);

          tableModel->Append(entry);
        }

        ++iter;
      }

      // Now see, if entries in the table have been removed and must
      // be deleted
      size_t i=1;

      while(i<=tableModel->GetRows()) {
        std::wstring                        directory=tableModel->GetEntry(i)->GetString(1);
        std::list<DiskInfo>::const_iterator iter;

        iter=list.begin();
        while (iter!=list.end()) {
          if (iter->dir==directory) {
            break;
          }

          ++iter;
        }

        if (iter==list.end()) {
          tableModel->Delete(i);
        }
        else {
          i++;
        }
      }

    }
  }

  void ShowDetail()
  {
    std::list<DiskInfo>::const_iterator iter;

    if (selection->HasSelection()){
      Lum::Model::ListTable::Entry *entry=tableModel->GetEntry(selection->GetLine());

      iter=list.begin();
      while (iter!=list.end() && iter->dir!=entry->GetString(1)) {
        ++iter;
      }
    }
    else {
      iter=list.end();
    }

    if (iter!=list.end()) {
      device->Set(iter->name);
      directory->Set(iter->dir);

      used->Set(DoubleToWString(iter->blocks-iter->free,iter->unit));
      free->Set(DoubleToWString(iter->free,iter->unit));
      reserved->Set(DoubleToWString(iter->free-iter->avail,iter->unit));
      available->Set(DoubleToWString(iter->avail,iter->unit));
      total->Set(DoubleToWString(iter->blocks,iter->unit));

      data->Enable();
      data->Set(0,(iter->blocks-iter->free)/iter->blocks);
      data->Set(1,iter->avail/iter->blocks);
      data->Set(2,(iter->free-iter->avail)/iter->blocks);
      data->Notify();
    }
    else {
      device->Set(L"");
      directory->Set(L"");

      used->Set(L"");
      free->Set(L"");
      reserved->Set(L"");
      available->Set(L"");
      total->Set(L"");

      data->Disable();
    }
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      UpdateDisplay();
      Lum::OS::display->AddTimer(refreshTime->Get(),0,timer);
    }
    else if (model==timer && timer->IsFinished()) {
      UpdateDisplay();
      Lum::OS::display->AddTimer(refreshTime->Get(),0,timer);
    }
    else if (model==refreshTime && IsOpen()) {
      Lum::OS::display->RemoveTimer(timer);
      if (refreshTime->Get()>0) {
        Lum::OS::display->AddTimer(refreshTime->Get(),0,timer);
      }
    }
    else if (model==showDetail && showDetail->IsFinished()) {
      ShowDetail();
    }

    Dialog::Resync(model,msg);
  }

};

LUM_MAIN(Lum::OS::MainDialog<MainWindow>,L"DiskUsage")
