/*
 * Biloba
 * Copyright (C) 2004-2005 Guillaume Demougeot, Colin Leroy
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

/**
 * Biloba - Q1 2005
	* Game by Guillaume Demougeot <dmgt@wanadoo.fr>
	* Code by Colin Leroy <colin@colino.net>
	*
	* This file contains all the font management code.
	*/

#include <stdlib.h>
#include <SDL.h>
#include <SDL_image.h>

#include <ctype.h>
#include <string.h>

#include "utils.h"
#include "font.h"

static SDL_Surface *font_pic = NULL;
static SDL_Surface *cursor_pic = NULL;

static void load_font(void)
{
	SDL_Surface *tmp = NULL;
	assert(font_pic == NULL);
	
	tmp = biloba_load_image("font.png");
	font_pic = SDL_DisplayFormatAlpha(tmp);
	SDL_FreeSurface(tmp);
	cursor_pic = biloba_load_image("cursor.png");

	assert(font_pic != NULL);	
	assert(cursor_pic != NULL);	
}

#define CHARWIDTH 10
void draw_message(const char *message, int x, int y, int cursor)
{
	draw_message_with_update(message, x, y, cursor, TRUE);
}

void draw_message_with_update(const char *message, int x, int y, int cursor, int update)
{
	int i = 0;
	int cur_x = x;
	static SDL_Rect *dst_rect = NULL;
	static SDL_Rect *src_rect = NULL;
	if (dst_rect == NULL)
		dst_rect = malloc(sizeof(SDL_Rect));
	if (src_rect == NULL)
		src_rect = malloc(sizeof(SDL_Rect));
	
	src_rect->x = CHARWIDTH; /* skip space */
	src_rect->y = 0;
	src_rect->w = CHARWIDTH;
	src_rect->h = 16;

	if (font_pic == NULL) {
		load_font();
	}
	
	for (i = 0; i < strlen(message); i ++) {
		dst_rect->x = cur_x;
		dst_rect->y = y+4;
		
		if (message[i] >='0' && message[i] <= '9')
			src_rect->x = CHARWIDTH + (message[i]-'0')*CHARWIDTH;	   /* skip space */
		else if (message[i] >='A' && message[i] <= 'Z')
			src_rect->x = 11*CHARWIDTH + (message[i]-'A')*CHARWIDTH; /* skip space+nums */
		else if (message[i] >='a' && message[i] <= 'z')
			src_rect->x = 37*CHARWIDTH + (message[i]-'a')*CHARWIDTH; /* skip space,nums,caps */
		else
			src_rect->x = 0;

#ifdef DEBUG
		printf("putting char %c (%d,%d) at (%d,%d)\n",
				message[i], src_rect->x, src_rect->y,
				dst_rect->x, dst_rect->y);
#endif
		SDL_BlitSurface(font_pic, src_rect, screen, dst_rect);	
		cur_x += CHARWIDTH;
	}
	if (cursor) {
		put_image(cursor_pic, cur_x, y+4);
	}
#ifdef DEBUG
	printf("updating %d,%d pixels from %d,%d\n",
			cur_x-x,16,x,y+4);
#endif
	if (update)
		SDL_UpdateRect(screen, x, y+4, cur_x - x + (cursor?CHARWIDTH:0), 16);
}

void clear_text(int length, int x, int y)
{
	static SDL_Rect *dst_rect = NULL;
	if (dst_rect == NULL)
		dst_rect = malloc(sizeof(SDL_Rect));
	
	dst_rect->x = x;
	dst_rect->y = y+4;
	dst_rect->w = CHARWIDTH * length;
	dst_rect->h = 16;

	SDL_FillRect(screen, dst_rect, 0x00000000);
	SDL_UpdateRect(screen, dst_rect->x, dst_rect->y, dst_rect->w, dst_rect->h);
}
