/*
 * Copyright (C) 2008 Adam Harwell, all rights reserved.
 *
 * Contact: Adam Harwell <adam.harwell@trinity.edu>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <gtk/gtk.h>
#include <glib-object.h>

#include <libhildondesktop/libhildondesktop.h>
#include <libhildondesktop/hildon-desktop-popup-window.h>
#include <hildon/hildon-controlbar.h>
#include <hildon/hildon-hvolumebar.h>
#include <hildon/hildon-sound.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include <gconf/gconf-client.h>
#define GKEY "/apps/osso/sound"
#define GVOLUME "/apps/osso/sound/master_volume"

#include "advanced-backlight.h"

/* Utility macro which defines the plugin dynamic type bits */
HD_DEFINE_PLUGIN (AdvancedBacklightPlugin, advanced_backlight_plugin, STATUSBAR_TYPE_ITEM);

static void platform_check();
static void pressed(GtkToggleButton *button);
static void window_init();
static void update_icon();
static void set_level_dsme(GtkRange *range, gpointer user_data);
void popup_place(HildonDesktopPopupWindow *menu, gint *x, gint *y, gpointer user_data);
static void set_volume_gconf(HildonVolumebar *localvbar, gpointer user_data);
static void mute_volume_gconf(HildonVolumebar *localvbar, gpointer user_data);
static int get_volume_gconf();
static void gconf_notify_func(GConfClient * gclient, guint cnxn_id, GConfEntry * entry, gpointer user_data);

/////////////////////////////////////////////////////////////////////////////////////////////////////
//// I'm new to GTK so I'm not sure how to pass certain things around... So they're global. Ick. ////
/////////////////////////////////////////////////////////////////////////////////////////////////////
GtkWidget* window;
int lastvalue;
GtkWidget *button;
GConfClient * ggclient;

char oldfile[90]; // To keep the statusbar icon up to date, I keep track of the current icon (oldfile)
                  // so I can update only when it is different (no flickering icon).
int n810 = 0; // Check for n810 hardware to allow only n810 owners to turn off backlight completely.

int popstate = 0;

/////////////////////////////////////////////////////////////////////
//// This is the initialization function. It is called on entry. ////
/////////////////////////////////////////////////////////////////////
static void advanced_backlight_plugin_init(AdvancedBacklightPlugin *statusbar_plugin)
{
	//GtkWidget *button; // The button wasn't always global...

	platform_check(); // See if we are on an n810

	FILE* myfile = fopen("/sys/devices/platform/omapfb/panel/backlight_level","r");
	fscanf(myfile,"%d",&lastvalue); // A failed attempt to get the current backlight level. Gets only the software level...
	fclose(myfile);

	button = gtk_button_new(); // The statusbar icon...

	//sprintf(oldfile,"init");
	//update_icon();
	gtk_button_set_image(GTK_BUTTON(button), gtk_image_new_from_file("/usr/share/icons/hicolor/40x40/hildon/qgn_stat_displaybright4.png"));
	gtk_widget_show_all(button);
	gtk_container_add(GTK_CONTAINER(statusbar_plugin), button); // Shove the button into the statusbar

	ggclient = gconf_client_get_default();

	window_init();
	g_signal_connect(button, "button-press-event", G_CALLBACK(pressed),NULL); // On click it will call the function "pressed()".
}

//////////////////////////////////////////////////
//// Check to see if we are on an n810 or not ////
//////////////////////////////////////////////////
static void platform_check()
{
	char version;

	FILE * fp = fopen("/proc/component_version","r");
	fseek(fp,sizeof(char)*15,0); // The important character is the 16th in the file.
	fread(&version,sizeof(char),1,fp);
	fclose(fp);

	if (version == '4') // 4 means n810, 3 means n800.
		n810 = 1;
	else // Err on the side of caution.
		n810 = 0;
}

static void popstatedown(GtkWidget *window)
{
	popstate = 0;
}

static void popstateup(GtkWidget *window)
{
	popstate = 1;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////
//// Called on statusbar icon click. Pops a whole window at the moment, which is less than ideal ////
/////////////////////////////////////////////////////////////////////////////////////////////////////
static void pressed(GtkToggleButton *btn)
{
	if (popstate) // If the window is already initialized, we want to kill it.
		hildon_desktop_popup_window_popdown(HILDON_DESKTOP_POPUP_WINDOW(window));
	else
		hildon_desktop_popup_window_popup(HILDON_DESKTOP_POPUP_WINDOW(window), popup_place, btn, 2);
}

static void window_init()
{
	if (window == NULL)
	{
		//window = gtk_window_new(GTK_WINDOW_POPUP); // This window is global so that we can check if it is open (above).
		window = hildon_desktop_popup_window_new(0, GTK_ORIENTATION_VERTICAL, HD_POPUP_WINDOW_DIRECTION_RIGHT_BOTTOM);
		GtkWidget *cbar = hildon_controlbar_new();
		GtkWidget *vbar = hildon_hvolumebar_new();

		gconf_client_add_dir(ggclient, GKEY, GCONF_CLIENT_PRELOAD_NONE, NULL);
		gconf_client_notify_add(ggclient, GVOLUME, gconf_notify_func, vbar, NULL, NULL);

		if (get_volume_gconf(ggclient) < 0.0)
			hildon_volumebar_set_mute(HILDON_VOLUMEBAR(vbar), 1);
		hildon_volumebar_set_level(HILDON_VOLUMEBAR(vbar), (double)get_volume_gconf());
		
			//gconf: /apps/osso/sound/master_volume

		if (n810)	// Only let n810 set brightness 0, since they have a transflexive screen
			hildon_controlbar_set_range(HILDON_CONTROLBAR (cbar), 0, 127);
		else		// An n800 user would lose their backlight entirely and be confused
			hildon_controlbar_set_range(HILDON_CONTROLBAR (cbar), 1, 127);

		hildon_controlbar_set_value(HILDON_CONTROLBAR(cbar), lastvalue); // Return the controlbar to the last specified value.
		gtk_range_set_update_policy(GTK_RANGE(cbar), GTK_UPDATE_DELAYED);

		update_icon();

		GtkWidget* vbox = gtk_vbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(vbox), cbar, 0,0,25);
		gtk_box_pack_start(GTK_BOX(vbox), vbar, 0,0,5);
		gtk_widget_show(cbar);
		gtk_widget_show(vbar);

		GtkWidget* hbox = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(hbox), vbox, 0,0,25);
		gtk_widget_show(vbox);
		gtk_widget_show(hbox);

		//GdkColor gdkblack;
		//gdkblack.red = 65535;
		//gdkblack.green = 0;
		//gdkblack.blue = 0;

		GtkWidget * myframe = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(myframe), GTK_SHADOW_OUT);
		gtk_widget_show(myframe);
		gtk_container_add(GTK_CONTAINER(myframe), hbox);

		gtk_container_add(GTK_CONTAINER(window), myframe);
		//hildon_desktop_popup_window_attach_widget(HILDON_DESKTOP_POPUP_WINDOW(window), cbar);

		/*
		   GtkRequisition req;
		   gtk_widget_size_request(GTK_WIDGET(window), &req);
		   gint sw = gdk_screen_get_width(gtk_widget_get_screen(window));

		   gint x,y;
		   gdk_window_get_position(GTK_WIDGET(btn)->window, &x, &y);

		   y += GTK_WIDGET(btn)->allocation.y + GTK_WIDGET(btn)->allocation.height + 12;
		   x += GTK_WIDGET(btn)->allocation.x;

		   if (x + req.width > sw) {
		   x -= req.width - GTK_WIDGET(btn)->allocation.width;
		   }

		   gtk_window_move(GTK_WINDOW(window), x, y);
		   gtk_window_set_has_frame(GTK_WINDOW(window),TRUE);
		   */
		//gtk_widget_show(window);
		//hildon_desktop_popup_window_popup(HILDON_DESKTOP_POPUP_WINDOW(window), popup_place, btn, 2);

		g_signal_connect(cbar, "value-changed", G_CALLBACK(set_level_dsme), NULL); // When the slider moves, call the function "set_level_dsme()"
		g_signal_connect(window, "popdown-window", G_CALLBACK(popstatedown), NULL); // When the slider moves, call the function "set_level_dsme()"
		g_signal_connect(window, "popup-window", G_CALLBACK(popstateup), NULL); // When the slider moves, call the function "set_level_dsme()"
		g_signal_connect (G_OBJECT(vbar), "mute_toggled", G_CALLBACK(mute_volume_gconf), NULL);
		g_signal_connect (G_OBJECT(vbar), "level_changed", G_CALLBACK(set_volume_gconf), NULL);
	}
}

static void set_volume_gconf(HildonVolumebar *localvbar, gpointer user_data)
{
	gconf_client_set_int(ggclient, GVOLUME, (int)hildon_volumebar_get_level(localvbar), NULL);
	update_icon();
}

static void mute_volume_gconf(HildonVolumebar *localvbar, gpointer user_data)
{
	if (hildon_volumebar_get_mute(localvbar))
		gconf_client_set_int(ggclient, GVOLUME, -1 * (int)hildon_volumebar_get_level(localvbar), NULL);
	else if (hildon_volumebar_get_mute(localvbar) == 0)
		gconf_client_set_int(ggclient, GVOLUME, (int)hildon_volumebar_get_level(localvbar), NULL);
	update_icon();
}

static int get_volume_gconf()
{
	return gconf_value_get_int(gconf_client_get(ggclient, GVOLUME, NULL));
}

static void gconf_notify_func(GConfClient * gclient, guint cnxn_id, GConfEntry * entry, gpointer vbar)
{
	if (vbar != NULL) {
		//GConfValue *value = gconf_entry_get_value(entry);
		//if (gconf_value_get_int(value) >= 0) {
		if (get_volume_gconf() >= 0)
		{
			if (hildon_volumebar_get_mute(vbar) == 0)
				hildon_volumebar_set_level(HILDON_VOLUMEBAR(vbar), (double)get_volume_gconf());
			else
				hildon_volumebar_set_mute(HILDON_VOLUMEBAR(vbar), 0);
		}
		else if (hildon_volumebar_get_mute(vbar) == 0)
			hildon_volumebar_set_mute(HILDON_VOLUMEBAR(vbar), 1);
	}
	//hildon_play_system_sound("/usr/share/sounds/ui-default_beep.wav");
}

static void update_icon()
{
	char newfile[90];
	int volume = get_volume_gconf();
	int brightness_num = (int)(round((double)lastvalue / 21.16) + 1);
	char volume_char;
	if (volume >= 0) {
		volume_char = (char)('0' + (int)(ceil((double)volume / 25.0)));
	}
	else {
		volume_char = 'm';
	}
	
	sprintf(newfile,"/usr/share/icons/hicolor/40x40/apps/adv-backlight/adv-backlight-icon-%d.%c.png", brightness_num, volume_char);

	if (strcmp(oldfile,newfile) != 0) { // If proposed icon is different than the current one, go ahead and change it.
		gtk_button_set_image(GTK_BUTTON (button), gtk_image_new_from_file(newfile));
		strncpy(oldfile,newfile,90); // Keep track of the current icon after the change.
	}
}
/////////////////////////////////////////////////////////
//// Does the actual work for setting the backlight. ////
/////////////////////////////////////////////////////////
static void set_level_dsme(GtkRange *range, gpointer user_data)
{
	gdouble myval = gtk_range_get_value(range); // Get the value out of the GtkRange.
	int myint = floor(myval); // We need an integer, not a double.
	lastvalue=myint; // Keep track of the last value set.

	myint = myint * 2 + 1;

	char command[80] = "sudo /usr/sbin/chroot /mnt/initfs/ dsmetest -l ";

	char brightness[4];
	snprintf(brightness, 4, "%d", myint);
	strncat(command, brightness, 4);
	//strncat(command, " 2>> /home/user/dsmelog", 23);
	g_spawn_command_line_async(command,NULL);

	/*
	GtkWidget * dialog = gtk_message_dialog_new (NULL,
			GTK_DIALOG_DESTROY_WITH_PARENT,
			GTK_MESSAGE_ERROR,
			GTK_BUTTONS_CLOSE, state);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
	*/

	// Propose a statusbar icon change based on the new brightness level. Available images are 1-4, empty-full.
	
	update_icon();
}

void popup_place(HildonDesktopPopupWindow *menu, gint *x, gint *y, gpointer user_data) {
	GtkRequisition req;
	GtkWidget *btn = GTK_WIDGET(user_data);
	gint sw;

	//(void)push_in;

	gtk_widget_size_request(GTK_WIDGET(menu), &req);
	sw = gdk_screen_get_width(gtk_widget_get_screen(btn));

	gdk_window_get_position(btn->window, x, y);

	*y += btn->allocation.y + btn->allocation.height + 10;
	*x += btn->allocation.x + 40;

	if (*x + req.width > sw) {
		*x -= req.width - btn->allocation.width;
	}
}


////////////////////////////////////////////////////////////////////////////
//// This is necessary, according to the statusbar template I followed. ////
////////////////////////////////////////////////////////////////////////////
static void advanced_backlight_plugin_class_init(AdvancedBacklightPluginClass *class)
{
}
