import gtk
import gtk.gdk
import gobject

import pango
import hildon

from youamp import IS_MAEMO, KNOWN_EXTS

from youamp.ui.list import ListView
from youamp.ui.elements import PlaylistLabel

class AddDialog(gtk.FileChooserDialog):
    def __init__(self):
        gtk.FileChooserDialog.__init__(self, title="Add Songs")
        self.add_button(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
        self.add_button(gtk.STOCK_OPEN, gtk.RESPONSE_ACCEPT)
        #self.set_action(gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER)

        filter = gtk.FileFilter()
        #filter.add_mime_type("inode/directory")
        
        for ext in KNOWN_EXTS:
            filter.add_pattern("*."+ext)
        
        self.set_filter(filter)

        self.set_local_only(True)
        self.set_select_multiple(True)

        self.set_default_response(gtk.RESPONSE_OK)

class SongsTab(gtk.VBox):
    def __init__(self, playlist, controller, menu, vmenu):
        gtk.VBox.__init__(self)
        self.set_spacing(5)
        
        self.playlist = SonglistView(playlist, controller, menu)
        self.label = PlaylistLabel(self.playlist)
        
        self._navi = gtk.HBox()
        self._navi.set_spacing(5)
        self.pack_start(self._navi, expand=False)

        hbox = gtk.HBox()
        self._navi.pack_start(hbox, expand=False)
        
        # Search Label
        self._mode_label = gtk.Label()
        self._mode_label.set_size_request(100, -1)
        hbox.pack_start(self._mode_label, expand=False)
        
        arrow = gtk.Button()
        arrow.set_relief(gtk.RELIEF_NONE)
        arrow.add(gtk.Arrow(gtk.ARROW_DOWN, gtk.SHADOW_IN))
        arrow.unset_flags(gtk.CAN_FOCUS)
        hbox.pack_end(arrow, expand=False)
          
        vmenu.attach_to_widget(arrow, lambda *args: None)
        arrow.connect("clicked", self._view_menu_pop, vmenu)
        
        self._mode_box = hbox

        # Order Combo
        self.shuffle = gtk.ToggleButton(_("shuffle"))
        self.shuffle.unset_flags(gtk.CAN_FOCUS)
        self._navi.pack_end(self.shuffle, expand=False)
        self._shndl = self.shuffle.connect("toggled", self._on_shuffle_toggled)

        # Scrolled View
        self._scroll = gtk.ScrolledWindow()
        self._scroll.set_policy(gtk.POLICY_NEVER, gtk.POLICY_ALWAYS)
        self._scroll.set_shadow_type(gtk.SHADOW_IN)
        hildon.hildon_helper_set_thumb_scrollbar(self._scroll, True)
        
        self._scroll.add(self.playlist)

        self.pack_start(self._scroll)

    def _update_shuffle_btn(self, state):
        self.shuffle.handler_block(self._shndl)
        self.shuffle.set_active(state)
        self.shuffle.handler_unblock(self._shndl)

    def _on_shuffle_toggled(self, caller):
        self.playlist.get_model().shuffle(caller.get_active())
        self.playlist.set_cursor(self.playlist.get_model().pos)

    def _view_menu_pop(self, button, m):
        a = button.get_allocation()
        ap = button.get_parent_window().get_position()

        m.popup(None, None, lambda *arg: (a.x + ap[0], a.y + a.height + ap[1], False), 0, 0)

class SonglistView(ListView):
    def __init__(self, playlist, controller, menu):
        ListView.__init__(self, playlist)

        self.set_headers_visible(True)

        transl = (("title", _("Title")),
                  ("artist", _("Artist")),
                  ("album", _("Album")))

        cell = gtk.CellRendererText()
        cell.set_property("ellipsize", pango.ELLIPSIZE_END)

        for key, title in transl:
            col = gtk.TreeViewColumn(title, cell)
            col.set_sizing(gtk.TREE_VIEW_COLUMN_FIXED)
            col.set_resizable(True)
            col.set_fixed_width(200)
            col.set_min_width(100)
            col.set_cell_data_func(cell, self._data_func, key)
            self.append_column(col)
        
        # menu
        self._menu = menu
        self.connect("button-press-event", self._button_press)
        
        self.controller = controller

        if IS_MAEMO:
            self.tap_and_hold_setup(self._menu._w)
            self.connect("tap-and-hold", self._set_song)

        # Signals
        self.connect("row-activated", controller.song_selected)
        
        # redirect to controller
        self._handle_uri_drop = lambda *args, **kwargs: controller.on_uri_drop(self.get_model(), *args, **kwargs)
    
    def restore(self):
        if self.get_model() and len(self.get_model()) > 0:
            self.select_current()

    def _popup_menu(self, ev):
        if not IS_MAEMO:
            self._set_song()
            
        self._menu.popup(None, None, None, ev.button, ev.time)

    def _set_song(self, *args):
        ptr = self.get_pointer()
        pos = self.get_path_at_pos(ptr[0], ptr[1] - 25)[0]
                
        self._menu.song = self.get_model()[pos]
        self._menu.playlist = self.get_model()
        self._menu.remove_act = self.remove_selection
        self._menu.add_act = self.send_selection
        self._menu.rem.set_sensitive(self.get_model().backend is not None)

    def remove_selection(self):
        model, paths = self.get_selection().get_selected_rows()
        paths = [model.get_iter(p) for p in paths]

        model.remove(paths)

    def send_selection(self):
        model, paths = self.get_selection().get_selected_rows()
        paths = [model.get_iter(p) for p in paths]
        
        self.controller.add_to_current(self.get_uris(paths))
          
    def get_uris(self, paths):
        return ["file://"+self.get_model()[p].uri for p in paths]
    
    def select_current(self):
        self.set_cursor(self.get_model().pos)

    def _data_func(self, col, cell, model, itr, key):
        v = model[itr][key]
        cell.set_property("text", v)

class PlaylistView(SongsTab):
    __gsignals__ = {"model-deleted": (gobject.SIGNAL_RUN_LAST, None, (object,))}
    
    def __init__(self, playlist, controller, song_menu, pl_menu, vmenu):
        SongsTab.__init__(self, playlist, controller, song_menu, vmenu)

        self._navi.pack_start(self.label, expand=False)
        self._mode_label.set_text(_("Playlist"))
        
        self.controller = controller
        
        self.label.drag_dest_set(gtk.DEST_DEFAULT_ALL, self.playlist.SINK[2:3], gtk.gdk.ACTION_COPY)
        self.label.connect("drag-data-received", self.playlist._recieve_drag_data)

        self.menu = pl_menu

        self.label.connect("button-press-event", self.__popup_menu)
        
        if IS_MAEMO:
            self.tap_and_hold_setup(pl_menu._w)
            self.connect("tap-and-hold", self._set_data)
        
        self.connect("key-press-event", self._on_key_press)
        
        self.show_all()
        self.label.show_init()
    
    def add(self):
        ad = AddDialog()

        if ad.run() == gtk.RESPONSE_ACCEPT:
            self.controller.on_uri_drop(self.playlist.get_model(), ad.get_uris())
        
        ad.destroy()

    def replace(self, model):
        self.playlist.set_model(model)
        self._update_shuffle_btn(model.shuffled)
        
        if model.title is not None: # FIXME call show_init at right location instead
            self.label.label.set_text(model.title)
    
    def restore(self):
        self.playlist.restore()
    
    def _on_key_press(self, caller, ev):
        key = gtk.gdk.keyval_name(ev.keyval)

        if key == "Delete":
            self.playlist.remove_selection()

    def remove(self):
        self.playlist.get_model().delete()
        self.emit("model-deleted", self.playlist.get_model())
    
    def _set_data(self, *a):
        self.menu.view = self
    
    def __popup_menu(self, caller, ev):
        if ev.button == 3 and not self.label.editing:
            if not IS_MAEMO:
                self._set_data()
                
            self.menu.popup(None, None, None, ev.button, ev.time)