import gtk
import gtk.gdk
import time

from youamp.ui.window import Window
from youamp.ui.preferences import Preferences
from youamp.ui.searchview import SearchView

from youamp.ui.playlist import PlaylistView, PlaylistMenu
from youamp.ui.popupmenu import SongMenu

from youamp.ui.elements import *
from youamp.ui import xml_escape

from youamp import VERSION, data_path

class UserInterface:
    NOTIFY_STR = "{0} <i>{1}</i>\n{2} <i>{3}</i>"

    def __init__(self, controller):
        player = controller.player
        config = controller.config
        library = controller.library
        scrobbler = controller.scrobbler
        self._player = player
        self._config = config
        self._press_time = {"F7": None, "F8": None}
        
        # Build Interface
        xml = GladeAdapter(data_path + "interface.glade")

        # Create Views
        smenu = SongMenu(config, player, xml)
        plmenu = PlaylistMenu(xml)

        self._view = [SearchView(controller.main_list, player, library, config, smenu, xml)]
        self._cur_view = self._view[0]

        #lists = [PlaylistView(l, player, library, smenu, plmenu) for l in library.get_playlists()]
        #self._view += lists
        
        # Windows
        self.window = Window(player, xml)
        self.window._w.set_menu(xml.get_object("main_menu"))
        
        about = AboutDialog(self.window._w)

        prefs = Preferences(config, scrobbler, xml)

        # Controls
        ctls = Controls(player, config, xml)
        
        # Menu
        if config["is-browser"]:
            xml.get_object("view_browse").set_active(True)
        else:
            xml.get_object("view_search").set_active(True)

        if not config["shuffle"]:
            xml.get_object("order_"+config["order-by"]).set_active(True)
               
        # Add {Search, Playlist}Views
        xml.get_object("main_vbox").pack_end(self._cur_view.view)
        
        if False:
            self.nb = xml.get_object("notebook1")
            
            for v in self._view:
                self.nb.append_page(v.view, v.label)
            
            for lv in lists:
                self.nb.set_tab_reorderable(lv.view, True)
    
            # disable implicit playlist change
            self.nb.connect("switch-page", self._change_playlist, player)
            self.nb.connect("page-reordered", self._move_lib_first)

        # Signals
        xml.connect_signals({"show-preferences": lambda *args: prefs.cshow(),
                             "show-about": lambda *args: about.show(),
                             "order-album": lambda caller: controller.set_list_order("album"),
                             "order-date": lambda caller: controller.set_list_order("date"),
                             "order-playcount": lambda caller: controller.set_list_order("playcount"),
                             "hide-on-delete": gtk.Widget.hide_on_delete,
                             "toggle": lambda caller: player.toggle(),
                             "previous": lambda caller: player.previous(),
                             "next": lambda caller: player.next(),
                             "view-search": lambda caller: self._cur_view.search_mode(),
                             "view-browse": lambda caller: self._cur_view.browse_mode(),
                             "select-current": lambda caller: self._cur_view.restore(),
                             "new-playlist": lambda caller: self.new_playlist(config, player, library, smenu, plmenu)})
        
        self.window._w.connect("key-press-event", self._handle_keypress)
        self.window._w.connect("key-release-event", self._handle_keyrelease)
        self.window._w.connect("destroy", controller.quit)
        ctls.pos.connect("change-value", lambda caller, *a: player.seek_to(caller.get_value()))
        
        self._toggle = xml.get_object("playback_item")
        self._thndl = self._toggle.connect("toggled", lambda caller: player.toggle())
        player.connect("toggled", self._watch_toggled)
        player.playlist.connect("list-switched", self._switch_to)
        
        # change to library on browsing
        # config.notify_add("is-browser", lambda *args: self.nb.set_current_page(0))

    def _switch_to(self, caller, model):
        i = [e.get_model() for e in self._view].index(model)
        n = self.nb.page_num(self._view[i].view)
        self.nb.set_current_page(n)

    def new_playlist(self, config, player, library, smenu, plmenu):
        pl = PlaylistView(library.get_new_playlist(), player, library, smenu, plmenu)
        self._view.append(pl)
        self.nb.append_page(pl.view, pl.label)
        self.nb.set_tab_reorderable(pl.view, True)
        self.nb.set_current_page(-1)

    def _move_lib_first(self, *args):
        self.nb.reorder_child(self._view[0].view, 0)

    def _change_playlist(self, nb, page, num, player):
        self._cur_view = nb.get_nth_page(num).top
        player.playlist.set(self._cur_view.view.playlist)

    def show_notification(self, song):
        return # no notifications on maemo for now...
        body = self.NOTIFY_STR.format(
                                    _("by"),
                                    xml_escape(song["artist"]),
                                    _("from"),
                                    xml_escape(song["album"]))
        self._notify.update(xml_escape(song["title"]), body)

        cover = song.cover_image((128, 128))
        
        if cover is None:
            cover = gtk.icon_theme_get_default().load_icon("audio-x-generic", 128, 0)
            
        self._notify.set_icon_from_pixbuf(cover)
        self._notify.show()
    
    def restore(self):
        for v in self._view:
            v.restore()
    
    def _watch_toggled(self, caller, state):
        self._toggle.handler_block(self._thndl)
        self._toggle.set_active(state)
        self._toggle.handler_unblock(self._thndl)
                  
    def _handle_keypress(self, widget, event):
        key = gtk.gdk.keyval_name(event.keyval)
        
        if key == "Right":
            self._player.next()
            return True # event handled
        elif key == "Left":
            self._player.previous()
            return True # event handled
        if key in ("F7", "F8"):
            self._press_time[key] = time.time()
        elif key == "F6":
            self.window.toggle_fullscreen()

    def _handle_keyrelease(self, widget, event):
        key = gtk.gdk.keyval_name(event.keyval)
        
        if key == "F7":
            if time.time() - self._press_time[key] < 1:
                self._on_vol_up()
            else:
                self._player.next()
        elif key == "F8":
            if time.time() - self._press_time[key] < 1:
                self._on_vol_down()
            else:
                self._player.previous()

    def _on_vol_down(self):
        vol = self._config["volume"]
        self._config["volume"] = max(0.0, vol - 0.1) # cut to range
    
    def _on_vol_up(self):
        vol = self._config["volume"]
        self._config["volume"] = min(1.0, vol + 0.1) # cut to range
