import gtk
import hildon
import gtk.gdk
import gtk.glade

from youamp import data_path, VERSION, IS_MAEMO
from youamp.ui import xml_escape

gtk.glade.textdomain("youamp")

GPLv3 = """YouAmp Music Player

Copyright (C) 2009 Pavel Rojtberg

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>."""

class GladeAdapter(gtk.glade.XML):
    def __init__(self, file):
        gtk.glade.XML.__init__(self, file)
    
    def get_object(self, name):
        return self.get_widget(name)
    
    def connect_signals(self, dict):
        self.signal_autoconnect(dict)
    
    def set_translation_domain(self, name):
        gtk.glade.textdomain(name)

class AboutDialog(gtk.AboutDialog):
    def __init__(self, parent):
        gtk.AboutDialog.__init__(self)
        
        self.set_transient_for(parent)
        
        self.set_name("YouAmp")
        self.set_version(VERSION)
                    
        self.set_copyright("Copyright (C) 2009 Pavel Rojtberg")
        self.set_license(GPLv3)
        self.set_authors(("Pavel Rojtberg",))
        self.set_artists(("Pavel Rojtberg", "Tango Icon Team"))
        self.set_translator_credits(_("translator-credits"))
        self.set_comments("A lightweight Music Player for the Internet Tablet")
        
        if IS_MAEMO:
            logo = gtk.gdk.pixbuf_new_from_file("/usr/share/pixmaps/youamp.png")
            self.set_logo(logo)
            
        self.connect("response", self.hide_on_delete)
        self.connect("delete-event", self.hide_on_delete)

class Controls:
    def __init__(self, player, config, xml):        
        # is playing
        self.img = dict()
        self.img[False] = gtk.image_new_from_file(data_path+"media-playback-start.png")
        self.img[True] = gtk.image_new_from_file(data_path+"media-playback-pause.png")
        
        self._config = config
        self._player = player
        
        controls = xml.get_object("controls_box")
        
        tb = xml.get_object("toggle_button")
        tb.set_image(gtk.image_new_from_pixbuf(self.img[False].get_pixbuf()))
        
        xml.get_object("button3").set_image(gtk.image_new_from_file(data_path+"media-skip-backward.png"))
        xml.get_object("button4").set_image(gtk.image_new_from_file(data_path+"media-skip-forward.png"))
        
        self.pos = hildon.Seekbar()
        self.pos.set_draw_value(False)
        self.pos.set_range(0.0, 1.0)
        self.pos.set_increments(0.01, 0.01)
        self.pos.unset_flags(gtk.CAN_FOCUS)
        self.pos.set_update_policy(gtk.UPDATE_DELAYED)
        controls.pack_start(self.pos)
                
        self.volume = hildon.HVolumebar()
        self.volume.set_size_request(200, -1)
        self.volume.set_level(config["volume"] * 100)
        self.volume.connect("level-changed", self._on_hvolume_changed)
        self.volume.connect("mute-toggled", self._on_mute)
        self.volume.unset_flags(gtk.CAN_FOCUS)
        controls.pack_end(self.volume, expand=False)
                        
        config.notify_add("volume", self._on_conf_volume_changed)
        player.connect("seek-changed", lambda caller, new_seek: self.pos.set_value(new_seek))
        player.connect("toggled", lambda c, playing: tb.set_image(gtk.image_new_from_pixbuf(self.img[playing].get_pixbuf())))
        
        controls.show_all()
   
    def _on_conf_volume_changed(self, client, cxn_id, entry, data):
        self.volume.set_level(entry.get_value().get_float()*100)
    
    def _on_mute(self, hvolume):
        if hvolume.get_mute():
            self._player.mute()
        else:
            self._config.notify("volume") # refresh volume
    
    def _on_hvolume_changed(self, hvolume):
        if(hvolume.get_mute()):
            self._player.mute()
            return
        else:
            val = hvolume.get_level()/100
        
        self._config["volume"] = val
