import gtk
import gtk.gdk
import md5

from youamp import VERSION

class PrefLabel(gtk.Label):
    def __init__(self, txt):
        gtk.Label.__init__(self)
        self.set_markup("<b>%s</b>" % txt)
        self.set_alignment(0, 0.5)

class Preferences(gtk.Window):
    def __init__(self, *args):
        self._created = False
        self._args = args
    
    def cshow(self):
        if not self._created:
            self._create(*self._args)
            self._created = True
        
        self.show()

    def _create(self, parent, config, scrobbler):
        gtk.Window.__init__(self)
        self.config = config
        self._scrobbler = scrobbler
        self._scrobbler.connect("error", self._update_lfm_state)
        
        self.set_title(_("Preferences"))
        self.set_transient_for(parent)
        self.set_type_hint(gtk.gdk.WINDOW_TYPE_HINT_DIALOG)
        self.set_default_size(500, -1)
        
        self._vbox = gtk.VBox()
        self.add(self._vbox)
        
        nb = gtk.Notebook()
        nb.set_border_width(5)
        self._vbox.pack_start(nb)
        
        general = gtk.VBox()
        general.set_border_width(5)
        nb.append_page(general, gtk.Label(_("General")))
        
        mf_label = PrefLabel( _("Music Folder"))
        general.pack_start(mf_label, expand=False, padding=5)
        
        music_folder = gtk.FileChooserButton(_("Music Folder"))
        music_folder.set_action(gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER)
        music_folder.set_local_only(True)
        music_folder.set_current_folder(self.config["music-folder"])
        music_folder.connect("current-folder-changed", self._on_mf_changed)
        general.pack_start(music_folder, padding=5, expand=False)
   
        rg_label = PrefLabel(_("Replaygain Preamp (db)"))
        general.pack_start(rg_label, expand=False, padding=5)
        
        rg_scale = gtk.HScale()
        rg_scale.set_range(-10.0, 10.0)
        rg_scale.set_value(self.config["rg-preamp"])
        rg_scale.connect("value-changed", self._on_rg_changed)
        rg_scale.set_update_policy(gtk.UPDATE_DELAYED)
        general.pack_start(rg_scale, expand=False)
        
        no_rg_label = PrefLabel(_("No Replaygain Amp (db)"))
        general.pack_start(no_rg_label, expand=False, padding=5)

        no_rg_scale = gtk.HScale()
        no_rg_scale.set_range(-10.0, 10.0)
        no_rg_scale.set_value(self.config["no-rg-preamp"])
        no_rg_scale.connect("value-changed", self._on_no_rg_changed)
        no_rg_scale.set_update_policy(gtk.UPDATE_DELAYED)
        general.pack_start(no_rg_scale, expand=False)
        
        lastfm = gtk.VBox()
        lastfm.set_border_width(5)
        nb.append_page(lastfm, gtk.Label("last.fm"))

        login_label = PrefLabel(_("Login"))
        lastfm.pack_start(login_label, expand=False, padding=5)

        login = gtk.Table(2, 2)
        login.set_col_spacings(10)
        lastfm.pack_start(login, expand=False, padding=5)
        
        user_label = gtk.Label(_("Username"))
        login.attach(user_label, 0, 1, 0, 1, xoptions=0)
        
        user_entry = gtk.Entry()
        user_entry.set_text(self.config["lastfm-user"])
        user_entry.connect("focus-out-event", self._on_lfm_user_changed)
        login.attach(user_entry, 1, 2, 0, 1, xoptions=0)

        pass_label = gtk.Label(_("Password"))
        pass_label
        login.attach(pass_label, 0, 1, 1, 2, xoptions=0)

        pass_entry = gtk.Entry()
        pass_entry.set_visibility(False)
        pass_entry.set_text("******" if self.config["lastfm-pass"] != "" else "")
        pass_entry.connect("focus-out-event", self._on_lfm_pass_changed)
        login.attach(pass_entry, 1, 2, 1, 2, xoptions=0)

        status_label = PrefLabel(_("Status"))
        lastfm.pack_start(status_label, expand=False, padding=5)
        
        ss = scrobbler.get_state()
        
        if ss == "OK":
            state = _("Connected")
        elif ss == "BADAUTH":
            state = _("Wrong login data")
        else:
            state = _("Disconnected")
        
        self._lastfm_state = gtk.Label(state)
        lastfm.pack_start(self._lastfm_state, expand=False)

        hbox = gtk.HBox()
        hbox.set_border_width(5)
        self._vbox.pack_start(hbox, expand=False)
        
        close = gtk.Button(stock=gtk.STOCK_CLOSE)
        close.connect("clicked", self.hide_on_delete)
        hbox.pack_end(close, expand=False)
        
        self.connect("delete-event", self.hide_on_delete)
        self.show_all()
    
    def _update_lfm_state(self, caller, e, msg):
        if msg == "BADAUTH":
            state = _("Wrong login data")
        else:
            state = _("Disconnected")
        
        self._lastfm_state.set_text(state)
    
    def _try_login(self):
        self._scrobbler.login(self.config["lastfm-user"], self.config["lastfm-pass"], ("you", VERSION))
        self._lastfm_state.set_text(_("Connected"))
            
    def _on_lfm_user_changed(self, caller, ev):
        txt = caller.get_text()
        
        if self.config["lastfm-user"] == txt:
            return
        
        self.config["lastfm-user"] = txt
        self._try_login()

    def _on_lfm_pass_changed(self, caller, ev):
        txt = caller.get_text()
        
        if txt == "******":
            return

        self.config["lastfm-pass"] = md5.md5(txt).hexdigest()
        self._try_login()
 
    def _on_mf_changed(self, caller):
        self.config["music-folder"] = caller.get_current_folder()
    
    def _on_rg_changed(self, caller):
        self.config["rg-preamp"] = caller.get_value()
    
    def _on_no_rg_changed(self, caller):
        self.config["no-rg-preamp"] = caller.get_value()
