import gtk
import gtk.gdk
import hildon
import pango

from youamp.ui.controls import Controls
from youamp.ui.searchview import SearchView
from youamp.ui.detailswindow import DetailsWindow
from youamp import IS_MAEMO, img_path

class Menu(gtk.Menu):
    def __init__(self, config):
        gtk.Menu.__init__(self)
        
        # view submenu
        view = gtk.MenuItem(_("View"))
        self.append(view)
        
        view_sub = gtk.Menu()
        view.set_submenu(view_sub)
        
        self.search = gtk.RadioMenuItem(None, _("Search"))
        view_sub.append(self.search)
        
        self.browse = gtk.RadioMenuItem(self.search, _("Browse"))
        view_sub.append(self.browse)
    
        # order submenu
        order = gtk.MenuItem(_("Order"))
        self.append(order)
        
        order_sub = gtk.Menu()
        order.set_submenu(order_sub)
        
        self.order = dict()
        self.order["artist"] = gtk.RadioMenuItem(None, _("Artist"))
        self.order["date"] = gtk.RadioMenuItem(self.order["artist"], _("Date added"))
        self.order["playcount"] = gtk.RadioMenuItem(self.order["artist"], _("Playcount"))

        for k in ("artist", "date", "playcount"):
            order_sub.append(self.order[k])
     
        self.prefs = gtk.ImageMenuItem(gtk.STOCK_PREFERENCES)
        self.append(self.prefs)
        
        self.about = gtk.ImageMenuItem(gtk.STOCK_ABOUT)
        self.append(self.about)
        
        # restore config
        if not config["shuffle"]:
            self.order[config["order-by"]].set_active(True)

        (self.browse if config["is-browser"] else self.search).set_active(True)

        self.show_all()

class CoverImage(gtk.Image):
    def __init__(self):
        gtk.Image.__init__(self)
        self.file = ""
    
    def set_generic(self):
        name = "audio-x-generic" if not IS_MAEMO else "gnome-mime-audio-x-mp3"
        pb = gtk.icon_theme_get_default().load_icon(name, self.size_request()[1], 0)
        self.set_from_pixbuf(pb)
        self.file = None
    
    def set_from_file(self, file):
        pb = gtk.gdk.pixbuf_new_from_file_at_size(file, *self.size_request())
        self.set_from_pixbuf(pb)
        self.file = file

class Window(hildon.Window):
    def __init__(self, player, config, library):
        hildon.Window.__init__(self)
        
        self._player = player
        self._config = config
        self._library = library

        self.set_default_size(800, 480)
        self.set_title(_("No Track"))
        self._is_fullscreen = False
        
        # Content Box
        vbox = gtk.VBox()
        vbox.set_spacing(5)
        
        hbox = gtk.HBox()
        hbox.set_spacing(5)
        vbox.pack_start(hbox, expand=False)
                
        # Cover
        evbox = gtk.EventBox()
        self._cover = CoverImage()
        evbox.add(self._cover)
        evbox.connect("button-press-event", self._display_details)
        hbox.pack_start(evbox, expand=False)
        
        vbox1 = gtk.VBox()
        hbox.pack_start(vbox1)
        
        # Details
        self._details = DetailsWindow.get_instance()
        
        # Label
        self._label = gtk.Label()
        self._label.set_alignment(0, 0)
        self._label.set_justify(gtk.JUSTIFY_LEFT)
        self._label.set_ellipsize(pango.ELLIPSIZE_END)
        self._label.set_selectable(False)
        self._label.set_markup("<b><big>%s</big></b>" % _("No Track"))
        vbox1.pack_start(self._label, expand=False)

        # Controls
        self._controls = Controls(self._player, self._config)
        vbox1.pack_start(self._controls, expand=False)
        
        hbox.show_all()
        
        csize = hbox.size_request()[1]
        self._cover.set_size_request(csize, csize)
        
        # View
        self.view = SearchView(self._config, self._player, self._library)
        vbox.pack_start(self.view)
        
        vbox.show()
        self.add(vbox)
    
        self._player.connect("song-changed", self._update_songinfo)
        
        self.show()

    def _display_details(self, *args):
        song = self._player.playlist[self._config["pos"]]      
        self._details.set_data(song)
        self._details.cshow()
   
    def set_menu(self, menu):
        hildon.Window.set_menu(self, menu)
        
        menu.search.connect("activate", lambda caller: self.view.search_mode())
        menu.browse.connect("activate", lambda caller: self.view.browse_mode())
    
    def toggle_fullscreen(self):
        if self._is_fullscreen:
            self.unfullscreen()
        else:
            self.fullscreen()
        
        self._is_fullscreen = not self._is_fullscreen
    
    def _update_songinfo(self, caller, newsong):
        # text
        self.set_title("%s - %s" % (newsong["title"], newsong["artist"]))
                
        label_txt = "<b><big>%s</big></b> %s <i>%s</i> %s <i>%s</i>" % (newsong["title"], _("by"), newsong["artist"], _("from"), newsong["album"])
        label_txt = label_txt.replace("&", "&amp;") # escape the xml escape char
        
        if not self._player.has_rg_info():
            label_txt = """<span foreground="red">%s</span>""" % label_txt
        
        self._label.set_markup(label_txt)
        
        # image
        cover_file = newsong.cover_uri()
        
        if cover_file == self._cover.file:
            return
        elif cover_file is None:
            self._cover.set_generic()
        else:
            self._cover.set_from_file(cover_file)
            
        if not IS_MAEMO:
            self.set_icon_list(self._cover.get_pixbuf())
        
        
