import gtk
import hildon
import gobject
import time
import urllib2
import sys

from youamp.ui.window import Window, Menu
from youamp.ui.about import AboutDialog
from youamp.ui.preferences import Preferences

from youamp import Playlist, IS_MAEMO, VERSION

from youamp.library import Library, check_db
from youamp.player import Player
from youamp.config import Config
import youamp.scrobbler as scrobbler

if IS_MAEMO:
    import osso
    
    osso_c = osso.Context("net.madman2k.youamp", VERSION, False)
        
class Controller:
    def __init__(self):
        check_db()
        self._config = Config("/apps/youamp/")
        self._check_config()
        self._player = Player(self._config)
        self._library = Library()
        
        self._refresh_playlist()
        
        self._window = Window(self._player, self._config, self._library)
        self._banner = None
        self._scrobbler = None
    
        # programm
        program = hildon.Program()
        program.add_window(self._window)
        program.set_property("can-hibernate", True)

        gobject.set_application_name("YouAmp")
   
        # last.fm submission
        if self._config["lastfm-user"] != "":
            try:
                self._scrobbler = scrobbler.Scrobbler(self._config["lastfm-user"], self._config["lastfm-pass"], ("you", VERSION))
            except (urllib2.URLError, scrobbler.AuthError), e:
                sys.stderr.write("lastfm.login: %s\n" % e)
            
        # RPC
        if IS_MAEMO:
            osso_rpc = osso.Rpc(osso_c)
            osso_rpc.set_rpc_callback("net.rojtberg.youamp", \
                                      "/net/rojtberg/youamp", \
                                      "net.rojtberg.youamp", self._handle_rpc)
        
        # dialogs
        self._about_dialog = None
        self._preferences = None
                    
        # menu
        menu = Menu(self._config)
        self._window.set_menu(menu)
                
        if self._config["shuffle"]:
            self._player.playlist.shuffle(True)

        # restore config
        if self._player.tracks() > 0:
            self._player.load_track()

        # menu signals
        menu.prefs.connect("activate", self._show_preferences)
        menu.about.connect("activate", self._show_about_dialog)
        
        menu.order["artist"].connect("activate", lambda caller: self._set_list_order("artist"))
        menu.order["date"].connect("activate", lambda caller: self._set_list_order("date"))
        menu.order["playcount"].connect("activate", lambda caller: self._set_list_order("playcount"))

        # signals
        self._window.connect("destroy", self.quit)
        self._window.connect("key-press-event", self._handle_keypress)
        
        self._player.connect("song-played", self._on_song_played)
        self._config.notify_add("music-folder", self._refresh_library)
        self._player.connect("song-changed", self._now_playing)

        self._library.connect("update-complete", self._on_library_updated)

        # start updating library
        self._refresh_library()
    
    def _refresh_library(self, *args):
        self._banner = hildon.hildon_banner_show_animation(self._window, "qgn_indi_pball_a", _("Updating database"))
        self._library.update(self._config["music-folder"])
        
    def _handle_rpc(self, interface, method, arguments, user_data):
        action = dict();
        action["toggle"] = self._window._controls.on_toggle
        action["prev"] = self._player.previous
        action["next"] = self._player.next
        
        action[method]()
    
    def _check_config(self):
        if not self._config.has_key("volume"):
            # write default config
            self._config["volume"] = 0.5
            self._config["search-str"] = ""
            self._config["search-artist"] = ""
            self._config["search-album"] = ""
            self._config["pos"] = 0
            self._config["shuffle"] = False
            self._config["is-browser"] = False
            self._config["rg-preamp"] = 0     # preamp to adjust the default of 89db (value: db)
            self._config["no-rg-preamp"] = -10  # amp value to be used if no rg info is available (value: db)
        
        # order by was added in v0.3.5, therefore check for it seperately
        if not self._config.has_key("order-by"):
            self._config["order-by"] = "artist"
        
        # music-folder was added in v0.3.8
        if not self._config.has_key("music-folder"):
            self._config["music-folder"] = "/home/user"
            
        # last.fm support was added in v0.4.0
        if not self._config.has_key("lastfm-user"):
            self._config["lastfm-user"] = ""
            self._config["lastfm-pass"] = ""

    def _set_list_order(self, order):
        self._config["order-by"] = order
        self._config["shuffle"] = False
        self._config["pos"] = 0
        
        self._refresh_playlist()

    def _refresh_playlist(self):
        playlist = self._library.get_tracks(self._config)
            
        self._player.playlist.update(playlist)

    def _on_song_played(self, player, song):
        played = player.get_time()
        
        # workaround for bugged meamo
        if not song.has_key("duration"):
            return
        
        if (played < song["duration"]/2 and played < 240) or song["duration"] < 30:
            return
        
        self._library.increment_played(song.uri)
        
        if self._scrobbler.is_connected():
            try:
                self._scrobbler.submit(song["artist"], song["title"], int(time.time()), album=song["album"], length=song["duration"])
            except urllib2.URLError, e:
                sys.stderr.write("lastfm.submit: %s\n" % e)

    def _now_playing(self, caller, song):
        if self._scrobbler.is_connected():
            try:
                self._scrobbler.now_playing(song["artist"], song["title"], song["album"])
            except urllib2.URLError, e:
                sys.stderr.write("lastfm.now_playing: %s\n" % e)
        
    def _show_about_dialog(self, caller):
        if self._about_dialog is None:
            self._about_dialog = AboutDialog(self._window)
        
        self._about_dialog.show()
    
    def _show_preferences(self, caller):
        if self._preferences is None:
            self._preferences = Preferences(self._window, self._config, self._scrobbler)
            self._preferences.show_all()
    
        self._preferences.show()

    def _on_library_updated(self, caller, was_updated):
        gobject.idle_add(self._banner.destroy)
        
        if was_updated:
            gobject.idle_add(self._refresh_playlist)
    
    def _handle_keypress(self, widget, event):
        key = gtk.gdk.keyval_name(event.keyval)
        
        if key == "Right":
            self._player.next()
            return True # event handled
        elif key == "Left":
            self._player.previous()
            return True # event handled
        if key == "F7":
            self._on_vol_up()
        elif key == "F8":
            self._on_vol_down()
        elif key == "F6":
            self._window.toggle_fullscreen()
    
    def _on_vol_down(self):
        vol = self._config["volume"]
        self._config["volume"] = max(0.0, vol - 0.1) # cut to range
    
    def _on_vol_up(self):
        vol = self._config["volume"]
        self._config["volume"] = min(1.0, vol + 0.1) # cut to range

    def start(self):
        gtk.main()
        
    def quit(self, caller=None):        
        if self._scrobbler.is_connected():
            try:
                self._scrobbler.flush()
            except urllib2.URLError, e:
                sys.stderr.write("lastfm.submit: %s\n" % e)
        
        gtk.main_quit()