/*
* This file is part of TuxPuck based on 2001-2002 Jacob Kroon's version
*
* Copyright (C) 2005 INdT - Instituto Nokia de Tecnologia
* http://www.indt.org/maemo
*
* This software is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public License
* as published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA
*
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <SDL.h>
#include "video.h"
#include "audio.h"
#include "tuxpuck.h"

/* defines */
#ifdef windows
#define SETTINGSFILE "tuxpuck.ini"
#else
#define SETTINGSFILE _settings_file
#endif

#ifdef MAEMO
	#if HGW_FUNC
	#include <hgw/hgw.h>
	    HgwContext *hgw_context = NULL;
	#endif

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

// where the plugin will save the users preferences
#define SETTINGS_OPPONENT "/apps/osso/tuxpuck/opponent"
#define SETTINGS_SOUND    "/apps/osso/tuxpuck/enable_sound"

#define BACKBORDERBOTTOM 450
#define BACKBORDERTOP	 415
#define BACKBORDERLEFT	  35
#define BACKBORDERRIGHT  135

//hildon-games-wrapper functions
/**
 * Pauses the game
 */
static int exit_callback(int errcode);
/**
 * Quit the game
 */
static int quit_callback(int errcode);
/**
 * Cleans saved game
 */
static int flush_callback(int errcode);

void save_state(int opponent, Uint8 score_player, Uint8 score_opponent);
int load_state(Uint8 *opponent, Uint8 *score_player, Uint8 *score_opponent);
#endif //MAEMO

/* externals */
extern time_t time(time_t *);
extern void run_intro(void);
extern AIPlayer *tux_create(Pad *, Puck *);
extern AIPlayer *arcana_create(Pad *, Puck *);

/* structs */
typedef struct {
  Uint8 sound;
  Uint8 fullscreen;
  Uint8 mouse_speed;
} Settings;

/* statics */
static Settings *_settings = NULL;
#ifndef windows
static char _settings_file[200];
#endif


/* functions */
static int _play_match(Uint8 opponent)
{
  int next_opponent;
  SDL_Event event;
  Uint8 loop = 1, scorer = 0;
  Uint32 elapsed_time = 0;
  char buffer[50];
  HumanPlayer *p1 = NULL;
  AIPlayer *p2 = NULL;
  Timer *timer = NULL;
  float alpha = 0.0;
  HgwMessage *msg;
  HgwMessageFlags flags;
#ifdef MAEMO
  int loaded_state = false; 	
  Uint8 score_player=0, score_opponent=0;
#else
  Menu *exit_menu;
#endif
  Menu *again_menu;
  memset(buffer, 0, 50);
  board_init();
  scoreboard_init();
  video_save();
  p1 = human_create(board_get_pad(1), "Human");

#ifdef MAEMO
  loaded_state = load_state(&opponent, &score_player, &score_opponent);
#endif

  switch (opponent) {
#ifndef MAEMO
	case 1:
	    p2 = tux_create(board_get_pad(2), board_get_puck());
	    break;
	case 2:
	    p2 = arcana_create(board_get_pad(2), board_get_puck());
	    break;
#else 	
	case 0:
	    p2 = tux_create(board_get_pad(2), board_get_puck());
	    break;
	case 1:
	    p2 = arcana_create(board_get_pad(2), board_get_puck());
	    break;
#endif //MAEMO
  }
#ifndef MAEMO
  exit_menu = menu_create(2);
  menu_add_field(exit_menu, 0, 1, "Continue");
  menu_add_field(exit_menu, 1, 1, "Surrender");
#endif  
  again_menu = menu_create(2);
  menu_add_field(again_menu, 0, 1, "Play Again");
#ifdef MAEMO
  menu_add_field(again_menu, 1, 1, "Exit");
#else
  menu_add_field(again_menu, 1, 1, "Main Menu");
#endif
  timer = tuxpuck_timer_create();
  timer_reset(timer);
  while (loop) {
    if ( hgw_msg_check_incoming(hgw_context, msg, flags) == HGW_ERR_COMMUNICATION ) {
      printf("message!\n");
    }
    while (SDL_PollEvent(&event))
      if (event.type == SDL_MOUSEBUTTONDOWN) {
	loop = 0;
	alpha = 1.0;
      }
    SDL_Delay(SLEEP);
    timer_update(timer);
    timer_reset(timer);
    elapsed_time = timer_elapsed(timer);
    alpha += elapsed_time * 0.001;
    if (alpha > 1.0) {
      loop = 0;
      alpha = 1.0;
    }
    board_clean_up();
    scoreboard_erase();
    aiplayer_erase(p2);
    aiplayer_set_alpha(p2, (Uint8) (alpha * 255));
    scoreboard_set_alpha((Uint8) (alpha * 255));
    entity_set_alpha((Entity *) board_get_puck(), (Uint8) (alpha * 255));
    entity_set_alpha(board_get_pad(1), (Uint8) (alpha * 255.0 / 2.0));
    entity_set_alpha(board_get_pad(2), (Uint8) (alpha * 255.0 / 2.0));
#ifdef MAEMO
	if (loaded_state == true) {
		p2->points = score_opponent;
		human_set_points(p1, score_player);
		scoreboard_set_points(score_player, score_opponent);
		_blit_all_points();
	}
#endif
    aiplayer_blit(p2);
    board_reblit();
    scoreboard_blit();
    video_update();
  }
#ifdef MAEMO
	if (loaded_state == true) {
		p2->points = score_opponent;
		human_set_points(p1, score_player);
		scoreboard_set_points(score_player, score_opponent);
		_blit_all_points();
	}
#endif
  loop = 1;
  board_clean_up();
  aiplayer_blit(p2);
  video_save();
  board_reblit();
  video_update();
  SDL_PumpEvents();
  SDL_GetRelativeMouseState(NULL, NULL);
#ifndef _DEBUG
  SDL_WM_GrabInput(SDL_GRAB_ON);
#endif
  human_set_speed(p1, _settings->mouse_speed);
  timer_reset(timer);
  while (loop) {
    while (SDL_PollEvent(&event))
      switch (event.type) {
      	case SDL_KEYDOWN:
			switch (event.key.keysym.sym) {
				case SDLK_ESCAPE:
      				timer_reset(timer);
			       #ifndef MAEMO
					if (menu_get_selected(exit_menu) == 1)
	    				loop = 0;
					#else
					save_state(opponent, human_get_points(p1), p2->points);
					exit_callback(0);
					return opponent;
					#endif
	  				break;
#ifndef MAEMO
				case SDLK_F1:
					_settings->sound = !_settings->sound;
					audio_set_mute(!_settings->sound);
	  				break;
#endif
				case SDLK_F5:
#ifndef MAEMO
	  				if (_settings->mouse_speed > 1)
					    _settings->mouse_speed--;
					human_set_speed(p1, _settings->mouse_speed);
					scoreboard_set_mousebar(_settings->mouse_speed);
#else
				case SDLK_F4:
					save_state(opponent, human_get_points(p1), p2->points);
					exit_callback(0);
					return opponent;
#endif
					break;
				case SDLK_F6:
#ifndef MAEMO
					if (_settings->mouse_speed < 10)
						_settings->mouse_speed++;
					human_set_speed(p1, _settings->mouse_speed);
					scoreboard_set_mousebar(_settings->mouse_speed);
#else
					save_state(opponent, human_get_points(p1), p2->points);
					exit_callback(0);
					return opponent;
#endif
					break;
#ifndef MAEMO	
				case SDLK_f:
				  _settings->fullscreen = !_settings->fullscreen;
				  video_toggle_fullscreen();
				  break;
#endif
				default:
	  				break;
			}
			break;
#ifdef MAEMO
		case SDL_MOUSEBUTTONDOWN:
			if (event.button.x>BACKBORDERLEFT 
				&& event.button.x<BACKBORDERRIGHT 
				&& event.button.y>BACKBORDERTOP 
				&& event.button.y<BACKBORDERBOTTOM)	
			{

				loop = 0;
				exit_callback(0);
				return opponent;
			}
			break;
#endif
        case SDL_QUIT:
			loop = 0;
			exit_callback(0);
 			return opponent;
			break;
      }
#ifndef MAEMO	
    SDL_Delay(SLEEP);
#endif
    timer_update(timer);
    timer_reset(timer);
    elapsed_time = timer_elapsed(timer);
    human_update(p1, elapsed_time);
    aiplayer_update(p2, elapsed_time);
    scoreboard_update(elapsed_time);
    if ((scorer = board_update(elapsed_time)) != 0) {
      scoreboard_add_point(scorer);
      if (scorer == 1) {
		human_give_point(p1);
		p2->set_state(p2, PLAYER_STATE_LOOSE_POINT);
      } else {
		p2->points++;
		p2->set_state(p2, PLAYER_STATE_WIN_POINT);
      }
      if (human_get_points(p1) >= 15 || p2->points >= 15) {
		if (human_get_points(p1) == 15)
			  p2->set_state(p2, PLAYER_STATE_LOOSE_GAME);
		else
			  p2->set_state(p2, PLAYER_STATE_WIN_GAME);
      }
    }
    board_clean_up();
    scoreboard_clean_up();
    scoreboard_reblit();
    if (p2->reblit(p2, elapsed_time) == 0)
      loop = 0;
    board_reblit();
    video_update();

  }
#ifndef _DEBUG
  SDL_WM_GrabInput(SDL_GRAB_OFF);
#endif
#ifndef MAEMO
  menu_free(exit_menu);
#endif
  timer_free(timer);
  human_free(p1);
  p2->free(p2);
  board_deinit();
  scoreboard_deinit();
  if (menu_get_selected(again_menu) == 0)
#ifndef MAEMO
    next_opponent = opponent;
#endif
	_play_match(opponent);
  else
    next_opponent = -1;
  menu_free(again_menu);
  return next_opponent;
}

static void _read_settings(void)
{
  FILE *file = NULL;
  char buffer[100], buffer2[100];
  Uint32 uint32 = 0;

  if ((file = fopen(SETTINGSFILE, "r")) == NULL)
    return;
  while (fgets(buffer, 100, file) != 0) {
    if (sscanf(buffer, "%s %d\n", buffer2, &uint32) != 2) {
      fclose(file);
      return;
    } else if (strcmp(buffer2, "SOUND") == 0)
      _settings->sound = (Uint8) uint32;
    else if (strcmp(buffer2, "FULLSCREEN") == 0)
      _settings->fullscreen = (Uint8) uint32;
    else if (strcmp(buffer2, "MOUSESPEED") == 0)
      _settings->mouse_speed = (Uint8) uint32;
  }
  fclose(file);
}

static void _save_settings(void)
{
  FILE *file = NULL;

  if ((file = fopen(SETTINGSFILE, "w")) == NULL)
    return;
  fprintf(file, "SOUND %d\n", _settings->sound);
  fprintf(file, "FULLSCREEN %d\n", _settings->fullscreen);
  fprintf(file, "MOUSESPEED %d\n", _settings->mouse_speed);
  fclose(file);
}

static void _tuxpuck_init(void)
{

    GConfClient *gc_client = NULL;
    g_type_init();
    gc_client = gconf_client_get_default();

#ifndef windows
  char *homeDir = NULL;
#endif
  srand(time(NULL));
  SDL_Init(SDL_INIT_VIDEO | SDL_INIT_AUDIO);
#ifndef MAEMO
  audio_init();
#endif
  video_init();

  video_save();
  _settings = (Settings *) malloc(sizeof(Settings));
  memset(_settings, 0, sizeof(Settings));
#ifndef MAEMO
  _settings->sound = 1;
#else
  _settings->sound = gconf_client_get_bool(gc_client, SETTINGS_SOUND, NULL);
#endif
	if (_settings->sound == true){
		audio_init();
	} 
#ifndef MAEMO
  _settings->fullscreen = 0;
#else
  _settings->fullscreen = 1;
#endif

  _settings->mouse_speed = 5;
#ifndef windows
  homeDir = getenv("HOME");
  sprintf(_settings_file, "%s/.tuxpuckrc", homeDir);
#endif
#ifndef MAEMO
  _read_settings();
  audio_set_mute(!_settings->sound);
#endif
  if (_settings->fullscreen)
    video_toggle_fullscreen();

#ifndef MAEMO
  run_intro();
  video_save();
#endif
}

static void _tuxpuck_deinit(void)
{
  audio_deinit();
  video_deinit();
  SDL_Quit();
  _save_settings();
  free(_settings);
}

int main(int argc, char *argv[])
{
    int next_opponent;

#ifdef MAEMO
    GConfClient *gc_client = NULL;
    g_type_init();
    gc_client = gconf_client_get_default();
    next_opponent = gconf_client_get_int(gc_client, SETTINGS_OPPONENT, NULL);    
    
    #if HGW_FUNC
    hgw_context = hgw_context_compat_init(argc, argv);

    if(hgw_context == NULL)
    {
        return 0;
    }
    hgw_compat_set_cb_exit(hgw_context, exit_callback);
    hgw_compat_set_cb_quit(hgw_context, quit_callback);
    hgw_compat_set_cb_flush(hgw_context, flush_callback);

    if(!hgw_context_compat_check(hgw_context))
    {
        return 0;
    }

    /* Shadow app part */
    hgw_msg_compat_receive(hgw_context, 0);
    usleep(100);
    #endif
#endif
  _tuxpuck_init();

#ifndef MAEMO
  Menu *main_menu, *op_menu;
   main_menu = menu_create(2);
  menu_add_field(main_menu, 0, 1, "Play Match");
  menu_add_field(main_menu, 1, 1, "Exit");
  op_menu = menu_create(3);
  menu_add_field(op_menu, 0, 0, "Opponent");
  menu_add_field(op_menu, 1, 1, "Tux");
  menu_add_field(op_menu, 2, 1, "Arcana");
  while (menu_get_selected(main_menu) == 0) {
    next_opponent = menu_get_selected(op_menu);
    while (next_opponent != -1)
      next_opponent = _play_match(next_opponent);
  }
#endif
	next_opponent = _play_match(next_opponent);
#ifndef MAEMO
  menu_free(op_menu);
  menu_free(main_menu);
#endif
  _tuxpuck_deinit();

#ifdef MAEMO
	/* Free memory */
	#if HGW_FUNC
	hgw_context_compat_destroy_deinit(hgw_context);
	#endif	
#endif	
  return 0;
}
#ifdef MAEMO
static int exit_callback(int errcode)
{
#if HGW_FUNC
//  hgw_notify_startup(hgw_context, HGW_SUN_PAUSE);

#endif

  return 0;
}

static int quit_callback(int errcode)
{
//  remove("/tmp/.tuxpuck-save");

#if HGW_FUNC
 // hgw_context_compat_destroy_quit(hgw_context);
#endif

  return 0;
}

static int flush_callback(int errcode)
{
  remove("/tmp/.tuxpuck-save");
  return 0;
}

void save_state(int opponent, Uint8 score_player, Uint8 score_opponent)
{
    FILE *pFile = fopen("/tmp/.tuxpuck-save", "wb");;
    if(pFile) {
        fwrite(&score_player, sizeof(Uint8), 1, pFile);
        fwrite(&score_opponent, sizeof(Uint8), 1, pFile);
        fwrite(&opponent, sizeof(Uint8), 1, pFile);
        fclose(pFile);
    }
}

int load_state(Uint8 *opponent, Uint8 *score_player, Uint8 *score_opponent) 
{
    FILE *pFile = fopen("/tmp/.tuxpuck-save", "rb");;
    if(pFile) {
        fread(score_player, sizeof(Uint8), 1, pFile);
        fread(score_opponent, sizeof(Uint8), 1, pFile);
        fread(opponent, sizeof(Uint8), 1, pFile);
        fclose(pFile);
	    remove("/tmp/.tuxpuck-save");
		return true;
    }
	return false;
}
#endif //MAEMO
