#include <sys/time.h>

#include <libosso.h>
#include <gconf/gconf-client.h>
#include <libhildondesktop/libhildondesktop.h>

#include "statusbarclock.h"

#define STATUSBARCLOCK_GET_PRIVATE(x) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((x), TYPE_STATUSBARCLOCK, StatusbarClockPrivate))

#define GCONF_CLOCK_MODE_DIR "/apps/osso/clock"
#define GCONF_CLOCK_MODE GCONF_CLOCK_MODE_DIR "/display-mode"


HD_DEFINE_PLUGIN (StatusbarClock,
                  statusbarclock,
                  STATUSBAR_TYPE_ITEM);


typedef struct _StatusbarClockPrivate StatusbarClockPrivate;
struct _StatusbarClockPrivate
{
  osso_context_t  *context;
  guint            timeout_id;

  GConfClient     *gconf_client;
  guint            gconf_notify;

  struct timeval   tv;
  struct tm        time_info;

  gboolean         analog;
};


static gboolean
statusbarclock_update_time (gpointer data)
{
  StatusbarClock *clock = STATUSBARCLOCK (data);
  StatusbarClockPrivate *pclock = STATUSBARCLOCK_GET_PRIVATE (clock);
  gchar          *tz_str;

  tz_str = g_strdup (g_getenv ("TZ"));
  g_unsetenv ("TZ");

  gettimeofday (&pclock->tv, NULL);
  localtime_r (&pclock->tv.tv_sec, &pclock->time_info);

  if (tz_str)
    {
      g_setenv ("TZ", tz_str, FALSE);
      g_free (tz_str);
    }

  gtk_widget_queue_draw (GTK_WIDGET (clock));

  if (pclock->timeout_id)
    g_source_remove (pclock->timeout_id);

  pclock->timeout_id = g_timeout_add ((60 - pclock->time_info.tm_sec) * 1000 + 10,
                                      statusbarclock_update_time,
                                      clock);

  return FALSE;
}


static void
statusbarclock_timezone_update (gpointer data)
{
  StatusbarClock *clock = STATUSBARCLOCK (data);

  statusbarclock_update_time (clock);
}


static void
statusbarclock_sleep_wakeup (osso_hw_state_t *state,
                             gpointer         data)
{
  StatusbarClock *clock = STATUSBARCLOCK (data);
  StatusbarClockPrivate *pclock = STATUSBARCLOCK_GET_PRIVATE (clock);

  if (pclock->timeout_id)
    g_source_remove (pclock->timeout_id);
  pclock->timeout_id = 0;

  if (state->system_inactivity_ind == FALSE)
    /* this also reregisters the timeout */
    statusbarclock_update_time (clock);
}


static void
statusbarclock_gconf_notify (GConfClient *client,
                             guint cnxn_id,
                             GConfEntry *entry,
                             gpointer data)
{
  StatusbarClock *clock = STATUSBARCLOCK (data);
  StatusbarClockPrivate *pclock = STATUSBARCLOCK_GET_PRIVATE (clock);
  GConfValue  *value;

  pclock->analog = TRUE;

  value = gconf_entry_get_value (entry);

  if (value && gconf_value_get_int (value) == 1)
    pclock->analog = FALSE;

  if (pclock->timeout_id)
    g_source_remove (pclock->timeout_id);
  pclock->timeout_id = 0;

  statusbarclock_update_time (clock);
}


static void
statusbarclock_show_date (GtkButton *button,
                          gpointer   data)
{
  StatusbarClock *clock = STATUSBARCLOCK (data);
  StatusbarClockPrivate *pclock = STATUSBARCLOCK_GET_PRIVATE (clock);
  gchar           message[50];

  strftime (message, G_N_ELEMENTS (message), "%A\n%x", &pclock->time_info);

  osso_system_note_infoprint (pclock->context, message, NULL);
}


static gboolean
statusbarclock_expose_event (GtkWidget *widget,
                             GdkEventExpose *event,
                             gpointer   data)
{
  StatusbarClock *clock = STATUSBARCLOCK (data);
  StatusbarClockPrivate *pclock = STATUSBARCLOCK_GET_PRIVATE (clock);
  gdouble         angle, height, left;
  char            timestr[10];
  int             i;

  cairo_t              *cr;
  cairo_text_extents_t  exts;
  cairo_font_options_t *fopt;

  cr = gdk_cairo_create (event->window);

  gdk_cairo_rectangle (cr, &event->area);
  cairo_clip (cr);
  cairo_translate (cr, widget->allocation.x, widget->allocation.y);

  cairo_set_source_rgb (cr, 1, 1, 1);

  if (pclock->analog)
    {
      cairo_translate (cr, widget->allocation.width / 2 - 0.5, widget->allocation.height / 2 - 0.5);
      cairo_scale (cr, MIN (widget->allocation.width / 2, widget->allocation.height / 2),
                   -MIN (widget->allocation.width / 2, widget->allocation.height / 2));
      cairo_scale (cr, 1.0/80, 1.0/80);
      cairo_arc (cr, 0, 0, 60, 0, 2 * G_PI);
      cairo_set_line_width (cr, 9);
      cairo_stroke (cr);
     
      cairo_set_line_width (cr, 4);
      for (i = 0; i < 12; i++)
        {
          cairo_save (cr);
          cairo_rotate (cr, i * G_PI / 6);
          cairo_move_to (cr, i % 3 ? 45 : 41, 0);
          cairo_line_to (cr, 60, 0);
          cairo_stroke (cr);
          cairo_restore (cr);
        }
     
      cairo_save (cr);
      angle = (gdouble) pclock->time_info.tm_min / 30.0 * G_PI;
      cairo_rotate (cr, -angle);
      cairo_move_to (cr, 0, -14);
      cairo_line_to (cr, 0, 45);
      cairo_set_line_width (cr, 6);
      cairo_stroke (cr);
      cairo_restore (cr);
     
      cairo_save (cr);
      angle = (gdouble) (pclock->time_info.tm_hour * 60 + pclock->time_info.tm_min) / 360.0 * G_PI;
      cairo_rotate (cr, -angle);
      cairo_move_to (cr, 0, -10);
      cairo_line_to (cr, 0, 33);
      cairo_set_line_width (cr, 10);
      cairo_stroke (cr);
      cairo_restore (cr);
    }
  else
    {
      cairo_select_font_face (cr, "Nokia Sans Cn",
                              CAIRO_FONT_SLANT_NORMAL,
                              CAIRO_FONT_WEIGHT_NORMAL);
      fopt = cairo_font_options_create ();
      cairo_font_options_set_hint_metrics (fopt, CAIRO_HINT_METRICS_OFF);
      cairo_set_font_options (cr, fopt);
      cairo_font_options_destroy (fopt);

      cairo_set_font_size (cr, 10);

      cairo_translate (cr, widget->allocation.width / 2,
                       widget->allocation.height / 2 - 2);

      cairo_text_extents (cr, "00:00", &exts);
      height = exts.y_bearing + exts.height / 2;

      strftime (timestr, 10, "%H", &pclock->time_info);
      cairo_text_extents (cr, timestr, &exts);
      left = exts.x_advance;
      strftime (timestr, 10, "%H:", &pclock->time_info);
      cairo_text_extents (cr, timestr, &exts);
      left += exts.x_advance;
      left /= 2;

      strftime (timestr, 10, "%H:%M", &pclock->time_info);
      cairo_text_extents (cr, timestr, &exts);
      cairo_scale (cr, (double) widget->allocation.width / exts.x_advance,
                   (double) widget->allocation.width / exts.x_advance);
      cairo_translate (cr, -left, -height);

      cairo_show_text (cr, timestr);
    }

  cairo_destroy (cr);

  return TRUE;
}


static void
statusbarclock_init (StatusbarClock *clock)
{
  StatusbarClockPrivate *pclock = STATUSBARCLOCK_GET_PRIVATE (clock);
  GError        *error = NULL;
  GtkWidget     *button = NULL;
  osso_return_t  ret;

  pclock->gconf_client = gconf_client_get_default ();

  pclock->analog = TRUE;

  if (gconf_client_get_int (pclock->gconf_client,
                            GCONF_CLOCK_MODE, NULL) == 1)
    pclock->analog = FALSE;

  pclock->context = osso_initialize (PACKAGE, VERSION,
                                     FALSE, NULL);
  if (!pclock->context) g_printerr ("osso_initialize() failed\n");

  button = gtk_button_new ();
  g_signal_connect (G_OBJECT (button), "clicked",
                    G_CALLBACK (statusbarclock_show_date),
                    clock);

  gtk_container_add (GTK_CONTAINER (clock),  button);

  g_signal_connect_after (G_OBJECT (button), "expose-event",
                          G_CALLBACK (statusbarclock_expose_event),
                          clock);

  gtk_widget_add_events (GTK_WIDGET (button), GDK_EXPOSURE_MASK);

  ret = osso_time_set_notification_cb (pclock->context,
                                       statusbarclock_timezone_update,
                                       clock);

  if (ret != OSSO_OK)
    g_printerr ("osso_time_set_notification_cb() failed\n");

  ret = osso_hw_set_event_cb (pclock->context,
                              NULL,
                              statusbarclock_sleep_wakeup,
                              clock);

  if (ret != OSSO_OK)
    g_printerr ("osso_hw_set_event_cb() failed\n");

  gconf_client_add_dir (pclock->gconf_client,
                        GCONF_CLOCK_MODE_DIR,
                        GCONF_CLIENT_PRELOAD_NONE,
                        &error);
  if (error)
    {
      g_printerr ("%s (gconf_client_add_dir)\n", error->message);
      g_error_free (error);
      error = NULL;
    }

  pclock->gconf_notify = gconf_client_notify_add (pclock->gconf_client,
                                                  GCONF_CLOCK_MODE,
                                                  statusbarclock_gconf_notify,
                                                  clock,
                                                  NULL, &error);
  if (error)
    {
      g_printerr ("%s (gconf_client_notify_add)\n", error->message);
      g_error_free (error);
      error = NULL;
    }

  /* start ticking */
  statusbarclock_update_time (clock);

  gtk_widget_show_all (GTK_WIDGET (clock));
}


static void
statusbarclock_dispose (GObject *clock)
{
  StatusbarClockPrivate *pclock = STATUSBARCLOCK_GET_PRIVATE (clock);

  if (pclock->gconf_client)
    {
      if (pclock->gconf_notify)
        gconf_client_notify_remove (pclock->gconf_client,
                                    pclock->gconf_notify);
      gconf_client_remove_dir (pclock->gconf_client,
                               GCONF_CLOCK_MODE_DIR, NULL);
      g_object_unref (pclock->gconf_client);
      pclock->gconf_client = NULL;
    }

  if (pclock->context)
    {
      osso_deinitialize (pclock->context);
      pclock->context = NULL;
    }

  if (pclock->timeout_id)
    {
      g_source_remove (pclock->timeout_id);
      pclock->timeout_id = 0;
    }
}


static void
statusbarclock_class_init (StatusbarClockClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS(klass);
  object_class->dispose = statusbarclock_dispose;
  g_type_class_add_private (klass, sizeof (StatusbarClockPrivate));
}


