#
# This is the ODF exporter module for SplitTheTab.
# Copyright (C) 2009, Henning Spruth
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor,
# Boston, MA  02110-1301, USA.
#


import sqlite3
import zipfile
import re

contentsHead='''<?xml version="1.0" encoding="UTF-8"?>
<office:document-content
  xmlns:office="urn:oasis:names:tc:opendocument:xmlns:office:1.0"
  xmlns:table="urn:oasis:names:tc:opendocument:xmlns:table:1.0"
  xmlns:of="urn:oasis:names:tc:opendocument:xmlns:of:1.2"
  xmlns:text="urn:oasis:names:tc:opendocument:xmlns:text:1.0"
>
<office:automatic-styles>
<style:style style:name="co1" style:family="table-column">
<style:table-column-properties fo:break-before="auto" style:column-width="0.8925in"/>
</style:style>
<number:number-style style:name="N2">
    <number:number number:decimal-places="2" number:min-integer-digits="1"/>
</number:number-style>
<style:style style:name="ce1" style:family="table-cell" style:parent-style-name="Default" style:data-style-name="N2"/>
</office:automatic-styles>

<office:body>
  <office:spreadsheet>
            <table:table table:name="Sheet1">
'''

contentsTail='''</table:table>
        </office:spreadsheet>
    </office:body>
</office:document-content>
'''

manifest='''<?xml version="1.0" encoding="UTF-8"?>
<manifest:manifest xmlns:manifest="urn:oasis:names:tc:opendocument:xmlns:manifest:1.0">
 <manifest:file-entry manifest:media-type="application/vnd.oasis.opendocument.spreadsheet" manifest:full-path="/"/>
 <manifest:file-entry manifest:media-type="text/xml" manifest:full-path="content.xml"/>
 <manifest:file-entry manifest:media-type="text/xml" manifest:full-path="meta.xml"/>
 <manifest:file-entry manifest:media-type="text/xml" manifest:full-path="styles.xml"/>
</manifest:manifest>
'''

meta='''<?xml version="1.0" encoding="UTF-8"?>
<office:document-meta
xmlns:office="urn:oasis:names:tc:opendocument:xmlns:office:1.0"
xmlns:meta="urn:oasis:names:tc:opendocument:xmlns:meta:1.0"
office:version="1.0">
<office:meta>
<meta:generator>SplitTheTab</meta:generator>
</office:meta>
</office:document-meta>
'''

styles='''<?xml version="1.0" encoding="UTF-8"?>
<office:document-styles
xmlns:office="urn:oasis:names:tc:opendocument:xmlns:office:1.0"
xmlns:style="urn:oasis:names:tc:opendocument:xmlns:style:1.0"
office:version="1.0">
<office:font-face-decls/>
<office:styles/>
</office:document-styles>
'''

cE='table:table-cell'

#      	A		B		C	D	E	F	G...	H...
#  1	Currencies
#  2	US$		1.0
#  3	EUR		1.4
#  4
#  5	Transactions
#  5	Date		Description	Amount	Curr	ConvAmt Payer	part	amount
#


class Exporter:

    def add(self,s):
        self.buffer+=s

    def addTextCell(self, s):
        s=re.sub('&', '&amp;', s)
        s=re.sub('<', '&lt;', s)
        self.add('<'+cE+' office:value-type="string"><text:p>'+s+'</text:p></'+cE+'>\n')

    def addFloatCell(self, f):
        self.add('<'+cE+' office:value-type="float" office:value="%f"/>\n' % f)

    def addFormulaCell(self, s):
        self.add('<table:table-cell table:formula="of:=%s" />\n' % s)
        #self.add('<table:table-cell table:formula="of:=%s" office:value-type="float"/>' % s)


    def openRow(self):
        self.add('<table:table-row>\n')

    def closeRow(self):
        self.add('</table:table-row>\n')
        self.rowNumber+=1


    def getContents(self):
        self.buffer=contentsHead
        self.rowNumber=1

        nameOfId={}
        accountIds=[]
        for row in self.sql.execute('SELECT id,name FROM account ORDER BY id;'):
            accountIds.append(row[0])
            nameOfId[row[0]]=row[1]
        accountCount=len(accountIds)

        columnCount=6+2*accountCount

        self.add('<table:table-column table:style-name="co1" table:number-columns-repeated="%d" table:default-cell-style-name="ce1"/>\n'%columnCount)

        # row 1
        self.openRow()
        self.addTextCell('Currencies')
        self.closeRow()

        # Currency rows
        self.rowOfCurrency={}
        for row in self.sql.execute('SELECT id,name,exchange_rate FROM currency ORDER BY id'):
            self.openRow()
            self.addTextCell(row[1])
            self.addFloatCell(row[2])
            # remember the row number of this currency ID
            self.rowOfCurrency[row[0]]=self.rowNumber
            self.closeRow()


        # empty row
        self.openRow()
        self.closeRow()

        # 
        self.openRow()
        self.addTextCell('Transactions')
        self.closeRow()

        # Header
        self.openRow()
        self.addTextCell('Date/Time')
        self.addTextCell('Description')
        self.addTextCell('Amount')
        self.addTextCell('Currency')
        self.addTextCell('Converted Amount')
        self.addTextCell('Paid By')
        for id in accountIds:
            self.addTextCell(nameOfId[id])
        i=0
        for id in accountIds:
            column='%c'%(71+i)
            #self.addTextCell(nameOfId[id])
            self.addFormulaCell('[.%s%d]'% (column, self.rowNumber))  # currency
            i+=1
        self.closeRow()

        # Transactions
        firstTransactionRow=self.rowNumber
        for row in self.sql.execute('SELECT id,timestamp,description,amount,currency_id,account_id FROM payment ORDER BY timestamp'):
            self.openRow()
            self.addTextCell(row[1]) # timestamp
            self.addTextCell(row[2]) # description
            self.addFloatCell(row[3]) # amount
            currencyRow=self.rowOfCurrency[row[4]]
            self.addFormulaCell('[.A%d]'%currencyRow)  # currency
            self.addFormulaCell('[.C%d]*[.B%d]'% (self.rowNumber, currencyRow))  # multiplied amount
            payer_id=row[5]
            self.addTextCell(nameOfId[payer_id])  # payer's name

            split={}
            for id in accountIds:
                split[id]=0

            sumOfSplits=0
            for row2 in self.sql.execute('SELECT account_id, split FROM split WHERE payment_id=?', (row[0],)):
                split[row2[0]]=row2[1]
                sumOfSplits+=row2[1]

            for id in accountIds:
                val=split[id]
                if (id==payer_id):
                    val-=sumOfSplits
                self.addFloatCell(val)

            firstSplitCol='%c'%(71)
            lastSplitCol='%c'%(70+len(accountIds))
            i=0
            for id in accountIds:
                column='%c'%(71+i)
                self.addFormulaCell('[.%s%d]*[.E%d]/%d' % (column,self.rowNumber,self.rowNumber, sumOfSplits))
                #self.addFormulaCell('[.%s%d]*[.E%d]/sum([.%s%d:.%s%d])' % (column,self.rowNumber,self.rowNumber, firstSplitCol, self.rowNumber, lastSplitCol, self.rowNumber))
                i+=1

            self.closeRow()

        lastTransactionRow=self.rowNumber-1

        # Totals
        self.openRow()
        for i in range(4):
            self.addTextCell('')
        self.addFormulaCell('sum([.E%d:.E%d])'%(firstTransactionRow,lastTransactionRow))
        self.addTextCell('')
        for i in range(len(accountIds)):
            self.addTextCell('')
        for i in range(len(accountIds)):
            columnName='%c'%(71+len(accountIds)+i)
            self.addFormulaCell('sum([.%s%d:.%s%d])'%(columnName, firstTransactionRow,columnName,lastTransactionRow))
        self.closeRow()

        self.add(contentsTail)
        return self.buffer


    def writeOdf(self, filename):
        zip=zipfile.ZipFile(filename, 'w')
        contents=self.getContents()
        #print contents
        zip.writestr('content.xml', contents)
        zip.writestr('meta.xml', meta)
        zip.writestr('styles.xml', styles)
        zip.writestr('mimetype', 'application/vnd.oasis.opendocument.spreadsheet')
        zip.writestr('META-INF/manifest.xml', manifest)
        zip.close()

    def __init__(self, sql):
        self.sql=sql
    
    


if __name__=='__main__':
    sql=sqlite3.connect('/tmp/peru2009.db')
    sql.text_factory=str
    exporter=Exporter(sql)
    exporter.writeOdf('/tmp/test.ods')

