// --------------------------------------------------------------------------
// Advanced Linux Sound Architecture (ALSA) specific audio driver interface.
// --------------------------------------------------------------------------
/***************************************************************************
 *  $Log: audiodrv.cpp,v $
 *  Revision 1.8  2008/07/19 13:58:04  wnd
 *  Almost complete rewrite to support new ALSA API.
 *
 *  Revision 1.7  2002/03/04 19:07:48  s_a_white
 *  Fix C++ use of nothrow.
 *
 *  Revision 1.6  2002/01/10 19:04:00  s_a_white
 *  Interface changes for audio drivers.
 *
 *  Revision 1.5  2001/12/11 19:38:13  s_a_white
 *  More GCC3 Fixes.
 *
 *  Revision 1.4  2001/01/29 01:17:30  jpaana
 *  Use int_least8_t instead of ubyte_sidt which is obsolete now
 *
 *  Revision 1.3  2001/01/23 21:23:23  s_a_white
 *  Replaced SID_HAVE_EXCEPTIONS with HAVE_EXCEPTIONS in new
 *  drivers.
 *
 *  Revision 1.2  2001/01/18 18:35:41  s_a_white
 *  Support for multiple drivers added.  C standard update applied (There
 *  should be no spaces before #)
 *
 *  Revision 1.1  2001/01/08 16:41:43  s_a_white
 *  App and Library Seperation
 *
 ***************************************************************************/

#include "audiodrv.h"
#ifdef HAVE_ALSA

#include <stdio.h>
#ifdef HAVE_EXCEPTIONS
#include <new>
#endif

#include <alsa/asoundlib.h>
#include <string.h>

#define ALSA_DEFAULT "default"

Audio_ALSA::Audio_ALSA()
{
	// Reset everything.
	outOfOrder();
}

Audio_ALSA::~Audio_ALSA()
{
	close ();
}

void
Audio_ALSA::outOfOrder()
{
	// Reset everything.
	_swapEndian = false;
	_bufShift = 0;
	_errorString = "None";
	_audioHandle = NULL;
}

bool
Audio_ALSA::fmt(AudioConfig &cfg, snd_pcm_hw_params_t *hw)
{
	struct precinfo {
		snd_pcm_format_t format;
		int encoding;
		int precision;
		int swap;
	};
	struct precinfo info[] = {
#if defined(WORDS_BIGENDIAN)
		{ SND_PCM_FORMAT_S16_BE, AUDIO_SIGNED_PCM, 16, false },
		{ SND_PCM_FORMAT_U16_BE, AUDIO_UNSIGNED_PCM, 16, false },
		{ SND_PCM_FORMAT_S16_LE, AUDIO_SIGNED_PCM, 16, true },
		{ SND_PCM_FORMAT_U16_LE, AUDIO_UNSIGNED_PCM, 16, true },
#else
		{ SND_PCM_FORMAT_S16_LE, AUDIO_SIGNED_PCM, 16, false },
		{ SND_PCM_FORMAT_U16_LE, AUDIO_UNSIGNED_PCM, 16, false },
		{ SND_PCM_FORMAT_S16_BE, AUDIO_SIGNED_PCM, 16, true },
		{ SND_PCM_FORMAT_U16_BE, AUDIO_UNSIGNED_PCM, 16, true },
#endif
		{ SND_PCM_FORMAT_S8, AUDIO_SIGNED_PCM, 8, false },
		{ SND_PCM_FORMAT_U8, AUDIO_UNSIGNED_PCM, 8, false }
	};

	switch (cfg.precision) {
		case 8:
			for (int i = 0; i < 2; i++) {
				struct precinfo t;
				memcpy(&t, &info[i], sizeof(t));
				memcpy(&info[i + 4], &info[i], sizeof(t));
				memcpy(&info[i], &t, sizeof(t));
			}
			break;
		case 16:
			break;
		default:
			fprintf(stderr, "Unsupported precision: %d bits\n",
					cfg.precision);
			return false;
	}

	int err;
	for (int i = 0; i < 6; i++) {
		err = snd_pcm_hw_params_set_format(_audioHandle, hw,
				info[i].format);
		if (err < 0) {
			continue;
		}

		cfg.precision = info[i].precision;
		cfg.encoding = info[i].encoding;
		_swapEndian = info[i].swap;
		_bufShift = info[i].precision == 8 ? 0 : 1;
		return true;
	}
	
	fprintf(stderr, "Cannot set sample format: %s\n", snd_strerror(err));
	return false;
}



bool
Audio_ALSA::init(AudioConfig &cfg, snd_pcm_hw_params_t *hw)
{
	int err;

	err = snd_pcm_hw_params_any(_audioHandle, hw);
	if (err < 0) {
		fprintf(stderr, "Cannot initialize hardware "
				"parameter structure: %s\n",
				snd_strerror(err));
		return false;
	}

	err = snd_pcm_hw_params_set_access(_audioHandle, hw,
			SND_PCM_ACCESS_RW_INTERLEAVED);
	if (err < 0) {
		fprintf(stderr, "Cannot set access type: %s\n",
				snd_strerror(err));
		return false;
	}

	/* try to find supported format/precision */
	if (! fmt(cfg, hw)) {
		return false;
	}

	/* use closest frequency */
	unsigned int realFreq = (unsigned int) cfg.frequency;
	err = snd_pcm_hw_params_set_rate_near(_audioHandle, hw, &realFreq, 0);
	if (err < 0) {
		fprintf(stderr, "Cannot set sample rate: %s\n",
				snd_strerror(err));
		return false;
	}
	cfg.frequency = realFreq;

	switch (cfg.channels) {
		case 2:
			_bufShift++;
			break;
		case 1:
			break;
		default:
			fprintf(stderr, "Unsupported channel count: %d\n",
					cfg.channels);
			return false;
	}
	err = snd_pcm_hw_params_set_channels(_audioHandle, hw,
			cfg.channels);
	if (err < 0) {
		fprintf(stderr, "Cannot set channel count: %s\n",
				snd_strerror(err));
		return false;
	}

	err = snd_pcm_hw_params(_audioHandle, hw);
	if (err < 0) {
		fprintf(stdout, "Cannot set parameters: %s\n",
				snd_strerror(err));
		return false;
	}

	return true;
}



void *
Audio_ALSA::open(AudioConfig &cfg, const char *name)
{
	AudioConfig tmpCfg = cfg;
	snd_pcm_hw_params_t *hw;
	int err;

	if (_audioHandle != NULL) {
		_errorString = "ERROR: Device already in use";
		return NULL;
	}

	err = snd_pcm_open(&_audioHandle, ALSA_DEFAULT,
			SND_PCM_STREAM_PLAYBACK, 0);
	if (err < 0) {
		_errorString = "ERROR: Could not open audio device.";
		fprintf(stderr, "Cannot open audio device: %s\n",
				snd_strerror(err));
		return false;
	}

	err = snd_pcm_hw_params_malloc(&hw);
	if (err < 0) {
		_errorString = "ERROR: Cannot initialise ALSA";
		fprintf(stderr, "Cannot allocate hardware "
				"parameter structure: %s\n",
				snd_strerror(err));
		return false;
	}

	bool init_ok = init(tmpCfg, hw);
	snd_pcm_hw_params_free(hw);
	if (! init_ok) {
		_errorString = "ERROR: Cannot initialise ALSA";
		return false;
	}

	err = snd_pcm_prepare(_audioHandle);
	if (err < 0) {
		_errorString = "Cannot initialise ALSA";
		fprintf(stderr, "cannot prepare audio interface for use: %s\n",
				snd_strerror(err));
		return false;
	}

	tmpCfg.bufSize = 4096;
#ifdef HAVE_EXCEPTIONS
	_sampleBuffer = new(std::nothrow) int_least8_t[tmpCfg.bufSize];
#else
	_sampleBuffer = new int_least8_t[tmpCfg.bufSize];
#endif

	if (!_sampleBuffer) {
		_errorString = "Unable to allocate memory for sample buffers";
		return NULL;
	}

	// Setup internal Config
	_settings = tmpCfg;
	// Update the users settings
	getConfig (cfg);
	return _sampleBuffer;
}



// Close an opened audio device, free any allocated buffers and
// reset any variables that reflect the current state.
void
Audio_ALSA::close()
{
	if (_audioHandle != NULL) {
		snd_pcm_close(_audioHandle);
		delete [] _sampleBuffer;
		outOfOrder ();
	}
}



void *
Audio_ALSA::reset ()
{
	return (void *) _sampleBuffer;   
}



void *
Audio_ALSA::write()
{
	if (_audioHandle == NULL)
	{
		_errorString = "ERROR: Device not open.";
		return NULL;
	}

	if (_swapEndian) {
		for (int n = 0; n < _settings.bufSize; n += 2) {
			char t = ((char *) _sampleBuffer)[n];
			((char *) _sampleBuffer)[n] =
				((char *) _sampleBuffer)[n + 1];
			((char *) _sampleBuffer)[n + 1] = t;
		}
	}
	snd_pcm_writei(_audioHandle, _sampleBuffer,
			_settings.bufSize >> _bufShift);
	return (void *) _sampleBuffer;
}

#endif // HAVE_ALSA
