// --------------------------------------------------------------------------
// ``Advanced Linux Sound Architecture (ALSA)'' specific audio driver interface.
// --------------------------------------------------------------------------

#include "audiodrv.h"

#include <alsa/asoundlib.h>

#define ALSA_DEFAULT	"default"


audioDriver::audioDriver()
{
	// Reset everything.
	alsa = NULL;
	errorString = "None";
	frequency = 0;
	channels = 0;
	precision = 0;
	bufShift = 0;
}



bool audioDriver::IsThere()
{
	// Check device availability and write permissions.
	snd_pcm_t *handle;
	int err;
	err = snd_pcm_open(&handle, ALSA_DEFAULT, SND_PCM_STREAM_PLAYBACK, 0);
	if (err < 0) {
		return false;
	} else if (handle != NULL) {
		snd_pcm_close(handle);
		return true;
	} else {
		return false;
	}
}



bool audioDriver::SetFormat(snd_pcm_t *handle,
		snd_pcm_hw_params_t *hw_params,
		int inPrecision)
{
	struct precinfo {
		snd_pcm_format_t format;
		int encoding;
		int precision;
		int swap;
	};
	struct precinfo info[] = {
#if defined(WORDS_BIGENDIAN)
		{ SND_PCM_FORMAT_S16_BE, SIDEMU_SIGNED_PCM,
			SIDEMU_16BIT, false },
		{ SND_PCM_FORMAT_U16_BE, SIDEMU_UNSIGNED_PCM,
			SIDEMU_16BIT, false },
		{ SND_PCM_FORMAT_S16_LE, SIDEMU_SIGNED_PCM,
			SIDEMU_16BIT, true },
		{ SND_PCM_FORMAT_U16_LE, SIDEMU_UNSIGNED_PCM,
			SIDEMU_16BIT, true },
#else
		{ SND_PCM_FORMAT_S16_LE, SIDEMU_SIGNED_PCM,
			SIDEMU_16BIT, false },
		{ SND_PCM_FORMAT_U16_LE, SIDEMU_UNSIGNED_PCM,
			SIDEMU_16BIT, false },
		{ SND_PCM_FORMAT_S16_BE, SIDEMU_SIGNED_PCM,
			SIDEMU_16BIT, true },
		{ SND_PCM_FORMAT_U16_BE, SIDEMU_UNSIGNED_PCM,
			SIDEMU_16BIT, true },
#endif
		{ SND_PCM_FORMAT_S8, SIDEMU_SIGNED_PCM,
			SIDEMU_8BIT, false },
		{ SND_PCM_FORMAT_U8, SIDEMU_UNSIGNED_PCM,
			SIDEMU_8BIT, false }
	};

	switch (inPrecision) {
		case SIDEMU_8BIT:
			for (int i = 0; i < 2; i++) {
				struct precinfo t;
				memcpy(&t, &info[i], sizeof(t));
				memcpy(&info[i + 4], &info[i], sizeof(t));
				memcpy(&info[i], &t, sizeof(t));
			}
			break;
		case SIDEMU_16BIT:
			break;
		default:
			fprintf(stderr, "unsupported precision: %d bits\n",
					inPrecision);
			return false;
	}

	int err;
	for (int i = 0; i < 6; i++) {
		err = snd_pcm_hw_params_set_format(handle, hw_params,
				info[i].format);
		if (err < 0) {
			continue;
		}

		precision = info[i].precision;
		encoding = info[i].encoding;
		swapEndian = info[i].swap;
		bufShift = info[i].precision == SIDEMU_8BIT ? 0 : 1;
		return true;
	}
	
	fprintf(stderr, "cannot set sample format: %s\n", snd_strerror(err));
	return false;
}



bool audioDriver::InitAlsa(snd_pcm_t *handle, int inPrecision,
		int inFrequency, int inChannels)
{
	snd_pcm_hw_params_t *hw_params;
	unsigned int get;
	int err;

	err = snd_pcm_hw_params_malloc(&hw_params);
	if (err < 0) {
		fprintf(stderr, "cannot allocate hardware "
				"parameter structure: %s\n",
				snd_strerror(err));
		return false;
	}

	err = snd_pcm_hw_params_any(handle, hw_params);
	if (err < 0) {
		fprintf(stderr, "cannot initialize hardware "
				"parameter structure: %s\n",
				snd_strerror(err));
		snd_pcm_hw_params_free(hw_params);
		return false;
	}

	err = snd_pcm_hw_params_set_access(handle, hw_params,
			SND_PCM_ACCESS_RW_INTERLEAVED);
	if (err < 0) {
		fprintf(stderr, "cannot set access type: %s\n",
				snd_strerror(err));
		snd_pcm_hw_params_free(hw_params);
		return false;
	}

	/* try to find supported format/precision */
	if (! SetFormat(handle, hw_params, inPrecision)) {
		snd_pcm_hw_params_free(hw_params);
		return false;
	}

	/* use closest frequency */
	get = (unsigned int) inFrequency;
	err = snd_pcm_hw_params_set_rate_near(handle, hw_params, &get, 0);
	if (err < 0) {
		fprintf(stderr, "cannot set sample rate: %s\n",
				snd_strerror(err));
		snd_pcm_hw_params_free(hw_params);
		return false;
	}
	if (get != (unsigned int) inFrequency) {
		frequency = get;
	}

	int chancount;
	switch (inChannels) {
		case SIDEMU_STEREO:
			chancount = 2;
			channels = SIDEMU_STEREO;
			bufShift++;
			break;
		case SIDEMU_MONO:
			chancount = 1;
			channels = SIDEMU_MONO;
			break;
		default:
			fprintf(stderr, "unsupported channel count: %d\n",
					inChannels);
			snd_pcm_hw_params_free(hw_params);
			return false;
	}
	err = snd_pcm_hw_params_set_channels(handle, hw_params, chancount);
	if (err < 0) {
		fprintf(stderr, "cannot set channel count: %s\n",
				snd_strerror(err));
		snd_pcm_hw_params_free(hw_params);
		return false;
	}

	err = snd_pcm_hw_params(handle, hw_params);
	if (err < 0) {
		fprintf(stderr, "cannot set parameters: %s\n",
				snd_strerror(err));
		snd_pcm_hw_params_free(hw_params);
		return false;
	}

	snd_pcm_hw_params_free(hw_params);

	err = snd_pcm_prepare(handle);
	if (err < 0) {
		fprintf(stderr, "cannot prepare audio interface for use: %s\n",
				snd_strerror(err));
		return false;
	}

	return true;
}



bool audioDriver::Open(udword inFreq, int inPrecision, int inChannels,
		int inFragments, int inFragBase)
{
	bool ret;
	int err;
	snd_pcm_t *handle;

	/* set wanted values, may be overwritten */
	frequency = inFreq;
	precision = inPrecision;
	channels = inChannels;

	err = snd_pcm_open(&handle, ALSA_DEFAULT, SND_PCM_STREAM_PLAYBACK, 0);
	if (err < 0) {
		fprintf(stderr, "cannot open audio device: %s\n",
				snd_strerror(err));
		errorString = "ERROR: Could not open audio device.";
		return false;
	}

	ret = InitAlsa(handle, inPrecision, inFreq, inChannels);
	if (ret == false) {
		errorString = "ERROR: Could not initialise audio device.";
		snd_pcm_close(handle);
		return false;
	}

	alsa = handle;
	blockSize = 4096;

	return true;
}



// Close an opened audio device, free any allocated buffers and
// reset any variables that reflect the current state.
void audioDriver::Close()
{
	if (alsa != NULL) {
		snd_pcm_close(alsa);
		alsa = NULL;
    	}
}



void audioDriver::Play(ubyte* pBuffer, int bufferSize)
{
	if (alsa != NULL) {
		if (swapEndian) {
			for (int n = 0; n < bufferSize; n += 2) {
				ubyte tmp = pBuffer[n + 0];
				pBuffer[n + 0] = pBuffer[n + 1];
				pBuffer[n + 1] = tmp;
			}
		}
		snd_pcm_writei(alsa, pBuffer, bufferSize >> bufShift);
	}
}
