using GLib;
using Gtk;
using Gdk;
using Glade;

#ifdef MAEMO
using Hildon;
#endif

public class GuiGtk : GLib.Object {

#define KEY_CANCEL     0xff1b
#define KEY_MENU       0xffc1
#define KEY_FULLSCREEN 0xffc3
#define KEY_ZOOMIN     0xffc4
#define KEY_ZOOMOUT    0xffc5

#ifdef DEV
  string BASE_PATH="..";
  const string GLADE_FILENAME="glade/guigtk.glade";
  const string BG_FILENAME="pixmaps/kshisen_bgnd.png";
  const string TILESET_FILENAME_CLASSIC="pixmaps/tileset.png";
  const string TILESET_FILENAME_LETTERS="pixmaps/tileset_letters.png";
#else
  string BASE_PATH=null;
  const string GLADE_FILENAME="guigtk.glade";
  const string BG_FILENAME="pixmaps/kshisen_bgnd.png";
  const string TILESET_FILENAME_CLASSIC="pixmaps/tileset.png";
  const string TILESET_FILENAME_LETTERS="pixmaps/tileset_letters.png";
#endif
  
  // cstate: Control state
  const int STATE_PLAY_UNINITIALIZED=0;  
  const int STATE_PLAY_IDLE=1;
  const int STATE_PLAY_SELECTED1=2;
  const int STATE_PLAY_SELECTED2=3;
  const int STATE_PLAY_GAMEOVER=4;
  
  // pstate: Paint states
  const int STATE_PAINT_BOARD=0;
  const int STATE_PAINT_SELECTED1=1;
  const int STATE_PAINT_SELECTED2=2;
  const int STATE_PAINT_MATCHED=3;  
  const int STATE_PAINT_WIN=4;
  const int STATE_PAINT_LOSE=5;
  const int STATE_PAINT_HINT=6;
  const int STATE_PAINT_TIME=7;
  
  const int NORMAL_SCREEN_WIDTH=696;
  const int NORMAL_SCREEN_HEIGHT=396;
  const int FULL_SCREEN_WIDTH=800;
  const int FULL_SCREEN_HEIGHT=480;
  const int DCSEC=100000;
  
  int SCREEN_WIDTH=NORMAL_SCREEN_WIDTH;
  int SCREEN_HEIGHT=NORMAL_SCREEN_HEIGHT;
  const int TILESET_ROWS=4;
  const int TILESET_COLS=9;
  int TILE_HEIGHT;
  int TILE_WIDTH;

  private Glade.XML xml;
  private Gtk.MenuShell menu;
  private Gtk.DrawingArea draw;
  private Gdk.Pixbuf bg;
  private Gdk.Pixbuf[] tile;
  private Gtk.Window winabout;
  private Gtk.ImageMenuItem undo;  
  
#ifdef MAEMO
  private Hildon.Program hprogram;
  private Hildon.Window hwin;
  private bool fullscreen;
#endif  
  
  private int[] selection1=new int[2];
  private int[] selection2=new int[2];
  private List<Point>? path=null;
  private List<Line>? pairs=null;
  private TimeVal start_time;
  private TimeVal play_time;
  private bool timer_registered=false;
                
  public ShishenSho app {get; set;}
  public int cstate;
  public int pstate;
  
  construct {
#ifdef DEV
      stdout.printf("Development mode\n");
#endif
    
    try {

#ifndef DEV
      string bp;
      foreach (string dir in Environment.get_system_data_dirs ()) {
        bp = dir + "shishensho";

        //Note older versions of vala used "File." now "FileUtils."
        if ( FileUtils.test (bp, FileTest.IS_DIR) ) {
          BASE_PATH = bp;
          break;
        }
      }
      if (BASE_PATH==null) BASE_PATH="/usr/share/shishensho";
#endif
    
      stderr.printf(BASE_PATH);
    
      xml = new Glade.XML(BASE_PATH+"/"+GLADE_FILENAME, null, null);
      xml.signal_autoconnect_full(connect_signals);
      
      draw=(Gtk.DrawingArea)xml.get_widget("draw");
      menu=(Gtk.MenuShell)xml.get_widget("menu");
      winabout=(Gtk.Window)xml.get_widget("winabout");
      load_background(BG_FILENAME);
      load_tileset(TILESET_FILENAME_CLASSIC);
      
      Gtk.VBox vbox=(Gtk.VBox)xml.get_widget("vbox");
      Gtk.Window wingame=(Gtk.Window)xml.get_widget("wingame");
#ifdef MAEMO
      hprogram=Hildon.Program.get_instance();
      hwin=new Hildon.Window();
      draw.ref();
      vbox.remove(draw);
      hwin.add(draw);
      draw.unref();
      connect_signals("gui_gtk_on_wingame_destroy", hwin, "destroy", null, null, false);
      connect_signals("gui_gtk_on_key_press", hwin, "key_press_event", null, null, false);
      fullscreen=false;
      hwin.set_menu((Gtk.Menu)menu);
      hprogram.add_window(hwin);
      Environment.set_application_name(wingame.get_title());
      hwin.show_all();
#else
      Gtk.MenuBar menubar=new Gtk.MenuBar();
      weak List<Gtk.MenuItem> children=(List<Gtk.MenuItem>)menu.children;
      List<Gtk.MenuItem> childrencopy=new List<Gtk.MenuItem>();
      foreach (Gtk.MenuItem menuitem in children) {
        childrencopy.append(menuitem);
      }
      foreach (Gtk.MenuItem menuitem in childrencopy) {
        ((Gtk.Container)menuitem.parent).remove((Gtk.Widget)menuitem);
        menubar.append(menuitem);
      }
      menu=menubar;
      menu.set("visible",true);
      vbox.pack_start(menu,false,false,0);
      wingame.set("visible",true);
#endif
      undo=(Gtk.ImageMenuItem)xml.get_widget("undo");
      cstate=STATE_PLAY_UNINITIALIZED;
    } catch (GLib.Error e) {
      stderr.printf("Error loading files\n");
    }
  }
  
  private void paint(int pstate) {
    this.pstate=pstate;
    
    // STATE_PAINT_BOARD repaints the widget. The other states paint over
    // the previously existing graphics of the widget and must not
    // trigger a begin_paint_region()
    switch (pstate) {
      case STATE_PAINT_BOARD:
      case STATE_PAINT_WIN:
      case STATE_PAINT_LOSE:
      case STATE_PAINT_TIME:
      case STATE_PAINT_HINT:
        draw.set_double_buffered(true);
        break;
      case STATE_PAINT_SELECTED1:
      case STATE_PAINT_SELECTED2:
      case STATE_PAINT_MATCHED:
        draw.set_double_buffered(false);
        break;
    }
    
    draw.queue_draw();
    draw.window.process_all_updates();
  }
  
  private void control(int cstate) {
    this.cstate=cstate;
  }
  
  private void sleep(int deci_seconds) {
    Thread.usleep(deci_seconds*DCSEC);
  }
  
  private void load_tileset(string filename) throws GLib.Error {
    // 4 rows x 9 columns of tiles in the same original image
    tile=new Gdk.Pixbuf[TILESET_ROWS*TILESET_COLS];
    Pixbuf tiles=new Gdk.Pixbuf.from_file(BASE_PATH+"/"+filename);
    
    TILE_HEIGHT=tiles.height/TILESET_ROWS;
    TILE_WIDTH=tiles.width/TILESET_COLS;
    
    int k=0;
    for (int i=0;i<TILESET_ROWS;i++) {
      for (int j=0;j<TILESET_COLS;j++) {
        // Tiles are 56px height, 40px width each
        tile[k]=new Gdk.Pixbuf.subpixbuf(tiles,j*TILE_WIDTH,i*TILE_HEIGHT,TILE_WIDTH,TILE_HEIGHT);
        k++;
      }
    }
  }
  
  private void load_background(string filename) throws GLib.Error {
    bg=new Gdk.Pixbuf.from_file(BASE_PATH+"/"+filename);
  }
  
  private void initialize_game() {
      app.newPlay();
      pairs=app.board.get_pairs(1);
      undo.sensitive=false;
      start_time=TimeVal();
      start_time.get_current_time();
      play_time.tv_usec=0;
      play_time.tv_sec=0;
      if (app.time_counter && !timer_registered) {
        Timeout.add(1000,on_update_time);
        timer_registered=true;
      }
      pstate=STATE_PAINT_BOARD;
      control(STATE_PLAY_IDLE);
  }
  
  public static void on_wingame_destroy(Gtk.Widget widget) {
    Gtk.main_quit();
  }
  
  [CCode (instance_pos = -1)]
  public void on_new_activate(Gtk.Widget widget) {
    control(STATE_PLAY_UNINITIALIZED);
    paint(STATE_PAINT_BOARD);
  }
  
  [CCode (instance_pos = -1)]
  public void on_hint_activate(Gtk.Widget widget) {
    if (cstate!=STATE_PLAY_GAMEOVER) {
      paint(STATE_PAINT_HINT);
      sleep(10);
      paint(STATE_PAINT_BOARD);
      control(STATE_PLAY_IDLE);
    }
  }
  
  [CCode (instance_pos = -1)]
  public void on_undo_activate(Gtk.Widget widget) {
    if (app.board.get_can_undo()) {
      if (cstate==STATE_PLAY_GAMEOVER && app.time_counter && !timer_registered) {
        // Reprogram the time update that had been
        // deactivated with the game over status
        Timeout.add(1000,on_update_time);
        timer_registered=true;
      }
      app.board.undo();
      paint(STATE_PAINT_BOARD);
      undo.sensitive=app.board.get_can_undo();
      pairs=app.board.get_pairs(1);
      control(STATE_PLAY_IDLE);
    }
  }
  
  [CCode (instance_pos = -1)]
  public void on_small_activate(Gtk.Widget widget) {
    app.board_size[0]=6+2;
    app.board_size[1]=10+2;
    on_new_activate(widget);
  }
  
  [CCode (instance_pos = -1)]
  public void on_medium_activate(Gtk.Widget widget) {
    app.board_size[0]=6+2;
    app.board_size[1]=12+2;
    on_new_activate(widget);
  }
  
  [CCode (instance_pos = -1)]
  public void on_large_activate(Gtk.Widget widget) {
    app.board_size[0]=6+2;
    app.board_size[1]=16+2;
    on_new_activate(widget);
  }
  
  [CCode (instance_pos = -1)]
  public void on_hard_activate(Gtk.Widget widget) {
    app.difficulty=1;
    on_new_activate(widget);
  }
    
  [CCode (instance_pos = -1)]
  public void on_easy_activate(Gtk.Widget widget) {
    app.difficulty=2;
    on_new_activate(widget);
  }
  
  [CCode (instance_pos = -1)]
  public void on_classic_activate(Gtk.Widget widget) {
    try {
      load_tileset(TILESET_FILENAME_CLASSIC);
      paint(STATE_PAINT_BOARD);
    } catch (GLib.Error e) {
      stderr.printf("Error loading files\n");
    }
  }

  [CCode (instance_pos = -1)]
  public void on_letters_activate(Gtk.Widget widget) {
    try {
      load_tileset(TILESET_FILENAME_LETTERS);
      paint(STATE_PAINT_BOARD);
    } catch (GLib.Error e) {
      stderr.printf("Error loading files\n");
    }
  }
  
  [CCode (instance_pos = -1)]
  public void on_gravity_activate(Gtk.Widget widget) {
    app.gravity=!app.gravity;
    on_new_activate(widget);
  }
  
  [CCode (instance_pos = -1)]
  public void on_time_counter_activate(Gtk.Widget widget) {
    app.time_counter=!app.time_counter;
    if (app.time_counter && cstate!=STATE_PLAY_GAMEOVER && !timer_registered) {
      // Reprogram the time update that had been
      // deactivated with the time_counter=false
      Timeout.add(1000,on_update_time);
      timer_registered=true;
    }
  }
  
  [CCode (instance_pos = -1)]
  public void on_about_activate(Gtk.Widget widget) {
    winabout.visible=true;
  }
  
  [CCode (instance_pos = -1)]
  public void on_aboutok_clicked(Gtk.Widget widget) {
    winabout.visible=false;
  }
  
#ifdef MAEMO  
  [CCode (instance_pos = -1)]
  public bool on_key_press(Gtk.Widget widget, Gdk.EventKey key) {
    bool result=false;
    switch (key.keyval) {
      // Capture F6 key event to toggle fullscreen mode
      case KEY_FULLSCREEN:
        if (fullscreen) {
          //Hildon.Banner.show_information(hwin, null, "Normal screen");
          hwin.unfullscreen();
          SCREEN_HEIGHT=NORMAL_SCREEN_HEIGHT;
          SCREEN_WIDTH=NORMAL_SCREEN_WIDTH;
        } else {
          //Hildon.Banner.show_information(hwin, null, "Full screen");
          hwin.fullscreen();
          SCREEN_HEIGHT=FULL_SCREEN_HEIGHT;
          SCREEN_WIDTH=FULL_SCREEN_WIDTH;
        }
        fullscreen=!fullscreen;
        result=true;
        break;
      case KEY_CANCEL:
        on_undo_activate(widget);
        break;
      case KEY_ZOOMIN:
        on_hint_activate(widget);
        break;
    }
    return result;
  }
#endif  
  
  public bool on_update_time() {
    int pstate0=pstate;
    paint(STATE_PAINT_TIME);
    if (pstate0==STATE_PAINT_BOARD) 
      pstate=pstate0;
    else paint(pstate0);
    timer_registered=app.time_counter && cstate!=STATE_PLAY_GAMEOVER;
    return timer_registered;
  }  
  
  [CCode (instance_pos = -1)]
  public void connect_signals(string handler_name, GLib.Object object, string signal_name, string? signal_data, GLib.Object? connect_object, bool after) {
    Module module = Module.open(null, ModuleFlags.BIND_LAZY);
    void* sym;
        
    if(!module.symbol(handler_name, out sym)) {
      stdout.printf("Symbol not found: %s\n",handler_name);
    } else {
      GLib.Signal.connect(object, signal_name, (GLib.Callback) sym, this);
    }
  }
  
  [CCode (instance_pos = -1)]
  public void on_draw_expose_event(Gtk.Widget widget, Gdk.EventExpose event) {
    
    return_if_fail(widget==draw);
    
    if (cstate==STATE_PLAY_UNINITIALIZED) initialize_game();
    
    // Board upper left corner on screen
    int x0=0;
    int y0=0;
    
    if (app!=null && app.board!=null) {
      x0=(SCREEN_WIDTH-app.board.board_size[1]*TILE_WIDTH)/2;
      y0=(SCREEN_HEIGHT-app.board.board_size[0]*TILE_HEIGHT)/2;
    }
        
    Gdk.Color red=Gdk.Color();
    Color.parse("#F00",out red);
    Gdk.Color orange=Gdk.Color();
    Color.parse("#FC0",out orange);
    Gdk.GC gc=new Gdk.GC(draw.window);
    
    // Background & board painting
    switch (pstate) {
      case STATE_PAINT_BOARD:
      case STATE_PAINT_TIME:
      case STATE_PAINT_WIN:
      case STATE_PAINT_LOSE:
      case STATE_PAINT_HINT:
      // Background painting    
        for (int i=0;i<SCREEN_HEIGHT/bg.height+1;i++) {
          for (int j=0;j<SCREEN_WIDTH/bg.width+1;j++) {
            draw_pixbuf(draw.window,gc,bg,0,0,j*bg.width,i*bg.height,-1,-1,Gdk.RgbDither.NONE,0,0);
          }
        }
        
        // Board painting
        // Max visible size: 7x17
        if (app!=null && app.board!=null) {
          for (int i=0;i<app.board.board_size[0];i++) {
            for (int j=0;j<app.board.board_size[1];j++) {
              // Tiles are 56px height, 40px width each
              uchar piece=app.board.board[i,j];
              if (piece!=0) {
                draw_pixbuf(draw.window,gc,tile[piece],0,0,x0+j*TILE_WIDTH,y0+i*TILE_HEIGHT,-1,-1,Gdk.RgbDither.NONE,0,0);
              }
            }
          }
        }
        break;
    }
      
    // Red rectangle for selection 1
    switch (pstate) {
      case STATE_PAINT_SELECTED1:
        gc.set_rgb_fg_color(red);
        gc.set_line_attributes(3,Gdk.LineStyle.SOLID,Gdk.CapStyle.ROUND,Gdk.JoinStyle.ROUND);
        draw_rectangle(draw.window,gc,false,x0+selection1[1]*TILE_WIDTH-2,y0+selection1[0]*TILE_HEIGHT-2,TILE_WIDTH+2*2,TILE_HEIGHT+2*2);
        break;
    }
    
    // Red rectangle for selection 2
    switch (pstate) {
      case STATE_PAINT_SELECTED2:
        gc.set_rgb_fg_color(red);
        gc.set_line_attributes(3,Gdk.LineStyle.SOLID,Gdk.CapStyle.ROUND,Gdk.JoinStyle.ROUND);
        draw_rectangle(draw.window,gc,false,x0+selection2[1]*TILE_WIDTH-2,y0+selection2[0]*TILE_HEIGHT-2,TILE_WIDTH+2*2,TILE_HEIGHT+2*2);
        break;
    }
  
    // Matching path
    switch (pstate) {
      case STATE_PAINT_MATCHED:
        gc.set_rgb_fg_color(red);
        gc.set_line_attributes(3,Gdk.LineStyle.SOLID,Gdk.CapStyle.ROUND,Gdk.JoinStyle.ROUND);
        if (path!=null) {
          weak Point? p0=null;
          foreach (Point p1 in path) {
            if (p0!=null) {
              draw_line(draw.window,gc,
                x0+p0.j*TILE_WIDTH-2+(TILE_WIDTH/2),y0+p0.i*TILE_HEIGHT-2+(TILE_HEIGHT/2),
                x0+p1.j*TILE_WIDTH-2+(TILE_WIDTH/2),y0+p1.i*TILE_HEIGHT-2+(TILE_HEIGHT/2));
            }
            p0=p1;
          }
        }
        break;
    }
    
    // Orange hint rectangles
    switch (pstate) {
      case STATE_PAINT_HINT:
        if (pairs!=null && pairs.length()>0) {
          weak Line pair=pairs.first().data;
          weak Point a=pair.a;
          weak Point b=pair.b;
          path=app.board.get_path(a,b);
          gc.set_rgb_fg_color(orange);
          gc.set_line_attributes(3,Gdk.LineStyle.ON_OFF_DASH,Gdk.CapStyle.ROUND,Gdk.JoinStyle.ROUND);
          
          draw_rectangle(draw.window,gc,false,x0+(int)a.j*TILE_WIDTH-2,y0+(int)a.i*TILE_HEIGHT-2,TILE_WIDTH+2*2,TILE_HEIGHT+2*2);
          
          if (path!=null) {
            weak Point? p0=null;
            foreach (Point p1 in path) {
              if (p0!=null) {
                draw_line(draw.window,gc,
                  x0+p0.j*TILE_WIDTH-2+(TILE_WIDTH/2),y0+p0.i*TILE_HEIGHT-2+(TILE_HEIGHT/2),
                  x0+p1.j*TILE_WIDTH-2+(TILE_WIDTH/2),y0+p1.i*TILE_HEIGHT-2+(TILE_HEIGHT/2));
              }
              p0=p1;
            }
            path=null;
          }
          
          draw_rectangle(draw.window,gc,false,x0+(int)b.j*TILE_WIDTH-2,y0+(int)b.i*TILE_HEIGHT-2,TILE_WIDTH+2*2,TILE_HEIGHT+2*2);
        }
        break;
    }
  
    // Win & loose notifications  
    switch (pstate) {
      case STATE_PAINT_WIN:
        draw_message(SCREEN_WIDTH/2,SCREEN_HEIGHT/2,true,"You Win!", "#FFF", "Sans Bold 100");
        break;
      case STATE_PAINT_LOSE:
        draw_message(SCREEN_WIDTH/2,SCREEN_HEIGHT/2,true,"Game Over", "#FFF", "Sans Bold 100");
        break;
    }
  
    if (app.time_counter) switch (pstate) {
      case STATE_PAINT_TIME:
      case STATE_PAINT_BOARD:
      case STATE_PAINT_WIN:
      case STATE_PAINT_LOSE:
      case STATE_PAINT_HINT:
        if (cstate!=STATE_PLAY_GAMEOVER) {
          play_time.get_current_time();
          play_time.tv_usec=0;
          play_time.tv_sec=play_time.tv_sec-start_time.tv_sec;
        }
        int hours=(int)(play_time.tv_sec/(60*60));
        int minutes=(int)((play_time.tv_sec/60)%60);
        int seconds=(int)(play_time.tv_sec%60);
        string time="%01d:%02d:%02d".printf(hours,minutes,seconds);
        
        int TIME_POS_X=SCREEN_WIDTH-100;
        int TIME_POS_Y=SCREEN_HEIGHT-30;
        
        draw_message(TIME_POS_X+3,TIME_POS_Y+3,false,time,"#000","Sans Bold 15");
        draw_message(TIME_POS_X,TIME_POS_Y,false,time,"#FFF","Sans Bold 15");
        break;
    }
  }
  
  [CCode (instance_pos = -1)]
  public void on_draw_button_press_event(Gtk.Widget widget, Gdk.EventButton event) {
    // The i,j coordinates are computed back here
    int x=(int)event.x;
    int y=(int)event.y;
    int i=(y-(SCREEN_HEIGHT-app.board.board_size[0]*TILE_HEIGHT)/2)/TILE_HEIGHT;
    int j=(x-(SCREEN_WIDTH-app.board.board_size[1]*TILE_WIDTH)/2)/TILE_WIDTH;

    switch (cstate) {
      case STATE_PLAY_IDLE:
        if (i>=0 && i<app.board.board_size[0]
          && j>=0 && j<app.board.board_size[1]
          && app.board.board[i,j]!=0) {
          selection1[0]=i;
          selection1[1]=j;
          paint(STATE_PAINT_SELECTED1);
          control(STATE_PLAY_SELECTED1);
        }
        break;
      case STATE_PLAY_SELECTED1:
        if (i>=0 && i<app.board.board_size[0]
          && j>=0 && j<app.board.board_size[1]
          && app.board.board[i,j]!=0) {
          if (i==selection1[0] && j==selection1[1]) {
            paint(STATE_PAINT_BOARD);
            control(STATE_PLAY_IDLE);
          } else {
            selection2[0]=i;
            selection2[1]=j;
            paint(STATE_PAINT_SELECTED2);
            
            Point a=new Point(selection1[0],selection1[1]);
            Point b=new Point(selection2[0],selection2[1]);
            path=app.board.get_path(a,b);
            paint(STATE_PAINT_MATCHED);
#ifndef MAEMO
            sleep(2);
#endif
            paint(STATE_PAINT_BOARD);
#ifndef MAEMO
            sleep(2);
#endif
            if (path.length()>0) {
              app.board.play(a,b);
            }
            path=null;
            paint(STATE_PAINT_BOARD);
            
            pairs=app.board.get_pairs(1);
            if (pairs.length()==0) {
              if (app.board.getNumPieces()==0) {
                paint(STATE_PAINT_WIN);
              } else {
                paint(STATE_PAINT_LOSE);
              }
              control(STATE_PLAY_GAMEOVER);
            } else {
              control(STATE_PLAY_IDLE);
            }
            undo.sensitive=app.board.get_can_undo();
          }
        }
        break;
      case STATE_PLAY_GAMEOVER:
        on_new_activate(draw);
        paint(STATE_PAINT_BOARD);
        break;
    }
    
  }
  
  public void draw_message(int x, int y, bool centered, string message, string color, string font)  {
        Gdk.Color c=Gdk.Color();
        Color.parse((string)color,out c);
        Gdk.GC gc=new Gdk.GC(draw.window);
        gc.set_rgb_fg_color(c);
        Pango.Context pc=pango_context_get_for_screen(draw.window.get_screen());
        Pango.Layout pl=new Pango.Layout(pc);
        pl.set_text(message,-1);
        pl.set_font_description(Pango.FontDescription.from_string(font));
        int width, height;
        pl.get_pixel_size(out width, out height);
        if (centered) {
          Gdk.draw_layout(draw.window,gc,x-width/2,y-height/2,pl);
        } else {
          Gdk.draw_layout(draw.window,gc,x,y,pl);
        }
  }
  
  
  public static void gui_init ([CCode (array_length_pos = 0.9)] ref string[] args) {
    Gtk.init(ref args);
  }
  
  public static void gui_main() {
    Gtk.main();
  }

}
