using GLib;

public class Board : GLib.Object {

  private static string charpieces = " abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
  private static ulong rndnext = -1;

  public int difficulty=1;
  public bool gravity=true;
  public int [] board_size;
  public uchar [,] board;
  public Queue<Move> history;







  public void initialize(int size_i, int size_j) {
    board_size = new int[2];
    board_size[0]=size_i;
    board_size[1]=size_j;
    board = new uchar[board_size[0],board_size[1]];
    for (int i=0;i<board_size[0];i++)
      for (int j=0;j<board_size[1];j++)
        board[i,j]=0;
    history=new Queue<Object>();
  }

  public static string piece_to_string(uchar piece) {
    return charpieces.substring(piece,1);
  }

  public static uchar string_to_piece(string piece) {
    uchar upiece;
    long charpieces_len=charpieces.size();
    for(upiece=0;(upiece<charpieces_len && charpieces.substring(upiece,1)!=piece);upiece++);
    if (upiece<charpieces_len) return upiece;
    else return 0;
  }

  public string to_string() {
    string result="  ";
    for (int j=0;j<board_size[1];j++) {
        if (j>0) result+=" ";
        result+="%d".printf(j%10);
    }
    result+="\n  "+string_repeat("--",board_size[1]);
    for (int i=0;i<board_size[0];i++) {
      result+="\n%d|".printf(i%10);
      for (int j=0;j<board_size[1];j++) {
        if (j>0) result+=" ";
        result+=charpieces.substring(board[i,j],1);
      }
      result+=" |\n";
      if (i<board_size[0]-1)
        result+=" |"+string_repeat("  ",board_size[1])+"|";
    }
    result+="  "+string_repeat("--",board_size[1])+"\n";
    return result;
  }

  public static Board build_random_board(int size_i, int size_j, int difficulty, bool gravity) {
    Board b=new Board();
    b.initialize(size_i,size_j);
    b.difficulty=difficulty;
    b.gravity=gravity;

    int num_different_pieces=((b.board_size[0]-2)*(b.board_size[1]-2)/((b.difficulty+1)*2))+1;
    for (int n=0;n<((b.difficulty+1)*2);n++) {
      for (char k=0;k<num_different_pieces;k++) {
        int i,j;
        do {
          j=(b.myrand() % (b.board_size[1]-2))+1;
          i=b.find_free_row(j);
        } while (i<1);
        b.board[i,j]=k;
      }
    }

    return b;
  }
  public List<Point>? get_path(Point a, Point b) {
    List<Point> result=new List<Point>();

    if (get_piece(a)!=get_piece(b)) return null;

    List<Line> h=get_horizontal_lines(a,b);
    List<Line> v=get_vertical_lines(a,b);
    Line ha=null, va=null, hb=null, vb=null;

    foreach (Line l in h) {
      if (l.contains(a)) ha=l;
      if (l.contains(b)) hb=l;
      if (ha!=null && hb!=null) break;
    }

    foreach (Line l in v) {
      if (l.contains(a)) va=l;
      if (l.contains(b)) vb=l;
      if (va!=null && vb!=null) break;
    }



    if ((ha==null && va==null) || (hb==null && vb==null))
      return null;

    if (ha.equals(hb) || va.equals(vb)) {
      result.append(a);
      result.append(b);
      return result;
    }

    Point ab;

    ab=ha.cuts(vb);


    if (ab!=null) {
      result.append(a);
      result.append(ab);
      result.append(b);
      return result;
    }

    ab=va.cuts(hb);


    if (ab!=null) {
      result.append(a);
      result.append(ab);
      result.append(b);
      return result;
    }

    foreach (Line l in v) {
      Point al=l.cuts(ha);
      Point bl=l.cuts(hb);




      if (al!=null && bl!=null) {
        result.append(a);
        result.append(al);
        result.append(bl);
        result.append(b);
        return result;
      }
    }

    foreach (Line l in h) {
      Point al=l.cuts(va);
      Point bl=l.cuts(vb);




      if (al!=null && bl!=null) {
        result.append(a);
        result.append(al);
        result.append(bl);
        result.append(b);
        return result;
      }
    }

    return null;
  }

  public uchar get_piece(Point p) {
    return board[p.i,p.j];
  }

  public void set_piece(Point p, uchar piece) {
    board[p.i,p.j]=piece;
  }

  public string get_str_piece(Point p) {
    uchar piece=board[p.i,p.j];
    return charpieces.substring(piece,1);
  }

  public void set_str_piece(Point p, string piece) {
    uchar upiece;
    long charpieces_len=charpieces.size();
    for(upiece=0;(upiece<charpieces_len && charpieces.substring(upiece,1)!=piece);upiece++);
    if (upiece<charpieces_len) board[p.i,p.j]=upiece;
  }

  public void play(Point a0, Point b0) {

    Point a=(a0.i<b0.i)?a0:b0;
    Point b=(a0.i<b0.i)?b0:a0;
    Move m=new Move(a,b,get_piece(a));
    history.push_head(m);
    set_piece(a,0);
    process_gravity(a);
    set_piece(b,0);
    process_gravity(b);
  }

  public bool get_can_undo() {
    return !history.is_empty();
  }

  public void undo() {
    if (!get_can_undo()) return;
    Move m=history.pop_head();
    undo_gravity(m.b);
    set_piece(m.b,m.piece);
    undo_gravity(m.a);
    set_piece(m.a,m.piece);
  }

  public List<Line> get_pairs(int max_results) {
    List<Line> result=new List<Line>();
    List<int> pieces=new List<int>();
    List<List> piece_points=new List<List>();
    for (int i=0;i<board_size[0];i++)
      for (int j=0;j<board_size[1];j++) {
        int piece=(int)board[i,j];
        if (piece==0) continue;
        int key=pieces.index(piece);
        Point *p=new Point(i,j);
        if (key==-1) {
          List<Point> points0=new List<Point>();
          points0.append(#p);
          pieces.append(piece);
          piece_points.append(#points0);

          key=pieces.index(piece);
          weak List<Point> points1=piece_points.nth_data(key);
        } else {
          weak List<Point> points1=piece_points.nth_data(key);
          points1.append(p);
        }
      }

    int nresults=0;
    int k=0;
    foreach (weak List<Point> points in piece_points) {
      int n=(int)points.length();
      for (int i=0;i<n;i++) {
        weak Point a=points.nth_data(i);
        for (int j=i+1;j<n;j++) {
          weak Point b=points.nth_data(j);
          List<Point> path=get_path(a.copy(),b.copy());
          if (path!=null && path.length()>0) {
            result.append(new Line(a,b));
            if (nresults++==max_results) break;
          }
        }
        if (nresults==max_results) break;
      }
      k++;
      foreach (Point *point in points) delete point;
      if (nresults==max_results) break;
    }
    return result;
  }

  public int getNumPieces() {
    int result=0;
    for (int j=0;j<board_size[1];j++) {
      for (int i=0;i<board_size[0];i++) {
        if (board[i,j]!=0) result++;
      }
    }
    return result;
  }





  private Board() {
  }


  private int myrand() {
      if (rndnext==-1) {

        TimeVal tv=TimeVal();
        tv.get_current_time();
        rndnext=((ulong)tv.tv_usec);
      }

      rndnext = rndnext * 1103515245 + 12345;
      return ((int)((ulong)(rndnext/65536) % 32768));
  }

  private string string_repeat(string s, int n) {
    string result="";
    for (int i=0;i<n;i++)
      result+=s;
    return result;
  }

  private int find_free_row(int j) {
    for (int i=1;i<board_size[0]-1;i++) {
      if (board[i,j]!=0) return (i-1);
    }
    return (board_size[0]-1-1);
  }

  private List<Line> get_horizontal_lines(Point exclude_a, Point exclude_b) {
    List<Line> result=new List<Line>();
    for (int i=0;i<board_size[0];i++) {
      int j0=-1;
      bool empty;
      for (int j=0;j<board_size[1];j++) {
        empty=(board[i,j]==0 || (i==exclude_a.i && j==exclude_a.j)
          || (i==exclude_b.i && j==exclude_b.j));
        if (j0==-1 && empty) {
          j0=j;
        } else if (j0!=-1 && !empty) {
          result.append(new Line(new Point(i,j0), new Point(i,j-1)));
          j0=-1;
        }
      }
      if (j0!=-1) result.append(new Line(new Point(i,j0), new Point(i,board_size[1]-1)));
    }





    return result;
  }

  private List<Line> get_vertical_lines(Point exclude_a, Point exclude_b) {
    List<Line> result=new List<Line>();
    for (int j=0;j<board_size[1];j++) {
      int i0=-1;
      bool empty;
      for (int i=0;i<board_size[0];i++) {
        empty=(board[i,j]==0 || (i==exclude_a.i && j==exclude_a.j)
          || (i==exclude_b.i && j==exclude_b.j));
        if (i0==-1 && empty) {
          i0=i;
        } else if (i0!=-1 && !empty) {
          result.append(new Line(new Point(i0,j), new Point(i-1,j)));
          i0=-1;
        }
      }
      if (i0!=-1) result.append(new Line(new Point(i0,j), new Point(board_size[0]-1,j)));
    }





    return result;
  }

  private void process_gravity(Point p) {
    if (gravity) for (int i=p.i;i>0;i--) board[i,p.j]=board[i-1,p.j];
  }

  private void undo_gravity(Point p) {
    if (gravity) for (int i=0;i<p.i;i++) board[i,p.j]=board[i+1,p.j];
  }

}
