/**
 * python-launcher-daemon.c
 * Daemon application to start Python and fork it when receiving
 * information from python-launcher (client).
 *
 * Copyright (C) 2005-2008 INdT - Instituto Nokia de Tecnologia
 *
 * Contact: Luciano Miguel Wolf <luciano.wolf@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include "Python.h"

#include "debug.h"

#include <stdlib.h>
#include <stdio.h>
#include <malloc.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <stddef.h>
#include <poll.h>
#include <alloca.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/un.h>
#include <sys/socket.h>

#include "python-launcher.h"

static int KEEP_RUNNING = 1;

/*
 * The 'select' used by daemon doesn't have a timeout, its necessary
 * to wakeup the socket to inform daemon that it's time to stop.
 */
static void stop_daemon(void)
{
    int sockfd, len, msg = -1;
    struct sockaddr_un address;

    sockfd = socket(PF_LOCAL, SOCK_STREAM, 0);
    address.sun_family = AF_LOCAL;

    len = sizeof(char) * strlen(PL_SOCK_PATH) + 1;
    strncpy(address.sun_path, PL_SOCK_PATH, len);
    len = SUN_LEN(&address);

    if (connect(sockfd, (struct sockaddr_un *)&address, len) < 0) {
        perror("Stop daemon:connect");
    } else {
        write(sockfd, &msg, sizeof(int));
        close(sockfd);
    }
}

/*
 * Callback to handle signals from a child process. It allows control
 * over daemon process, to stop it when requested.
 */
static void handle_child(int sign)
{
    pid_t child;
    int exit_status;

    child = wait(&exit_status);
    if ((child > 0) && (WIFEXITED(exit_status)))
        if (WEXITSTATUS(exit_status) == PL_STOP_DAEMON) {
            KEEP_RUNNING = 0;
            stop_daemon();
        }
}

/*
 * Initialize the Python interpreter. All configurations are
 * read from conf_file.
 *
 * Returns 0 on success or -1 on failure.
 */
static int load_python(const char *conf_file) {
    FILE *py_conf;
    int res;

    Py_Initialize();

    py_conf = fopen(conf_file, "r");
    if (py_conf)
        res = PyRun_SimpleFileEx(py_conf, conf_file, 1);
    else
        res = PyRun_SimpleString(DAEMON_CONF);

    return res;
}

/*
 * Fork and run the interpreter using args as parameters.
 *
 * Returns 0 on success or -1 on error.
 */
static int run_script(const int argc, const char **args)
{
    char *buffer, *path_index, *path, *script;
    char IMPORTS[] = "import os; os.chdir('%s')";
    int var_size;
    FILE *fd = NULL;

    int i;
    DEBUG ("argc: %d", argc);
    for (i = 0; i < argc; i++)
        DEBUG ("args[%d]: '%s'", i, args[i]);

    script = (char *)args[0];
    fd = fopen(script, "r");
    if (!fd) {
        perror("Open script file");
        return -1;
    }

    path_index = rindex(script, '/');
    var_size = strlen(script) - strlen(path_index);
    path = malloc(sizeof(char) * (var_size + 1));
    if (!path) {
        perror("Malloc path");
        free(script);
        return -1;
    }
    memcpy(path, script, var_size);
    path[var_size] = '\0';

    PyOS_AfterFork();
    PySys_SetArgv(argc, (char **)args);
    PyRun_SimpleString("gtk.init_check()");

    var_size = strlen(IMPORTS) + strlen(path) - 1;
    buffer = malloc(sizeof(char) * var_size);
    if (!buffer) {
        perror("Malloc IMPORTS buffer");
        free(script);
        free(path);
        return -1;
    }
    snprintf(buffer, var_size, IMPORTS, path);
    PyRun_SimpleString(buffer);
    PyRun_SimpleFile(fd, script);

    fclose(fd);
    free(buffer);
    free(path);
    return 0;
}

/**
 * Parse the serialized string array.
 *
 * The string is unparsed *IN PLACE* in given @p buf.
 *
 * @param buf buffer to operate on. It will be modified to contain the
 *        string array. Must not be NULL.
 * @param argc returns the number of elements in the string array.
 *
 * @return parsed string array. It's a single block of memory (actually
 *         it's the same as @p buf) and should be freed like it.
 */
static char **unmarshal_string_array(char *buf, int *argc)
{
    char **offsets;
    int i;

    offsets = (char **)buf;
    *argc = (int)offsets[0];
    offsets[0] = buf + sizeof(char *) * *argc;
    for (i = 1; i < *argc; i++)
         offsets[i] += (int)offsets[0];

    return offsets;
}

/*
 * Unpack buf to get an array of strings.
 *
 * Returns an array of strings, to be freed by the caller.
 */
static char **get_args(const int c_sock, int *elems)
{
    char *buf = NULL;
    int len;

    *elems = 0;
    if (read(c_sock, &len, sizeof(len)) < 0)
        return NULL;
    if (len < 0) {
        *elems = len;
        return NULL;
    }
    buf = malloc(sizeof(char) * len);
    if (!buf) {
        perror("Get_args:malloc");
        return NULL;
    }

    if (read(c_sock, buf, len) < 0) {
        free(buf);
        return NULL;
    }

    return unmarshal_string_array(buf, elems);
}

/*
 * Wait for connections.
 *
 * Returns a new (client) socket number on success, 0 on timeout or
 * -1 on failure.
 */
static int wait_clients(const int d_sock)
{
    int fdcount, c_sock;
    struct sockaddr_un c_sockad;
    socklen_t addrlen;
    struct pollfd pollfds;

    memset(&c_sockad, 0x00, sizeof(c_sockad));
    addrlen = SUN_LEN(&c_sockad);

    pollfds.fd = d_sock;
    pollfds.events = POLLIN;
    fdcount = poll(&pollfds, 1, -1);

    if ((fdcount > 0) && (pollfds.revents & POLLIN))
        c_sock = accept(d_sock, &c_sockad, &addrlen);
    else
        c_sock = fdcount;

    return c_sock;
}

/*
 * Watch socket to process all incoming requests.
 *
 * Returns a new pid (0 = child / -1 = stop the daemon)
 */
static pid_t handle_requests(const int d_sock, const int fd)
{
    char **args = NULL;
    int c_sock, elems;
    pid_t pid = 1;

    listen(d_sock, 3);
    signal(SIGCHLD, handle_child);

    if (fd >= 0) {
        if (write(fd, PL_SOCK_READY, sizeof(PL_SOCK_READY)) < 0) {
            perror("Write on client's pipe");
            return -1;
        }
    }

    while(KEEP_RUNNING && pid > 0) {
        c_sock = wait_clients(d_sock);
        if (c_sock <= 0)
            continue;

        pid = fork();
        if (pid) {
            close(c_sock);
        } else {
            signal(SIGCHLD, SIG_DFL);
            close(d_sock);

            args = get_args(c_sock, &elems);
            close(c_sock);
            if (!args) {
                DEBUG ();
                pid = elems;
            } else {
                DEBUG ();
                if (run_script(elems, (const char **)args) < 0)
                    fprintf(stderr, "Error running script %s\n", args[0]);
                free(args);
            }
        }
    }
    return pid;
}

/*
 * Try to create a socket, allowing server to listen commands.
 *
 * Returns a new socket number on success or -1 on failure.
 */
static int create_socket(void)
{
    int d_sock;
    socklen_t addrlen;
    struct sockaddr_un d_sockad = {
            AF_LOCAL, PL_SOCK_PATH
    };

    d_sock = socket(PF_LOCAL, SOCK_STREAM , 0);
    if (d_sock < 0) {
        perror("Create socket");
        return -1;
    }

    addrlen = SUN_LEN(&d_sockad);

    unlink(PL_SOCK_PATH);
    if (bind(d_sock, (struct sockaddr *)&d_sockad, addrlen) < 0) {
        perror("Bind socket");
        close(d_sock);
        d_sock = -1;
    }

    return d_sock;
}

int main(int argc, char *argv[])
{
    int d_sock, option, fd = -1;
    pid_t pid = -1;

    DEBUG ();

    if (daemon(0, 1) < 0) {
        perror("Daemon");
        return -1;
    }

    option = getopt(argc, argv, "sf:");
    switch (option) {
    case 's':
        stop_daemon();
        return 0;
    case 'f':
        fd = atoi(optarg);
        break;
    case '?':
        return -1;
    default:
        break;
    }

    if (pl_get_load_opt() != PL_STARTUP && fd < 0)
        return 0;

    d_sock = create_socket();
    if (d_sock < 0) {
        if (errno == EADDRINUSE) {
            fprintf(stderr, "Another daemon already running!\n");
            return 0;
        } else {
            return 1;
        }
    }
    if (load_python(DAEMON_CONF_FILE))
        fprintf(stderr, "Error loading python config file!\n");
    else
        pid = handle_requests(d_sock, fd);

    if (fd >= 0)
        close(fd);

    Py_Finalize();
    if (pid > 0) {
        close(d_sock);
        unlink(PL_SOCK_PATH);
        fprintf(stderr, "\nPyLauncher daemon stopped\n");
    } else if (pid < 0) {
        exit(PL_STOP_DAEMON);
    }
    return 0;
}

/* vim:et:sw=4:ts=4:co=80
 */
