#!/usr/bin/env python2.5
# -*- coding: UTF8 -*-
# Copyright (C) 2008 by Daniel Martin Yerga
# <dyerga@gmail.com>

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# Pyrecipe: Recipe Manager for Internet Tablets
# Version 0.1
#

import gtk
import gtk.glade
import hildon
import gobject
import sqlite3
import pango
import os
import osso
import pyrecipe_utils

osso_c = osso.Context("pyrecipe_service", "0.1", False)

#Columns for recipe list
(ID, TITLE) = range(2)
#Columns for ingredient list
(AMOUNT, UNIT, INGREDIENT) = range(3)
   
#See if is running ins scratchbox, and get the directories path in consequence
if os.path.exists('/scratchbox'):
    scratchbox = True
else:
    scratchbox = False

if scratchbox:
    if os.path.exists('database'):
        first_time_run = False
    else:
        first_time_run = True
    imagesdir = icondir = 'pixmaps/'
    sqlite_conn = sqlite3.connect('database') 
    glade_file = "pyrecipe.glade" 
else:
    if os.path.exists('/home/user/.pyrecipe/database'):
        first_time_run = False
    else:
        first_time_run = True
    imagesdir = '/usr/share/pyrecipe/pixmaps/'
    icondir = '/usr/share/icons/hicolor/scalable/hildon/'
    sqlite_conn = sqlite3.connect('/home/user/.pyrecipe/database')
    glade_file = "/usr/share/pyrecipe/pyrecipe.glade"  

#In first run, create the database with the parameters
if first_time_run:
    cursor_sql = sqlite_conn.cursor()
    cursor_sql.execute('CREATE TABLE recipes (id INTEGER PRIMARY KEY, title text, \
    category, prep_time, cook_time, servings, ingredients text, description text, \
    notes text, imagesrc text)')
    cursor_sql.close()
    print 'creating database'

class MainWindow(hildon.Program):
    def __init__(self):
        self.program = hildon.Program()
        self.program.__init__()
        gtk.set_application_name("")

        self.window = hildon.Window()
        self.window.set_title('Pyrecipe - Recipe List')
        self.window.set_default_size(800, 480)
        self.program.add_window(self.window)
        self.window.connect("destroy", self.quit)
        self.window.connect("key-press-event", self.on_key_press)
        self.window.connect("window-state-event", self.on_window_state_change)
        self.window_in_fullscreen = False

        #Timer and shopping list isn't running        
        self.timer_running = False
        self.shoplist_running = False
        
        self.glade=gtk.glade.XML(glade_file)
        vbox1 = self.glade.get_widget("vbox1")
        vbox1.reparent(self.window)       

        ## Create menu  ##
        #get menu from glade and reparent as common menu in hildon program
        mainMenu = self.glade.get_widget("menubar1")
        menu = gtk.Menu()
        for child in mainMenu.get_children():
            child.reparent(menu)
        self.window.set_menu(menu)
        mainMenu.destroy()

        ## Set images for menu and buttons ##
        import_img = self.glade.get_widget("import_img")
        import_img.set_from_file(imagesdir + 'import.png')
        export_img = self.glade.get_widget("export_img")
        export_img.set_from_file(imagesdir + 'export.png')
        help_img = self.glade.get_widget("help_img")
        help_img.set_from_file(imagesdir + 'help.png')
        about_img = self.glade.get_widget("about_img")
        about_img.set_from_file(imagesdir + 'about.png')
        quit_img = self.glade.get_widget("quit_img")
        quit_img.set_from_file(imagesdir + 'quit.png')
        recipe_list_img = self.glade.get_widget("recipe_list_img")
        recipe_list_img.set_from_file(imagesdir + 'list.png')
        new_img = self.glade.get_widget("new_img")
        new_img.set_from_file(imagesdir + 'new.png')
        view_img = self.glade.get_widget("view_img")
        view_img.set_from_file(imagesdir + 'view.png')
        edit_img = self.glade.get_widget("edit_img")
        edit_img.set_from_file(imagesdir + 'edit.png')
        save_img = self.glade.get_widget("save_img")
        save_img.set_from_file(imagesdir + 'save.png')
        delete_img = self.glade.get_widget("delete_img")
        delete_img.set_from_file(imagesdir + 'delete.png')
        search_img = self.glade.get_widget("search_img")
        search_img.set_from_file(imagesdir + 'search.png')
        clear_search_img = self.glade.get_widget("clear_search_img")
        clear_search_img.set_from_file(imagesdir + 'clear.png')
        clear_image_img = self.glade.get_widget("clear_image_img")
        clear_image_img.set_from_file(imagesdir + 'clear.png')
        add_img = self.glade.get_widget("add_img")
        add_img.set_from_file(imagesdir + 'add.png')
        add_ing_toggle_img = self.glade.get_widget("add_ing_toggle_img")
        add_ing_toggle_img.set_from_file(imagesdir + 'add.png')        
        clear_ing_img = self.glade.get_widget("clear_ing_img")
        clear_ing_img.set_from_file(imagesdir + 'clear.png')
        remove_ing_img = self.glade.get_widget("remove_ing_img")
        remove_ing_img.set_from_file(imagesdir + 'remove.png')
        clear_ing_list_img = self.glade.get_widget("clear_ing_list_img")
        clear_ing_list_img.set_from_file(imagesdir + 'clear.png')
        menu_shop_list_img = self.glade.get_widget("menu_shop_list_img")
        menu_shop_list_img.set_from_file(imagesdir + 'shop.png')
        timer_menu_img = self.glade.get_widget("timer_menu_img")
        timer_menu_img.set_from_file(imagesdir + 'timer.png')
        backup_menu_img = self.glade.get_widget("backup_menu_img")
        backup_menu_img.set_from_file(imagesdir + 'backup.png')        

        main_notebook = self.glade.get_widget("notebook1")
        main_notebook.connect("switch_page", self.on_main_notebook_changed)
        main_notebook.set_show_tabs(False)
        recipe_image = self.glade.get_widget("recipe_image")

        view_recipe_sw = self.glade.get_widget("view_recipe_swin")
        recipe_list_sw = self.glade.get_widget("recipe_list_sw")
        hildon.hildon_helper_set_thumb_scrollbar(view_recipe_sw, True)
        hildon.hildon_helper_set_thumb_scrollbar(recipe_list_sw, True) 
        
        # create tree model
        recipe_list = pyrecipe_utils.get_recipe_list(sqlite_conn)
        recipe_list_model = self.create_recipe_list_model(recipe_list)

        #Define the recipe list treeview with multiple selections, search, etc.
        recipe_list_treeview = self.glade.get_widget("recipe_list_treeview")
        recipe_list_treeview.set_model(recipe_list_model)
        recipe_list_treeview.set_rules_hint(True)
        recipe_list_treeview.set_headers_visible(True)
        recipe_list_treeview.set_search_column(TITLE)
        recipe_list_treeview.set_enable_search(True)
        recipe_list_treeview.set_rubber_banding(True)
        selection = recipe_list_treeview.get_selection()
        selection.set_mode(gtk.SELECTION_MULTIPLE)        
        search_entry = self.glade.get_widget("search_entry")
        recipe_list_treeview.set_search_entry(entry=search_entry)
        
		# add columns to the tree view
        self.add_columns_to_recipe_list(recipe_list_treeview)

		# ingredients treeview
        ingredient_list_model = self.create_ingredient_list_model()

        ingredient_list_treeview = self.glade.get_widget("ingredient_list_treeview")
        ingredient_list_treeview.set_model(ingredient_list_model)
        ingredient_list_treeview.set_rules_hint(True)
        ingredient_list_treeview.set_headers_visible(True)

		# add columns to the tree view
        self.add_columns_to_ingredient_list(ingredient_list_treeview)

        self.recipe_list_button = self.glade.get_widget("recipe_list_button")
        self.recipe_list_button.connect("clicked", self.show_recipe_list, \
                                        main_notebook)       
        
        self.new_recipe_button = self.glade.get_widget("new_recipe_button")
        self.new_recipe_button.connect("clicked", self.new_recipe, main_notebook,\
                                       recipe_image, ingredient_list_model)

        self.view_recipe_button = self.glade.get_widget("view_recipe_button")
        self.view_recipe_button.connect("clicked", self.view_recipe, main_notebook,\
                                        recipe_list_treeview, recipe_list_model)

        self.edit_recipe_button = self.glade.get_widget("edit_recipe_button")
        self.edit_recipe_button.connect("clicked", self.edit_recipe, main_notebook,\
                            recipe_list_treeview, recipe_list_model, recipe_image,\
                            ingredient_list_model)
               
        self.save_recipe_button = self.glade.get_widget("save_recipe_button")
        self.save_recipe_button.connect("clicked", self.save_recipe, \
                                        recipe_list_model, recipe_list_treeview,\
                                        ingredient_list_model)
                 
        self.delete_recipe_button = self.glade.get_widget("delete_recipe_button")
        self.delete_recipe_button.connect("clicked", self.delete_recipe, \
                                          recipe_list_treeview, main_notebook)
                                                                             
        self.search_button = self.glade.get_widget("search_button")
        self.search_button.connect("toggled", self.show_search_toolbar)
        self.search_toolbar = self.glade.get_widget("search_toolbar")
        clear_search_button = self.glade.get_widget("clear_search_button")
        clear_search_button.connect("clicked", self.clear_search, search_entry)
        
        self.recipe_textview = self.glade.get_widget("recipe_textview")
        self.recipe_view_buffer = self.recipe_textview.get_buffer()
        self.fontsize = 15
        self.recipe_textview.modify_font(pango.FontDescription("Monospace "\
                                                          + str(self.fontsize)))
 
        edit_recipe_notebook = self.glade.get_widget("edit_recipe_notebook")
        edit_recipe_notebook.set_show_tabs(False)

        instructions_sw = self.glade.get_widget("instructions_sw")
        notes_sw = self.glade.get_widget("notes_sw")
        ingredients_sw = self.glade.get_widget("ingredients_sw")
        hildon.hildon_helper_set_thumb_scrollbar(instructions_sw, True)
        hildon.hildon_helper_set_thumb_scrollbar(notes_sw, True)        
        hildon.hildon_helper_set_thumb_scrollbar(ingredients_sw, True) 
                
        set_description_button = self.glade.get_widget("set_description_button")
        set_description_button.connect("clicked", self.on_show_description, \
                                    edit_recipe_notebook) 

        set_image_button = self.glade.get_widget("set_image_button")
        set_image_button.connect("clicked", self.on_show_set_image, \
                                    edit_recipe_notebook)        
               
        set_ingredients_button = self.glade.get_widget("set_ingredients_button")
        set_ingredients_button.connect("clicked", self.on_show_ingredients, \
                                    edit_recipe_notebook) 
        
        set_instructions_button = self.glade.get_widget("set_instructions_button")
        set_instructions_button.connect("clicked", self.on_show_instructions, \
                                        edit_recipe_notebook) 
        
        set_notes_button = self.glade.get_widget("set_notes_button")
        set_notes_button.connect("clicked", self.on_show_notes, \
                                edit_recipe_notebook)

        self.title_entry = self.glade.get_widget("title_entry")        
        self.category_entry = self.glade.get_widget("category_entry") 
               
        ##Finger friendly widgets number editor##        
        hbox4 =  self.glade.get_widget("hbox4")        
        gobject.type_register(pyrecipe_utils.FingerNE)
        
        self.servingsNE = pyrecipe_utils.FingerNE('horizontal')        
        self.preptimeNE = pyrecipe_utils.FingerNE('horizontal')
        self.preptimeNE.set_tap_and_hold()
        self.cooktimeNE = pyrecipe_utils.FingerNE('horizontal')
        self.cooktimeNE.set_tap_and_hold()

        NE_list = [self.cooktimeNE, self.preptimeNE, self.servingsNE]
        for i in range(len(NE_list)):
            NE_list[i].set_minimum(0)
            NE_list[i].set_size([60, 50, 60, 60])
            NE_list[i].set_image_buttons(imagesdir + 'add.png', imagesdir + \
                                        'remove.png')

        hbox4.pack_start(self.servingsNE, False, False, 0)
        hbox4.reorder_child(self.servingsNE, 0)

        hbox4.pack_start(self.preptimeNE, False, False, 0)
        hbox4.reorder_child(self.preptimeNE, 1)
              
        hbox4.pack_start(self.cooktimeNE, False, False, 0)
        hbox4.reorder_child(self.cooktimeNE, 2)	         
                
        
        image_button = self.glade.get_widget("image_button")
        image_button.connect("clicked", self.select_image, recipe_image)
        
        clear_img_button = self.glade.get_widget("clear_img_button")
        clear_img_button.connect("clicked", self.clear_image, recipe_image) 
                      
        instructions_tv = self.glade.get_widget("instructions_textview")
        self.instructions_buffer = instructions_tv.get_buffer()
        
        notes_tv = self.glade.get_widget("notes_textview")
        self.notes_buffer = notes_tv.get_buffer()

        self.ingredient_entry = self.glade.get_widget("ingredient_entry")
        self.amount_entry = self.glade.get_widget("amount_entry")
        self.unit_entry = self.glade.get_widget("unit_entry")

        add_ing_toolbar = self.glade.get_widget("add_ing_toolbar")
        add_ing_toggle_btn = self.glade.get_widget("add_ing_toggle_btn")
        add_ing_toggle_btn.connect("toggled", self.show_add_toolbar, add_ing_toolbar) 

        add_ing_button = self.glade.get_widget("add_ing_button")
        add_ing_button.connect("clicked", self.on_add_ingredient, \
                                ingredient_list_model)

        clear_ing_button = self.glade.get_widget("clear_ing_button")
        clear_ing_button.connect("clicked", self.on_clear_ingredient)
                                
        delete_ing_button = self.glade.get_widget("delete_ing_button")
        delete_ing_button.connect("clicked", self.on_delete_ingredient, \
                                  ingredient_list_treeview)
        clear_ing_list_button = self.glade.get_widget("clear_ing_list_button")
        clear_ing_list_button.connect("clicked", self.on_clear_ingredient_list, \
                                     ingredient_list_model)       
        
        ## Connect menu signals ##
        import_menu = self.glade.get_widget("import_menu")
        import_menu.connect("activate", self.on_import, recipe_list_model)
        export_menu = self.glade.get_widget("export_menu")
        export_menu.connect("activate", self.show_export_dialog, \
                            recipe_list_treeview, recipe_list_model)
        shop_menu = self.glade.get_widget("shop_menu")
        shop_menu.connect("activate", self.on_show_shopping_list)
        timer_menu = self.glade.get_widget("timer_menu")    
        timer_menu.connect("activate", self.on_show_timer)
        backup_menu = self.glade.get_widget("backup_menu")    
        backup_menu.connect("activate", self.on_show_backup, recipe_list_model)                     
        help_menu = self.glade.get_widget("help_menu")
        help_menu.hide()
        #help_menu.connect("activate", self.on_show_help)
        about_menu = self.glade.get_widget("about_menu")
        about_menu.connect("activate", self.on_show_about)
        quit_menu = self.glade.get_widget("quit_menu")
        quit_menu.connect("activate", self.quit)

        self.window.show_all()

    #Depending on the window shown, hide/show buttons and set titles
    def on_main_notebook_changed(self, widget, dummy, pagenum):              
        if pagenum == 0:
            self.recipe_list_button.hide()
            self.new_recipe_button.show() 
            self.view_recipe_button.show()             
            self.edit_recipe_button.show()
            self.save_recipe_button.hide()
            self.delete_recipe_button.show()            
            self.search_button.show()
            self.window.set_title('Pyrecipe - Recipe List')

        elif pagenum == 1:
            self.recipe_list_button.show()
            self.new_recipe_button.show() 
            self.view_recipe_button.hide()             
            self.edit_recipe_button.show()
            self.save_recipe_button.hide()
            self.delete_recipe_button.show()            
            self.search_button.hide()
            self.window.set_title('Pyrecipe - View Recipe')

        elif pagenum == 2:
            self.recipe_list_button.show()
            self.new_recipe_button.show() 
            self.view_recipe_button.hide()             
            self.edit_recipe_button.hide()
            self.save_recipe_button.show()
            self.delete_recipe_button.hide()            
            self.search_button.hide()
            self.window.set_title('Pyrecipe - New or Edit Recipe')

    def show_recipe_list(self, widget, notebook):
         notebook.set_current_page(0)  

    #Pressing in the new recipe button
    def new_recipe(self, widget, notebook, recipe_image, ingredient_list_model):  
        self.mode = 'new'
        if notebook.get_current_page() == 2:
            self.clear_recipe(widget, recipe_image, ingredient_list_model)
        else:
            self.clear_recipe(widget, recipe_image, ingredient_list_model)
            notebook.set_current_page(2)        

    #Clear all widgets in the recipe editor        
    def clear_recipe(self, widget, recipe_image, ingredient_list_model):        
        self.title_entry.set_text('')
        self.category_entry.set_text('')
        self.preptimeNE.set_minimum(0)       
        self.cooktimeNE.set_minimum(0)       
        self.servingsNE.set_minimum(0)
        
        self.image_filename = ''
        missing_img = gtk.gdk.pixbuf_new_from_file(imagesdir + 'image.png')
        recipe_image.set_from_pixbuf(missing_img) 
        self.instructions_buffer.set_text('')
        self.notes_buffer.set_text('')
           
        self.on_clear_ingredient(widget)       
        ingredient_list_model.clear()        
        
        hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                             'New Recipe')            

    #View the selected recipe        
    def view_recipe(self, widget, notebook, recipe_list_treeview, recipe_list_model):
        selection = recipe_list_treeview.get_selection()
        
        if selection.count_selected_rows() > 1:
            hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                             'Select only one recipe') 
            return
        elif selection.count_selected_rows() == 0:
            hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                             'No selected recipe') 
            return

        select_model, selected = selection.get_selected_rows()
        
        iters = [select_model.get_iter(path) for path in selected]

        select_iter = iters[0]

        if select_iter:
            recipe_id = recipe_list_model.get_value(select_iter, ID)
            
        lista = []
        for row in sqlite_conn.execute('select * from recipes where id=%s' %\
                                      int(recipe_id)):
            lista.append(row)
        
        #print lista
        #for i in range(len(lista)):
        #    if lista[i][0] == int(recipe_id):
        #        id_index = i
        
        self.recipe_view_buffer.set_text("")
        try:
            self.recipe_view_buffer.create_tag("monospace", family="monospace")
            self.recipe_view_buffer.create_tag("big", scale=pango.SCALE_LARGE)
            self.recipe_view_buffer.create_tag("bold", weight=pango.WEIGHT_BOLD )                
            self.recipe_view_buffer.create_tag("center", justification=gtk.JUSTIFY_CENTER)
            self.recipe_view_buffer.create_tag("underline", underline=pango.UNDERLINE_SINGLE)
        except:
            print 'tags created yet.'
        textiter = self.recipe_view_buffer.get_iter_at_offset(0)		
        self.recipe_view_buffer.insert_with_tags_by_name(textiter, lista[0][1]+'\n\n', \
                                              "bold", "center", "big", "monospace")

        if lista[0][2] == '':
            pass
        else:                                              
            self.recipe_view_buffer.insert_with_tags_by_name(textiter, 'Category: ' \
                                            + lista[0][2] + '\n', "monospace")

        if lista[0][3] == '0 minutes':
            pass
        else:
            self.recipe_view_buffer.insert_with_tags_by_name(textiter, 'Preparation time: '\
                                               + lista[0][3] +'\n', "monospace")
        if lista[0][4] == '0 minutes':
            pass
        else:
            self.recipe_view_buffer.insert_with_tags_by_name(textiter, 'Cooking time: '\
                                            + lista[0][4] + '\n', "monospace")
        if lista[0][5] == '0':
            pass
        else:
            self.recipe_view_buffer.insert_with_tags_by_name(textiter, 'Servings: '\
                                            + lista[0][5] + '\n\n', "monospace")

        try:
            pixbuf = gtk.gdk.pixbuf_new_from_file(lista[0][9])
            w, h = self.set_thumbnail_size(pixbuf, 200)
            scaled_pix = pixbuf.scale_simple(int(w), int(h), gtk.gdk.INTERP_BILINEAR)
            del pixbuf        
            self.recipe_view_buffer.insert_pixbuf(textiter, scaled_pix)
            self.recipe_view_buffer.insert_with_tags_by_name(textiter, \
                        '\n\nINGREDIENTS\n\n', "bold", "underline", "monospace")            
        except:
            self.recipe_view_buffer.insert_with_tags_by_name(textiter, \
                            'INGREDIENTS\n\n', "bold", "underline", "monospace")        
            print 'No recipe image.'
            
        if lista[0][6] != '':
            ##tratamiento a ingredientes:
            il_l = lista[0][6].split('\n')
            #print 'split_lista: ', il_l
            new_list = []
            for i in range(len(il_l)):
                new = il_l[i].split('||')
                #print 'new: ', new
                if new == ['']:
                    pass
                else:
                    if new[0] == new[1] == '':
                        new_list.append(new[0]+ '' + new[1] + '' + new[2])              
                    elif new[0] == '':
                        new_list.append(new[0]+ '' + new[1] + ' ' + new[2])            
                    elif new[1] == '':
                        new_list.append(new[0]+ ' ' + new[1] + '' + new[2])
                    else:           
                        new_list.append(new[0]+ ' ' + new[1] + ' ' + new[2])

            lista_ing = ''
            for  i in range(len(new_list)):
                lista_ing += new_list[i]
                if i < len(new_list)-1:
                    lista_ing += '\n'
        else:
            lista_ing = ''
                            
        self.recipe_view_buffer.insert_with_tags_by_name(textiter, lista_ing+'\n\n', \
                                                    "monospace")		
        self.recipe_view_buffer.insert_with_tags_by_name(textiter, 'INSTRUCTIONS\n\n', \
                                                    "bold", "underline", "monospace")
        self.recipe_view_buffer.insert_with_tags_by_name(textiter, lista[0][7]+'\n\n', \
                                                    "monospace")
        if lista[0][8] == '':
            pass
        else:
            self.recipe_view_buffer.insert_with_tags_by_name(textiter, 'NOTES\n\n', \
                                                    "bold", "underline", "monospace")
            self.recipe_view_buffer.insert_with_tags_by_name(textiter, lista[0][8], \
                                                    "monospace")
                             
        notebook.set_current_page(1)

    #Edit the selected recipe        
    def edit_recipe(self, widget, notebook, recipe_list_treeview, recipe_list_model,\
                    recipe_image, ingredient_list_model):

        self.mode = 'edit'
        selection = recipe_list_treeview.get_selection()
                
        if selection.count_selected_rows() > 1:
            hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                             'Select only one recipe') 
            return
        elif selection.count_selected_rows() == 0:
            hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                             'No selected recipe') 
            return

        select_model, selected = selection.get_selected_rows()        
        iters = [select_model.get_iter(path) for path in selected]
        select_iter = iters[0]

        #get id        
        if select_iter:
            self.edit_recipe_id = recipe_list_model.get_value(select_iter, ID)
            
        lista = []
        for row in sqlite_conn.execute('select * from recipes where id=%s' %\
                                      int(self.edit_recipe_id)):
            lista.append(row)        
     
        self.title_entry.set_text(lista[0][1])
        self.category_entry.set_text(lista[0][2])
        self.preptimeNE.set_value(lista[0][3].replace(' minutes', ''))
        self.cooktimeNE.set_value(lista[0][4].replace(' minutes', ''))      
        self.servingsNE.set_value(lista[0][5])

        try:
            self.image_filename = lista[0][9]
            new_pixbuf = gtk.gdk.pixbuf_new_from_file(self.image_filename)
            w, h = self.set_thumbnail_size(new_pixbuf, 200)
            scale_pixbuf = new_pixbuf.scale_simple(int(w), int(h), \
                                               gtk.gdk.INTERP_BILINEAR)
            del new_pixbuf
            recipe_image.set_from_pixbuf(scale_pixbuf)
            del scale_pixbuf
            recipe_image.show() 
        except:
            missing_img = gtk.gdk.pixbuf_new_from_file(imagesdir + 'image.png')
            recipe_image.set_from_pixbuf(missing_img)          
                       
        self.instructions_buffer.set_text(lista[0][7])
        self.notes_buffer.set_text(lista[0][8])
        
        ##tratamiento a ingredientes:
        if lista[0][6] != '':
            il_l = lista[0][6].split('\n')
            #print lista[0][2]
            #new[0]-->amount
            #new[1]-->unit
            #new[2]-->ingredient
            ingredient_list_model.clear()
            for i in range(len(il_l)):
                new = il_l[i].split('||')
                if new == ['']:
                    pass
                else:
                    new_item = (new[0], new[1], new[2])
                    newiter = ingredient_list_model.append()           
                    ingredient_list_model.set(newiter, AMOUNT, new_item[AMOUNT], \
                        UNIT, new_item[UNIT], INGREDIENT, new_item[INGREDIENT])      

        notebook.set_current_page(2)

    #Save the edited recipe        
    def save_recipe(self, widget, recipe_list_model, recipe_list_treeview, \
                    ingredient_list_model):       
        title_txt = self.title_entry.get_text()
        
        if title_txt == '':
            hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                                  'Recipes must have title')
            return
                
        cat_txt = self.category_entry.get_text()
        prep_time_txt = self.preptimeNE.get_value() + ' minutes'
        cook_time_txt = self.cooktimeNE.get_value() + ' minutes'
        serv_txt = self.servingsNE.get_value()        
       
        start, end = self.instructions_buffer.get_bounds()
        instructions_txt = self.instructions_buffer.get_text(start, end)
        
        start, end = self.notes_buffer.get_bounds()    
        notes_txt = self.notes_buffer.get_text(start, end)        
        
        ##get ingredients from the treeview
        ingredient_list = []
        final_ing_list = ''
        
        for row in ingredient_list_model:
            amount = row[0]
            unit = row[1]
            ingredient = row[2]            
            new_ing = amount + '||' + unit + '||' + ingredient
            ingredient_list.append(new_ing)        

        for i in range(len(ingredient_list)):
            final_ing_list += ingredient_list[i]
            if i < len(ingredient_list)-1:
                final_ing_list += '\n'

        ##dump it to sqlite database
        #if it's a new recipe
        if self.mode == 'new':
            sqlite_cursor = sqlite_conn.cursor()
            sqlite_cursor.execute('INSERT INTO recipes VALUES (null, ?, ?, ?, ?,\
            ?, ?, ?, ?, ?)', (title_txt, cat_txt, prep_time_txt, cook_time_txt, \
            serv_txt, final_ing_list, instructions_txt, notes_txt, \
            self.image_filename))
                
            sqlite_conn.commit()
           
            new_recipe_id = sqlite_cursor.lastrowid
            sqlite_cursor.close()
        
            new_item = [new_recipe_id, title_txt]

            newiter = recipe_list_model.append()
            recipe_list_model.set(newiter,
                ID, new_item[ID],
                TITLE, new_item[TITLE])
            hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                             'New recipe saved')
        #if it's an edited recipe
        elif self.mode == 'edit':
            print 'edit mode'
            recipe_id = self.edit_recipe_id
            sqlite_cursor = sqlite_conn.cursor()
            sqlite_cursor.execute('REPLACE INTO recipes VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)',\
            (recipe_id, title_txt, cat_txt, prep_time_txt, cook_time_txt, serv_txt,\
            final_ing_list,instructions_txt, notes_txt, self.image_filename))
                
            sqlite_conn.commit()
            sqlite_cursor.close()
        
            new_item = [recipe_id, title_txt]

            ##update item in recipe list treeview        
            selection = recipe_list_treeview.get_selection()
            select_model, selected = selection.get_selected_rows()
            
            iters = [select_model.get_iter(path) for path in selected]
            select_iter = iters[0]            
            
            if select_iter:            
                recipe_id = recipe_list_model.get_value(select_iter, ID)           
                title = recipe_list_model.get_value(select_iter, TITLE)          
                recipe_list_model.set(select_iter, ID, new_item[ID], TITLE, \
                                      new_item[TITLE])           
            else:
                hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                             'Error updating the recipe list')           
                 
            hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                             'Edited recipe saved')
        self.image_filename = ''       

    #Delete the selected recipes           
    def delete_recipe(self, widget, recipe_list_treeview, notebook):
        selection = recipe_list_treeview.get_selection()
        if selection.count_selected_rows() == 0:
            hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                             'No selected recipe') 
            return

        select_model, selected = selection.get_selected_rows()
        iters = [select_model.get_iter(path) for path in selected]   
        
        #show confirmation dialog
        confirmation = pyrecipe_utils.on_confirmation(self.window, \
                                    'Are you sure?\nIt will delete the recipes') 
        if confirmation == True:
        
            for i in iters:
                recipe_id = select_model.get_value(i, ID)                
                select_model.remove(i)      
        
                #update sqlite database
                sqlite_conn.execute("delete from recipes where id=%s" % int(recipe_id))        
                sqlite_conn.commit()
                
            if notebook.get_current_page() == 1:
                notebook.set_current_page(0)
                       
            hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint', \
                                                  'Recipes deleted')  
        else:
            return       
        
    def show_search_toolbar(self, widget):
        if widget.get_active() == True:
           self.search_toolbar.show()
        else:
           self.search_toolbar.hide()            

    def on_show_description(self, widget, notebook):
        notebook.set_current_page(0)

    def on_show_ingredients(self, widget, notebook):
        notebook.set_current_page(1)

    def on_show_instructions(self, widget, notebook):
        notebook.set_current_page(2)

    def on_show_notes(self, widget, notebook):
        notebook.set_current_page(3)

    def on_show_set_image (self, widget, notebook):
        notebook.set_current_page(4) 

    def show_add_toolbar(self, widget, add_ing_toolbar):
        if widget.get_active() == True:
           add_ing_toolbar.show()
        else:
           add_ing_toolbar.hide()  
              
	#Functions for fullscreen
    def on_window_state_change(self, widget, event, *args):           
        if event.new_window_state & gtk.gdk.WINDOW_STATE_FULLSCREEN:
            self.window_in_fullscreen = True
        else:
            self.window_in_fullscreen = False

    #F6 fullscreen, F7 bigger font, F8 smaller font
    def on_key_press(self, widget, event, *args):            
        if event.keyval == gtk.keysyms.F6:
            if self.window_in_fullscreen:
                self.window.unfullscreen ()
            else:
                self.window.fullscreen ()
        elif event.keyval == gtk.keysyms.F7:
            if self.fontsize >= 36:
                hildon.hildon_banner_show_information(widget, \
                                    'qgn_note_infoprint', 'Maximun font reached')
                return

            self.fontsize += 2
            self.recipe_textview.modify_font(pango.FontDescription("Monospace "\
                                                          + str(self.fontsize)))
	
        elif event.keyval == gtk.keysyms.F8:
            if self.fontsize <= 11:
                hildon.hildon_banner_show_information(widget, \
                                    'qgn_note_infoprint', 'Minimal font reached')
                return
            self.fontsize -= 2
            self.recipe_textview.modify_font(pango.FontDescription("Monospace "\
                                                          + str(self.fontsize)))
                
    def create_recipe_list_model(self, recipe_list):
        lstore = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_STRING)

        for item in recipe_list:
            lstore_iter = lstore.append()        
            lstore.set(lstore_iter, ID, item[ID], TITLE, item[TITLE])
        return lstore    
    
    def add_columns_to_recipe_list(self, treeview):
        model = treeview.get_model()

        # column for ID        
        column = gtk.TreeViewColumn('ID', gtk.CellRendererText(), text=ID)
        column.set_visible(False)      
        treeview.append_column(column)

        # column for title
        renderer = gtk.CellRendererText()
        renderer.set_property('scale', 1.3)
        renderer.set_property('wrap-mode', gtk.WRAP_WORD)
        renderer.set_property('wrap-width', 500)        
        column = gtk.TreeViewColumn('Recipe List', renderer, text=TITLE)
        column.set_sort_column_id(TITLE)
        column.set_property("expand", True) 
        treeview.append_column(column)

    def create_ingredient_list_model(self):
        lstore = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_STRING, \
                               gobject.TYPE_STRING)
        return lstore    
    
    def add_columns_to_ingredient_list(self, treeview):
        model = treeview.get_model()

        column = gtk.TreeViewColumn('Amount', gtk.CellRendererText(), \
                                    text=AMOUNT)
        treeview.append_column(column)
        
        column = gtk.TreeViewColumn('Unit', gtk.CellRendererText(), text=UNIT)       
        treeview.append_column(column)        

        column = gtk.TreeViewColumn('Ingredient', gtk.CellRendererText(), \
                                    text=INGREDIENT)
        column.set_resizable(True)               
        treeview.append_column(column)  

    def on_add_ingredient(self, widget, ingredient_list_model):
        ingredient = self.ingredient_entry.get_text()
        
        if ingredient == '':
            hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                                  'Ingredients must have title')
            return            
        
        amount = self.amount_entry.get_text()
        unit = self.unit_entry.get_text()

        new_item = [amount, unit, ingredient]

        newiter = ingredient_list_model.append()
        ingredient_list_model.set(newiter, AMOUNT, new_item[AMOUNT], UNIT, \
                              new_item[UNIT], INGREDIENT, new_item[INGREDIENT])

        self.on_clear_ingredient(widget)

    def on_clear_ingredient(self, widget):
        self.ingredient_entry.set_text('')
        self.amount_entry.set_text('')
        self.unit_entry.set_text('')         
          
    def on_delete_ingredient(self, widget, ingredient_list_treeview):
        selection = ingredient_list_treeview.get_selection()
        select_model, select_iter = selection.get_selected()

        if select_iter:
            select_model.remove(select_iter)
        else:
            hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                                  'No selected ingredient')
            return         

    def on_clear_ingredient_list(self, widget, ingredient_list_model):
        ingredient_list_model.clear()

    def clear_search(self, widget, search_entry):
        search_entry.set_text('')

    #Select image in the recipe editor
    def select_image(self, widget, recipe_image):
        dialog = hildon.FileChooserDialog(self.window, gtk.FILE_CHOOSER_ACTION_OPEN)
        dialog.set_title("Browse Image")
        dialog.set_default_response(gtk.RESPONSE_OK)
        dialog.set_current_folder('/home/user/MyDocs/.images')

        result = ""
        if dialog.run() == gtk.RESPONSE_OK:
            try:
                result = dialog.get_filename()
                new_pixbuf = gtk.gdk.pixbuf_new_from_file(result)
                self.image_filename = result
                
                w, h = self.set_thumbnail_size(new_pixbuf, 200)
                scale_pixbuf = new_pixbuf.scale_simple(int(w), int(h), \
                                                      gtk.gdk.INTERP_BILINEAR)
                del new_pixbuf
                recipe_image.set_from_pixbuf(scale_pixbuf)
                del scale_pixbuf
                recipe_image.show()
            except:
                print 'No image file selected.'
                result = None
        dialog.destroy()

    def clear_image(self, widget, recipe_image):    
        recipe_image.set_from_file(imagesdir + 'image.png')

    #Calculate the thumbnail size depending of its actual size        
    def set_thumbnail_size(self, pixbuf, size):
        pix_width = pixbuf.get_width()
        pix_height = pixbuf.get_height()
		
        if pix_width > pix_height:
            w = size
            h = float(pix_height)/float(pix_width) * size
        elif pix_height > pix_width:
            h = size
            w = float(pix_width)/float(pix_height) * size
        elif pix_height == pix_width:
            h = size
            w = size
        else:
            print "Pixbuf size can't be calculated."

        return w, h

    #Get all parameters from a selected recipe
    def get_recipe_from_selection(self, widget, recipe_list_treeview, \
                                    recipe_list_model):
        selection = recipe_list_treeview.get_selection()
        if selection.count_selected_rows() == 0:
            hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                             'No selected recipe') 
            return

        select_model, selected = selection.get_selected_rows()
        iters = [select_model.get_iter(path) for path in selected]
        
        lista = []        
        for select_iter in iters:
            recipe_id = recipe_list_model.get_value(select_iter, ID)
            for row in sqlite_conn.execute('select * from recipes where id=%s' %\
                                        int(recipe_id)):
                lista.append(row)   

        return lista

    #Importing recipes from some compatible format
    def on_import(self, widget, recipe_list_model):
        filename = pyrecipe_utils.show_open_dialog(self.window, 'Import Recipe')
        
        if filename == None:
            print 'No importing recipe.'
            return None
        
        else:
            file_ext = os.path.splitext(filename)[1]
        
        krecipes_ext = ['.kreml', '.kre']
        gourmet_ext = ['.grmt']
        #mealmaster_ext = ['.mmf']
            
        if file_ext in krecipes_ext:
            print 'importing as krecipes'
            from filters import krecipe_import as krecipeI
            krec_imp = krecipeI.KRecipes()
            # recipe_info == (title, category, preptime, cooktime, servings, 
            # final_ing_list, instructions, notes)
            recipes_info = krec_imp.initiation(filename)            
        elif file_ext in gourmet_ext:
            print 'importing as gourmet'
            from filters import gourmet_xml_import as gxml_import
            gxml_imp = gxml_import.GourmetXML()
            recipes_info = gxml_imp.initiation(filename)
        #elif file_ext in mealmaster_ext:
        #    print 'importing as mealmaster'
        #    from filters import mealmaster_import as mm_import
        #    mm_importer = mm_import.MM()
        #    recipes_info = mm_importer.initiation(filename)        
        #    print recipes_info
        else:
            msg1 = "Pyrecipe doesn't recognize the format of the file"
            msg2 = 'It can import only Gourmet .grmt\nand Krecipes .kreml files'
            pyrecipe_utils.info_dialog(self.window, msg1, msg2)
            return
        
        if recipes_info == None:
            return
                    
        sqlite_cursor = sqlite_conn.cursor()
        #img_filename = None yet.
        
        for i in range(len(recipes_info[0])):
        
            sqlite_cursor.execute('INSERT INTO recipes VALUES (null, ?, ?, ?, ?,\
                ?, ?, ?, ?, ?)', (recipes_info[0][i][1], recipes_info[1][i][1], \
                recipes_info[2][i][1], recipes_info[3][i][1], recipes_info[4][i][1], \
                recipes_info[5][i][1], recipes_info[6][i][1], recipes_info[7][i][1], None))

            sqlite_conn.commit()
           
            new_recipe_id = sqlite_cursor.lastrowid
            sqlite_cursor.close()
        
            new_item = [new_recipe_id, recipes_info[0][i][1]]

            newiter = recipe_list_model.append()
            recipe_list_model.set(newiter, ID, new_item[ID], TITLE, new_item[TITLE])
        
        
        hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                             'New recipes imported')


    #Show the export dialog with its format options
    def show_export_dialog(self, widget, recipe_list_treeview, recipe_list_model):
        export_dlg = gtk.Dialog(title='Export Formats', parent=None, \
                              buttons=("Cancel", gtk.RESPONSE_CLOSE))
        export_dlg.set_size_request(350, 300)  			

        hbox1 = gtk.HBox()
        hbox1.set_homogeneous(True)
        
        html_btn = gtk.Button('HTML')
        html_btn.connect("clicked", self.on_export, widget, export_dlg, 'html', \
                        recipe_list_treeview, recipe_list_model)
        gxml_btn = gtk.Button('Gourmet XML')
        gxml_btn .connect("clicked", self.on_export, widget, export_dlg, 'gxml', \
                        recipe_list_treeview, recipe_list_model) 
                     
        hbox1.pack_start(html_btn, True, True, 0)
        hbox1.pack_start(gxml_btn, True, True, 0) 
             
        hbox2 = gtk.HBox()
        hbox2.set_homogeneous(True)
        
        krecipes_btn = gtk.Button('KRecipes')
        krecipes_btn.connect("clicked", self.on_export, widget, export_dlg, \
                            'krecipes', recipe_list_treeview, recipe_list_model)       
        mealm_btn = gtk.Button('MealMaster')
        mealm_btn.connect("clicked", self.on_export, widget, export_dlg, \
                        'mealmaster', recipe_list_treeview, recipe_list_model)

        hbox2.pack_start(krecipes_btn, True, True, 0)
        hbox2.pack_start(mealm_btn, True, True, 0) 

        export_dlg.vbox.pack_start(hbox1, True, True, 0)
        export_dlg.vbox.pack_start(hbox2, True, True, 0)       

        export_dlg.show_all()
		
        result = export_dlg.run()
        export_dlg.destroy()    

    #Export the selected recipes depending of its chosen format
    def on_export(self, button, widget, dialog, format, recipe_list_treeview, \
                  recipe_list_model):
        dialog.destroy()
                
        exp_recipe_list = self.get_recipe_from_selection(widget, \
                                        recipe_list_treeview, recipe_list_model)
        if format == 'html':
            from filters import html_export
            html_export.HTML(exp_recipe_list, self.window, widget)
        elif format == 'gxml':
            from filters import gourmet_xml_export as gxml_export
            gxml_export.GourmetXML(exp_recipe_list, self.window, widget)          
        elif format == 'krecipes':
            from filters import krecipe_export        
            krecipe_export.KRecipes(exp_recipe_list, self.window, widget)
        
        elif format == 'email':
            pass
        
        elif format == 'mealmaster':
            from filters import mealmaster_export  
            mealmaster_export.MM(exp_recipe_list, self.window, widget)     

    #Create shopping list, if it's running show it
    def on_show_shopping_list(self, widget):
        import shopping_list
        if self.shoplist_running:
            self.shoplist.show()
        else:
            self.shoplist_running = True
            self.shoplist = shopping_list.ShopGui()

    #Create the timer, if it's running show it        
    def on_show_timer(self, widget):
        import timer
        if self.timer_running:
            self.mytimer.show()
        else:
            self.timer_running = True
            self.mytimer = timer.Timer(imagesdir, osso_c)

    #Release the backup system and 
    #if it's importing a backup, add the recipes to the recipe list
    def on_show_backup(self, widget, recipe_list_model):
        import backup
        backup_type = backup.BackupGui(self.window)
        
        if backup_type.backup_type == 'import':
            #print 'titulos differents: ', backup_type.backup_titles
            #print 'ids: ', backup_type.recipe_ids            
            for i in range(len(backup_type.backup_titles)):
        
                new_item = [backup_type.recipe_ids[i], backup_type.backup_titles[i]]

                newiter = recipe_list_model.append()
                recipe_list_model.set(newiter, ID, new_item[ID], TITLE, new_item[TITLE])
        
        
            hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                             'Backup imported')
            return
    
    def on_show_help(self, widget):
        pass
        
    def on_show_about(self, widget):
        dialog = gtk.AboutDialog()
        dialog.set_name("PyRecipe")
        dialog.set_version("0.1")
        dialog.set_copyright("Copyright © 2008")
        dialog.set_website("http://pyrecipe.garage.maemo.org")
        dialog.set_authors(["Daniel Martin Yerga <dyerga@gmail.com>"])
        logo = gtk.gdk.pixbuf_new_from_file(icondir + "pyrecipe.png") 		
        dialog.set_logo(logo)
        dialog.set_license("""This program is released under the GNU
GeneralPublic License. Please visit
http://www.gnu.org/copyleft/gpl.html
for details.""")
        dialog.set_artists(["""Logo by Daniel Martin Yerga.
Icons from ALLBLACK iconset by Mandarancio (http://mandarancio.deviantart.com/)
"""])
        #dialog.set_translator_credits("""English = Daniel Martin Yerga
#Spanish = Daniel Martin Yerga""")
        dialog.run()
        dialog.destroy()
  
    def quit(self, widget):
        gtk.main_quit()
        
if __name__ == "__main__":
    app = MainWindow()
    gtk.main()
