#ifndef SOKOBAN_H
#define SOKOBAN_H

/*
  PushIt - A simple Sokoban solving game
  Copyright (C) 2006  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stack>
#include <string>
#include <vector>

class Sokoban
{
private:
  class UndoCell
  {
  public:
    size_t        pos;
    unsigned char type;
  };

  class UndoEntry
  {
  public:
    std::vector<UndoCell> cells;
  };

public:
  enum Type {
    typeWall        =  1,
    typeFloor       =  2,
    typePackage     =  4,
    typeGoal        =  8,
    typeSokoban     = 16
  };

private:
  size_t                     width;
  size_t                     height;
  std::vector<unsigned char> area;
  std::stack<UndoEntry>      undo;

private:
  void SetPackage(size_t x, size_t y);
  void ClearPackage(size_t x, size_t y);

  void SetSokoban(size_t x, size_t y);
  void ClearSokoban(size_t x, size_t y);

  void StartUndo();
  void AddCellToUndoEntry(size_t x, size_t y);

public:
  Sokoban();

  void SetAreaSize(size_t width, size_t height);

  void Set(size_t pos, unsigned char type);
  void Set(size_t x, size_t y, unsigned char type);
  unsigned char Get(size_t x, size_t y) const;

  size_t GetWidth() const;
  size_t GetHeight() const;

  void GetSokoban(size_t &x, size_t &y) const;

  bool IsIn(size_t x, size_t y) const;
  bool IsFloor(size_t x, size_t y) const;
  bool IsWall(size_t x, size_t y) const;
  bool IsPackage(size_t x, size_t y) const;
  bool IsGoal(size_t x, size_t y) const;
  bool IsSokoban(size_t x, size_t y) const;

  bool CanMoveLeft() const;
  bool CanMoveRight() const;
  bool CanMoveUp() const;
  bool CanMoveDown() const;

  void MoveLeft();
  void MoveRight();
  void MoveUp();
  void MoveDown();

  bool IsFinished() const;

  void Undo();

  // Helper
  void SetLineAsText(size_t line, const std::wstring& text);
  void Postprocess();
};

#endif
