/*
 * Copyright: (C) 2009 Bruce W. Forsberg
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 *   Bruce Forsberg  bruce.forsberg@gmail.com
 *
 */

#include "otrplayerfile.h"

#include <qfile.h>
#include <qfileinfo.h>
#include <qdir.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qtextstream.h>
#include <qdatetime.h>

#include <cstdio>
#include <cstdlib>
#include <iostream>

//TBD If we delete all entries in the Dir list then all default entries get added back in


#define FIRST_LINE_STRING_V10 "This is a data file for the otrplayer program V1.0"
#define FIRST_LINE_STRING "This is a data file for the otrplayer program V2.0"

//
// Data Layout of file
// A - indicates an entry for a file. It contains 3 lines, file name, location, date
// B - button entry
// D - directory entry
// S - advance entry
// F - font entry
// L - last played file



// Constructor
otrplayerfile::otrplayerfile()
{
   QString  dir_str;
   QDir     dir;

   convertDirectory();

   if (getenv("HOME"))
      dir_str = getenv("HOME");
   dir_str.append("/MyDocs/");
   dir.mkdir(dir_str);
   dir_str.append(".otrplayer/");
   dir.mkdir(dir_str);
   dir_str.append("otrplayer.cfg");

   _file = new QFile(dir_str);

   convertV20();
}


// Destructor
otrplayerfile::~otrplayerfile()
{
   delete _file;
}

// If otrplayer directory is present then convert it to .otrplayer
void
otrplayerfile::convertDirectory()
{
   QString  old_dir;
   QString  new_dir;
   QDir  dir;

   if (getenv("HOME"))
   {
      old_dir = getenv("HOME");
      new_dir = getenv("HOME");
   }
   old_dir.append("/MyDocs/otrplayer/");
   new_dir.append("/MyDocs/.otrplayer/");

   // IF the directory exists then we move it
   if (dir.exists(old_dir) == true)
   {
      dir.rename(old_dir, new_dir);
   }
}


void
otrplayerfile::process(
   QStringList&  lst,
   QStringList&  offsets)
{
   QString str;
   QStringList  io_files;
   QStringList  io_offset;
   QStringList  io_date;
   QStringList::Iterator   it;
   int          index;


#if 0
   std::cout << "otrplayerfile::process called" << std::endl;
#endif

   readEntries(io_files, io_offset, io_date);

   // Add new entries to list update existing entries if needed
   for (it = lst.begin(); it != lst.end(); ++it)
   {
#if 0
      std::cout << "Reading entry:" << (*it).toStdString() << std::endl;
#endif

      QFileInfo  info(*it);
      // Search for file, IF not found
      if (io_files.find(*it) == io_files.end())
      {
#if 0
         std::cout << "Not found adding." << std::endl;
#endif
         io_files.append(*it);
         io_offset.append("0");
         io_date.append(info.lastModified().toString());
         // Update offset for returned list
         offsets.append("0");
      }
      // ELSE if found check date
      else
      {
#if 0
         std::cout << "Found checking date." << std::endl;
#endif
         index = io_files.findIndex(*it);

         // IF date is different then update entry and zero offset
         if (io_date.at(index) != info.lastModified().toString())
         {
            io_files.removeAt(index);
            io_offset.removeAt(index);
            io_date.removeAt(index);

            io_files.append(*it);
            io_offset.append("0");
            io_date.append(info.lastModified().toString());

            // Update offset for returned list
            offsets.append("0");
         }
         else
         {
            // Update offset for returned list
            offsets.append(io_offset.at(index));
         }
      }
   }

   // Remove old entries no longer needed
   // TBD what if we added new entries above will the end iterator be OK?
   it = io_files.begin();
   while (it != io_files.end())
   {
#if 0
      std::cout << "Looking for entry:" << (*it).toStdString() << std::endl;
#endif
      // IF we can't find this entry 
      if (lst.find(*it) == lst.end())
      {
         index = io_files.findIndex(*it);
#if 0
         std::cout << "We are going to delete entry:" << (*it).toStdString() << " at index:" << index << std::endl;
         std::cout << "Files size is: " << io_files.size() << std::endl;
         std::cout << "Offset size is: " << io_offset.size() << std::endl;
         std::cout << "Date size is: " << io_date.size() << std::endl;
#endif
         io_files.removeAt(index);
         io_offset.removeAt(index);
         io_date.removeAt(index);
         // Since we just destroyed the iterator start over
         it = io_files.begin();
      }
      else
         it++;
   }

   writeEntries(io_files, io_offset, io_date);
}


long
otrplayerfile::getOffset(QString& file)
{
   QString str;
   long offset = 0;


   if (_file->open(IO_ReadOnly))
   {
      // Read all entries from file
      QTextStream t(_file);
      str = t.readLine();
      if (str == FIRST_LINE_STRING)
      {
         while (!t.atEnd())
         {
            str = t.readLine();
            if (str == "A")
            {
               str = t.readLine();
               if (str == file)
               {
                  str = t.readLine();
                  offset = str.toLong();
                  break;
               }
               else
               {
                  str = t.readLine();
                  str = t.readLine();
               }
            }
         }
      }
      _file->close();
   }
   return (offset);
}


void
otrplayerfile::storeOffset(
   QString& file,
   long     offset)
{
   QStringList  io_files;
   QStringList  io_offset;
   QStringList  io_date;
   QStringList::Iterator  it;
   QString      str;
   int          index;


   readEntries(io_files, io_offset, io_date);

#if 0
   std::cout << "Before update entry" << std::endl;
   std::cout << "Number of entries in io_files is: " << io_files.size() << std::endl;
   std::cout << "Number of entries in io_offset is: " << io_offset.size() << std::endl;
   std::cout << "Number of entries in io_date is: " << io_date.size() << std::endl;
#endif

   it = io_files.find(file);
   if (it != io_files.end())
   {
      QFileInfo  info(file);
      str.setNum(offset);

      // Remove old entry
      index = io_files.findIndex(file);
      io_files.removeAt(index);
      io_offset.removeAt(index);
      io_date.removeAt(index);

#if 0
   std::cout << "After remove entry" << std::endl;
   std::cout << "Number of entries in io_files is: " << io_files.size() << std::endl;
   std::cout << "Number of entries in io_offset is: " << io_offset.size() << std::endl;
   std::cout << "Number of entries in io_date is: " << io_date.size() << std::endl;
#endif

      // Add new entry with updated offset
      io_files.append(file);
      io_offset.append(str);
      io_date.append(info.lastModified().toString());
   }
#if 0
   std::cout << "After update entry" << std::endl;
   std::cout << "Number of entries in io_files is: " << io_files.size() << std::endl;
   std::cout << "Number of entries in io_offset is: " << io_offset.size() << std::endl;
   std::cout << "Number of entries in io_date is: " << io_date.size() << std::endl;
#endif

   writeEntries(io_files, io_offset, io_date);
}


void
otrplayerfile::storePlayedFile(
   QString& file)
{
   QStringList  lst;


   // Open file to read
   if (_file->open(IO_ReadOnly))
   {
      QTextStream t(_file);

      // Read first line and store
      lst.append(t.readLine());
      // Read all entries
      while (!t.atEnd())
      {
         QString  l_str = t.readLine();
         QChar chr = l_str.at(0);
         if (chr == 'L')
         {
         }
         else if (chr == 'A')
         {
               lst.append(l_str);
               l_str = t.readLine();
               lst.append(l_str);
               l_str = t.readLine();
               lst.append(l_str);
               l_str = t.readLine();
               lst.append(l_str);
         }
         else
            lst.append(l_str);
      }
      _file->close();
      _file->remove();

      // Dump the entries to the original file
      if (_file->open(IO_WriteOnly))
      {
         QTextStream tt(_file);
         for ( QStringList::Iterator it = lst.begin(); it != lst.end(); ++it )
            tt << (*it) << endl;

         // Write new played file value
         tt << "L " << file << endl;

         _file->close();
      }
   }
   // ELSE no file then create new one
   else
   {
      // Dump the entries to the original file
      if (_file->open(IO_WriteOnly))
      {
         QTextStream tt(_file);

         tt << FIRST_LINE_STRING << endl;

         // Write new played file value
         tt << "L " << file << endl;

         _file->close();
      }
   }
}


void
otrplayerfile::getPlayedFile(
   QString& file)
{
   QString str, tag;


   if (_file->open(IO_ReadOnly))
   {
      // Read all entries from file
      QTextStream t(_file);
      str = t.readLine();
      if (str == FIRST_LINE_STRING)
      {
         while (!t.atEnd())
         {
            QString  str1 = t.readLine();
            QChar chr = str1.at(0);

            if (chr == 'L')
            {
               QTextStream s(&str1, IO_ReadOnly);
               s >> tag >> file;
            }
            else if (chr == 'A')
            {
               // Skip all 4 lines if a file entry
               str = t.readLine();
               str = t.readLine();
               str = t.readLine();
            }
         }
      }
      _file->close();
   }
}


void
otrplayerfile::readEntries(
   QStringList& io_files,
   QStringList& io_offset,
   QStringList& io_date)
{
   QString str;


   if (_file->open(IO_ReadOnly))
   {
      // Read all entries from file
      QTextStream t(_file);
      str = t.readLine();
      if (str == FIRST_LINE_STRING)
      {
         while (!t.atEnd())
         {
            str = t.readLine();
            if (!t.atEnd())
            {
               if (str == "A")
               {
                  str = t.readLine();
                  io_files.append(str);
                  str = t.readLine();
                  io_offset.append(str);
                  str = t.readLine();
                  io_date.append(str);
               }
            }
         }
      }
      _file->close();
   }
}


void
otrplayerfile::writeEntries(
   QStringList& io_files,
   QStringList& io_offset,
   QStringList& io_date)
{
   QStringList::Iterator   it1, it2, it3;
   QStringList  lst;


   if (_file->open(IO_ReadOnly))
   {
      QTextStream t(_file);

      // Read first line and discard
      t.readLine();
      // Read all entries
      while (!t.atEnd())
      {
         QString  l_str = t.readLine();
         QChar chr = l_str.at(0);
         if (chr == 'A')
         {
               // Skip writting these
               l_str = t.readLine();
               l_str = t.readLine();
               l_str = t.readLine();
         }
         else if (chr == '/')
         {
               // Skip writting these
               l_str = t.readLine();
               l_str = t.readLine();
         }
         else
            lst.append(l_str);
      }
      _file->close();
      _file->remove();
   }

   // Dump the entries to the original file that are not file entries
   if (_file->open(IO_WriteOnly))
   {
      QTextStream tt(_file);

      tt << FIRST_LINE_STRING << endl;
      for ( QStringList::Iterator it = lst.begin(); it != lst.end(); ++it )
         tt << (*it) << endl;

      // Write out all file entries

      // Write all entries to file
      QTextStream t(_file);
      for (it1 = io_files.begin(),
           it2 = io_offset.begin(),
           it3 = io_date.begin(); it1 != io_files.end(); ++it1, ++it2, ++it3)
      {
         t << "A" << endl << (*it1) << endl << (*it2) << endl << (*it3) << endl;
      }
      _file->close();
   }
}


void
otrplayerfile::setButton(
   key_enum  key_but,
   int       value)
{
   QStringList  lst;
   QString  tag, item;
   QString  str_but;


   if (key_but == otrplayerfile::CALENDAR_KEY)
      str_but = "Calendar";
   else if (key_but == otrplayerfile::ADDRESS_KEY)
      str_but = "Address";
   else if (key_but == otrplayerfile::HOME_KEY)
      str_but = "Home";
   else if (key_but == otrplayerfile::MENU_KEY)
      str_but = "Menu";
   else if (key_but == otrplayerfile::EMAIL_KEY)
      str_but = "Email";
   else if (key_but == otrplayerfile::UP_KEY)
      str_but = "Up";
   else if (key_but == otrplayerfile::DOWN_KEY)
      str_but = "Down";
   else if (key_but == otrplayerfile::SPACE_KEY)
      str_but = "Space";

   // Open file to read
   if (_file->open(IO_ReadOnly))
   {
      QTextStream t(_file);

      // Read first line and store
      lst.append(t.readLine());
      // Read all entries
      while (!t.atEnd())
      {
         QString  l_str = t.readLine();
         QChar chr = l_str.at(0);
         if (chr == 'B')
         {
            QTextStream s(&l_str, IO_ReadOnly);
            s >> tag;
            s >> item;

            // Dump all entries that are not the correct button
            if (str_but != item)
               lst.append(l_str);
         }
         else if (chr == 'A')
         {
               lst.append(l_str);
               l_str = t.readLine();
               lst.append(l_str);
               l_str = t.readLine();
               lst.append(l_str);
               l_str = t.readLine();
               lst.append(l_str);
         }
         else
            lst.append(l_str);
      }
      _file->close();
      _file->remove();

      // Dump the entries to the original file
      if (_file->open(IO_WriteOnly))
      {
         QTextStream tt(_file);
         for ( QStringList::Iterator it = lst.begin(); it != lst.end(); ++it )
            tt << (*it) << endl;

         // Write new button value
         tt << "B " << str_but << " " << value << endl;

         _file->close();
      }
   }
}


int
otrplayerfile::getButton(key_enum key_but)
{
   QString str, str_but;
   int     ret_value = 0;


   if (key_but == otrplayerfile::CALENDAR_KEY)
      str_but = "Calendar";
   else if (key_but == otrplayerfile::ADDRESS_KEY)
      str_but = "Address";
   else if (key_but == otrplayerfile::HOME_KEY)
      str_but = "Home";
   else if (key_but == otrplayerfile::MENU_KEY)
      str_but = "Menu";
   else if (key_but == otrplayerfile::EMAIL_KEY)
      str_but = "Email";
   else if (key_but == otrplayerfile::UP_KEY)
      str_but = "Up";
   else if (key_but == otrplayerfile::DOWN_KEY)
      str_but = "Down";
   else if (key_but == otrplayerfile::SPACE_KEY)
      str_but = "Space";

   if (_file->open(IO_ReadOnly))
   {
      // Read all entries from file
      QTextStream t(_file);
      str = t.readLine();
      if (str == FIRST_LINE_STRING)
      {
         while (!t.atEnd())
         {
            QChar chr;
            t >> chr;
            if (chr == 'B')
            {
               QString   str1;
               t.skipWhiteSpace();
               t >> str1;
               if (str_but == str1)
               {
                  t >> ret_value;
               }
            }
            else if (chr == 'A')
            {
               // Skip all 4 lines if a file entry
               str = t.readLine();
               str = t.readLine();
               str = t.readLine();
            }
            str = t.readLine();
         }
      }
      _file->close();
   }
   return (ret_value);
}


void
otrplayerfile::convertV20()
{
   QString str, first_str;
   QStringList  io_files;
   QStringList  io_offset;
   QStringList  io_date;


   if (_file->open(IO_ReadOnly))
   {
      // Read all entries from file
      QTextStream t(_file);
      first_str = t.readLine();
      // Check if V1.0 version of data file. If so load data
      if (first_str == FIRST_LINE_STRING_V10)
      {
         while (!t.atEnd())
         {
            str = t.readLine();
            if (!t.atEnd())
            {
               io_files.append(str);
               str = t.readLine();
               io_offset.append(str);
               str = t.readLine();
               io_date.append(str);
            }
         }
      }
      _file->close();

      // IF file was old version lets write new version
      if (first_str == FIRST_LINE_STRING_V10)
      {
         writeEntries(io_files, io_offset, io_date);
      }
   }
}


void
otrplayerfile::getDirs(
   QStringList& lst,
   QStringList& tree)
{
   QString  str;


   if (_file->open(IO_ReadOnly))
   {
      // Read all entries from file
      QTextStream t(_file);
      str = t.readLine();
      if (str == FIRST_LINE_STRING)
      {
         while (!t.atEnd())
         {
            QChar chr;
            t >> chr;
            if (chr == 'D')
            {
               QString   str1, str2;
               t.skipWhiteSpace();
               t >> str1;
               t.skipWhiteSpace();
               t >> str2;
               lst.append(str1);
               tree.append(str2);
            }
            else if (chr == 'A')
            {
               // Skip all 4 lines if a file entry
               str = t.readLine();
               str = t.readLine();
               str = t.readLine();
            }
            str = t.readLine();
         }
      }
      _file->close();
   }
}


void
otrplayerfile::setDir(
   QString dir,
   QString tree)
{
   QStringList  lst;


   deleteDir(dir);

   if (_file->open(IO_WriteOnly | IO_Append))
   {
      QTextStream tt(_file);

      // If an empty file then write first line string
      if (_file->size() == 0)
         tt << FIRST_LINE_STRING << endl;

      // Write new dir value
      tt << "D " << dir << " " << tree << endl;

      _file->close();
   }
}


void
otrplayerfile::deleteDir(
   QString  dir)
{
   QStringList  lst;


   // Open file to read
   if (_file->open(IO_ReadOnly))
   {
      QTextStream t(_file);

      // Read first line and store
      lst.append(t.readLine());
      // Read all entries
      while (!t.atEnd())
      {
         QString tag, str1, str2;

         QString  l_str = t.readLine();
         QChar chr = l_str.at(0);
         if (chr == 'D')
         {
            QTextStream s(&l_str, IO_ReadOnly);
            s >> tag;
            s >> str1;
            s >> str2;

            // Dump all entries that are not the correct button
            if (str1 != dir)
               lst.append(l_str);
         }
         else if (chr == 'A')
         {
               lst.append(l_str);
               l_str = t.readLine();
               lst.append(l_str);
               l_str = t.readLine();
               lst.append(l_str);
               l_str = t.readLine();
               lst.append(l_str);
         }
         else
            lst.append(l_str);
      }
      _file->close();
      _file->remove();

      // Dump the entries to the original file
      if (_file->open(IO_WriteOnly))
      {
         QTextStream tt(_file);
         for ( QStringList::Iterator it = lst.begin(); it != lst.end(); ++it )
            tt << (*it) << endl;

         _file->close();
      }
   }
}


void
otrplayerfile::setAdvance(
   int  mode,
   int  sec)
{
   QStringList  lst;
   QString  tag;
   int      orig_mode;


   // Open file to read
   if (_file->open(IO_ReadOnly))
   {
      QTextStream t(_file);

      // Read first line and store
      lst.append(t.readLine());
      // Read all entries
      while (!t.atEnd())
      {
         QString  l_str = t.readLine();
         QChar chr = l_str.at(0);
         if (chr == 'S')
         {
            QTextStream s(&l_str, IO_ReadOnly);
            s >> tag;
            s >> orig_mode;

            // Dump all entries that are not the correct button
            if (mode != orig_mode)
               lst.append(l_str);
         }
         else if (chr == 'A')
         {
               lst.append(l_str);
               l_str = t.readLine();
               lst.append(l_str);
               l_str = t.readLine();
               lst.append(l_str);
               l_str = t.readLine();
               lst.append(l_str);
         }
         else
            lst.append(l_str);
      }
      _file->close();
      _file->remove();

      // Dump the entries to the original file
      if (_file->open(IO_WriteOnly))
      {
         QTextStream tt(_file);
         for ( QStringList::Iterator it = lst.begin(); it != lst.end(); ++it )
            tt << (*it) << endl;

         // Write new button value
         tt << "S " << mode << " " << sec << endl;

         _file->close();
      }
   }
}


bool
otrplayerfile::getAdvance(
   int  mode,
   int& value)
{
   int     orig_mode;
   int     ret_value = false;
   QString str, tag;


   if (_file->open(IO_ReadOnly))
   {
      // Read all entries from file
      QTextStream t(_file);
      str = t.readLine();
      if (str == FIRST_LINE_STRING)
      {
         while (!t.atEnd())
         {
            QString  str1 = t.readLine();
            QChar chr = str1.at(0);

            if (chr == 'S')
            {
               QTextStream s(&str1, IO_ReadOnly);
               s >> tag >> orig_mode;
               if (orig_mode == mode)
               {
                  s >> value;
                  ret_value = true;
               }
            }
            else if (chr == 'A')
            {
               // Skip all 4 lines if a file entry
               str = t.readLine();
               str = t.readLine();
               str = t.readLine();
            }
         }
      }
      _file->close();
   }
   return (ret_value);
}


void
otrplayerfile::setFont(
   int  font)
{
   QStringList  lst;
   int      orig_font;


   // Open file to read
   if (_file->open(IO_ReadOnly))
   {
      QTextStream t(_file);

      // Read first line and store
      lst.append(t.readLine());
      // Read all entries
      while (!t.atEnd())
      {
         QString  l_str = t.readLine();
         QChar chr = l_str.at(0);
         if (chr == 'F')
         {
            // We need to remove the previous F entry
         }
         else if (chr == 'A')
         {
               lst.append(l_str);
               l_str = t.readLine();
               lst.append(l_str);
               l_str = t.readLine();
               lst.append(l_str);
               l_str = t.readLine();
               lst.append(l_str);
         }
         else
            lst.append(l_str);
      }
      _file->close();
      _file->remove();

      // Dump the entries to the original file
      if (_file->open(IO_WriteOnly))
      {
         QTextStream tt(_file);
         for ( QStringList::Iterator it = lst.begin(); it != lst.end(); ++it )
            tt << (*it) << endl;

         // Write new button value
         tt << "F " << font << endl;

         _file->close();
      }
   }
   // ELSE no file then create new one
   else
   {
      // Dump the entries to the original file
      if (_file->open(IO_WriteOnly))
      {
         QTextStream tt(_file);

         tt << FIRST_LINE_STRING << endl;

         // Write new button value
         tt << "F " << font << endl;

         _file->close();
      }
   }
}


bool
otrplayerfile::getFont(
   int& font)
{
   int     ret_value = false;
   QString str, tag;


   if (_file->open(IO_ReadOnly))
   {
      // Read all entries from file
      QTextStream t(_file);
      str = t.readLine();
      if (str == FIRST_LINE_STRING)
      {
         while (!t.atEnd())
         {
            QString  str1 = t.readLine();
            QChar chr = str1.at(0);

            if (chr == 'F')
            {
               QTextStream s(&str1, IO_ReadOnly);
               s >> tag >> font;
               ret_value = true;
            }
            else if (chr == 'A')
            {
               // Skip all 4 lines if a file entry
               str = t.readLine();
               str = t.readLine();
               str = t.readLine();
            }
         }
      }
      _file->close();
   }
   return (ret_value);
}


