/*
 * Copyright: (C) 2008 Bruce W. Forsberg
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 *   Bruce Forsberg  bruce.forsberg@gmail.com
 *
 */


#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <dirent.h>
#include <stdlib.h>
#include <glib.h>

#include "mileagefile.h"
#include "mileagecfg.h"

#define CAR_DATA_CFG  "mileage.cfg"

extern void open_mileagecfg();
extern void close_mileagecfg();

/*
 Below is how the data is defined
 First line is units
 P - Person entry
 D - Destination entry
 C - Cost entry
*/

static FILE *fp = NULL;
static GKeyFile  *key_file = NULL;


void
open_mileagecfg()
{
   GString *path;
   GError  *error = NULL;


   path = g_string_new(NULL);
   if (fp == NULL)
   {
      /* Create dir and file string for configuration file */
      openCarDir(path);
      g_string_append(path, CAR_DATA_CFG);

      key_file = g_key_file_new();
      g_key_file_load_from_file(key_file, path->str, G_KEY_FILE_KEEP_COMMENTS, &error);

      if (error != NULL)
      {
         g_error_free(error);
      }
   }
   g_string_free(path, TRUE);
}

/* This function will close the configuration file */
void
close_mileagecfg()
{
   gchar *str;
   gsize    len;
   GError  *error = NULL;
   GString *path;

   path = g_string_new(NULL);
   if (key_file != NULL)
   {
      str = g_key_file_to_data(key_file, &len, &error);
      if (error != NULL)
      {
         g_error_free(error);
      }
      if (str)
      {
         /* Create dir and file string for configuration file */
         openCarDir(path);
         g_string_append(path, CAR_DATA_CFG);
         /* Create file if it is not created */
         fp = fopen(path->str, "w+");
         if (fp == NULL)
         {
         }
         else
         {
            fwrite(str, len, 1, fp);
            fclose(fp);
            fp = NULL;
         }
         g_free(str);
         str = NULL;
      }
      g_key_file_free(key_file);
      key_file = NULL;
   }
   g_string_free(path, TRUE);
}

/* This function will return the units such as MPG that is stored in the configuraiton file */
int
returnUnits()
{
   int  units = 0;
   GError *error = NULL;

   open_mileagecfg();
   units = g_key_file_get_integer(key_file, "CONFIG", "units", &error);
   if (error != NULL)
      g_error_free(error);
   close_mileagecfg();

   return units;
}

/* This stores the units into the configuration file */
void
setUnits(int units)
{
   open_mileagecfg();
   g_key_file_set_integer(key_file, "CONFIG", "units", units);
   close_mileagecfg();
 }

int
returnCostUnits()
{
   int units = 0;
   GError *error = NULL;

   open_mileagecfg();
   units = g_key_file_get_integer(key_file, "CONFIG", "cost", &error);
   if (error != NULL)
      g_error_free(error);
   close_mileagecfg();

   return units;
}

void
setCostUnits(int units)
{
   open_mileagecfg();
   g_key_file_set_integer(key_file, "CONFIG", "cost", units);
   close_mileagecfg();
}

gchar **
returnPersons()
{
   GError *error = NULL;
   gsize len;
   gchar **persons;

   open_mileagecfg();
   persons = g_key_file_get_string_list(key_file, "CONFIG", "person_list", &len, &error);
   if (error != NULL)
   {
      g_error_free(error);
   }
   close_mileagecfg();

   return persons;
}

void setPerson(char *person)
{
   GError *error = NULL;
   gsize len = 0;
   gchar **persons;
   gchar **persons_new;
   int i;
   int found_person = 0;

   open_mileagecfg();
   persons = g_key_file_get_string_list(key_file, "CONFIG", "person_list", &len, &error);
   if (error != NULL)
   {
      g_error_free(error);
   }
   close_mileagecfg();

   /* IF the person already exists don't add it again */
   for (i = 0; i < len; i++)
   {
      if (strcmp(person, persons[i]) == 0)
         found_person = 1;
   }

   /* IF person does not already exist and it is not an empty string */
   if ((found_person == 0) && (strlen(person) != 0))
   {
      /* Allocated space for existing list + NULL + new string */
      persons_new = (gchar **)g_malloc(sizeof(gchar *) * (len + 2));
      for (i = 0; i < len; i++)
      {
         persons_new[i] = g_malloc(strlen(persons[i]) + 1);
         g_stpcpy(persons_new[i], persons[i]);
      }
      persons_new[i] = g_malloc(strlen(person) + 1);
      g_stpcpy(persons_new[i++], person);
      persons_new[i] = NULL;

      open_mileagecfg();
      g_key_file_set_string_list(key_file, "CONFIG", "person_list", persons_new, len + 1);
      close_mileagecfg();
      g_strfreev(persons_new);
   }

   g_strfreev(persons);
}

gchar **
returnDestinations()
{
   GError *error = NULL;
   gsize len;
   gchar **dest;

   open_mileagecfg();
   dest = g_key_file_get_string_list(key_file, "CONFIG", "destination_list", &len, &error);
   if (error != NULL)
   {
      g_error_free(error);
   }
   close_mileagecfg();

   return dest;
}

void setDestinations(char *dest)
{
   GError *error = NULL;
   gsize len = 0;
   gchar **dests;
   gchar **dest_new;
   int i;
   int found_dest = 0;

   open_mileagecfg();
   dests = g_key_file_get_string_list(key_file, "CONFIG", "destination_list", &len, &error);
   if (error != NULL)
   {
      g_error_free(error);
   }
   close_mileagecfg();

   /* IF the destination already exists don't add it again */
   for (i = 0; i < len; i++)
   {
      if (strcmp(dest, dests[i]) == 0)
         found_dest = 1;
   }

   /* IF destination does not already exist and it is not an empty string */
   if ((found_dest == 0) && (strlen(dest) != 0))
   {
       /* Allocated space for existing list + NULL + new string */
       dest_new = (gchar **)g_malloc(sizeof(gchar *) * (len + 2));
       for (i = 0; i < len; i++)
       {
          dest_new[i] = g_malloc(strlen(dests[i]) + 1);
          g_stpcpy(dest_new[i], dests[i]);
       }
       dest_new[i] = g_malloc(strlen(dest) + 1);
       g_stpcpy(dest_new[i++], dest);
       dest_new[i] = NULL;

       open_mileagecfg();
       g_key_file_set_string_list(key_file, "CONFIG", "destination_list", dest_new, len + 1);
       close_mileagecfg();
      g_strfreev(dest_new);
   }

   g_strfreev(dests);
}

