/*
 * Copyright: (C) 2010 Bruce W. Forsberg
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 *   Bruce Forsberg  bruce.forsberg@gmail.com
 *
 */


#include <gtk/gtkmain.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtktoolbar.h>
#include <gtk/gtk.h>

#ifdef MAEMO
#if GTK_CHECK_VERSION(2,10,0)
#include <hildon/hildon-program.h>
#include <hildon/hildon-help.h>
#else
#include <hildon-widgets/hildon-program.h>
#endif
#include <libosso.h>
#endif

#include <stdlib.h>

#include "car_tab.h"
#include "entry_tab.h"
#include "data_tab.h"
#include "care_tab.h"
#include "trip_tab.h"
#include "graph_tab.h"
#include "common_tab.h"

#define MILEAGE_VERSION "1.5.1"

static gint delete_event_cb(GtkWidget* w, GdkEventAny* e, gpointer data);
static void create_main_menu(GtkWidget *w);

#ifdef MAEMO
static void about_event_cb(GtkWidget* w, GdkEventAny* e, gpointer data);
static void help_event_cb(GtkWidget* w, GdkEventAny* e, gpointer data);
static  osso_context_t *osso_context = NULL;
static gboolean key_press_cb(GtkWidget *, GdkEventKey *, HildonWindow *);
#endif

#ifdef MAEMO
static HildonWindow *window;
#else
static GtkWidget  *window;
#endif
static int fullscreen = 0;

int main(int argc, char *argv[])
{
   /* Create needed variables */
#ifdef MAEMO
   HildonProgram *program;
#endif
   GtkWidget *car_tab, *entry_tab, *data_tab, *care_tab, *trip_tab, *graph_tab;
   GtkWidget *car_label, *entry_label, *data_label, *care_label, *trip_label, *graph_label;
   GtkNotebook *note;

   /* Initialize the GTK. */
   gtk_init(&argc, &argv);

#ifdef MAEMO
   /* Create the hildon program and setup the title */
   program = HILDON_PROGRAM(hildon_program_get_instance());
   g_set_application_name("Car Mileage Calculator");

   osso_context = osso_initialize("mileage", MILEAGE_VERSION, FALSE, NULL);
   if (osso_context == NULL)
   {
      fprintf(stdout, "Can't initialize osso context.\n");
     /* return 1; */
   }

   /* Create HildonWindow and set it to HildonProgram */
   window = HILDON_WINDOW(hildon_window_new());
   hildon_program_add_window(program, window);
#else
   window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
#endif

   create_main_menu(GTK_WIDGET(window));

   car_tab = gtk_hbox_new(FALSE, 0);
   entry_tab = gtk_hbox_new(FALSE, 0);
   data_tab = gtk_vbox_new(TRUE, 0);
   care_tab = gtk_hbox_new(FALSE, 0);
   trip_tab = gtk_hbox_new(FALSE, 0);
   graph_tab = gtk_vbox_new(TRUE, 0);

   car_label = gtk_label_new("Car");
   entry_label = gtk_label_new("Entry");
   data_label = gtk_label_new("Data");
   care_label = gtk_label_new("Care");
   trip_label = gtk_label_new("Trip");
   graph_label = gtk_label_new("Graph");

   note = (GtkNotebook *)gtk_notebook_new();
   gtk_container_add(GTK_CONTAINER(window), (GtkWidget *)note);

   gtk_notebook_insert_page(note, car_tab, car_label, 0);
   gtk_notebook_insert_page(note, entry_tab, entry_label, 1);
   gtk_notebook_insert_page(note, data_tab, data_label, 2);
   gtk_notebook_insert_page(note, care_tab, care_label, 3);
   gtk_notebook_insert_page(note, trip_tab, trip_label, 4);
   gtk_notebook_insert_page(note, graph_tab, graph_label, 5);

   car_tab_layout(car_tab);
   entry_tab_layout(entry_tab);
   data_tab_layout(data_tab);
   care_tab_layout(care_tab);
   trip_tab_layout(trip_tab);
   graph_tab_layout(graph_tab);

   common_tab_updateUnits();
   common_tab_updateCars();

#ifdef MAEMO
   /* Add hardware button listener to application */
   g_signal_connect(G_OBJECT(window),
		   "key_press_event",
		   G_CALLBACK(key_press_cb),
		   NULL);
#endif

   gtk_signal_connect(GTK_OBJECT(window),
                     "delete_event",
                     GTK_SIGNAL_FUNC(delete_event_cb),
                     NULL);

/*   gtk_window_set_default_size (GTK_WINDOW (window), 696, 396); */

   /* Begin the main application */
   gtk_widget_show_all(GTK_WIDGET(window));

   gtk_main();

#ifdef MAEMO
   osso_deinitialize(osso_context);
#endif

   /* Exit */
   return 0;
}

gint 
delete_event_cb(GtkWidget* window, GdkEventAny* e, gpointer data)
{
  gtk_main_quit();
  return FALSE;
}

#ifdef MAEMO
void
help_event_cb(GtkWidget* window, GdkEventAny* e, gpointer data)
{
#if GTK_CHECK_VERSION (2,10,0)
   hildon_help_show(osso_context, "Applications_mileage_mainview", HILDON_HELP_SHOW_DIALOG);
#endif
}
#endif

void
about_event_cb(GtkWidget* window, GdkEventAny* e, gpointer data)
{
   GtkWidget *dialog;
   GdkPixbuf *logo;
   GError    *error = NULL;

   dialog = gtk_about_dialog_new();

#ifdef MAEMO
   logo = gdk_pixbuf_new_from_file("/usr/share/icons/hicolor/scalable/hildon/mileage.png", &error);
#else
   logo = gdk_pixbuf_new_from_file("../data/icons/scalable/mileage.png", &error);
#endif

   if (error == NULL)
      gtk_about_dialog_set_logo(GTK_ABOUT_DIALOG(dialog), logo);

   gtk_about_dialog_set_name(GTK_ABOUT_DIALOG(dialog), "Mileage");
   gtk_about_dialog_set_version(GTK_ABOUT_DIALOG(dialog), MILEAGE_VERSION);
   gtk_about_dialog_set_copyright(GTK_ABOUT_DIALOG(dialog), "(C) 2010 Bruce Forsberg");
   gtk_about_dialog_set_comments(GTK_ABOUT_DIALOG(dialog), "Car Mileage Calculator");
   gtk_about_dialog_set_website(GTK_ABOUT_DIALOG(dialog), "http://mileage.garage.maemo.org");
   gtk_about_dialog_set_website_label(GTK_ABOUT_DIALOG(dialog), "mileage.garage.maemo.org");

   gtk_dialog_run(GTK_DIALOG(dialog));
   gtk_widget_destroy(dialog);
}

void
create_main_menu(GtkWidget *w)
{
#ifdef MAEMO
   GtkWidget  *main_menu;
   GtkWidget  *item_close;
   GtkWidget  *item_help;
   GtkWidget  *item_about;
#if GTK_CHECK_VERSION(2,10,0)
   GtkWidget  *item_backup;
   GtkWidget  *item_import;
   GtkWidget  *item_export;
#endif

   main_menu = gtk_menu_new();

   item_about = gtk_menu_item_new_with_label("About");
   item_help = gtk_menu_item_new_with_label("Help");
   item_close = gtk_menu_item_new_with_label("Close");
#if GTK_CHECK_VERSION(2,10,0)
   item_backup = gtk_menu_item_new_with_label("Backup via Email");
   item_import = gtk_menu_item_new_with_label("Import");
   item_export = gtk_menu_item_new_with_label("Export");

   gtk_menu_append(main_menu, item_backup);
   gtk_menu_append(main_menu, item_import);
   gtk_menu_append(main_menu, item_export);
#endif
   gtk_menu_append(main_menu, item_about);
   gtk_menu_append(main_menu, item_help);
   gtk_menu_append(main_menu, item_close);

   hildon_window_set_menu(HILDON_WINDOW(w), GTK_MENU(main_menu));
#if GTK_CHECK_VERSION(2,10,0)
   g_signal_connect(G_OBJECT(item_backup), "activate", GTK_SIGNAL_FUNC(car_backup_clicked_cb), NULL);
   g_signal_connect(G_OBJECT(item_import), "activate", GTK_SIGNAL_FUNC(car_import_clicked_cb), NULL);
   g_signal_connect(G_OBJECT(item_export), "activate", GTK_SIGNAL_FUNC(car_export_clicked_cb), NULL);
#endif
   g_signal_connect(G_OBJECT(item_about), "activate", GTK_SIGNAL_FUNC(about_event_cb), NULL);
   g_signal_connect(G_OBJECT(item_help), "activate", GTK_SIGNAL_FUNC(help_event_cb), NULL);
   g_signal_connect(G_OBJECT(item_close), "activate", GTK_SIGNAL_FUNC(delete_event_cb), NULL);

   gtk_widget_show_all(GTK_WIDGET(main_menu));
#endif
}

#ifdef MAEMO
osso_context_t *
mileage_get_osso()
{
   return osso_context;
}

gboolean
key_press_cb(GtkWidget * widget, GdkEventKey * event,
		HildonWindow * w)
{
   GtkWindow *win;
   switch (event->keyval)
   {
      case GDK_F6:
	 win = GTK_WINDOW(window);
	 if (fullscreen == 0)
         {
            fullscreen = 1;
            gtk_window_fullscreen(win);
         }
	 else
         {
            fullscreen = 0;
            gtk_window_unfullscreen(win);
         }
	 data_tab_fullscreen(fullscreen);
	 care_tab_fullscreen(fullscreen);
	 trip_tab_fullscreen(fullscreen);
    graph_tab_fullscreen(fullscreen);
         return TRUE;
   }
   return FALSE;
}
#endif

