/*
 * Copyright: (C) 2008 Bruce W. Forsberg
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 *   Bruce Forsberg  bruce.forsberg@gmail.com
 *
 */


#include <gtk/gtkmain.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtktoolbar.h>
#include <gtk/gtk.h>

#include <stdlib.h>
#include <string.h>

#include "mileage.h"
#include "car_tab.h"
#include "common_tab.h"
#include "data_tab.h"
#include "mileagecfg.h"
#include "mileagefile.h"

#ifdef MAEMO
#include <libosso.h>

#if GTK_CHECK_VERSION(2,10,0)
#include <tablet-browser-interface.h>
#include <conbtdialogs-dbus.h>
#include <libmodest-dbus-client/libmodest-dbus-client.h>
#endif

#endif


typedef struct {
   GtkWidget            *widget;
   gint                  index;
   const gchar          *title;
#if GTK_CHECK_VERSION(2,10,0)
   GtkAssistantPageType  type;
#endif
   gboolean              complete;
} PageInfo;

static GtkWidget *car_selector = NULL;
static GtkWidget *tablel = NULL;
static int        car_inserted = FALSE;
static GtkWidget *selector2;
static GtkWidget *selector3;
static GtkWidget *selector4;
static GtkWidget *lastMPG;
static GtkWidget *lifeMPG;
static GtkWidget *milesMonth;
static GtkWidget *main_window;
static GtkWidget *label3;
static GtkWidget *label4;
static GtkWidget *label5;
static GtkTextBuffer *text_buf;
static GtkWidget *date_text;
static GtkWidget *odo_text;
static GtkWidget *fuel_text;
static GtkWidget *cost_text;
static GtkWidget *assist_text_buf;
static GtkWidget *assist_import;
static GtkWidget *assist_export;
static GtkWidget *assist_apply;
static GtkWidget *export_file_text;
static GtkWidget *import_file_text;
#ifdef MAEMO
static GtkWidget *email_but;
#endif
static int        running_export = 0;
static int        running_import = 0;

static void car_selector_clicked_cb(GtkWidget* w, gpointer data);
static void car_units_selected_clicked_cb(GtkWidget* w, gpointer data);
static void car_units_entry_selected_clicked_cb(GtkWidget* w, gpointer data);
static void car_cost_units_clicked_cb(GtkWidget* w, gpointer data);
static void car_new_clicked_cb(GtkWidget* w, gpointer data);
static void car_delete_clicked_cb(GtkWidget* w, gpointer data);
static void car_tab_assist_cancel(GtkWidget* w, gpointer data);
static void car_tab_assist_export_cancel(GtkWidget* w, gpointer data);
static void car_tab_assist_apply(GtkWidget* w, gpointer data);
static void car_tab_assist_export_apply(GtkWidget* w, gpointer data);
static void car_import_select_file(GtkWidget* w, gpointer data);
static void car_import_data_now(GtkWidget* w, gpointer data);
static void car_export_select_file(GtkWidget* w, gpointer data);

void car_tab_layout(GtkWidget *window)
{
   /* Create needed variables */
   GtkWidget *vboxl;
   GtkWidget *vboxr;
   GtkWidget *vsep;
   GtkWidget *label1, *label2, *label6, *label7;
   GtkWidget *car_entry;
   GtkWidget *button1, *button2;
   GtkTextView *text_box;
   GtkWidget *tabler;
   GtkWidget *sw;

   main_window = window;

   vboxl = gtk_vbox_new(FALSE, 10);
   gtk_container_add(GTK_CONTAINER(window), vboxl);

   vsep = gtk_vseparator_new();
   gtk_container_add(GTK_CONTAINER(window), vsep);

   vboxr = gtk_vbox_new(FALSE, 10);
   gtk_container_add(GTK_CONTAINER(window), vboxr);

   tablel = gtk_table_new(5,9,FALSE);
   gtk_container_add(GTK_CONTAINER(vboxl), (GtkWidget *)tablel);

   car_selector = gtk_combo_box_new_text();
   gtk_table_attach(GTK_TABLE(tablel), car_selector, 0, 4, 0, 1, GTK_FILL, GTK_FILL, 2, 2);
   gtk_signal_connect(GTK_OBJECT(car_selector),
                     "changed",
                     GTK_SIGNAL_FUNC(car_selector_clicked_cb),
                     car_selector);

   tabler = gtk_table_new(5,8,FALSE);
   gtk_container_add(GTK_CONTAINER(vboxr), (GtkWidget *)tabler);

   label1 = gtk_label_new("New Car");
   gtk_table_attach(GTK_TABLE(tabler), label1, 0, 2, 0, 1, GTK_FILL, GTK_FILL, 2, 2);

   car_entry = gtk_entry_new_with_max_length(20);
   gtk_table_attach(GTK_TABLE(tabler), car_entry, 2, 6, 0, 1, GTK_FILL, GTK_FILL, 2, 2);

   button1 = gtk_button_new_with_label("Create New Car");
   gtk_signal_connect(GTK_OBJECT(button1),
                     "clicked",
                     GTK_SIGNAL_FUNC(car_new_clicked_cb),
                     car_entry);
   gtk_table_attach(GTK_TABLE(tabler), button1, 0, 4, 1, 2, GTK_FILL, GTK_FILL, 2, 2);

   button2= gtk_button_new_with_label("Delete Current Car");
   gtk_signal_connect(GTK_OBJECT(button2),
                     "clicked",
                     GTK_SIGNAL_FUNC(car_delete_clicked_cb),
                     NULL);
   gtk_table_attach(GTK_TABLE(tabler), button2, 0, 4, 2, 3, GTK_FILL, GTK_FILL, 2, 2);

#ifndef MAEMO
#if GTK_CHECK_VERSION(2,10,0)
   GtkWidget  *button3, *button4;

   button3= gtk_button_new_with_label("Import Mileage Data");
   gtk_signal_connect(GTK_OBJECT(button3),
                     "clicked",
                     GTK_SIGNAL_FUNC(car_import_clicked_cb),
                     NULL);
   gtk_table_attach(GTK_TABLE(tabler), button3, 0, 4, 3, 4, GTK_FILL, GTK_FILL, 2, 2);

   button4= gtk_button_new_with_label("Export Mileage Data");
   gtk_signal_connect(GTK_OBJECT(button4),
                     "clicked",
                     GTK_SIGNAL_FUNC(car_export_clicked_cb),
                     NULL);
   gtk_table_attach(GTK_TABLE(tabler), button4, 0, 4, 4, 5, GTK_FILL, GTK_FILL, 2, 2);
#endif
#endif

   label2 = gtk_label_new("Display Units");
   gtk_table_attach(GTK_TABLE(tablel), label2, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 2, 2);

   selector2 = gtk_combo_box_new_text();
   gtk_combo_box_insert_text((GtkComboBox *)selector2, CAR_MPG_US_INDEX, CAR_MPG_US_TEXT);
   gtk_combo_box_insert_text((GtkComboBox *)selector2, CAR_MPG_IMP_INDEX, CAR_MPG_IMP_TEXT);
   gtk_combo_box_insert_text((GtkComboBox *)selector2, CAR_LP100KM_INDEX, CAR_LP100KM_TEXT);
   gtk_combo_box_insert_text((GtkComboBox *)selector2, CAR_LP10KM_INDEX, CAR_LP10KM_TEXT);
   gtk_combo_box_insert_text((GtkComboBox *)selector2, CAR_MPL_INDEX, CAR_MPL_TEXT);
   gtk_combo_box_insert_text((GtkComboBox *)selector2, CAR_KMPL_INDEX, CAR_KMPL_TEXT);
   gtk_table_attach(GTK_TABLE(tablel), selector2, 1, 4, 1, 2, GTK_FILL, GTK_FILL, 2, 2);
   gtk_signal_connect(GTK_OBJECT(selector2),
                     "changed",
                     GTK_SIGNAL_FUNC(car_units_selected_clicked_cb),
                     selector2);

   label7 = gtk_label_new("Entry Units");
   gtk_table_attach(GTK_TABLE(tablel), label7, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 2, 2);

   selector4 = gtk_combo_box_new_text();
   gtk_combo_box_insert_text((GtkComboBox *)selector4, CAR_MPG_US_INDEX, CAR_MPG_US_TEXT);
   gtk_combo_box_insert_text((GtkComboBox *)selector4, CAR_MPG_IMP_INDEX, CAR_MPG_IMP_TEXT);
   gtk_combo_box_insert_text((GtkComboBox *)selector4, CAR_LP100KM_INDEX, CAR_LP100KM_TEXT);
   gtk_combo_box_insert_text((GtkComboBox *)selector4, CAR_LP10KM_INDEX, CAR_LP10KM_TEXT);
   gtk_combo_box_insert_text((GtkComboBox *)selector4, CAR_MPL_INDEX, CAR_MPL_TEXT);
   gtk_combo_box_insert_text((GtkComboBox *)selector4, CAR_KMPL_INDEX, CAR_KMPL_TEXT);
   gtk_table_attach(GTK_TABLE(tablel), selector4, 1, 4, 2, 3, GTK_FILL, GTK_FILL, 2, 2);
   gtk_signal_connect(GTK_OBJECT(selector4),
                      "changed",
                      GTK_SIGNAL_FUNC(car_units_entry_selected_clicked_cb),
                      selector4);

   label6 = gtk_label_new("Cost Entries");
   gtk_table_attach(GTK_TABLE(tablel), label6, 0, 1, 3, 4, GTK_FILL, GTK_FILL, 2, 2);

   selector3 = gtk_combo_box_new_text();
   gtk_combo_box_insert_text((GtkComboBox *)selector3, 0, "Fillup");
   gtk_combo_box_insert_text((GtkComboBox *)selector3, 1, "Per Unit");
   gtk_table_attach(GTK_TABLE(tablel), selector3, 1, 3, 3, 4, GTK_FILL, GTK_FILL, 2, 2);
   gtk_signal_connect(GTK_OBJECT(selector3),
                     "changed",
                     GTK_SIGNAL_FUNC(car_cost_units_clicked_cb),
                     selector3);

   label3 = gtk_label_new("Lifetime MPG");
   gtk_table_attach(GTK_TABLE(tablel), label3, 0, 1, 4, 5, GTK_FILL, GTK_FILL, 2, 2);

   lifeMPG = gtk_entry_new_with_max_length(20);
   gtk_editable_set_editable(GTK_EDITABLE(lifeMPG), FALSE);
   gtk_table_attach(GTK_TABLE(tablel), lifeMPG, 1, 3, 4, 5, GTK_FILL, GTK_FILL, 2, 2);

   label4 = gtk_label_new("Last MPG");
   gtk_table_attach(GTK_TABLE(tablel), label4, 0, 1, 5, 6, GTK_FILL, GTK_FILL, 2, 2);

   lastMPG = gtk_entry_new_with_max_length(20);
   gtk_editable_set_editable(GTK_EDITABLE(lastMPG), FALSE);
   gtk_table_attach(GTK_TABLE(tablel), lastMPG, 1, 3, 5, 6, GTK_FILL, GTK_FILL, 2, 2);

   label5 = gtk_label_new("Miles / Month");
   gtk_table_attach(GTK_TABLE(tablel), label5, 0, 1, 6, 7, GTK_FILL, GTK_FILL, 2, 2);

   milesMonth = gtk_entry_new_with_max_length(20);
   gtk_editable_set_editable(GTK_EDITABLE(milesMonth), FALSE);
   gtk_table_attach(GTK_TABLE(tablel), milesMonth, 1, 3, 6, 7, GTK_FILL, GTK_FILL, 2, 2);

   text_buf = gtk_text_buffer_new(NULL);
   gtk_text_buffer_insert_at_cursor(text_buf, "No Alerts for this Car\n\n", 24);

   text_box = (GtkTextView *)gtk_text_view_new();
   gtk_text_view_set_editable(GTK_TEXT_VIEW(text_box), FALSE);
   gtk_text_view_set_buffer(text_box, text_buf);

   sw = gtk_scrolled_window_new(NULL, NULL);
   gtk_container_add(GTK_CONTAINER(sw), GTK_WIDGET(text_box));
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
   gtk_table_attach(GTK_TABLE(tablel), sw, 0, 5, 7, 9, GTK_FILL, GTK_FILL, 2, 2);
}

void
car_tab_load_car(char * car)
{
   if (car == NULL)
   {
      gtk_widget_destroy(car_selector);
      car_selector = gtk_combo_box_new_text();
      gtk_table_attach(GTK_TABLE(tablel), car_selector, 0, 4, 0, 1, GTK_FILL, GTK_FILL, 2, 2);
      gtk_signal_connect(GTK_OBJECT(car_selector),
                     "changed",
                     GTK_SIGNAL_FUNC(car_selector_clicked_cb),
                     car_selector);
      gtk_widget_show_all(car_selector);
      car_inserted = FALSE;
   }
   else
   {
      gtk_combo_box_append_text(GTK_COMBO_BOX(car_selector), car);
      car_inserted = TRUE;
   }
}

GtkWidget *
car_tab_get_car_selector()
{
   return car_selector;
}

int
car_tab_get_car_inserted()
{
   return car_inserted;
}

void
car_tab_update_units(int units)
{
   gtk_combo_box_set_active(GTK_COMBO_BOX(selector2), units);
}

void
car_tab_update_units_entry(int units)
{
   gtk_combo_box_set_active(GTK_COMBO_BOX(selector4), units);
}

void
car_tab_update_cost_units(int units)
{
   gtk_combo_box_set_active(GTK_COMBO_BOX(selector3), units);
}

void
car_tab_update_car_combo_box(int value)
{
   gtk_combo_box_set_active((GtkComboBox *)car_selector, value);
}

void
car_tab_set_life_MPG(char *text)
{
   gtk_entry_set_text((GtkEntry *)lifeMPG, text);
}

void
car_tab_set_last_MPG(char *text)
{
   gtk_entry_set_text((GtkEntry *)lastMPG, text);
}

void
car_tab_set_miles_month(char *text)
{
   gtk_entry_set_text((GtkEntry *)milesMonth, text);
}


void
car_selector_clicked_cb(GtkWidget* w, gpointer data)
{  
   char *car;
   
   /* Update all combo boxes to the same value */
   common_tab_update_car_combo_box(gtk_combo_box_get_active((GtkComboBox *)car_selector));

   if (data)
   {
      car = gtk_combo_box_get_active_text(data);
      if (car[strlen(car) - 1] == '\n')
      {
         car[strlen(car) - 1] = '\0';
      }

#ifdef DEBUG
      fprintf(stdout, "Car Selector choosen %s.\n", car);
#endif
     /* We only call load_data here and not in the other tabs since all similiar */
     /* functions will be called for each tab. We only need to load once */
      common_tab_load_data(car);

      if (car)
         free(car);
   }
}

void
car_new_clicked_cb(GtkWidget* w, gpointer data)
{
   char *new_car;
   GtkWidget *dialog;

#ifdef DEBUG
   fprintf(stdout, "New clicked\n");
#endif
   new_car = (char *)gtk_entry_get_text(GTK_ENTRY(data));
   if (new_car == NULL || strlen(new_car) == 0)
   {
      dialog = gtk_message_dialog_new (NULL,
                                  GTK_DIALOG_DESTROY_WITH_PARENT,
                                  GTK_MESSAGE_ERROR,
                                  GTK_BUTTONS_CLOSE,
                                  "Warning, Must enter a name for the car.");
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
   }
   else
   {
      /* Search for car */
      if (searchForCar(new_car) == 0)
      {
         if (createNewCar(new_car) == 0)
         {
            dialog = gtk_message_dialog_new (NULL,
                               GTK_DIALOG_DESTROY_WITH_PARENT,
                               GTK_MESSAGE_ERROR,
                               GTK_BUTTONS_CLOSE,
                               "Warning, Unable to create Car.");
            gtk_dialog_run (GTK_DIALOG (dialog));
            gtk_widget_destroy (dialog);
         }
         else
         {
            common_tab_updateCars();
            common_tab_select_car(new_car);
            /* Clear Entry */
            gtk_entry_set_text(GTK_ENTRY(data), "");
         }
      }
      else
      {
         dialog = gtk_message_dialog_new (NULL,
                               GTK_DIALOG_DESTROY_WITH_PARENT,
                               GTK_MESSAGE_ERROR,
                               GTK_BUTTONS_CLOSE,
                               "Warning, This car has already been entered.");
         gtk_dialog_run (GTK_DIALOG (dialog));
         gtk_widget_destroy (dialog);
      }
   }
}

void
car_delete_clicked_cb(GtkWidget* w, gpointer data)
{
   GtkWidget *dialog;
   int        result;
   char      *car = NULL;

#ifdef DEBUG
   fprintf(stdout, "Delete clicked\n");
#endif

   car = gtk_combo_box_get_active_text(GTK_COMBO_BOX(car_selector));
   if (car == NULL)
      return;
   if (car[strlen(car) - 1] == '\n')
   {
      car[strlen(car) - 1] = '\0';
   }

   dialog = gtk_message_dialog_new (NULL,
                GTK_DIALOG_DESTROY_WITH_PARENT,
                GTK_MESSAGE_ERROR,
                GTK_BUTTONS_YES_NO,
                "Warning, This will delete the car: %s\n This will remove all data for this car,\n Are you sure?", car);
   result = gtk_dialog_run (GTK_DIALOG (dialog));
   gtk_widget_destroy (dialog);

   switch (result)
   {
      case GTK_RESPONSE_YES:
#ifdef DEBUG
         fprintf(stdout, "Delete Processed\n");
#endif

         deleteCar(car);

         /* Update all entries after delete */
         common_tab_updateCars();
      break;

      default:
#ifdef DEBUG
         fprintf(stdout, "Delete Cancelled\n");
#endif
      break;
   }

   if (car)
      free(car);
}

#if GTK_CHECK_VERSION(2,10,0)
void
car_export_clicked_cb(GtkWidget* w, gpointer data)
{
   char      *car = NULL;
   GtkWidget *dialog;
   PageInfo page[3] = {
    {NULL, -1, "Export Instructions",   GTK_ASSISTANT_PAGE_INTRO, TRUE},
    {NULL, -1, "Select Export Format",    GTK_ASSISTANT_PAGE_CONTENT, TRUE},
    {NULL, -1, "Finished",         GTK_ASSISTANT_PAGE_CONFIRM, TRUE},
    };
   GtkWidget *file_label;
   GtkWidget *csv_but, *html_but;
#ifdef MAEMO
   GtkWidget *email_label;
#endif
   int i;

   if (running_export == 1)
      return;
   running_export = 1;

#ifdef DEBUG
   fprintf(stdout, "Export clicked\n");
#endif

   car = gtk_combo_box_get_active_text(GTK_COMBO_BOX(car_selector));
   if (car == NULL)
   {
      dialog = gtk_message_dialog_new (NULL,
         GTK_DIALOG_DESTROY_WITH_PARENT,
         GTK_MESSAGE_ERROR,
         GTK_BUTTONS_CLOSE,
         "Warning, You must select a Car to export data for first.");
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      running_export = 0;
      return;
   }

   page[0].widget = gtk_label_new(
       "This allows one to export data from the currently selected car\n\
to either CVS format or HTML format. The data to be exported will be\n\
the currently selected data type on the data tab.");
   page[1].widget = gtk_table_new(5,3,FALSE);
   page[2].widget = gtk_label_new("Please click Apply. If you selected CSV it will write\n\
out the data to the specified file. If you specified HTML\n\
then the html file will be loaded into a browser viewer");

   file_label = gtk_button_new_with_label("Select file");
   gtk_signal_connect(GTK_OBJECT(file_label),
                     "clicked",
                     GTK_SIGNAL_FUNC(car_export_select_file),
                     NULL);
   gtk_table_attach(GTK_TABLE(page[1].widget), file_label, 0, 1, 0, 1, GTK_FILL, GTK_SHRINK, 2, 2);

   export_file_text = gtk_entry_new_with_max_length(200);
   gtk_table_attach(GTK_TABLE(page[1].widget), export_file_text, 1, 3, 0, 1, GTK_FILL, GTK_SHRINK, 2, 2);

   csv_but = gtk_radio_button_new_with_label(NULL, "CSV Format");
   gtk_table_attach(GTK_TABLE(page[1].widget), csv_but, 0, 1, 1, 2, GTK_FILL, GTK_SHRINK, 2,2);

   html_but = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(csv_but), "HTML Format");
   gtk_table_attach(GTK_TABLE(page[1].widget), html_but, 1, 2, 1, 2, GTK_FILL, GTK_SHRINK,2, 2);
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(csv_but), TRUE);

#ifdef MAEMO
   email_label = gtk_label_new("Email file");
   gtk_table_attach(GTK_TABLE(page[1].widget), email_label, 0, 1, 2, 3, GTK_FILL, GTK_SHRINK, 2, 2);

   email_but = gtk_check_button_new();
   gtk_table_attach(GTK_TABLE(page[1].widget), email_but, 1, 2, 2, 3, GTK_FILL, GTK_SHRINK, 2, 2);
#endif

   assist_export = gtk_assistant_new();
   for (i = 0; i < sizeof(page) / sizeof(PageInfo); i++)
   {
      page[i].index = gtk_assistant_append_page(GTK_ASSISTANT(assist_export), page[i].widget);
      gtk_assistant_set_page_title(GTK_ASSISTANT(assist_export), page[i].widget, page[i].title);
      gtk_assistant_set_page_type(GTK_ASSISTANT(assist_export), page[i].widget, page[i].type);
      gtk_assistant_set_page_complete(GTK_ASSISTANT(assist_export), page[i].widget, page[i].complete);
   }

   g_signal_connect(G_OBJECT(assist_export), "cancel", G_CALLBACK(car_tab_assist_export_cancel), assist_export);
   g_signal_connect(G_OBJECT(assist_export), "close", G_CALLBACK(car_tab_assist_export_apply), csv_but);
   gtk_widget_show_all(assist_export);

   if (car)
      free(car);
}

void
car_import_clicked_cb(GtkWidget* w, gpointer data)
{
   char      *car = NULL;
   GtkWidget *dialog;
   PageInfo page[3] = {
	{NULL, -1, "Import Instructions", 	GTK_ASSISTANT_PAGE_INTRO, TRUE},
	{NULL, -1, "Input Field Values", 	GTK_ASSISTANT_PAGE_CONTENT, TRUE},
    {NULL, -1, "Verify Values",  		GTK_ASSISTANT_PAGE_CONFIRM, FALSE},
    };
   int i;
   GtkWidget  *date_label;
   GtkWidget  *file_label;
   GtkWidget  *odo_label;
   GtkWidget  *fuel_label;
   GtkWidget  *cost_label;
   GtkWidget  *text_box, *sw;
   GtkWidget  *import_button;
   GtkWidget  *label1;

#ifdef DEBUG
   fprintf(stdout, "Import clicked\n");
#endif

   if (running_import == 1)
      return;
   running_import = 1;

   car = gtk_combo_box_get_active_text(GTK_COMBO_BOX(car_selector));
   if (car == NULL)
   {
      dialog = gtk_message_dialog_new (NULL,
         GTK_DIALOG_DESTROY_WITH_PARENT,
         GTK_MESSAGE_ERROR,
         GTK_BUTTONS_CLOSE,
         "Warning, You must select a Car to import data to first.");
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      running_import = 0;
      return;
   }

   page[0].widget = gtk_label_new(
       "This allows one to import car mileage data from another car\n\
mileage program. It expects data to be in CSV (Comma Seperated Values)\n\
format. Four values must be present. 1. Date in some reasonable format,\n\
2. odometer reading, 3. fuel used, 4. an optional cost value. The data\n\
will be imported into the currently selected car. If this data is for a\n\
new car you should create a new car first and select it. If there are gaps\n\
in the mileage data then enter each sequence seperately. In the next\n\
GUI enter the position of each field into the box with the first\n\
field being called 0.");
   page[1].widget = gtk_table_new(5,2,FALSE);
   page[2].widget = gtk_table_new(5,3,FALSE);
   assist_apply = page[2].widget;

   file_label = gtk_button_new_with_label("Select CSV file location");
   gtk_signal_connect(GTK_OBJECT(file_label),
                     "clicked",
                     GTK_SIGNAL_FUNC(car_import_select_file),
                     NULL);
   gtk_table_attach(GTK_TABLE(page[1].widget), file_label, 0, 1, 0, 1, GTK_FILL, GTK_SHRINK, 2, 2);

   import_file_text = gtk_entry_new_with_max_length(55);
   gtk_table_attach(GTK_TABLE(page[1].widget), import_file_text, 1, 2, 0, 1, GTK_FILL, GTK_SHRINK, 2, 2);

   date_label = gtk_label_new("Enter field number with date");
   gtk_table_attach(GTK_TABLE(page[1].widget), date_label, 0, 1, 1, 2, GTK_FILL, GTK_SHRINK, 2, 2);

   date_text = gtk_entry_new_with_max_length(5);
   gtk_table_attach(GTK_TABLE(page[1].widget), date_text, 1, 2, 1, 2, GTK_FILL, GTK_SHRINK, 2, 2);

   odo_label = gtk_label_new("Enter field number with odometer");
   gtk_table_attach(GTK_TABLE(page[1].widget), odo_label, 0, 1, 2, 3, GTK_FILL, GTK_SHRINK, 2, 2);

   odo_text = gtk_entry_new_with_max_length(5);
   gtk_table_attach(GTK_TABLE(page[1].widget), odo_text, 1, 2, 2, 3, GTK_FILL, GTK_SHRINK, 2, 2);

   fuel_label = gtk_label_new("Enter field number with fuel amount");
   gtk_table_attach(GTK_TABLE(page[1].widget), fuel_label, 0, 1, 3, 4, GTK_FILL, GTK_SHRINK, 2, 2);

   fuel_text = gtk_entry_new_with_max_length(5);
   gtk_table_attach(GTK_TABLE(page[1].widget), fuel_text, 1, 2, 3, 4, GTK_FILL, GTK_SHRINK, 2, 2);

   cost_label = gtk_label_new("Enter field number with cost(-1 if not present)");
   gtk_table_attach(GTK_TABLE(page[1].widget), cost_label, 0, 1, 4, 5, GTK_FILL, GTK_SHRINK, 2, 2);

   cost_text = gtk_entry_new_with_max_length(5);
   gtk_table_attach(GTK_TABLE(page[1].widget), cost_text, 1, 2, 4, 5, GTK_FILL, GTK_SHRINK, 2, 2);

   import_button = gtk_button_new_with_label("View Car Data Now");
   gtk_signal_connect(GTK_OBJECT(import_button),
                     "clicked",
                     GTK_SIGNAL_FUNC(car_import_data_now),
                     import_file_text);
   gtk_table_attach(GTK_TABLE(page[2].widget), import_button, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 2, 2);

   assist_text_buf = (GtkWidget *)gtk_text_buffer_new(NULL);

   text_box = gtk_text_view_new();
   gtk_text_view_set_editable(GTK_TEXT_VIEW(text_box), FALSE);
   gtk_text_view_set_buffer(GTK_TEXT_VIEW(text_box), GTK_TEXT_BUFFER(assist_text_buf));

   sw = gtk_scrolled_window_new(NULL, NULL);
   gtk_container_add(GTK_CONTAINER(sw), GTK_WIDGET(text_box));
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
      GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
   gtk_widget_set_size_request(sw, 400, 200);
   gtk_table_attach(GTK_TABLE(page[2].widget), sw, 0, 2, 1, 5, GTK_FILL, GTK_FILL, 2, 2);

   label1 = gtk_label_new("Click on the View button.\n\
Verify the data. If it is\n\
correct then click Apply.\n\
This will import the data\n\
into the currently selected\n\
car. If not correct then go\n\
back and correct the problem.");
   gtk_table_attach(GTK_TABLE(page[2].widget), label1, 2, 3, 1, 5, GTK_FILL, GTK_FILL, 2, 2);

   assist_import = gtk_assistant_new();
   for (i = 0; i < sizeof(page) / sizeof(PageInfo); i++)
   {
      page[i].index = gtk_assistant_append_page(GTK_ASSISTANT(assist_import), page[i].widget);
      gtk_assistant_set_page_title(GTK_ASSISTANT(assist_import), page[i].widget, page[i].title);
      gtk_assistant_set_page_type(GTK_ASSISTANT(assist_import), page[i].widget, page[i].type);
      gtk_assistant_set_page_complete(GTK_ASSISTANT(assist_import), page[i].widget, page[i].complete);
   }

   g_signal_connect(G_OBJECT(assist_import), "cancel", G_CALLBACK(car_tab_assist_cancel), assist_import);
   g_signal_connect(G_OBJECT(assist_import), "close", G_CALLBACK(car_tab_assist_apply), page[2].widget);
   gtk_widget_show_all(assist_import);

   if (car)
      free(car);
}

void
car_import_select_file(GtkWidget* w, gpointer data)
{
   GtkWidget *dialog;

   dialog = gtk_file_chooser_dialog_new ("Select Import CSV File", NULL,
             GTK_FILE_CHOOSER_ACTION_SAVE,
             GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
             GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
             NULL);

   if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
   {
      char *filename; 

      filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      gtk_entry_set_text(GTK_ENTRY(import_file_text), filename);
      g_free (filename);
   }

   gtk_widget_destroy (dialog);
}

void
car_export_select_file(GtkWidget* w, gpointer data)
{
   GtkWidget *dialog;

   dialog = gtk_file_chooser_dialog_new ("Select Export File", NULL,
             GTK_FILE_CHOOSER_ACTION_SAVE,
             GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
             GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
             NULL);

   if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
   {
      char *filename;

      filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
      gtk_entry_set_text(GTK_ENTRY(export_file_text), filename);
      g_free (filename);
   }

   gtk_widget_destroy (dialog);
}

void
car_import_data_now(GtkWidget* w, gpointer data)
{
   char *file_text;
   FILE *fd;
   int  date_index, odo_index, fuel_index, cost_index;
   char buffer[10000];
   char dialog_buffer[10000];
   char *saveptr;
   int  index;
   char *field;
   char *date_str, *odo_str, *fuel_str, *cost_str;
   char combo_buffer[10000];
   int  len;
   GtkTextIter  start;
   GtkTextIter  end;
   GtkWidget *dialog;
   int  return_status = TRUE;

#ifdef DEBUG
      fprintf(stdout, "car_import_data_now\n");
#endif
   if (strlen(gtk_entry_get_text(GTK_ENTRY(date_text))) == 0)
   {
      dialog = gtk_message_dialog_new (NULL,
         GTK_DIALOG_DESTROY_WITH_PARENT,
         GTK_MESSAGE_ERROR,
         GTK_BUTTONS_CLOSE,
         "Error, No date field entered\nProcess Aborted.");
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      return_status = FALSE;
   }

   if (strlen(gtk_entry_get_text(GTK_ENTRY(odo_text))) == 0)
   {
      dialog = gtk_message_dialog_new (NULL,
         GTK_DIALOG_DESTROY_WITH_PARENT,
         GTK_MESSAGE_ERROR,
         GTK_BUTTONS_CLOSE,
         "Error, No odometer field entered\nProcess Aborted.");
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      return_status = FALSE;
   }

   if (strlen(gtk_entry_get_text(GTK_ENTRY(fuel_text))) == 0)
   {
      dialog = gtk_message_dialog_new (NULL,
         GTK_DIALOG_DESTROY_WITH_PARENT,
         GTK_MESSAGE_ERROR,
         GTK_BUTTONS_CLOSE,
         "Error, No fuel field entered\nProcess Aborted.");
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      return_status = FALSE;
   }

   file_text = (char *)gtk_entry_get_text(GTK_ENTRY(data));
   date_index = atoi((char *)gtk_entry_get_text(GTK_ENTRY(date_text)));
   odo_index = atoi((char *)gtk_entry_get_text(GTK_ENTRY(odo_text)));
   fuel_index = atoi((char *)gtk_entry_get_text(GTK_ENTRY(fuel_text)));
   cost_index = atoi((char *)gtk_entry_get_text(GTK_ENTRY(cost_text)));

   /* IF no cost entry is present then flag it */
   if (strlen(gtk_entry_get_text(GTK_ENTRY(cost_text))) == 0)
   {
      cost_index = -1;
   }

   /* Erase buffer contents and put in header string */
   gtk_text_buffer_get_start_iter(GTK_TEXT_BUFFER(assist_text_buf), &start);
   gtk_text_buffer_get_end_iter(GTK_TEXT_BUFFER(assist_text_buf), &end);
   gtk_text_buffer_delete(GTK_TEXT_BUFFER(assist_text_buf), &start, &end);
   sprintf(combo_buffer, "Date  Odometer  Fuel  Cost");
   gtk_text_buffer_insert_at_cursor(GTK_TEXT_BUFFER(assist_text_buf), combo_buffer, strlen(combo_buffer));

   if (file_text && return_status == TRUE)
   {
#ifdef DEBUG
      fprintf(stdout, "Opening file %s\n", file_text);
#endif
      fd = fopen(file_text, "r");
      if (fd)
      {
         while (fgets(buffer, 10000, fd) != NULL)
         {
            strncpy(dialog_buffer, buffer, 10000);
            index = 0;
            date_str = NULL;
            odo_str = NULL;
            fuel_str = NULL;
            cost_str = NULL;
            field = (char *)strtok_r(buffer, ",", &saveptr);
            while(field != NULL)
            {
               if (index == date_index)
                  date_str = field;
               if (index == odo_index)
                  odo_str = field;
               if (index == fuel_index)
                  fuel_str = field;
               if (index == cost_index)
                  cost_str = field;
               field = (char *)strtok_r(NULL, ",", &saveptr);
               index++;
            }
            /* IF valid values */
            if (date_str && odo_str && fuel_str)
            {
               GDate date;
               g_date_set_parse(&date, date_str);
               if (g_date_valid(&date) == FALSE)
               {
                  dialog = gtk_message_dialog_new (NULL,
                     GTK_DIALOG_DESTROY_WITH_PARENT,
                     GTK_MESSAGE_ERROR,
                     GTK_BUTTONS_CLOSE,
                     "Error, Can't understand date for entry\n%s\nPlease use MM/DD/YYYY format.\nProcess Aborted.", dialog_buffer);
                  gtk_dialog_run (GTK_DIALOG (dialog));
                  gtk_widget_destroy (dialog);
                  gtk_text_buffer_get_start_iter(GTK_TEXT_BUFFER(assist_text_buf), &start);
                  gtk_text_buffer_get_end_iter(GTK_TEXT_BUFFER(assist_text_buf), &end);
                  gtk_text_buffer_delete(GTK_TEXT_BUFFER(assist_text_buf), &start, &end);
                  return_status = FALSE;
                  break;
               }
               else if (atof(odo_str) <= 0.0)
               {
                  dialog = gtk_message_dialog_new (NULL,
                     GTK_DIALOG_DESTROY_WITH_PARENT,
                     GTK_MESSAGE_ERROR,
                     GTK_BUTTONS_CLOSE,
                     "Error, Can't understand odometer for entry\n%s\nPlease use number format > 0.\nProcess Aborted.", dialog_buffer);
                  gtk_dialog_run (GTK_DIALOG (dialog));
                  gtk_widget_destroy (dialog);
                  gtk_text_buffer_get_start_iter(GTK_TEXT_BUFFER(assist_text_buf), &start);
                  gtk_text_buffer_get_end_iter(GTK_TEXT_BUFFER(assist_text_buf), &end);
                  gtk_text_buffer_delete(GTK_TEXT_BUFFER(assist_text_buf), &start, &end);
                  return_status = FALSE;
                  break;
               }
               else if (atof(fuel_str) <= 0.0)
               {
                  dialog = gtk_message_dialog_new (NULL,
                     GTK_DIALOG_DESTROY_WITH_PARENT,
                     GTK_MESSAGE_ERROR,
                     GTK_BUTTONS_CLOSE,
                     "Error, Can't understand fuel for entry\n%s\nPlease use number format > 0.\nProcess Aborted.", dialog_buffer);
                  gtk_dialog_run (GTK_DIALOG (dialog));
                  gtk_widget_destroy (dialog);
                  gtk_text_buffer_get_start_iter(GTK_TEXT_BUFFER(assist_text_buf), &start);
                  gtk_text_buffer_get_end_iter(GTK_TEXT_BUFFER(assist_text_buf), &end);
                  gtk_text_buffer_delete(GTK_TEXT_BUFFER(assist_text_buf), &start, &end);
                  return_status = FALSE;
                  break;
               }
               else
               {

               /* IF last character is newline then remove it */
               len = strlen(date_str);
               if (len > 0 && (date_str[len-1] == '\n' || date_str[len-1] == '\r'))
                  date_str[len-1] = '\0';
               if (len > 1 && (date_str[len-2] == '\n' || date_str[len-2] == '\r'))
                  date_str[len-2] = '\0';
               len = strlen(odo_str);
               if (len > 0 && (odo_str[len-1] == '\n' || odo_str[len-1] == '\r'))
                  odo_str[len-1] = '\0';
               if (len > 1 && (odo_str[len-2] == '\n' || odo_str[len-2] == '\r'))
                  odo_str[len-2] = '\0';
               len = strlen(fuel_str);
               if (len > 0 && (fuel_str[len-1] == '\n' || fuel_str[len-1] == '\r'))
                  fuel_str[len-1] = '\0';
               if (len > 1 && (fuel_str[len-2] == '\n' || fuel_str[len-2] == '\r'))
                  fuel_str[len-2] = '\0';
               if (cost_str)
               {
                  len = strlen(cost_str);
                  if (len > 0 && (cost_str[len-1] == '\n' || cost_str[len-1] == '\r'))
                     cost_str[len-1] = '\0';
                  if (len > 1 && (cost_str[len-2] == '\n' || cost_str[len-2] == '\r'))
                     cost_str[len-2] = '\0';
               }
               if (cost_str != NULL)
                   sprintf(combo_buffer, "\n%s %s %s %s", date_str, odo_str, fuel_str, cost_str);
               else
                   sprintf(combo_buffer, "\n%s %s %s", date_str, odo_str, fuel_str);
               gtk_text_buffer_insert_at_cursor(GTK_TEXT_BUFFER(assist_text_buf), combo_buffer, strlen(combo_buffer));
               }
            }
         }
         fclose(fd);
      }
      else
      {
         dialog = gtk_message_dialog_new (NULL,
            GTK_DIALOG_DESTROY_WITH_PARENT,
            GTK_MESSAGE_ERROR,
            GTK_BUTTONS_CLOSE,
            "Error, Can't open file:\n%s.", file_text);
         gtk_dialog_run (GTK_DIALOG (dialog));
         gtk_widget_destroy (dialog);
         return_status = FALSE;
      }
   }
   if (return_status == TRUE)
      gtk_assistant_set_page_complete(GTK_ASSISTANT(assist_import), assist_apply, TRUE);
}

void
car_tab_assist_cancel(GtkWidget* w, gpointer data)
{
   running_import = 0;
   gtk_widget_destroy(GTK_WIDGET(data));
}

void
car_tab_assist_export_cancel(GtkWidget* w, gpointer data)
{
   running_export = 0;
   gtk_widget_destroy(GTK_WIDGET(data));
}

void
car_tab_assist_export_apply(GtkWidget* w, gpointer data)
{
   int              data_selected;
   GtkListStore    *list_store;
   GtkTreeIter      iter;
   gboolean         valid;
   char            *date_str;
   char            *odo_str;
   char            *fuel_str;
   char            *mpg_str;
   char            *cost_str;
   char            *file_text;
   char            *extra1_str;
   char            *extra2_str;
   FILE            *fd;
   GtkWidget       *dialog;
   char            buf[10000];
   int             units;
   char            *units_str;
   char            *car;
 

#ifdef DEBUG
   fprintf(stdout, "Export Apply selected.\n");
#endif

   car = gtk_combo_box_get_active_text(GTK_COMBO_BOX(car_selector));

   units = returnEntryUnits(car);
   /* If entry units not defined then use display units */
   if (units == -1)
      units = returnUnits();

   if (units == CAR_MPG_US_INDEX)
      units_str = "MPG";
	else if (units == CAR_MPG_IMP_INDEX)
      units_str = "MPG";
   else if (units == CAR_LP100KM_INDEX)
      units_str = "L/100km";
   else if (units == CAR_LP10KM_INDEX)
      units_str = "L/10km";
   else if (units == CAR_MPL_INDEX)
      units_str = "Miles/L";
   else if (units == CAR_KMPL_INDEX)
      units_str = "km/L";
   else
      units_str = "UNKNOWN";

   file_text = (char *)gtk_entry_get_text(GTK_ENTRY(export_file_text));

   if (strlen(file_text) == 0)
   {
      dialog = gtk_message_dialog_new (NULL,
         GTK_DIALOG_DESTROY_WITH_PARENT,
         GTK_MESSAGE_ERROR,
         GTK_BUTTONS_CLOSE,
         "Error, Please enter a file name.");
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      gtk_widget_destroy(GTK_WIDGET(assist_export));
      return;
   }

   fd = fopen(file_text, "w+");
   if (fd == NULL)
   {
      dialog = gtk_message_dialog_new (NULL,
         GTK_DIALOG_DESTROY_WITH_PARENT,
         GTK_MESSAGE_ERROR,
         GTK_BUTTONS_CLOSE,
         "Error, Can't open file:\n%s.", file_text);
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
   }
   else
   {
      /* IF CSV button is selected */
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(data)) == TRUE)
      {
         data_selected = data_tab_get_display_selector();
         if (data_selected == DATA_DISPLAY_MILEAGE)
         {
            sprintf(buf, "Date,Odometer,Fuel,%s,Cost\n", units_str);
            fwrite(buf, strlen(buf), 1, fd);
            list_store = data_tab_get_store();
            valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(list_store), &iter);
            while (valid)
            {
               gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter,
                 DATA_DATE_ROW, &date_str,
                 DATA_ODO_ROW,  &odo_str,
                 DATA_FUEL_ROW, &fuel_str,
                 DATA_MPG_ROW,  &mpg_str,
                 DATA_COST_ROW, &cost_str,
                 -1);
               sprintf(buf, "%s,%s,%s,%s,%s\n",
                  date_str, odo_str, fuel_str, mpg_str, cost_str);
               fwrite(buf, strlen(buf), 1, fd);
               if (date_str)
                  g_free(date_str);
               if (odo_str)
                  g_free(odo_str);
               if (fuel_str)
                  g_free(fuel_str);
               if (mpg_str)
                  g_free(mpg_str);
               if (cost_str)
                  g_free(cost_str);
               valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(list_store), &iter);
            }
         }
         else if (data_selected == DATA_DISPLAY_ACTIVITY)
         {
            sprintf(buf, "Date,Odometer,Activity,Comment,Cost\n");
            fwrite(buf, strlen(buf), 1, fd);
            list_store = data_tab_get_store();
            valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(list_store), &iter);
            while (valid)
            {
               gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter,
                 DATA_DATE_ROW, &date_str,
                 DATA_ODO_ROW,  &odo_str,
                 DATA_FUEL_ROW, &fuel_str,
                 DATA_MPG_ROW,  &mpg_str,
                 DATA_COST_ROW, &cost_str,
                 -1);
               sprintf(buf, "%s,%s,%s,%s,%s\n",
                  date_str, odo_str, fuel_str, mpg_str,cost_str);
               fwrite(buf, strlen(buf), 1, fd);
               if (date_str)
                  g_free(date_str);
               if (odo_str)
                  g_free(odo_str);
               if (fuel_str)
                  g_free(fuel_str);
               if (mpg_str)
                  g_free(mpg_str);
               if (cost_str)
                  g_free(cost_str);
               valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(list_store), &iter);
            }
         }
         else if (data_selected == DATA_DISPLAY_ALERTS)
         {
            sprintf(buf, "Odometer,Repeat Every,Activity,Comment\n");
            fwrite(buf, strlen(buf), 1, fd);
            list_store = data_tab_get_store();
            valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(list_store), &iter);
            while (valid)
            {
               gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter,
                 DATA_DATE_ROW, &date_str,
                 DATA_ODO_ROW,  &odo_str,
                 DATA_FUEL_ROW, &fuel_str,
                 DATA_MPG_ROW,  &mpg_str,
                 -1);
               sprintf(buf, "%s %s %s %s\n",
                  date_str, odo_str, fuel_str, mpg_str);
               fwrite(buf, strlen(buf), 1, fd);
               if (date_str)
                  g_free(date_str);
               if (odo_str)
                  g_free(odo_str);
               if (fuel_str)
                  g_free(fuel_str);
               if (mpg_str)
                  g_free(mpg_str);
               valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(list_store), &iter);
            }
         }
         else if (data_selected == DATA_DISPLAY_TRIPS)
         {
            sprintf(buf, "Start Date,Start Odometer,End Date,End Odometer,Person,Destination,Purpose\n");
            fwrite(buf, strlen(buf), 1, fd);
            list_store = data_tab_get_store();
            valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(list_store), &iter);
            while (valid)
            {
               gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter,
                 DATA_DATE_ROW, &date_str,
                 DATA_ODO_ROW,  &odo_str,
                 DATA_FUEL_ROW, &fuel_str,
                 DATA_MPG_ROW,  &mpg_str,
                 DATA_COST_ROW, &cost_str,
                 DATA_EXTRA1_ROW, &extra1_str,
                 DATA_EXTRA2_ROW, &extra2_str,
                 -1);
               /* If end date and end odometer are "Empty then make them empty */
               if (strcmp("Empty", fuel_str) == 0)
                  fuel_str = "";
               if (strcmp("Empty", mpg_str) == 0)
                  mpg_str = "";

               sprintf(buf, "%s,%s,%s,%s,%s,%s,%s\n",
                  date_str, odo_str, fuel_str, mpg_str, cost_str, extra1_str, extra2_str);
               fwrite(buf, strlen(buf), 1, fd);
               if (date_str)
                  g_free(date_str);
               if (odo_str)
                  g_free(odo_str);
               if (fuel_str)
                  g_free(fuel_str);
               if (mpg_str)
                  g_free(mpg_str);
               if (cost_str)
                  g_free(cost_str);
               if (extra1_str)
                  g_free(extra1_str);
               if (extra2_str)
                  g_free(extra2_str);
               valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(list_store), &iter);
            }
         }
         fclose(fd);
      }
      else
      {
         sprintf(buf, "<html>\n");
         fwrite(buf, strlen(buf), 1, fd);
         sprintf(buf, "<h1>Car Mileage Export</h1>\n");
         fwrite(buf, strlen(buf), 1, fd);
         data_selected = data_tab_get_display_selector();
         if (data_selected == DATA_DISPLAY_MILEAGE)
         {
            sprintf(buf, "<h2>Car Mileage Data for: %s</h2>\n", car);
            fwrite(buf, strlen(buf), 1, fd);
#if 0
            sprintf(buf, "<PRE><h2>Date    Odometer    Fuel    %s    Cost</h2></PRE>\n", units_str);
            fwrite(buf, strlen(buf), 1, fd);
#endif
            list_store = data_tab_get_store();
            valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(list_store), &iter);
            sprintf(buf, "<SCRIPT LANGUAGE='Javascript'>\n");
            fwrite(buf, strlen(buf), 1, fd);
            sprintf(buf, "document.write(\"<XMP>\")\n");
            fwrite(buf, strlen(buf), 1, fd);
            sprintf(buf, "document.write(\"Date      \\tOdometer\\tFuel    \\t%s     \\tCost\\r\")\n", units_str);
            fwrite(buf, strlen(buf), 1, fd);
            while (valid)
            {
               gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter,
                 DATA_DATE_ROW, &date_str,
                 DATA_ODO_ROW,  &odo_str,
                 DATA_FUEL_ROW, &fuel_str,
                 DATA_MPG_ROW,  &mpg_str,
                 DATA_COST_ROW, &cost_str,
                 -1);
               sprintf(buf, "document.write(\"%s\\t%8s\\t%8s\\t%8s\\t%8s\\r\")\n",
                  date_str, odo_str, fuel_str, mpg_str, cost_str);
               fwrite(buf, strlen(buf), 1, fd);
               if (date_str)
                  g_free(date_str);
               if (odo_str)
                  g_free(odo_str);
               if (fuel_str)
                  g_free(fuel_str);
               if (mpg_str)
                  g_free(mpg_str);
               if (cost_str)
                  g_free(cost_str);
               valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(list_store), &iter);
            }
            sprintf(buf, "document.write(\"</XMP>\")\n");
            fwrite(buf, strlen(buf), 1, fd);
            sprintf(buf, "</SCRIPT>\n");
            fwrite(buf, strlen(buf), 1, fd);
         }
         else if (data_selected == DATA_DISPLAY_ACTIVITY)
         {
            sprintf(buf, "<h2>Car Activity Data for: %s</h2>\n", car);
            fwrite(buf, strlen(buf), 1, fd);
#if 0
            sprintf(buf, "<h2>Date  Odometer  Activity  Comment  Cost</h2>\n");
            fwrite(buf, strlen(buf), 1, fd);
#endif
            list_store = data_tab_get_store();
            valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(list_store), &iter);
            sprintf(buf, "<SCRIPT LANGUAGE='Javascript'>\n");
            fwrite(buf, strlen(buf), 1, fd);
            sprintf(buf, "document.write(\"<XMP>\")\n");
            fwrite(buf, strlen(buf), 1, fd);
            sprintf(buf, "document.write(\"Date    \\tOdometer  \\tActivity                 \\tComment    \\tCost    \\r\")\n");
            fwrite(buf, strlen(buf), 1, fd);
            while (valid)
            {
               gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter,
                 DATA_DATE_ROW, &date_str,
                 DATA_ODO_ROW,  &odo_str,
                 DATA_FUEL_ROW, &fuel_str,
                 DATA_MPG_ROW,  &mpg_str,
                 DATA_COST_ROW, &cost_str,
                 -1);
               sprintf(buf, "document.write(\"%s\\t%8s\\t%25s\\t%s\\t%10s\\r\")\n",
                  date_str, odo_str, fuel_str, mpg_str, cost_str);
               fwrite(buf, strlen(buf), 1, fd);
               if (date_str)
                  g_free(date_str);
               if (odo_str)
                  g_free(odo_str);
               if (fuel_str)
                  g_free(fuel_str);
               if (mpg_str)
                  g_free(mpg_str);
               if (cost_str)
                  g_free(cost_str);
               valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(list_store), &iter);
            }
            sprintf(buf, "document.write(\"</XMP>\")\n");
            fwrite(buf, strlen(buf), 1, fd);
            sprintf(buf, "</SCRIPT>\n");
            fwrite(buf, strlen(buf), 1, fd);
         }
         else if (data_selected == DATA_DISPLAY_ALERTS)
         {
            sprintf(buf, "<h2>Car Alert Data for: %s</h2>\n", car);
            fwrite(buf, strlen(buf), 1, fd);
#if 0
            sprintf(buf, "<h2>Odometer  Repeat Every  Activity  Comment</h2>\n");
            fwrite(buf, strlen(buf), 1, fd);
#endif
            list_store = data_tab_get_store();
            valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(list_store), &iter);
            sprintf(buf, "<SCRIPT LANGUAGE='Javascript'>\n");
            fwrite(buf, strlen(buf), 1, fd);
            sprintf(buf, "document.write(\"<XMP>\")\n");
            fwrite(buf, strlen(buf), 1, fd);
            sprintf(buf, "document.write(\"Odometer  \\tRepeat Every\\tActivity        \\tComment\\r\")\n");
            fwrite(buf, strlen(buf), 1, fd);
            while (valid)
            {
               gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter,
                 DATA_DATE_ROW, &date_str,
                 DATA_ODO_ROW,  &odo_str,
                 DATA_FUEL_ROW, &fuel_str,
                 DATA_MPG_ROW,  &mpg_str,
                 -1);
               sprintf(buf, "document.write(\"%s\\t%8s\\t%25s\\t%s\\r\")\n",
                  date_str, odo_str, fuel_str, mpg_str);
               fwrite(buf, strlen(buf), 1, fd);
               if (date_str)
                  g_free(date_str);
               if (odo_str)
                  g_free(odo_str);
               if (fuel_str)
                  g_free(fuel_str);
               if (mpg_str)
                  g_free(mpg_str);
               valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(list_store), &iter);
            }
            sprintf(buf, "document.write(\"</XMP>\")\n");
            fwrite(buf, strlen(buf), 1, fd);
            sprintf(buf, "</SCRIPT>\n");
            fwrite(buf, strlen(buf), 1, fd);
         }
         else if (data_selected == DATA_DISPLAY_TRIPS)
         {
            sprintf(buf, "<h2>Car Trip Data for: %s</h2>\n", car);
            fwrite(buf, strlen(buf), 1, fd);
#if 0
            sprintf(buf, "<h2>Start Date  Start Odometer  End Date  End Odometer  Person  Destination  Purpose</h2>\n");
            fwrite(buf, strlen(buf), 1, fd);
#endif
            list_store = data_tab_get_store();
            valid = gtk_tree_model_get_iter_first (GTK_TREE_MODEL(list_store), &iter);
            sprintf(buf, "<SCRIPT LANGUAGE='Javascript'>\n");
            fwrite(buf, strlen(buf), 1, fd);
            sprintf(buf, "document.write(\"<XMP>\")\n");
            fwrite(buf, strlen(buf), 1, fd);
            sprintf(buf, "document.write(\"Start Date\\tStart Odometer\\tEnd Date\\tEnd Odometer  \\tPerson    \\tDestination   \\tPurpose\\r\")\n");
            fwrite(buf, strlen(buf), 1, fd);
            while (valid)
            {
               gtk_tree_model_get(GTK_TREE_MODEL(list_store), &iter,
                 DATA_DATE_ROW, &date_str,
                 DATA_ODO_ROW,  &odo_str,
                 DATA_FUEL_ROW, &fuel_str,
                 DATA_MPG_ROW,  &mpg_str,
                 DATA_COST_ROW, &cost_str,
                 DATA_EXTRA1_ROW, &extra1_str,
                 DATA_EXTRA2_ROW, &extra2_str,
                 -1);
               /* If end date and end odometer are "Empty then make them empty */
               if (strcmp("Empty", fuel_str) == 0)
                  fuel_str = "";
               if (strcmp("Empty", mpg_str) == 0)
                  mpg_str = "";
               sprintf(buf, "document.write(\"%11s\\t%8s\\t%11s\\t%8s\\t%s\\t%s\\t%s\\r\")\n",
                  date_str, odo_str, fuel_str, mpg_str, cost_str, extra1_str, extra2_str);
               fwrite(buf, strlen(buf), 1, fd);
               if (date_str)
                  g_free(date_str);
               if (odo_str)
                  g_free(odo_str);
               if (fuel_str)
                  g_free(fuel_str);
               if (mpg_str)
                  g_free(mpg_str);
               if (cost_str)
                  g_free(cost_str);
               if (extra1_str)
                  g_free(extra1_str);
               if (extra2_str)
                  g_free(extra2_str);
               valid = gtk_tree_model_iter_next (GTK_TREE_MODEL(list_store), &iter);
            }
            sprintf(buf, "document.write(\"</XMP>\")\n");
            fwrite(buf, strlen(buf), 1, fd);
            sprintf(buf, "</SCRIPT>\n");
            fwrite(buf, strlen(buf), 1, fd);
         }
         fclose(fd);
         sprintf(buf, "file://%s", file_text);
#ifdef MAEMO
         /* Put the html file into the browser */
         osso_return_t ret;
         ret = osso_rpc_run_with_defaults(mileage_get_osso(), "osso_browser",
           OSSO_BROWSER_OPEN_NEW_WINDOW_REQ, NULL,
           DBUS_TYPE_STRING, buf, DBUS_TYPE_BOOLEAN, FALSE, DBUS_TYPE_INVALID);
#endif
      }

#ifdef MAEMO
      /* EMAIL the file if the user requested it */
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(email_but)))
      {
         gboolean result = TRUE;
         GSList *list = NULL;
         sprintf(buf, "Attached to this email is the output from the Car Mileage Calculator for car %s", car);
         /* Attach the saved file (and not the one currently on screen). If the file
         * has not yet been saved, nothing will be attached */
         list = g_slist_append(list, file_text);
         result = libmodest_dbus_client_compose_mail(mileage_get_osso(), /*osso_context_t*/
            NULL, /*to*/
            NULL, /*cc*/
            NULL, /*bcc*/
            "Car Mileage Calculator - Export", /*subject*/
            buf, /*body*/
            list /*attachments*/);
         g_slist_free(list);
         if (result == FALSE)
         {
            dialog = gtk_message_dialog_new (NULL,
               GTK_DIALOG_DESTROY_WITH_PARENT,
               GTK_MESSAGE_ERROR,
               GTK_BUTTONS_CLOSE,
               "Error, Can't email file");
            gtk_dialog_run (GTK_DIALOG (dialog));
            gtk_widget_destroy (dialog);
         }
      }
#endif
   }

   gtk_widget_destroy(GTK_WIDGET(assist_export));
   running_export = 0;
   if (car)
      free(car);
}

void
car_tab_assist_apply(GtkWidget* w, gpointer data)
{
   char  *car;
   gint  lines;
   GtkTextIter  start;
   GtkTextIter  end;
   char  *total_str;
   int i;
   int len;

#ifdef DEBUG
   fprintf(stdout, "Apply selected.\n");
#endif

   lines = gtk_text_buffer_get_line_count(GTK_TEXT_BUFFER(assist_text_buf));

   car = gtk_combo_box_get_active_text(GTK_COMBO_BOX(car_selector));
   if (car != NULL)
   {
      if (car[strlen(car) - 1] == '\n')
      {
         car[strlen(car) - 1] = '\0';
      }
      /* Loop thru each entry but skip header */
      for (i = 1; i < lines; i++)
      {
         gtk_text_buffer_get_iter_at_line(GTK_TEXT_BUFFER(assist_text_buf), &start, i);
         /* IF last line in file then get an end iterator */
         if (lines == (i + 1))
            gtk_text_buffer_get_end_iter(GTK_TEXT_BUFFER(assist_text_buf), &end);
         else
            gtk_text_buffer_get_iter_at_line(GTK_TEXT_BUFFER(assist_text_buf), &end, i + 1);
         total_str = gtk_text_buffer_get_text(GTK_TEXT_BUFFER(assist_text_buf), &start, &end, FALSE);

         /* Make sure this is not a blank line */
         if (strlen(total_str) > 3)
         {
            len = strlen(total_str);
            if (total_str[len-1] == '\n')
               total_str[len-1] = '\0';
            if (i == 1)
               addMileEntry(car, total_str, TRUE);
            else
               addMileEntry(car, total_str, FALSE);
         }
         g_free(total_str);
      }
   }

   common_tab_updateDisplayTab(car);
   graph_tab_load_data(car);

   if (car)
      free(car);

   running_import = 0;
   gtk_widget_destroy(GTK_WIDGET(assist_import));
}
#endif

void
car_units_selected_clicked_cb(GtkWidget* w, gpointer data)
{
   int  value;
   char *car;

#ifdef DEBUG
   fprintf(stdout, "Units Selected\n");
#endif
   value = gtk_combo_box_get_active(GTK_COMBO_BOX(data));
   if (value != -1)
      setUnits(value);

   common_tab_updateUnits();
   car = gtk_combo_box_get_active_text(GTK_COMBO_BOX(car_selector));
   if (car != NULL)
   {
      if (car[strlen(car) - 1] == '\n')
      {
         car[strlen(car) - 1] = '\0';
      }
      /* Update all entries after delete */
      common_tab_load_data(car);
      common_tab_updateDisplayTab(car);
   }
   if (car)
      g_free(car);
}

void
car_units_entry_selected_clicked_cb(GtkWidget* w, gpointer data)
{
   int  value;
   char *car;

#ifdef DEBUG
   fprintf(stdout, "Entry Units Selected\n");
#endif

   car = gtk_combo_box_get_active_text(GTK_COMBO_BOX(car_selector));
   if (car != NULL)
   {
      if (car[strlen(car) - 1] == '\n')
      {
         car[strlen(car) - 1] = '\0';
      }
      value = gtk_combo_box_get_active(GTK_COMBO_BOX(data));
      if (value != -1)
         addEntryUnits(car, value);

      /* Update all entries after delete */
      common_tab_load_data(car);
      common_tab_updateDisplayTab(car);
   }
   if (car)
      g_free(car);
}

void
car_cost_units_clicked_cb(GtkWidget* w, gpointer data)
{
   int  value;

#ifdef DEBUG
   fprintf(stdout, "Cost Units Selected\n");
#endif
   value = gtk_combo_box_get_active(GTK_COMBO_BOX(data));
   if (value != -1)
      setCostUnits(value);

   common_tab_updateUnits();
}

void
car_tab_update_text_labels(
   char     *lifetime_label,
   char     *last_label,
   char     *miles_month_label)
{
   gtk_label_set_text(GTK_LABEL(label3), lifetime_label);
   gtk_label_set_text(GTK_LABEL(label4), last_label);
   gtk_label_set_text(GTK_LABEL(label5), miles_month_label);
}

GtkWidget *
car_tab_get_text_buffer()
{
   return (GTK_WIDGET(text_buf));
}

void
car_backup_clicked_cb(GtkWidget* w, gpointer data)
{
   GDate  *date = NULL;
   GDateYear  year;
   GDateMonth month;
   GDateDay   day;
   char       str[512];

   str[0] = '0';
   date = g_date_new();
   if (date) 
   {
      /* IF GLIB 2.10 or later */
      if (GLIB_CHECK_VERSION(2,10,0))
         g_date_set_time_t(date, time(NULL));
      else
         g_date_set_time(date, (GTime)time(NULL));
      year = g_date_get_year(date);
      month= g_date_get_month(date);
      day  = g_date_get_day(date);
      g_date_free(date);
      sprintf(str, "%d/%d/%d", (int)year, (int)month, (int)day);
   }

#ifdef MAEMO
   char       buf[1024];
   GtkWidget       *dialog;
   gboolean result = TRUE;
   GSList *list = returnCarFiles();
   sprintf(buf, "Attached to this email are the data files from the Car Mileage Calculator generated on %s.\n\
To restore place these files into the mileage directory under the MyDocs directory. Be careful not to \
write over any files that are already there.", str);
   /* Attach the saved file (and not the one currently on screen). If the file
   * has not yet been saved, nothing will be attached */
   result = libmodest_dbus_client_compose_mail(mileage_get_osso(), /*osso_context_t*/
      NULL, /*to*/
      NULL, /*cc*/
      NULL, /*bcc*/
      "Car Mileage Calculator - Backup", /*subject*/
      buf, /*body*/
      list /*attachments*/);

   g_slist_foreach(list, common_delete_data, NULL);
   g_slist_free(list);
   if (result == FALSE)
   {
      dialog = gtk_message_dialog_new (NULL,
         GTK_DIALOG_DESTROY_WITH_PARENT,
         GTK_MESSAGE_ERROR,
         GTK_BUTTONS_CLOSE,
         "Error, Can't email file");
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
   }
#endif
}

