/*
 * Copyright: (C) 2008 Bruce W. Forsberg
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 *   Bruce Forsberg  bruce.forsberg@gmail.com
 *
 */


#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <dirent.h>
#include <stdlib.h>
#include <unistd.h>

#include <glib.h>

#include "mileagefile.h"

#define CAR_DATA_EXT  ".car"
#define FIRST_LINE_STRING "This is a data file for the Car Mileage program V1.0"

static GSList * returnEntries(char *, char);
static void deleteCommonEntry(char *, char *, int);


/*
 Below is how the data is defined
 M - Mileage entry, date, odometer, fuel, and optional cost
 P - Mileage entry as M but no previous entry
 A - Name of car
 C - Alert - Odometer, repeat (0 if no repeat), Activity, Comments
 D - Activity - Date, Odometer, Activity, Comments
 R - Person category for Trip tab
 E - Destination category for Trip tab
 F - Trip entry

*/


int
returnCars(char **car_array)
{
   GString *path_to_cars = NULL;
   GString *path_plus_car = NULL;
   DIR  *open_dir;
   struct dirent  *dir;
   char line[1024];
   char *amount;
   int   car_num = 0;
   char *car_entry;
   FILE *fp;
   char *ptr;

   *car_array = NULL;

   path_to_cars = g_string_new(NULL);
   path_plus_car = g_string_new(NULL);
   openCarDir(path_to_cars);

#ifdef DEBUG
   fprintf(stdout, "Opening dir %s\n", path_to_cars->str);
#endif
   open_dir = opendir(path_to_cars->str);
   if (open_dir)
   {
      while ((dir = readdir(open_dir)) != NULL)
      {
         if ((strcmp (dir->d_name, ".") != 0) && (strcmp(dir->d_name, "..") != 0))
         {
            g_string_erase(path_plus_car, 0, -1);
            g_string_append(path_plus_car, path_to_cars->str);
            g_string_append(path_plus_car, "/");
            g_string_append(path_plus_car, dir->d_name);
#ifdef DEBUG
            fprintf(stdout, "looking at file %s\n", path_plus_car->str);
#endif
            fp = fopen(path_plus_car->str, "r");
            if (fp != NULL)
            {
               amount = fgets(line, 1024, fp);
               if (amount != NULL)
               {
#ifdef DEBUG
                  fprintf(stdout, "first line reads:%s\n", line);
#endif
                  /* Is this a car file */
                  if (strncmp(line, FIRST_LINE_STRING, strlen(FIRST_LINE_STRING)) == 0)
                  {
                     amount = fgets(line, 1024, fp);
                     if (strlen(line) > 2)
                     {
                        /* If this is one of our files then add to car list */
                        if (line[0] == 'A')
                        {
                           car_num++;
                           *car_array = realloc(*car_array, car_num * sizeof (void *));
                           car_entry = malloc(strlen(&(line[2])) + 1);
                           ptr = *car_array + (car_num - 1) * sizeof (void *);
                           memcpy(ptr, &car_entry, sizeof(void *));
                           /* Don't copy the last \n character */
                           strncpy(car_entry, &line[2], strlen(&line[2]) - 1);
                           car_entry[strlen(&line[2])-1] = '\0';
#ifdef DEBUG
                           fprintf(stdout, "adding car entry %s\n", car_entry);
#endif
                        }
                     }
                  }
               }
               fclose(fp);
            }
         }
      }
      closedir(open_dir);
   }

   g_string_free(path_to_cars, TRUE);
   g_string_free(path_plus_car, TRUE);
   return car_num;
}

/*  This returns all mileage entries */
GSList *
returnMileEntries(
   char *car)
{
   char line[1024];
   FILE *fp;
   char *str;
   GSList  *lst = NULL;

   fp = openCarFile(car);
   if (fp != NULL)
   {
      fseek(fp, 0, SEEK_SET);
      while (fgets(line, 1024, fp) != NULL)
      {
         if (line[strlen(line) - 1] == '\n')
         {
            line[strlen(line) - 1] = '\0';
         }
         if (line[0] == 'M')
         {
            str = calloc(strlen(line) + 1, 1);
            strcpy(str, line + 2);
            strcat(str, " 0");
            lst = g_slist_prepend(lst, str);
         }
         else if (line[0] == 'P')
         {
            str = calloc(strlen(line) + 1, 1);
            strcpy(str, line + 2);
            strcat(str, " 1");
            lst = g_slist_prepend(lst, str);
         }
      }
      fclose(fp);
      lst = g_slist_reverse(lst);
   }

   return (lst);
}

/*  This returns all entries for specified value */
GSList *
returnEntries(
   char *car,
   char value)
{
   char line[1024];
   FILE *fp;
   char *str;
   GSList  *lst = NULL;

   fp = openCarFile(car);
   if (fp != NULL)
   {
      fseek(fp, 0, SEEK_SET);
      while (fgets(line, 1024, fp) != NULL)
      {
         if (line[strlen(line) - 1] == '\n')
         {
            line[strlen(line) - 1] = '\0';
         }
         if (line[0] == value)
         {
            str = calloc(strlen(line) + 1, 1);
            strcpy(str, line + 2);
            lst = g_slist_prepend(lst, str);
         }
      }
      fclose(fp);
      lst = g_slist_reverse(lst);
   }

   return (lst);
}

/* Searches for car. 1 = car found, 0 = car not found */
int
searchForCar(const char *carname)
{
   GString *path_to_cars = NULL;
   DIR  *open_dir;
   struct dirent  *dir;
   int   return_val = 0;
   GString *car_plus_ext = NULL;


   path_to_cars = g_string_new(NULL);
   car_plus_ext = g_string_new(NULL);

   /* Form file name to search for by appending extension */
   g_string_append(car_plus_ext, carname);
   g_string_append(car_plus_ext, CAR_DATA_EXT);

   openCarDir(path_to_cars);

#ifdef DEBUG
   fprintf(stdout, "Opening dir %s\n", path_to_cars->str);
#endif
   open_dir = opendir(path_to_cars->str);
   if (open_dir)
   {
      while ((dir = readdir(open_dir)) != NULL)
      {
         if ((strcmp (dir->d_name, ".") != 0) && (strcmp(dir->d_name, "..") != 0))
         {
            if (strcmp(dir->d_name, car_plus_ext->str) == 0)
            {
               return_val = 1;
               break;
            }
         }
      }
      closedir(open_dir);
   }

   g_string_free(path_to_cars, TRUE);
   g_string_free(car_plus_ext, TRUE);
   return return_val;
}

/* Creates a new car. 1 = car created, 0 = car not created */
int
createNewCar(const char *carname)
{
   GString *path_to_cars = NULL;
   int   return_val = 0;
   FILE  *fd;


   path_to_cars = g_string_new(NULL);
   createCarName(carname, path_to_cars);

   if ((fd = fopen(path_to_cars->str, "w")) != NULL)
   {
      fwrite(FIRST_LINE_STRING, strlen(FIRST_LINE_STRING), 1, fd);
      fwrite("\n", 1, 1, fd);
      fwrite("A ", 2, 1, fd);
      fwrite(carname, strlen(carname), 1, fd);
      fwrite("\n", 1, 1, fd);

      fclose(fd);
      return_val = 1;
   }

   g_string_free(path_to_cars, TRUE);
   return return_val;
}

void
deleteCar(const char *carname)
{
   GString *path_to_cars = NULL;

   path_to_cars = g_string_new(NULL);
   createCarName(carname, path_to_cars);

   unlink(path_to_cars->str);
   g_string_free(path_to_cars, TRUE);
}

void
deleteEntry(
   char *car,
   char *mileage)
{
   deleteCommonEntry(car, mileage, 1);
}

/* 1 = method delete mileage entry
   2 = method delete trip entry */
void
deleteCommonEntry(
   char *car,
   char *mileage,
   int   method)
{
   GSList  *lst = NULL;
   char line[1024];
   FILE *fp;
   char *str;
   char  odo[1024];
   char  date[1024];
   char  marker[1024];
   GString *path_to_cars = NULL;

   path_to_cars = g_string_new(NULL);

   if (method == 1)
   {
#ifdef DEBUG
   fprintf(stdout, "Deleting mileage entry %s\n", mileage);
#endif
   }
   else if (method == 2)
   {
#ifdef DEBUG
   fprintf(stdout, "Deleting trip entry %s\n", mileage);
#endif
   }

   /* TBD - what should we do if user deletes P entry */
   /*       maybe add a note in help log */

   /* Put all entries into the list that don't match the specified */
   /* mileage entry */
   fp = openCarFile(car);
   if (fp != NULL)
   {
      fseek(fp, 0, SEEK_SET);
      while (fgets(line, 1024, fp) != NULL)
      {
         if ( ((method == 1) && (line[0] == 'M' || line[0] == 'P')) ||
             ((method == 2) && (line[0] == 'F')) )
         {
            sscanf(line, "%s %s %s", marker, date, odo);
            if (strcmp(odo, mileage) != 0)
            {
               str = calloc(strlen(line) + 1, 1);
               strcpy(str, line);
               lst = g_slist_prepend(lst, str);
            }
         }
         else
         {
            str = calloc(strlen(line) + 1, 1);
            strcpy(str, line);
            lst = g_slist_prepend(lst, str);
         }
      }
      lst = g_slist_reverse(lst);
      fclose(fp);
   }

   /* Now lets dump all the entries back out to the file */
   createCarName(car, path_to_cars);

   if ((fp = fopen(path_to_cars->str, "w")) != NULL)
   {
      GSList *next = lst;

      do
      {
         if (next->data)
         {
            fwrite(next->data, strlen(next->data), 1, fp);
            free(next->data);
         }
      } while ((next = g_slist_next(next)) != NULL);
      fclose(fp);
   }
   g_string_free(path_to_cars, TRUE);
   g_slist_free(lst);
}

/* entry = 0 for Activity, 1 for Alert */
void
deleteActivityAlertEntry(
   char *car,
   char *first,
   char *second,
   char *third,
   int   entry)
{
   GSList  *lst = NULL;
   char line[1024];
   char buf[1024];
   FILE *fp;
   char *str;
   char  *odo;
   char  *repeat;
   char  *read_date;
   char  *act;
   char  *marker;
   char  *saveptr;
   char  *length;
   int   len;
   GString *path_to_cars = NULL;

   path_to_cars = g_string_new(NULL);

#ifdef DEBUG
   fprintf(stdout, "Deleting activity/alert entry %s\n", first);
#endif

   /* Put all entries into the list that don't match the specified */
   /* activity entry */
   fp = openCarFile(car);
   if (fp != NULL)
   {
      fseek(fp, 0, SEEK_SET);
      while (fgets(line, 1024, fp) != NULL)
      {
         if ((entry == 0) && (line[0] == 'D'))
         {
            strncpy(buf, line, 1024);
            marker = (char *)strtok_r(buf, " ", &saveptr);
            read_date = (char *)strtok_r(NULL, " ", &saveptr);
            odo = (char *)strtok_r(NULL, " ", &saveptr);
            length = (char *)strtok_r(NULL, " ", &saveptr);
            len = atoi(length);
            act = saveptr;
            saveptr[len] = '\0';

            /* Odometer, date, and activity must all match to delete entry */
            if ((strcmp(second, odo) != 0) || (strcmp(first, read_date) != 0) || (strcmp(third, act) != 0))
            {
               str = calloc(strlen(line) + 1, 1);
               strcpy(str, line);
               lst = g_slist_prepend(lst, str);
            }
         }
         else if ((entry == 1) && (line[0] == 'C'))
         {
            strncpy(buf, line, 1024);
            marker = (char *)strtok_r(buf, " ", &saveptr);
            odo = (char *)strtok_r(NULL, " ", &saveptr);
            repeat = (char *)strtok_r(NULL, " ", &saveptr);
            length = (char *)strtok_r(NULL, " ", &saveptr);
            len = atoi(length);
            act = saveptr;
            saveptr[len] = '\0';

            /* Odometer, date, and activity must all match to delete entry */
            if ((strcmp(first, odo) != 0) || (strcmp(second, repeat) != 0) || (strcmp(third, act) != 0))
            {
               str = calloc(strlen(line) + 1, 1);
               strcpy(str, line);
               lst = g_slist_prepend(lst, str);
            }
         }
         else
         {
            str = calloc(strlen(line) + 1, 1);
            strcpy(str, line);
            lst = g_slist_prepend(lst, str);
         }
      }
      lst = g_slist_reverse(lst);
      fclose(fp);
   }

   /* Now lets dump all the entries back out to the file */
   createCarName(car, path_to_cars);

   if ((fp = fopen(path_to_cars->str, "w")) != NULL)
   {
      GSList *next = lst;

      do
      {
         if (next->data)
         {
            fwrite(next->data, strlen(next->data), 1, fp);
            free(next->data);
         }
      } while ((next = g_slist_next(next)) != NULL);
      fclose(fp);
   }
   g_string_free(path_to_cars, TRUE);
   g_slist_free(lst);
}

void
deleteTripEntry(
   char *car,
   char *mileage)
{
   deleteCommonEntry(car, mileage, 2);
}

void
addMileEntry(char *car, char *str, int previous)
{
   FILE *fp;

   fp = openCarFile(car);
   if (fp != NULL)
   {
      fseek(fp, 0, SEEK_END);

      if (previous == TRUE)
         fwrite("P ", 2, 1, fp);
      else
         fwrite("M ", 2, 1, fp);
      fwrite(str, strlen(str), 1, fp);
      fwrite("\n", 1, 1, fp);
      fclose(fp);
   }
}

FILE *
openCarFile(char *car)
{
   FILE * fp = NULL;
   GString *path_to_car = NULL;

   if (car == NULL)
      return NULL;

   path_to_car = g_string_new(NULL);

   openCarDir(path_to_car);

   g_string_append(path_to_car, car);
   g_string_append(path_to_car, ".car");

   fp = fopen(path_to_car->str, "a+");

   g_string_free(path_to_car, TRUE);

   return fp;
}

void
openCarDir(GString *path)
{
   if (getenv("HOME"))
   {
      g_string_append(path, getenv("HOME"));
      g_string_append(path, "/");
   }

   g_string_append(path, CAR_DATA_DIR);

   /* Create the directory path if not created */
   g_mkdir_with_parents(path->str, 0777);
}


void
createCarName(const char *car, GString *path_to_car)
{
   /* Form file name to search for by appending extension */
   openCarDir(path_to_car);

   g_string_append(path_to_car, car);
   g_string_append(path_to_car, CAR_DATA_EXT);
}

void
addActivityEntry(
   char *car,
   char *date_str,
   char *odo_str,
   char *act_str,
   char *comment_str)
{
   FILE *fp;
   char  buf[1024];
   

   fp = openCarFile(car);
   if (fp != NULL) 
   {
      fseek(fp, 0, SEEK_END);
   
      fwrite("D ", 2, 1, fp);

      fwrite(date_str, strlen(date_str), 1, fp);
      fwrite(" ", 1, 1, fp);

      fwrite(odo_str, strlen(odo_str), 1, fp);
      fwrite(" ", 1, 1, fp);

      sprintf(buf, "%d", strlen(act_str));
      fwrite(buf, strlen(buf), 1, fp);
      fwrite(" ", 1, 1, fp);
      fwrite(act_str, strlen(act_str), 1, fp);
      fwrite(" ", 1, 1, fp);

      sprintf(buf, "%d", strlen(comment_str));
      fwrite(buf, strlen(buf), 1, fp);
      fwrite(" ", 1, 1, fp);
      fwrite(comment_str, strlen(comment_str), 1, fp);

      fwrite("\n", 1, 1, fp);
      fclose(fp);
   }
}

void
addAlertEntry(
   char *car,
   char *odo_str,
   char *repeat,
   char *act_str,
   char *comment_str)
{
   FILE *fp;
   char  buf[1024];


   fp = openCarFile(car);
   if (fp != NULL)
   {
      fseek(fp, 0, SEEK_END);

      fwrite("C ", 2, 1, fp);

      fwrite(odo_str, strlen(odo_str), 1, fp);
      fwrite(" ", 1, 1, fp);

      fwrite(repeat, strlen(repeat), 1, fp);
      fwrite(" ", 1, 1, fp);

      sprintf(buf, "%d", strlen(act_str));
      fwrite(buf, strlen(buf), 1, fp);
      fwrite(" ", 1, 1, fp);
      fwrite(act_str, strlen(act_str), 1, fp);
      fwrite(" ", 1, 1, fp);

      sprintf(buf, "%d", strlen(comment_str));
      fwrite(buf, strlen(buf), 1, fp);
      fwrite(" ", 1, 1, fp);
      fwrite(comment_str, strlen(comment_str), 1, fp);

      fwrite("\n", 1, 1, fp);
      fclose(fp);
   }
}

GSList *
getNextAlert(
   char *car,
   int   delta)
{
   char line[1024];
   char buf[1024];
   FILE *fp;
   char *str;
   GSList  *lst = NULL;
   int  max_odo = -1;
   int  min_odo = 0;
   char  *saveptr;
   char  *marker;
   char  *read_date;
   char  *odo;
   char  *act;
   char  *comment;
   char  *repeat;
   char  *length;
   int    odo_int;
   int    rep_int;
   int    len;


   /* Get the min and max odometer readings to search for alerts */
   fp = openCarFile(car);
   if (fp != NULL)
   {
      fseek(fp, 0, SEEK_SET);
      while (fgets(line, 1024, fp) != NULL)
      {
         if (line[strlen(line) - 1] == '\n')
         {
            line[strlen(line) - 1] = '\0';
         }
         if ((line[0] == 'M') || (line[0] == 'P'))
         {
            marker = (char *)strtok_r(line, " ", &saveptr);
            read_date = (char *)strtok_r(NULL, " ", &saveptr);
            odo = (char *)strtok_r(NULL, " ", &saveptr);
            odo_int = atoi(odo);
            if (odo_int > min_odo)
               min_odo = odo_int;
         }
      }
      max_odo = min_odo + delta;
      fclose(fp);
   }

   /* Now we search for alerts that meet the min and max odo readings */
   fp = openCarFile(car);
   if (fp != NULL)
   {
      fseek(fp, 0, SEEK_SET);
      while (fgets(line, 1024, fp) != NULL)
      {
         if (line[strlen(line) - 1] == '\n')
         {
            line[strlen(line) - 1] = '\0';
         }
         if (line[0] == 'C')
         {
            strncpy(buf, line, 1024);
            marker = (char *)strtok_r(buf, " ", &saveptr);
            odo = (char *)strtok_r(NULL, " ", &saveptr);
            repeat = (char *)strtok_r(NULL, " ", &saveptr);
            length = (char *)strtok_r(NULL, " ", &saveptr);
            len = atoi(length);
            act = saveptr;
            saveptr[len] = '\0';
            saveptr += (len + 1);
            length = (char *)strtok_r(NULL, " ", &saveptr);
            len = atoi(length);
            comment = saveptr;
            saveptr[len] = '\0';
            odo_int = atoi(odo);
            rep_int = atoi(repeat);

            /* If a non repeating event */
            if (rep_int == 0)
            {
               /* We are looking for all the odometer readings between */
               /* the last fillup and delta odometer readings */
               if ((odo_int <= max_odo) && (odo_int >= min_odo))
               {
                  /* Send odometer reading, activity, and comment */
                  str = calloc(strlen(line), 1);
                  sprintf(str, "%d %d %s %d %s", odo_int, strlen(act), act, strlen(comment), comment);
                  lst = g_slist_prepend(lst, str);
               }
            }
            /* ELSE a repeating event */
	    else
            {
               /* First check to see if base odometer reading is below current */
               if (odo_int <= max_odo)
               {
                  /* Now fine the base or repeating value closest to current */
                  int mult = 0;
                  int new_odo;
                  do
                  {
                     if ((mult * rep_int + odo_int) > min_odo)
                        break;
                     mult++;
                  }
                  while (TRUE);
                  new_odo = mult * rep_int + odo_int;

                  /* IF this value between delta and max_odo */
                  if (new_odo <= max_odo)
                  {
                     /* Send odometer reading and activity */
                     str = calloc(strlen(line), 1);
                     sprintf(str, "%d %d %s %d %s", new_odo, strlen(act), act, strlen(comment), comment);
                     lst = g_slist_prepend(lst, str);
                  }
               }
            }
         }
      }
      lst = g_slist_reverse(lst);
      fclose(fp);
   }
   return (lst);
}

/* This returns all activity entries */
GSList *
returnActivityEntries(char *car)
{
   return (returnEntries(car, 'D'));
}

/* This returns all alert entries */
GSList *
returnAlertEntries(char *car)
{
   return returnEntries(car, 'C');
}

/* This returns all trip entries */
GSList *
returnTripEntries(char *car)
{
   return returnEntries(car, 'F');
}


void
addTripEntry(
   char *car,
   char *start_date,
   char *start_odo,
   char *end_date,
   char *end_odo,
   char *person_str,
   char *dest_text,
   char *purpose_text)
{
   FILE *fp;
   char  buf[1024];


   /* In case we are updateing a previous entry then we delete the existing one */
   deleteTripEntry(car, start_odo);

   fp = openCarFile(car);
   if (fp != NULL)
   {

      fseek(fp, 0, SEEK_END);

      fwrite("F ", 2, 1, fp);

      fwrite(start_date, strlen(start_date), 1, fp);
      fwrite(" ", 1, 1, fp);

      fwrite(start_odo, strlen(start_odo), 1, fp);
      fwrite(" ", 1, 1, fp);

      /* If no end date and put in "Empty" as a placeholder */
      if (strlen(end_date) == 0)
         fwrite("Empty", 5, 1, fp);
      else
         fwrite(end_date, strlen(end_date), 1, fp);
      fwrite(" ", 1, 1, fp);

      /* If no end date and put in "Empty" as a placeholder */
      if (strlen(end_odo) == 0)
         fwrite("Empty", 5, 1, fp);
      else
         fwrite(end_odo, strlen(end_odo), 1, fp);
      fwrite(" ", 1, 1, fp);

      sprintf(buf, "%d", strlen(person_str));
      fwrite(buf, strlen(buf), 1, fp);
      fwrite(" ", 1, 1, fp);
      fwrite(person_str, strlen(person_str), 1, fp);
      fwrite(" ", 1, 1, fp);

      sprintf(buf, "%d", strlen(dest_text));
      fwrite(buf, strlen(buf), 1, fp);
      fwrite(" ", 1, 1, fp);
      fwrite(dest_text, strlen(dest_text), 1, fp);
      fwrite(" ", 1, 1, fp);

      sprintf(buf, "%d", strlen(purpose_text));
      fwrite(buf, strlen(buf), 1, fp);
      fwrite(" ", 1, 1, fp);
      fwrite(purpose_text, strlen(purpose_text), 1, fp);

      fwrite("\n", 1, 1, fp);
      fclose(fp);
   }
}

