/*****************************************************************************
 *** Mauku - Jaiku client for Maemo devices 
 ***
 *** Copyright (c) 2007 Henrik Hedberg <hhedberg@innologies.fi>
 ***
 *** Licensed under the Apache License, Version 2.0 (the "License");
 *** you may not use this file except in compliance with the License.
 *** You may obtain a copy of the License at
 ***
 ***     http://www.apache.org/licenses/LICENSE-2.0
 ***
 *** Unless required by applicable law or agreed to in writing, software
 *** distributed under the License is distributed on an "AS IS" BASIS,
 *** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *** See the License for the specific language governing permissions and
 *** limitations under the License.
 ***
 *****************************************************************************/
 
#include "config.h"

#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <errno.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <string.h>

#if HILDON == 1

#include <hildon/hildon.h>

#else

#include <hildon-widgets/hildon-banner.h>
#include <hildon-widgets/hildon-program.h>
#include <hildon-widgets/hildon-defines.h>
#include <hildon-widgets/hildon-note.h>

#endif

#include <miaouw/miaouw.h>

#include "mauku.h"

typedef struct {
	ViewItem* failed_session_item;
	time_t last_successful_session_timestamp;
} SessionInfo;

struct _View {
	GtkWidget* window;
	gchar* unique_id;
	UiSensitiveActionType sensitive_actions;
	gboolean descending_timestamps;
	void (*update_callback)(View* view);
	void (*destroy_callback)(View* view);
	GData* user_data;
	GArray* items;
	GtkWidget* back_vbox;
	GtkWidget* top_bottom_widget;
	GtkWidget* scrolled_window;
	GtkWidget* vbox;
	guint show_scrollbar_notify_id;
	guint first_item_shaded; /* 0 or 1 */
	GSList* adding_sessions;
	GData* session_infos;
};

struct _ViewAddingSession {
	View* view;
	ViewItemSource* source;
	guint index;
	GList* speech_list;
	gboolean speech_activated;
	guint timeout_id;
};

static void add_item(View* view, ViewItem* item, guint index, ViewAddingSession* session);
static gint compare_unique_ids(gconstpointer a, gconstpointer b);
static void do_adding_session_end(ViewAddingSession* session, gboolean successful, gboolean keep_old_items);
static void do_gesture_event(MiaouwGestureEventBox* event_box, MiaouwGestureEvent* event, View* view, ViewItem* item);
static gboolean on_adding_session_timeout(gpointer user_data);
static gboolean on_button_press_event(GtkWidget* widget, GdkEventButton* event, ViewItem* item);
static gboolean on_delete_event(GtkWidget* widget, GdkEvent* event, View* view);
static void on_gesture_event(MiaouwGestureEventBox* event_box, MiaouwGestureEvent* event, ViewItem* item);
static void on_gesture_notify(MiaouwGestureEventBox* event_box, MiaouwGestureEvent* event, ViewItem* item);
static gboolean on_key_press_event(GtkWidget* widget, GdkEventKey* event, View* view);
static void on_mark_menu_item_activate(GtkMenuItem* menu_item, gpointer user_data);
static void on_marked_items_notify(GConfClient* client, guint cnxn_id, GConfEntry *entry, gpointer user_data);
static void on_show_scrollbar_notify(GConfClient* client, guint cnxn_id, GConfEntry *entry, gpointer user_data);
static void on_size_allocate(GtkWidget* widget, GtkAllocation* allocation, gpointer user_data); 
static gboolean on_update_timeout(gpointer data);
static void on_view_gesture_event(MiaouwGestureEventBox* event_box, MiaouwGestureEvent* event, View* view);
static gboolean on_visibility_notify_event(GtkWidget* widget, GdkEventVisibility* event, View* view);
static void remove_item(View* view, guint index);
static void substract_time(struct tm* t1, struct tm* t2);
static void update_marked_item_status(ViewItem* item);
static guint update_text(ViewItem* item);

static GList* views = NULL;
static View* topmost_view = NULL;
static GdkCursor* cursor_gesture_jump_top = NULL;
static GdkCursor* cursor_gesture_jump_bottom = NULL;
static GdkCursor* cursor_gesture_window_menu = NULL;
static GdkCursor* cursor_gesture_context_menu = NULL;
static GdkCursor* cursor_gesture_close = NULL;
static GdkCursor* cursor_gesture_open = NULL;
static GHashTable* marked_items = NULL;
static guint marked_items_notify_id = 0;

static ViewItemSource internal_item_source = {
	NULL
};

static ViewItemClass internal_item_class = {
	&internal_item_source,
	{ 0, 0xeeee, 0xeeee, 0xeeee },
	{ 0, 0xdddd, 0xdddd, 0xdddd },
	NULL, NULL
};

#define DOUBLE_BUTTON_PRESS_TIME_MIN 150
#define DOUBLE_BUTTON_PRESS_TIME_MAX 600

#define ADDING_SESSION_TIMEOUT 60

View* view_new(gchar* unique_id, gchar* title, UiSensitiveActionType sensitive_actions,
               void (*update_callback)(View* view), void (*destroy_callback)(View* view)) {
	View* view;
	GdkPixbuf* pixbuf;
	GtkWidget* event_box;

	if ((view = view_get_view_by_unique_id(unique_id))) {
		gtk_window_present(GTK_WINDOW(view_get_window(view)));

		return view;
	}

	if (!cursor_gesture_jump_top) {
		pixbuf = gdk_pixbuf_new_from_file("/usr/share/icons/hicolor/34x34/hildon/mauku_gesture_jump_top.tif", NULL);
		if (pixbuf) {
			cursor_gesture_jump_top = gdk_cursor_new_from_pixbuf(gdk_display_get_default(), pixbuf, 16, 33);
			gdk_pixbuf_unref(pixbuf);
		}
		pixbuf = gdk_pixbuf_new_from_file("/usr/share/icons/hicolor/34x34/hildon/mauku_gesture_jump_bottom.tif", NULL);
		if (pixbuf) {
			cursor_gesture_jump_bottom = gdk_cursor_new_from_pixbuf(gdk_display_get_default(), pixbuf, 16, 33);
			gdk_pixbuf_unref(pixbuf);
		}
		pixbuf = gdk_pixbuf_new_from_file("/usr/share/icons/hicolor/34x34/hildon/mauku_gesture_window_menu.tif", NULL);
		if (pixbuf) {
			cursor_gesture_window_menu = gdk_cursor_new_from_pixbuf(gdk_display_get_default(), pixbuf, 16, 33);
			gdk_pixbuf_unref(pixbuf);
		}
		pixbuf = gdk_pixbuf_new_from_file("/usr/share/icons/hicolor/34x34/hildon/mauku_gesture_context_menu.tif", NULL);
		if (pixbuf) {
			cursor_gesture_context_menu = gdk_cursor_new_from_pixbuf(gdk_display_get_default(), pixbuf, 16, 33);
			gdk_pixbuf_unref(pixbuf);
		}
		pixbuf = gdk_pixbuf_new_from_file("/usr/share/icons/hicolor/34x34/hildon/mauku_gesture_close.tif", NULL);
		if (pixbuf) {
			cursor_gesture_close = gdk_cursor_new_from_pixbuf(gdk_display_get_default(), pixbuf, 16, 33);
			gdk_pixbuf_unref(pixbuf);
		}
		pixbuf = gdk_pixbuf_new_from_file("/usr/share/icons/hicolor/34x34/hildon/mauku_gesture_open.tif", NULL);
		if (pixbuf) {
			cursor_gesture_open = gdk_cursor_new_from_pixbuf(gdk_display_get_default(), pixbuf, 16, 33);
			gdk_pixbuf_unref(pixbuf);
		}
	}

	view = (View*)g_malloc0(sizeof(View));
	view->unique_id = g_strdup(unique_id);
	view->sensitive_actions = sensitive_actions;
	view->descending_timestamps = FALSE;
	view->update_callback = update_callback;
	view->destroy_callback = destroy_callback;
	g_datalist_init(&view->user_data);
	g_datalist_init(&view->session_infos);
	view->items = g_array_new(FALSE, FALSE, sizeof(ViewItem*));
	view->back_vbox = gtk_vbox_new(FALSE, 0);
	view->vbox = miaouw_box_new_vertical();
	
	view->window = miaouw_window_new();
	hildon_program_add_window(HILDON_PROGRAM(hildon_program_get_instance()), HILDON_WINDOW(view->window));

	gtk_window_set_title(GTK_WINDOW(view->window), title);
	g_signal_connect(G_OBJECT(view->window), "delete_event", G_CALLBACK(on_delete_event), view);
	g_signal_connect(G_OBJECT(view->window), "key_press_event", G_CALLBACK(on_key_press_event), view);
	g_signal_connect(G_OBJECT(view->window), "visibility-notify-event", G_CALLBACK(on_visibility_notify_event), view);

	if (topmost_view) {
		if (miaouw_window_is_fullscreen(MIAOUW_WINDOW(topmost_view->window))) {
			gtk_window_fullscreen(GTK_WINDOW(view->window));
		}
		miaouw_window_set_font_size(GTK_WINDOW(view->window), miaouw_window_get_font_size(MIAOUW_WINDOW(topmost_view->window)));
	}

	event_box = miaouw_gesture_event_box_new();
	g_signal_connect(event_box, "gesture-notify", G_CALLBACK(on_gesture_notify), NULL);
	g_signal_connect(event_box, "gesture-event", G_CALLBACK(on_view_gesture_event), view);

	view->scrolled_window = miaouw_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(view->scrolled_window), GTK_POLICY_NEVER, GTK_POLICY_NEVER);
	miaouw_scrolled_window_set_scrolling_hints(MIAOUW_SCROLLED_WINDOW(view->scrolled_window), TRUE);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(view->scrolled_window), GTK_WIDGET(view->vbox));
	gtk_container_add(GTK_CONTAINER(view->window), view->back_vbox);
	gtk_container_add(GTK_CONTAINER(view->back_vbox), event_box);
	gtk_container_add(GTK_CONTAINER(event_box), view->scrolled_window);
	miaouw_window_set_vadjustment(MIAOUW_WINDOW(view->window), gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(view->scrolled_window)));

	view->show_scrollbar_notify_id = gconf_client_notify_add(gconf, MAUKU_GCONF_KEY_SHOW_SCROLLBAR, on_show_scrollbar_notify, view, NULL, NULL);
	gconf_client_notify(gconf, MAUKU_GCONF_KEY_SHOW_SCROLLBAR);
	if (!marked_items_notify_id) {
		marked_items_notify_id = gconf_client_notify_add(gconf, MAUKU_GCONF_KEY_MARKED_ITEMS, on_marked_items_notify, NULL, NULL, NULL);
		gconf_client_notify(gconf, MAUKU_GCONF_KEY_MARKED_ITEMS);
	}
	
	gtk_widget_show_all(view->window);

	views = g_list_prepend(views, view);

	return view;
}

void view_destroy(View* view) {
	GSList* slist_item;
	
	if (view->destroy_callback) {
		view->destroy_callback(view);
	}
	views = g_list_remove(views, view);
	
	gconf_client_notify_remove(gconf, view->show_scrollbar_notify_id);

	while (view->items->len) {
		remove_item(view, view->items->len - 1);
	}
	g_array_free(view->items, TRUE);

	for (slist_item = view->adding_sessions; slist_item; slist_item = g_slist_next(slist_item)) {
		((ViewAddingSession*)slist_item->data)->view = NULL;
	}
	g_slist_free(view->adding_sessions);
	
	g_datalist_clear(&view->user_data);
	gtk_widget_destroy(view->window);
	g_free(view->unique_id);
	g_free(view);
}

gint view_get_item_count(View* view) {
	return view->items->len;
}

gchar* view_get_unique_id(View* view) {
	return view->unique_id;
}

gpointer view_get_user_data(View* view, const gchar* key) {
	return g_datalist_get_data(&view->user_data, key);
}

gpointer view_get_user_data_by_id(View* view, const GQuark id) {
	return g_datalist_id_get_data(&view->user_data, id);
}

gboolean view_get_visibility(View* view) {
	return GTK_WIDGET_VISIBLE(view->window);
}

GtkWidget* view_get_window(View* view) {
	return view->window;
}


gboolean view_exists(View* view) {
	if (!g_list_find(views, view)) {
		return FALSE;
	}
	
	return TRUE;
}

/* set widget to NULL to remove */
void view_set_top_bottom_widget(View* view, GtkWidget* widget, gboolean top) {
	if (view->top_bottom_widget) {
		gtk_container_remove(GTK_CONTAINER(view->back_vbox), view->top_bottom_widget);
	}
	if (widget) {
		gtk_box_pack_start(GTK_BOX(view->back_vbox), widget, FALSE, FALSE, 2);
		if (top) {
			gtk_box_reorder_child(GTK_BOX(view->back_vbox), widget, 0);
		}
		view->top_bottom_widget = widget;
	} else {
		view->top_bottom_widget = NULL;
	}
	ui_set_toolbar_hidden(topmost_view == view && view->top_bottom_widget ? TRUE : FALSE);
}

void view_set_user_data(View* view, const gchar* key, gpointer value, GDestroyNotify destroy_func) {
	return g_datalist_set_data_full(&view->user_data, key, value, destroy_func);
}

void view_set_user_data_by_id(View* view, const GQuark id, gpointer value, GDestroyNotify destroy_func) {
	return g_datalist_id_set_data_full(&view->user_data, id, value, destroy_func);
}

void view_set_visibility(View* view, gboolean is_visible) {
	if (is_visible) {
		gtk_widget_show(view->window);
		gtk_window_present(GTK_WINDOW(view->window));
	} else {
		gtk_widget_hide(view->window);
	}
}

void view_start_update(View* view, gboolean user_initiated) {
	if (view->update_callback) {
		view->update_callback(view);
	}
}

void view_adding_session_add_item(ViewAddingSession* session, ViewItem* item) {
	ViewItem* other_item;

	while (session->index > 0) {
		other_item = g_array_index(session->view->items, ViewItem*, session->index - 1);
		if (session->source == other_item->item_class->source || item->timestamp < other_item->timestamp ||
		    other_item->item_class->source == &internal_item_source) {
			break; 	
		}

		session->index--;
	}
	while (session->index < session->view->items->len) {
		other_item = g_array_index(session->view->items, ViewItem*, session->index);
		if (session->source == other_item->item_class->source || item->timestamp > other_item->timestamp ||
		    other_item->item_class->source == &internal_item_source) {
			break; 	
		}

		session->index++;
	}

	add_item(session->view, item, session->index, session);
	session->index++;
	
	if (gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_NEW_ITEM_NOTIFICATION, NULL)) {
		miaouw_window_activate_urgency_hint(MIAOUW_WINDOW(session->view->window));
		if (gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_BACKLIGHT, NULL)) {
			osso_display_state_on(osso_context);
		}
		if (gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_RAISE_WINDOW, NULL)) {
			gtk_widget_show(session->view->window);
			gtk_window_present(GTK_WINDOW(session->view->window));
		}
	}
	
	if (item->speech) {
		if (session->speech_activated) {
			session->speech_list = g_list_prepend(session->speech_list, item->speech);
		} else {
			g_free(item->speech);
		}
		item->speech = NULL;
	}
	
	g_source_remove(session->timeout_id);
	session->timeout_id = g_timeout_add(ADDING_SESSION_TIMEOUT * 1000, on_adding_session_timeout, session);
}

ViewAddingSession* view_begin_adding_session(View* view, ViewItemSource* source) {
	ViewAddingSession* session;
	GSList* slist;
	ViewItem* item;

	for (slist = view->adding_sessions; slist; slist = g_slist_next(slist)) {
		if (((ViewAddingSession*)slist->data)->source == source) {
		
			return NULL;
		}
	}

	view->sensitive_actions &= ~UI_SENSITIVE_ACTION_UPDATE;
	if (view == topmost_view) {
	 	ui_set_sensitive_actions(view->sensitive_actions);
	}
	
	session = (ViewAddingSession*)g_malloc0(sizeof(ViewAddingSession));
	session->view = view;
	session->source = source;
	session->speech_list = NULL;
	session->speech_activated = (view_get_item_count(view) > 0);
	session->timeout_id = g_timeout_add(ADDING_SESSION_TIMEOUT * 1000, on_adding_session_timeout, session);

	for (session->index = 0; session->index < session->view->items->len; session->index++) {
		item = g_array_index(session->view->items, ViewItem*, session->index);
		if (session->source == item->item_class->source) {
			break; 	
		}
	}

	view->adding_sessions = g_slist_prepend(view->adding_sessions, session);

	return session;
}

void view_adding_session_end(ViewAddingSession* session, gboolean successful, gboolean keep_old_items) {	
	if (session->view) {
		do_adding_session_end(session, successful, keep_old_items);
	}
	g_source_remove(session->timeout_id);
	g_free(session);
}

ViewItem* view_adding_session_get_item_and_remove_previous_items(ViewAddingSession* session, const gchar* unique_id) {
	GdkColor color;
	guint index, i;
	ViewItem* item;
	GSList* slist_item;
	ViewAddingSession* other_session;

	g_assert(session->view != NULL);

	color.red = color.green = color.blue = 0x0000;
	for (index = session->index; index < session->view->items->len; index++) {
		item = g_array_index(session->view->items, ViewItem*, index);
		if (session->source == item->item_class->source && !strcmp(unique_id, item->unique_id)) {
			for (i = session->index; i < index; i++) {
				if (g_array_index(session->view->items, ViewItem*, session->index)->item_class->source == session->source) {
					remove_item(session->view, session->index);
					
					for (slist_item = session->view->adding_sessions; slist_item; slist_item = g_slist_next(slist_item)) {
						other_session = (ViewAddingSession*)slist_item->data;
						if (other_session != session && other_session->index > session->index) {
							other_session->index--;
						}
					}
				} else {
					session->index++;
				}
			}
			
			if (item->flags & VIEW_ITEM_FLAG_OLD) {
				item->flags &= ~VIEW_ITEM_FLAG_OLD;
				gtk_widget_modify_fg(item->label, GTK_STATE_NORMAL, &color);
			}
			
			break;
		}
	}
	
	g_source_remove(session->timeout_id);
	session->timeout_id = g_timeout_add(ADDING_SESSION_TIMEOUT * 1000, on_adding_session_timeout, session);
		
	if (index >= session->view->items->len) {

		return NULL;
	} else {	
		session->index++;

		return item;
	}
}

View* view_adding_session_get_view(ViewAddingSession* session) {

	return session->view;
}

gboolean view_adding_session_is_valid(ViewAddingSession* session) {

	return session->view != NULL;
}

void view_item_update_texts(ViewItem* item, gchar* begin, gchar* end) {
	gboolean needs_updating;

	needs_updating = FALSE;
	if (begin != item->text_begin && strcmp(begin, item->text_begin)) {
		g_free(item->text_begin);
		item->text_begin = g_strdup(begin);
		needs_updating = TRUE;
	}
	if ((end && item->text_end && end != item->text_end && strcmp(end, item->text_end)) ||
	    (end && !item->text_end)) {
		g_free(item->text_end);
		item->text_end = g_strdup(end);
		needs_updating = TRUE;
	} else if (!end && item->text_end) {
		g_free(item->text_end);
		item->text_end = NULL;
		needs_updating = TRUE;
	}
	if (needs_updating) {
		update_text(item);
	}
}

/* Helper function for the next function. */
static void add_key_space_value_to_slist_sorted(gpointer key, gpointer value, gpointer user_data) {
	GSList** slist_pointer;
	gchar* s;
	
	slist_pointer = (GSList**)user_data;
	s = g_strdup_printf("%s %s", key, value);
	*slist_pointer = g_slist_insert_sorted(*slist_pointer, s, (GCompareFunc)strcmp);
}

GSList* view_get_marked_items() {
	GSList* slist = NULL;
	
	g_hash_table_foreach(marked_items, add_key_space_value_to_slist_sorted, &slist);
	
	return slist;
}

View* view_get_topmost_view() {
	return topmost_view;
}

View* view_get_view_by_unique_id(gchar* unique_id) {
	GList* item;
	
	if ((item = g_list_find_custom(views, unique_id, compare_unique_ids))) {
		return (View*)(item->data);
	}
	
	return NULL;
}

guint view_get_view_count() {
	return g_list_length(views);
}

static void add_item(View* view, ViewItem* item, guint index, ViewAddingSession* session) {
	GtkWidget* event_box;
	GtkWidget* label;
	GtkWidget* image;
	GtkWidget* frame;
	GdkColor color;
	PangoFontDescription* font;
	gboolean marked;
	GList* list;
	ViewItemSelection* selection;
	GtkWidget* menu_item;
	GSList* slist_item;
	ViewAddingSession* other_session;

	g_assert(view != NULL);
	
	item->flags = VIEW_ITEM_FLAG_NEW;
	item->view = view;

	item->event_box = miaouw_gesture_event_box_new();
	miaouw_scrolled_window_activate_scrolling(MIAOUW_SCROLLED_WINDOW(view->scrolled_window), item->event_box);
	g_signal_connect(item->event_box, "button-press-event", G_CALLBACK(on_button_press_event), item);
	g_signal_connect(item->event_box, "gesture-notify", G_CALLBACK(on_gesture_notify), item);
	g_signal_connect(item->event_box, "gesture-event", G_CALLBACK(on_gesture_event), item);
	item->hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(item->event_box), item->hbox);
	
	if (item->item_class->source->name) {
		event_box = miaouw_gesture_event_box_new();
		gtk_widget_modify_bg(event_box, GTK_STATE_NORMAL, &item->item_class->source->bg_color);
		label = gtk_label_new(item->item_class->source->name);
		gtk_box_pack_start(GTK_BOX(item->hbox), event_box, FALSE, FALSE, 0);
		gtk_widget_set_size_request(label, 18, -1);
		gtk_widget_modify_fg(label, GTK_STATE_NORMAL, &item->item_class->source->fg_color);
		gtk_label_set_angle(GTK_LABEL(label), 90.0);
		font = pango_font_description_new();
		pango_font_description_set_size(font, 10 * PANGO_SCALE);
		gtk_widget_modify_font(label, font);
		pango_font_description_free(font);
		gtk_container_add(GTK_CONTAINER(event_box), label);
	}
	
	if (item->avatar) {
		image = gtk_image_new_from_pixbuf(item->avatar);
		gtk_widget_set_size_request(image, 50, 50);
		gtk_misc_set_alignment(GTK_MISC(image), 0.5, 0.5);
		if (item->item_class->source->name) {
			frame = gtk_aspect_frame_new(NULL, 0.5, 0.5, 1, FALSE);
			gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
			gtk_container_set_border_width(GTK_CONTAINER(frame), 4);
			gtk_container_add(GTK_CONTAINER(frame), image);
			gtk_box_pack_start(GTK_BOX(item->hbox), frame, FALSE, FALSE, 0);
		} else {
			gtk_box_pack_start(GTK_BOX(item->hbox), image, FALSE, FALSE, 4);
		}
	}

	if (item->icon) {
		image = gtk_image_new_from_pixbuf(item->icon);
		gtk_misc_set_alignment(GTK_MISC(image), 0.5, 0.5);
		gtk_misc_set_padding(GTK_MISC(image), 2, 4);
		gtk_box_pack_start(GTK_BOX(item->hbox), image, FALSE, FALSE, 0);
	}

	item->label = gtk_label_new(NULL);
	color.red = color.green = color.blue = 0x0000;
	gtk_widget_modify_fg(item->label, GTK_STATE_NORMAL, &color);
	item->change_font_size_signal_id = g_signal_connect_swapped(view->window, "change-font-size",
	                                                            G_CALLBACK(gtk_widget_modify_font), item->label);

	/* A workaround for GTK+ bug causing markupped label to take too much vertical space */
	g_signal_connect_after(G_OBJECT(item->label), "size-allocate", G_CALLBACK(on_size_allocate), NULL);

	font = pango_font_description_new();
	pango_font_description_set_size(font, miaouw_window_get_font_size(MIAOUW_WINDOW(view->window)) * PANGO_SCALE);
	gtk_widget_modify_font(item->label, font);
	pango_font_description_free(font);
	gtk_label_set_line_wrap(GTK_LABEL(item->label), TRUE);
	pango_layout_set_wrap(gtk_label_get_layout(GTK_LABEL(item->label)), PANGO_WRAP_WORD_CHAR);
	gtk_misc_set_alignment(GTK_MISC(item->label), 0.0, 0.5);
	gtk_box_pack_start(GTK_BOX(item->hbox), item->label, TRUE, TRUE, 0);

	item->menu = gtk_menu_new();
	font = pango_font_description_new();
	pango_font_description_set_size(font, 20 * PANGO_SCALE);
	if (item->selections) {
		gtk_menu_shell_append(GTK_MENU_SHELL(item->menu), gtk_separator_menu_item_new());
		for (list = g_list_first(item->selections); list; list = g_list_next(list)) {
			selection = (ViewItemSelection*)list->data;
			if (selection) {
				menu_item = gtk_menu_item_new_with_label(selection->title);
				gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(menu_item)), font);
				g_signal_connect(menu_item, "activate", selection->callback, selection->data);
				gtk_menu_shell_append(GTK_MENU_SHELL(item->menu), menu_item);
				g_free(selection->title);
				g_free(selection);
				list->data = NULL;
			} else {
				gtk_menu_shell_append(GTK_MENU_SHELL(item->menu), gtk_separator_menu_item_new());
			}
		}
		g_list_free(item->selections);
		item->selections = NULL;
	}
	pango_font_description_free(font);

	update_marked_item_status(item);

	gtk_widget_tap_and_hold_setup(item->event_box, item->menu, NULL, 0);
	gtk_widget_show_all(item->menu);		
	gtk_widget_show_all(item->event_box);

	color.red = 0xffff;
	color.green = color.blue = 0xeeee;
	gtk_widget_modify_bg(item->event_box, GTK_STATE_NORMAL, &color);

	item->timeout_id = g_timeout_add_full(G_PRIORITY_DEFAULT_IDLE, 1000 * update_text(item) + 250, on_update_timeout, item, NULL);

	if (index < view->items->len) {
		miaouw_box_add_before(MIAOUW_BOX(view->vbox), item->event_box, g_array_index(view->items, ViewItem*, index)->event_box);
	} else {
		miaouw_box_add_after(MIAOUW_BOX(view->vbox), item->event_box, NULL);
	}
	g_array_insert_val(view->items, index, item);

	if (index == 0) {
		view->first_item_shaded = (view->first_item_shaded + 1) % 2;
	}

	for (slist_item = view->adding_sessions; slist_item; slist_item = g_slist_next(slist_item)) {
		other_session = (ViewAddingSession*)slist_item->data;
		if (other_session != session && other_session->index > index) {
			other_session->index++;
		}
	}
}

static gint compare_unique_ids(gconstpointer a, gconstpointer b) {	
	return strcmp(((View*)a)->unique_id, (gchar*)b);
}
static gboolean on_adding_session_timeout(gpointer user_data) {	
	ViewAddingSession* session;
	
	session = (ViewAddingSession*)user_data;
	gdk_threads_enter();
	if (session->view) {
		do_adding_session_end(session, FALSE, TRUE);
	}
	gdk_threads_leave();

	return FALSE;
}

static gboolean on_button_press_event(GtkWidget* widget, GdkEventButton* event, ViewItem* item) {
	static guint32 previous_button_press_time = 0;
	static gint previous_button_press_x = 0;
	static gint previous_button_press_y = 0;

	if (event->type == GDK_BUTTON_PRESS && item->item_class->clicked_callback && /* && !miaouw_scrolled_window_is_dragged(item->view->scrolled_window)) { */
	    event->time - previous_button_press_time >= DOUBLE_BUTTON_PRESS_TIME_MIN &&
	    event->time - previous_button_press_time <= DOUBLE_BUTTON_PRESS_TIME_MAX &&
	    !gtk_drag_check_threshold(widget, previous_button_press_x, previous_button_press_y, event->x, event->y)) {
		if (event->x < gdk_pixbuf_get_width(item->avatar) + 25) {
			item->item_class->clicked_callback(item->view, item, TRUE);
		} else {
			item->item_class->clicked_callback(item->view, item, FALSE);
		}	
	}

	previous_button_press_time = event->time;
	previous_button_press_x = event->x;
	previous_button_press_y = event->y;

	return FALSE;
}

static void do_adding_session_end(ViewAddingSession* session, gboolean successful, gboolean keep_old_items) {	
	static GdkPixbuf* pixbuf = NULL;
	GList* list_element;
	ViewItem* item;
	GdkColor color;
	time_t t;
	gboolean shaded;
	guint i;
	GSList* slist_item;
	ViewAddingSession* other_session;
	SessionInfo* session_info;

	g_assert(session->view != NULL);

	if (!pixbuf) {
		pixbuf = gdk_pixbuf_new_from_file("/usr/share/icons/hicolor/50x50/hildon/qgn_note_gene_syswarning.png", NULL);
	}
	
		if (!(session_info = (SessionInfo*)g_datalist_get_data(&session->view->session_infos, session->source->name))) {
		session_info = (SessionInfo*)g_malloc0(sizeof(SessionInfo));
		g_datalist_set_data_full(&session->view->session_infos, session->source->name, session_info, g_free);
	}
	if (successful) {
		if (session_info->failed_session_item) {
			for (i = 0; i < session->view->items->len; i++) {
				item = g_array_index(session->view->items, ViewItem*, i);
				if (item == session_info->failed_session_item) {
					remove_item(session->view, i);
					session_info->failed_session_item = NULL;

					for (slist_item = session->view->adding_sessions; slist_item; slist_item = g_slist_next(slist_item)) {
						other_session = (ViewAddingSession*)slist_item->data;
						if (other_session->index > i) {
							other_session->index--;
						}
					}

					break;
				}
			}
		}
		session_info->last_successful_session_timestamp = time(NULL);
	} else if (!session_info->failed_session_item) {
		session_info->failed_session_item = (ViewItem*)g_malloc0(sizeof(ViewItem));
		session_info->failed_session_item->item_class = &internal_item_class;
		session_info->failed_session_item->unique_id = g_strdup(session->source->name);
		if (session_info->last_successful_session_timestamp) {
			session_info->failed_session_item->text_begin = g_strconcat("Failed to update ", session->source->name, " items.\n<small>Last update ", NULL);
			session_info->failed_session_item->text_end = g_strdup(".</small>");
			session_info->failed_session_item->timestamp = session_info->last_successful_session_timestamp;
		} else {
			session_info->failed_session_item->text_begin = g_strconcat("Failed to update ", session->source->name, " items.", NULL);
		}
		session_info->failed_session_item->avatar = pixbuf;
		add_item(session->view, session_info->failed_session_item, 0, NULL);
	}

	for (list_element = g_list_first(session->speech_list); list_element; list_element = g_list_next(list_element)) {
		if (gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_SPEECH, NULL)) {
			text_to_speech_service_speak(list_element->data);
		}
		g_free(list_element->data);
	}
	g_list_free(session->speech_list);
	session->speech_list = NULL;

	t = time(NULL) - 60 * 60 * gconf_client_get_int(gconf, MAUKU_GCONF_KEY_KEEP_OLD_ITEMS, NULL);
	color.red = 0x7777;
	color.green = color.blue = 0x4444;
	while (session->index < session->view->items->len) {
		item = g_array_index(session->view->items, ViewItem*, session->index);
		if (item->item_class->source != session->source) {
			session->index++;
		} else if (keep_old_items && item->timestamp > t) {
			if (!(item->flags & VIEW_ITEM_FLAG_OLD)) {
				item->flags |= VIEW_ITEM_FLAG_OLD;
				gtk_widget_modify_fg(item->label, GTK_STATE_NORMAL, &color);
			}
			session->index++;
		} else {
			remove_item(session->view, session->index);

			for (slist_item = session->view->adding_sessions; slist_item; slist_item = g_slist_next(slist_item)) {
				other_session = (ViewAddingSession*)slist_item->data;
				if (other_session != session && other_session->index > session->index) {
					other_session->index--;
				}
			}
		}
	}
	
	for (i = 0; i < session->view->items->len; i++) {
		item = g_array_index(session->view->items, ViewItem*, i);
		shaded = (i + session->view->first_item_shaded) % 2;
		if (((item->item_class->source == session->source || item->item_class->source == &internal_item_source) ||
		     item->flags & VIEW_ITEM_FLAG_NEW) || (!(item->flags & VIEW_ITEM_FLAG_NEW) && item->shaded != shaded)) {
			item->flags &= ~VIEW_ITEM_FLAG_NEW;
			item->shaded = shaded;
			if (shaded) {
				color = g_array_index(session->view->items, ViewItem*, i)->item_class->color_shaded;
			} else {
				color = g_array_index(session->view->items, ViewItem*, i)->item_class->color_normal;
			}
			gtk_widget_modify_bg(item->event_box, GTK_STATE_NORMAL, &color);
		}				
	}
	
	session->view->adding_sessions = g_slist_remove(session->view->adding_sessions, session);
	if (!session->view->adding_sessions) {
		session->view->sensitive_actions |= UI_SENSITIVE_ACTION_UPDATE;	
		if (session->view == topmost_view) {
 			ui_set_sensitive_actions(session->view->sensitive_actions);
		}
	}
	session->view = NULL;
}

static void do_gesture_event(MiaouwGestureEventBox* event_box, MiaouwGestureEvent* event, View* view, ViewItem* item) {
	GdkColor* color_pointer;
	GtkAdjustment* adjustment;
	
	miaouw_scrolled_window_set_drag_scrolling(MIAOUW_SCROLLED_WINDOW(view->scrolled_window), TRUE);
	gdk_window_set_cursor(gtk_widget_get_parent_window(GTK_WIDGET(event_box)), NULL);

	if (item && !(item->flags & VIEW_ITEM_FLAG_NEW)) {
		if (item->shaded) {
			color_pointer = &item->item_class->color_shaded;
		} else {
			color_pointer = &item->item_class->color_normal;
		}
		gtk_widget_modify_bg(item->event_box, GTK_STATE_NORMAL, color_pointer);
	}

	if (event->gesture == MIAOUW_GESTURE_LEFT && event->previous_gesture == MIAOUW_GESTURE_NONE) {
		ui_close_view(view);
	} else if (event->gesture == MIAOUW_GESTURE_UP && event->previous_gesture == MIAOUW_GESTURE_LEFT) {
		adjustment = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(view->scrolled_window));
		gtk_adjustment_set_value(adjustment, adjustment->lower);
	} else if (event->gesture == MIAOUW_GESTURE_DOWN && event->previous_gesture == MIAOUW_GESTURE_LEFT) {
		adjustment = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(view->scrolled_window));
		gtk_adjustment_set_value(adjustment, adjustment->upper - adjustment->page_size);
	} else if (event->gesture == MIAOUW_GESTURE_UP && event->previous_gesture == MIAOUW_GESTURE_RIGHT) {
		ui_popup_menu();
	} else if (item && event->gesture == MIAOUW_GESTURE_DOWN && event->previous_gesture == MIAOUW_GESTURE_RIGHT) {
		gtk_menu_popup(GTK_MENU(item->menu), NULL, NULL, NULL, NULL, 0, gtk_get_current_event_time());
	}
}

static gboolean on_delete_event(GtkWidget* widget, GdkEvent* event, View* view) {
	ui_close_view(view);
	
	return TRUE;
}

static void on_gesture_event(MiaouwGestureEventBox* event_box, MiaouwGestureEvent* event, ViewItem* item) {
	do_gesture_event(event_box, event, item->view, item);

	if (event->gesture == MIAOUW_GESTURE_RIGHT && event->previous_gesture == MIAOUW_GESTURE_NONE &&
	    item->item_class->clicked_callback) { /* && !miaouw_scrolled_window_is_dragged(item->view->scrolled_window)) { */
		if (event->start_x < gdk_pixbuf_get_width(item->avatar) + 25) {
			item->item_class->clicked_callback(item->view, item, TRUE);
		} else {
			item->item_class->clicked_callback(item->view, item, FALSE);
		}
	}
}

static void on_gesture_notify(MiaouwGestureEventBox* event_box, MiaouwGestureEvent* event, ViewItem* item) {
	gboolean scrolling;
	GdkCursor* cursor;
	GdkColor color;
	GdkColor* color_pointer;

	scrolling = FALSE;
	cursor = NULL;
	color_pointer = NULL;

	if (event->gesture == MIAOUW_GESTURE_LEFT && event->previous_gesture == MIAOUW_GESTURE_NONE) {
		cursor = cursor_gesture_close;
	} else if (event->gesture == MIAOUW_GESTURE_RIGHT && event->previous_gesture == MIAOUW_GESTURE_NONE) {
		if (item && item->item_class->clicked_callback) {
			cursor = cursor_gesture_open;
			color.red = 0xffff;
			color.green = color.blue = 0xeeee;
			color_pointer = &color;
		}
	} else if (event->gesture == MIAOUW_GESTURE_UP && event->previous_gesture == MIAOUW_GESTURE_LEFT) {
		cursor = cursor_gesture_jump_top;
	} else if (event->gesture == MIAOUW_GESTURE_DOWN && event->previous_gesture == MIAOUW_GESTURE_LEFT) {
		cursor = cursor_gesture_jump_bottom;
	} else if (event->gesture == MIAOUW_GESTURE_UP && event->previous_gesture == MIAOUW_GESTURE_RIGHT) {
		cursor = cursor_gesture_window_menu;
		if (item && !(item->flags & VIEW_ITEM_FLAG_NEW)) {
			if (item->shaded) {
				color_pointer = &item->item_class->color_shaded;
			} else {
				color_pointer = &item->item_class->color_normal;
			}
		}
	} else if (item && event->gesture == MIAOUW_GESTURE_DOWN && event->previous_gesture == MIAOUW_GESTURE_RIGHT) {
		cursor = cursor_gesture_context_menu;
		if (item->shaded && !(item->flags & VIEW_ITEM_FLAG_NEW)) {
			color_pointer = &item->item_class->color_shaded;
		} else {
			color_pointer = &item->item_class->color_normal;
		}
	} else { /* MIAOUW_GESTURE_FAILED, for example. */
		scrolling = TRUE;
		if (item && !(item->flags & VIEW_ITEM_FLAG_NEW)) {
			if (item->shaded) {
				color_pointer = &item->item_class->color_shaded;
			} else {
				color_pointer = &item->item_class->color_normal;
			}
		}	
	}

	if (item) {
		miaouw_scrolled_window_set_drag_scrolling(MIAOUW_SCROLLED_WINDOW(item->view->scrolled_window), scrolling);
	}
	gdk_window_set_cursor(gtk_widget_get_parent_window(GTK_WIDGET(event_box)), cursor);
	if (color_pointer && item) {
		gtk_widget_modify_bg(item->event_box, GTK_STATE_NORMAL, color_pointer);
	}
}

static gboolean on_key_press_event(GtkWidget* widget, GdkEventKey* event, View* view) {
	GtkAdjustment* adjustment;
	GList* list;
	
	switch (event->keyval) {
		case HILDON_HARDKEY_FULLSCREEN:
			/* Hack to make the View resize itself. */
			gtk_widget_set_size_request(view->vbox, 100, -1);
			break;
		case HILDON_HARDKEY_ESC:

			return on_delete_event(view->window, NULL, view);
		case HILDON_HARDKEY_LEFT:
			if (!view->top_bottom_widget && (list = g_list_find(views, view)) && (list = g_list_next(list)) &&
			    GTK_WIDGET_VISIBLE(((View*)list->data)->window)) {
				gtk_window_present(GTK_WINDOW(((View*)list->data)->window));
			}
			break;
		case HILDON_HARDKEY_RIGHT:
			if (!view->top_bottom_widget && (list = g_list_find(views, view)) && (list = g_list_previous(list))) {
				gtk_window_present(GTK_WINDOW(((View*)list->data)->window));
			}
			break;
		case HILDON_HARDKEY_SELECT:
			adjustment = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(view->scrolled_window));
			gtk_adjustment_set_value(adjustment, adjustment->lower);			
			view_start_update(view, TRUE);
			break;
	}
	
	return FALSE;
}

/* Helper function for the next function. */
static void add_key_space_value_to_slist(gpointer key, gpointer value, gpointer user_data) {
	GSList** slist_pointer;
	gchar* s;
	
	slist_pointer = (GSList**)user_data;
	s = g_strdup_printf("%s %s", key, value);
	*slist_pointer = g_slist_prepend(*slist_pointer, s);
}

static void on_mark_menu_item_activate(GtkMenuItem* menu_item, gpointer user_data) {
	ViewItem* item;
	GSList* slist;

	item = (ViewItem*)user_data;
	if (item->marked_image) {
		g_hash_table_remove(marked_items, (item->referred_unique_id ? item->referred_unique_id : item->unique_id));
	} else {
		g_hash_table_insert(marked_items, g_strdup(item->referred_unique_id ? item->referred_unique_id : item->unique_id), item->url);
	}

	slist = NULL;
	g_hash_table_foreach(marked_items, add_key_space_value_to_slist, &slist);
	gconf_client_set_list(gconf, MAUKU_GCONF_KEY_MARKED_ITEMS, GCONF_VALUE_STRING, slist, NULL);
	g_slist_foreach(slist, (GFunc)g_free, NULL);
	g_slist_free(slist);

	/* Hack: Caching is not working properly. */
	gconf_client_suggest_sync(gconf, NULL);
	gconf_client_clear_cache(gconf);
}

static void on_marked_items_notify(GConfClient* client, guint cnxn_id, GConfEntry* entry, gpointer user_data) {
	GSList* slist;
	GSList* slist_item;
	GList* list;
	View* view;
	gchar* s;
	gint i;

	if (marked_items) {
		g_hash_table_destroy(marked_items);
	}
	marked_items = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
	slist = gconf_client_get_list(gconf, MAUKU_GCONF_KEY_MARKED_ITEMS, GCONF_VALUE_STRING, NULL);
	for (slist_item = slist; slist_item; slist_item = g_slist_next(slist_item)) {
		if ((s = strchr(slist_item->data, ' '))) {
			*s = 0;
			s++;
		} else {
			s = "";
		}
		g_hash_table_insert(marked_items, g_strdup(slist_item->data), g_strdup(s));
		g_free(slist_item->data);

	}
	g_slist_free(slist);

	for (list = g_list_first(views); list; list = g_list_next(list)) {
		view = (View*)list->data;
		for (i = 0; i < view->items->len; i++) {
			update_marked_item_status(g_array_index(view->items, ViewItem*, i));
		}
	}
}

static void on_show_scrollbar_notify(GConfClient* client, guint cnxn_id, GConfEntry* entry, gpointer user_data) {
	View* view;

	view = (View*)user_data;
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(view->scrolled_window), GTK_POLICY_NEVER, (gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_SHOW_SCROLLBAR, NULL) ? GTK_POLICY_ALWAYS : GTK_POLICY_NEVER));
	gtk_widget_set_size_request(view->vbox, 100, -1);
}

/* A workaround for GTK+ bug causing markupped label to take too much vertical space */
static void on_size_allocate(GtkWidget* widget, GtkAllocation* allocation, gpointer user_data) {		 
	gtk_widget_set_size_request(widget, allocation->width, -1);
}

static gboolean on_update_timeout(gpointer data) {
	ViewItem* item;

	gdk_threads_enter();
	item = (ViewItem*)data;
	g_source_remove(item->timeout_id);
	item->timeout_id = g_timeout_add_full(G_PRIORITY_DEFAULT_IDLE, 1000 * update_text(item) + 250, on_update_timeout, item, NULL);
	gdk_threads_leave();

	return FALSE;
}

static void on_view_gesture_event(MiaouwGestureEventBox* event_box, MiaouwGestureEvent* event, View* view) {
	do_gesture_event(event_box, event, view, NULL);
}

static gboolean on_visibility_notify_event(GtkWidget* widget, GdkEventVisibility* event, View* view) {
        if (event->state != GDK_VISIBILITY_FULLY_OBSCURED) {
		topmost_view = view;
		ui_set_sensitive_actions(view->sensitive_actions);
		ui_set_toolbar_hidden(view->top_bottom_widget ? TRUE : FALSE);
        }

        return FALSE;
}

static void remove_item(View* view, guint index) {
	ViewItem* item;

	item = g_array_index(view->items, ViewItem*, index);
	g_source_remove(item->timeout_id);
	g_signal_handler_disconnect(view->window, item->change_font_size_signal_id);

	if (item->menu) {
		gtk_widget_destroy(item->menu);
	}

	gtk_widget_destroy(item->event_box);	
	g_array_remove_index(view->items, index);
	if (item->item_class->destroy_callback) {
		item->item_class->destroy_callback(item);
	}
	g_free(item->unique_id);
	g_free(item->referred_unique_id);
	g_free(item->url);
	g_free(item->text_begin);
	g_free(item->text_end);
	g_free(item);
	
	if (index == 0) {
		view->first_item_shaded = (view->first_item_shaded + 1) % 2;
	}
}

/* t1 = t1 - t2; */
static void substract_time(struct tm* t1, struct tm* t2) {
	t1->tm_sec -= t2->tm_sec;
	if (t1->tm_sec < 0) {
		t1->tm_sec += 60;
		t1->tm_min -= 1;
	}
	t1->tm_min -= t2->tm_min;
	if (t1->tm_min < 0) {
		t1->tm_min += 60;
		t1->tm_hour -= 1;
	}
	t1->tm_hour -= t2->tm_hour;
	if (t1->tm_hour < 0) {
		t1->tm_hour += 24;
		t1->tm_mday -= 1;
	}
	t1->tm_mday -= t2->tm_mday;
	if (t1->tm_mday < 0) {
		t1->tm_mday += 30; /* FIXME */
		t1->tm_mon -= 1;
	}		
	t1->tm_mon -= t2->tm_mon;
	if (t1->tm_mon < 0) {
		t1->tm_mon += 12;
		t1->tm_year -= 1;
	}
	t1->tm_year -= t2->tm_year;
}

static void update_marked_item_status(ViewItem* item) {
	gboolean marked;
	PangoFontDescription* font;
	
	marked = (item->referred_unique_id && g_hash_table_lookup(marked_items, item->referred_unique_id)) ||
	         (!item->referred_unique_id && g_hash_table_lookup(marked_items, item->unique_id));

	if (marked && !item->marked_image) {
		if (item->mark_menu_item) {
			gtk_widget_destroy(item->mark_menu_item);
			item->mark_menu_item = NULL;
		}
		item->marked_image = gtk_image_new_from_icon_name("qgn_list_gene_favor", GTK_ICON_SIZE_MENU);
		gtk_misc_set_alignment(GTK_MISC(item->marked_image), 0.5, 0.5);
		gtk_misc_set_padding(GTK_MISC(item->marked_image), 2, 4);
		gtk_box_pack_start(GTK_BOX(item->hbox), item->marked_image, FALSE, FALSE, 0);
		gtk_widget_show(item->marked_image);
	} else if (!marked && item->marked_image) {
		if (item->mark_menu_item) {
			gtk_widget_destroy(item->mark_menu_item);
			item->mark_menu_item = NULL;
		}
		gtk_widget_destroy(item->marked_image);
		item->marked_image = NULL;
	}
	if (!item->mark_menu_item) {
		if (marked) {
			item->mark_menu_item = gtk_menu_item_new_with_label(item->referred_unique_id ? "Unmark a referred item" : "Unmark an item");
		} else {
			item->mark_menu_item = gtk_menu_item_new_with_label(item->referred_unique_id ? "Mark a referred item" : "Mark an item");
		}
		font = pango_font_description_new();
		pango_font_description_set_size(font, 20 * PANGO_SCALE);
		gtk_widget_modify_font(gtk_bin_get_child(GTK_BIN(item->mark_menu_item)), font);
		pango_font_description_free(font);
		g_signal_connect(item->mark_menu_item, "activate", G_CALLBACK(on_mark_menu_item_activate), item);
		gtk_menu_shell_prepend(GTK_MENU_SHELL(item->menu), item->mark_menu_item);
		gtk_widget_show(item->mark_menu_item);
	}
}

static guint update_text(ViewItem* item) {
	GString* string;
	time_t t;
	struct tm now;
	struct tm tm;
	guint secs_to_next = 30;
	
	if (item->text_end) {
		string = g_string_new(item->text_begin);	
		t = time(NULL);
		gmtime_r(&t, &now);
		gmtime_r(&item->timestamp, &tm);
		substract_time(&now, &tm);
		if (now.tm_year < 0) {
			g_string_append(string, "in future");
			secs_to_next = item->timestamp - t;
		} else if (now.tm_year > 0) {
			if (now.tm_mon > 0) {
				g_string_append_printf(string, "%d %s, %d %s ago", now.tm_year, (now.tm_year == 1 ? "year" : "years"), now.tm_mon, (now.tm_mon == 1 ? "month" : "months"));
			} else {
				g_string_append_printf(string, "%d %s ago", now.tm_year, (now.tm_year == 1 ? "year" : "years"));
			}
			secs_to_next = 24 * 60 * 60 - (now.tm_hour * 60 * 60 + now.tm_min * 60 + now.tm_sec);
		} else if (now.tm_mon > 0) {
			if (now.tm_mday > 0) {
				g_string_append_printf(string, "%d %s, %d %s ago", now.tm_mon, (now.tm_mon == 1 ? "month" : "months"), now.tm_mday, (now.tm_mday == 1 ? "day" : "days"));
			} else {
				g_string_append_printf(string, "%d %s ago", now.tm_mon, (now.tm_mon == 1 ? "month" : "months"));
			}
			secs_to_next = 24 * 60 * 60 - (now.tm_hour * 60 * 60 + now.tm_min * 60 + now.tm_sec);
		} else if (now.tm_mday > 0) {
			if (now.tm_hour > 0) {
				g_string_append_printf(string, "%d %s, %d %s ago", now.tm_mday, (now.tm_mday == 1 ? "day" : "days"), now.tm_hour, (now.tm_hour == 1 ? "hour" : "hours"));
			} else {
				g_string_append_printf(string, "%d days ago", now.tm_mday);
			}
			secs_to_next = 60 * 60 - (now.tm_min * 60 + now.tm_sec);
		} else if (now.tm_hour > 0) {
			if (now.tm_min > 0) {
				g_string_append_printf(string, "%d %s, %d %s ago", now.tm_hour, (now.tm_hour == 1 ? "hour" : "hours"), now.tm_min, (now.tm_min == 1 ? "minute" : "minutes"));
			} else {
				g_string_append_printf(string, "%d %s ago", now.tm_hour, (now.tm_hour == 1 ? "hour" : "hours"));
			}
			secs_to_next = 60 - now.tm_sec;
		} else if (now.tm_min > 0) {
			g_string_append_printf(string, "%d %s ago", now.tm_min, (now.tm_min == 1 ? "minute" : "minutes"));
			secs_to_next = 60 - now.tm_sec;
		} else {
			g_string_append(string, "a moment ago");
			secs_to_next = 60 - now.tm_sec;
		}

		g_string_append(string, item->text_end);	
		gtk_label_set_markup(GTK_LABEL(item->label), string->str);
		g_string_free(string, TRUE);
	} else {
		gtk_label_set_markup(GTK_LABEL(item->label), item->text_begin);
	}

	return secs_to_next;
}
