/*****************************************************************************
 *** Mauku - Jaiku client for Maemo devices 
 ***
 *** Email backend
 ***
 *** Copyright (c) 2007-2008 Henrik Hedberg <hhedberg@innologies.fi>
 ***
 *** Licensed under the Apache License, Version 2.0 (the "License");
 *** you may not use this file except in compliance with the License.
 *** You may obtain a copy of the License at
 ***
 ***     http://www.apache.org/licenses/LICENSE-2.0
 ***
 *** Unless required by applicable law or agreed to in writing, software
 *** distributed under the License is distributed on an "AS IS" BASIS,
 *** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *** See the License for the specific language governing permissions and
 *** limitations under the License.
 ***
 *****************************************************************************/

#include "config.h"


#include <glib.h>
#include <gtk/gtk.h>
#include <gconf/gconf-client.h>
#include <json.h>
#include <time.h>

#include "mauku.h"

ViewItemSource email_item_source = {
	"email",
	{ 0, 0xffff, 0xffff, 0xffff },
	{ 0, 0x6666, 0x6666, 0x6666 }
};

static ViewItemClass email_item_class = {
	&email_item_source,
	{ 0, 0xeeee, 0xeeee, 0xffff },
	{ 0, 0xdddd, 0xdddd, 0xffff },
	NULL,
	NULL
};

#if LIBTINYMAIL && LIBTINYMAIL_MAEMO

#include <tny.h>
#include "accountstore.h"
#include "folderobserver.h"

static void get_headers(TnyFolder *self, gboolean cancelled, TnyList *headers, GError *err, gpointer user_data) {
	static GdkPixbuf* pixbuf = NULL;
	View* view;
	gboolean exists;
	ViewAddingSession* session;
	TnyIterator* iter;
	TnyHeader* header;
	gchar* unique_id;
	ViewItem* item;
	gchar* from;
	gchar* subject;
	gchar* s;
	struct tm struct_tm;
	
	if (!pixbuf) {
		pixbuf = gdk_pixbuf_new_from_file("/usr/share/icons/hicolor/26x26/hildon/qgn_list_gene_mail.png", NULL);	
	}
	
	view = (View*)user_data;
	gdk_threads_enter();
	if ((exists = view_exists(view))) {
		session = view_begin_adding_session(view, &email_item_source);
	}
	gdk_threads_leave();
	if (!exists || !session) {
		return;
	}

	iter = tny_list_create_iterator(headers);
	while (!tny_iterator_is_done(iter)) {
		header = TNY_HEADER(tny_iterator_get_current(iter));
		if (!(tny_header_get_flags(header) & TNY_HEADER_FLAG_DELETED)) {
			unique_id = tny_header_dup_uid(header);

			gdk_threads_enter();
			if (view_adding_session_is_valid(session)) {
				if (view_adding_session_get_item_and_remove_previous_items(session, unique_id)) {
					g_free(unique_id);
				} else {
					gdk_threads_leave();

					from = tny_header_dup_from(header);
					subject = tny_header_dup_subject(header);

					if ((s = strchr(from, '<'))) {
						*s-- = 0;
						while (s > from && *s == ' ') {
							*s-- = 0;
						}
					}

					item = g_malloc0(sizeof(ViewItem));
					item->item_class = &email_item_class;
					item->unique_id = unique_id;
					item->avatar = pixbuf;
					item->text_begin = g_markup_printf_escaped("%s\n<small>Email from %s ", subject, from);
					item->text_end = g_strdup(".</small>");
					if (!(item->timestamp = tny_header_get_date_received(header))) {
						if (!(item->timestamp = tny_header_get_date_sent(header))) {
							item->timestamp = time(NULL);
						} 
					}

					g_free(from);
					g_free(subject);

					gdk_threads_enter();
					if (view_adding_session_is_valid(session)) {
						view_adding_session_add_item(session, item);
					}
				}
			}
			gdk_threads_leave();
		}

		tny_iterator_next(iter);
	}
	g_object_unref(iter);

	gdk_threads_enter();
	if (view_adding_session_is_valid(session)) {
		view_adding_session_end(session, TRUE, FALSE);
	}
	gdk_threads_leave();
}

static void on_folder_updated(MaukuTnyFolderObserver* folder_observer, TnyFolderChange* folder_change, gpointer user_data) {
	static gboolean is_first_time = TRUE;
	TnyFolderChangeChanged changed;
	TnyFolder* folder;
	TnyList* list;
	TnyIterator* iter;

	changed = tny_folder_change_get_changed(folder_change);
	printf("Change: %d\n", changed);
	if (is_first_time ||
	    changed & TNY_FOLDER_CHANGE_CHANGED_ADDED_HEADERS ||
	    changed & TNY_FOLDER_CHANGE_CHANGED_EXPUNGED_HEADERS) {
		folder = tny_folder_change_get_folder(folder_change);
		list = tny_simple_list_new();
		tny_folder_get_headers_async(folder, list, FALSE, get_headers, NULL, user_data);
		g_object_unref(list);
		g_object_unref(folder);
	}
	is_first_time = FALSE;
}

static void here() {
	printf("Here\n");
}

static void on_connection_status_changed(TnyAccount* account, gint status, gpointer user_data) {
	printf("Connection status: %d\n", status);
}

static gpointer update_thread(gpointer data) {
	static TnyAccountStore* account_store = NULL;
	static TnyFolder* folder = NULL;
	TnyDevice* device;
	TnyAccount* account;
	TnyList* list;
	TnyIterator* iter;
	MaukuTnyFolderObserver* folder_observer;

	if (!account_store) {
		account_store = TNY_ACCOUNT_STORE(mauku_tny_account_store_new());
	}
	device = tny_account_store_get_device(account_store);
	tny_device_force_offline(device);
	tny_device_force_online(device);
	g_object_unref(device);
	if (!folder) {
		list = tny_simple_list_new();
		tny_account_store_get_accounts(account_store, list, TNY_ACCOUNT_STORE_STORE_ACCOUNTS);
		iter = tny_list_create_iterator(list);
		if (tny_iterator_is_done(iter)) {
			printf("No accounts\n");

			return NULL;
		}
		account = TNY_ACCOUNT(tny_iterator_get_current(iter));
		g_signal_connect(account, "connection-status-changed", G_CALLBACK(on_connection_status_changed), data);
		g_object_unref(iter);
		g_object_unref(list);

		folder = tny_store_account_find_folder(TNY_STORE_ACCOUNT(account), "imap://hhedberg@tols16.oulu.fi/INBOX", NULL);
		if (!folder) {
			printf("No INBOX folder\n");

			return NULL;
		}
		folder_observer = mauku_tny_folder_observer_new();
		g_signal_connect(folder_observer, "folder-updated", G_CALLBACK(on_folder_updated), data);
		tny_folder_add_observer(folder, TNY_FOLDER_OBSERVER(folder_observer));
	}
	tny_folder_refresh_async(folder, here, NULL, NULL);
	
	return NULL;
}

void email_update_view(View* view) {
	if (http_is_connected()) {
		if (g_thread_create(update_thread, view, FALSE, NULL) == NULL) {
			/* TODO: Failed... */
		}
	}
}

#else

gboolean email_handle_marked_item(ViewAddingSession* session, MicrofeedHttp* http, gchar* unique_id, gchar* url) {
	
	return FALSE;
}

void email_update_view(View* view) {
}

#endif
