/*****************************************************************************
 *** Mauku - Jaiku client for Maemo devices 
 ***
 *** Copyright (c) 2007 Henrik Hedberg <hhedberg@innologies.fi>
 ***
 *** Licensed under the Apache License, Version 2.0 (the "License");
 *** you may not use this file except in compliance with the License.
 *** You may obtain a copy of the License at
 ***
 ***     http://www.apache.org/licenses/LICENSE-2.0
 ***
 *** Unless required by applicable law or agreed to in writing, software
 *** distributed under the License is distributed on an "AS IS" BASIS,
 *** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *** See the License for the specific language governing permissions and
 *** limitations under the License.
 ***
 *****************************************************************************/

#include "config.h"
 
#include <glib.h>
#include <gtk/gtk.h>
#include <gconf/gconf-client.h>
#include <json.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <string.h>
#include <stdlib.h>

#if HILDON == 1

#include <hildon/hildon.h>

#else

#include <hildon-widgets/hildon-banner.h>
#include <hildon-widgets/hildon-note.h>

#endif

#include "miaouwwindow.h"
#include "miaouwscrolledwindow.h"

#include "mauku.h"

typedef struct {
	const gchar* location;
	gboolean location_found;
} SaveLocationData;

static gboolean foreach_location(GtkTreeModel* model, GtkTreePath* path, GtkTreeIter* iter, gpointer user_data);
static void save_location(const gchar* location);

static GtkListStore* locations = NULL;
static GStaticMutex locations_mutex = G_STATIC_MUTEX_INIT;
static gchar* latest_message = NULL;
static GStaticMutex latest_message_mutex = G_STATIC_MUTEX_INIT;

void jaiku_change_location() {
	GtkWidget* note;
	GtkWidget* dialog;
	GtkWidget* label;
	GtkWidget* entry;
	GtkTreeIter iter;
	
	if (!gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_JAIKU_ENABLED, NULL)) {
		note = hildon_note_new_information(NULL, "Location is supported only for Jaiku.");
		gtk_dialog_run(GTK_DIALOG(note));
		gtk_widget_destroy(note);
	
		return;
	}
	
	if (!locations) {
		locations = gtk_list_store_new(1, G_TYPE_STRING);
      		gtk_list_store_append (locations, &iter);
		gtk_list_store_set (locations, &iter, 0, "Somewhere with Mauku", -1);
	}		

	dialog = gtk_dialog_new_with_buttons("Change location", NULL, GTK_DIALOG_MODAL|GTK_DIALOG_DESTROY_WITH_PARENT,
	                                     GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);

	label = gtk_label_new("Type your new location here:");
	gtk_misc_set_alignment(GTK_MISC(label), 0.5, 1.0);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), label, FALSE, FALSE, 0);
	entry = gtk_combo_box_entry_new_with_model(GTK_TREE_MODEL(locations), 0);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), entry, FALSE, FALSE, 16);

	gtk_widget_show_all(dialog);
	if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {
		jaiku_set_location(gtk_combo_box_get_active_text(GTK_COMBO_BOX(entry)));
	}

	gtk_widget_destroy(dialog);
}
void jaiku_set_location(const gchar* location) {
	gchar* name = NULL;
	gchar* key = NULL;
	GtkWidget* note;
	MicrofeedHttp* http;
	GString* string;
	char buffer[1024];
	size_t len;
	
	if (network_make_connected() && (http = microfeed_http_new())) {
/*	    	jaiku_start_login(); */
		snprintf(buffer, 1024, "%s/.mauku/cookies", (getenv("HOME") ? getenv("HOME") : "/home/user"));
		microfeed_http_set_cookie_file(http, buffer);
		ui_set_location(location);
		save_location(location);
		
		string = g_string_new("user=");
		g_string_append(string, name);
		g_string_append(string, "&personal_key=");
		g_string_append(string, key);
		g_static_mutex_lock(&latest_message_mutex);
		if (latest_message) {
			g_string_append(string, "&message=");
			g_string_append(string, latest_message);
		}	
		g_static_mutex_unlock(&latest_message_mutex);
		g_string_append(string, "&location=");
		g_string_append(string, location);
		snprintf(buffer, 1024, "http://jaiku.com/?user=%s&personal_key=%s", name, key);
		if (!microfeed_http_post_data(http, buffer, &len, string->str)) {
			note = hildon_note_new_information(NULL, "Could not send a new Jaiku.");
			gtk_dialog_run(GTK_DIALOG(note));
			gtk_widget_destroy(note);
		}			

		g_string_free(string, TRUE);
		microfeed_http_free(http);
	}
	
	g_free(name);
	g_free(key);
}

void jaiku_update_location(MicrofeedHttp* http) {
	gchar* name = NULL;
	GtkWidget* note;
	char buffer[1024];
	JsonObject* presence;
	JsonObject* location;
	JsonObject* object;
	GString* string;
	gboolean comma;
	
	if (!(name = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, NULL)) || !*name) {

		return;
	}
	
	snprintf(buffer, 1024, "http://www.jaiku.com/user/%s/presence/json",  name);
	if ((presence = microfeed_http_get_json_object(http, buffer))) {
		if ((object = json_object_object_get(presence, "line"))) {
			g_static_mutex_lock(&latest_message_mutex);
			if (latest_message) {
				g_free(latest_message);
			}
			if (*json_object_get_string(object)) {
				latest_message = g_strdup(json_object_get_string(object));
			} else {
				latest_message = NULL;
			}
			g_static_mutex_unlock(&latest_message_mutex);
		}
		if ((location = json_object_object_get(presence, "location"))) {
			string = g_string_new("");
			comma = FALSE;
			if ((object = json_object_object_get(location, "neighbourhood")) &&
			    *json_object_get_string(object)) {
				g_string_append(string, json_object_get_string(object));
				comma = TRUE;
			}
			if ((object = json_object_object_get(location, "city")) &&
			    *json_object_get_string(object)) {
				if (comma) {
					g_string_append(string, ", ");
				}
				g_string_append(string, json_object_get_string(object));
				comma = TRUE;
			}
			if ((object = json_object_object_get(location, "country")) &&
			    *json_object_get_string(object)) {
				if (comma) {
					g_string_append(string, ", ");
				}
				g_string_append(string, json_object_get_string(object));
			}

			if (string->len > 0) {
				gdk_threads_enter();
				ui_set_location(string->str);
				save_location(string->str);
				gdk_threads_leave();
			} else {
				printf("*** No location string.\n");
			}

			g_string_free(string, TRUE);
		} else {
			printf("*** No location object.\n");
		}		

		json_object_put(presence);
	}
	
	g_free(name);
	g_free(key);
}

static gboolean foreach_location(GtkTreeModel* model, GtkTreePath* path, GtkTreeIter* iter, gpointer user_data) {
	gchar* location;
	SaveLocationData* data = (SaveLocationData*)user_data;
	
	gtk_tree_model_get(model, iter, 0, &location, -1);
	if (!strcmp(location, data->location)) {
		gtk_list_store_move_after(GTK_LIST_STORE(model), iter, NULL);
		data->location_found = TRUE;
	}
	g_free(location);
	
	return data->location_found;
}

static void save_location(const gchar* location) {
	GtkTreeIter iter;
	SaveLocationData data = { location, FALSE };

	g_static_mutex_lock(&locations_mutex);
	if (!locations) {
		locations = gtk_list_store_new(1, G_TYPE_STRING);
      		gtk_list_store_append(locations, &iter);
		gtk_list_store_set (locations, &iter, 0, location, -1);
	} else {
		gtk_tree_model_foreach(GTK_TREE_MODEL(locations), foreach_location, &data);
		if (!data.location_found) {
		    	gtk_list_store_prepend(locations, &iter);
			gtk_list_store_set (locations, &iter, 0, location, -1);
		}
	}
	g_static_mutex_unlock(&locations_mutex);
}
