/*****************************************************************************
 *** Mauku - Jaiku client for Maemo devices 
 ***
 *** Generic backend functionality
 ***
 *** Copyright (c) 2007-2008 Henrik Hedberg <hhedberg@innologies.fi>
 ***
 *** Licensed under the Apache License, Version 2.0 (the "License");
 *** you may not use this file except in compliance with the License.
 *** You may obtain a copy of the License at
 ***
 ***     http://www.apache.org/licenses/LICENSE-2.0
 ***
 *** Unless required by applicable law or agreed to in writing, software
 *** distributed under the License is distributed on an "AS IS" BASIS,
 *** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *** See the License for the specific language governing permissions and
 *** limitations under the License.
 ***
 *****************************************************************************/

#include "config.h"

#include "mauku.h"

static void* update_marked_items_thread(void* data);
static void* update_thread(void* args);

typedef struct {
	View* view;
	gchar* url;
	gchar* userpass;
	ViewItemSource* item_source;
	gboolean keep_old_items;
	UpdateThreadCallback callback;
	InitializeHttpCallback initialize_http_callback;
	gpointer user_data;
} UpdateThreadData;
	
GList* backend_append_links_to_selections(GList* selections, gchar* text, GCallback callback) {
	gchar* start;
	gchar* end;
	gchar* temp;
	ViewItemSelection* selection;
	gboolean first = TRUE;
	
	end = text;
	while (((start = strstr(end, "http://")) && start[7] && !g_ascii_isspace(start[7])) ||
	     ((start = strstr(end, "www.")) && start[4] && !g_ascii_isspace(start[4]))) {
		if (*start == 'h' && (temp = strstr(end, "www.")) && temp[4] && !g_ascii_isspace(temp[4]) && temp < start) {
			start = temp;
		} else if (*start == 'w' && (temp = strstr(end, "http://")) && temp[7] && !g_ascii_isspace(temp[7]) && temp < start) {
			start = temp;
		}
		for (end = start; *end; end++) {
			if (g_ascii_isspace(*end) || *end == ')') {
				break;
			}
		}
		while (end > start && *(end - 1) == '.') {
			end--;
		}
		selection = g_new0(ViewItemSelection, 1);
		selection->title = g_strndup(start, end - start);
		selection->callback = callback;
		selection->data = NULL;
		if (first) {
			first = FALSE;
			selections = g_list_append(selections, NULL);
		}
		selections = g_list_append(selections, selection);
	}
	
	return selections;
}

void backend_start_update(View* view, gchar* url, gchar* userpass, ViewItemSource* item_source, gboolean keep_old_items, UpdateThreadCallback callback, InitializeHttpCallback initialize_http_callback, gpointer user_data) {
	UpdateThreadData* data;

	data = (UpdateThreadData*)g_malloc0(sizeof(UpdateThreadData));
	data->view = view;
	data->url = g_strdup(url);
	data->userpass = g_strdup(userpass);
	data->item_source = item_source;
	data->keep_old_items = keep_old_items;
	data->callback = callback;
	data->initialize_http_callback = initialize_http_callback;
	data->user_data = user_data;

	if (g_thread_create(update_thread, data, FALSE, NULL) == NULL) {
		/* TODO: Failed... */
	}
}

void backend_update_marked_items_view(View* view) {
	if (g_thread_create(update_marked_items_thread, view, FALSE, NULL) == NULL) {
		/* TODO: Failed... */
	}
}

static void* update_marked_items_thread(void* data) {
	View* view;
	gboolean exists;
	ViewAddingSession* jaiku_session = NULL;
	ViewAddingSession* twitter_session = NULL;
	gchar* url;
	gchar* unique_id;
	MicrofeedHttp* http;
	GSList* slist;
	GSList* slist_item;
	gboolean jaiku_successful = TRUE;
	gboolean twitter_successful = TRUE;
	
	view = (View*)data;
	
	gdk_threads_enter();
	if ((exists = view_exists(view))) {
		jaiku_session = view_begin_adding_session(view, &jaiku_item_source);
		twitter_session = view_begin_adding_session(view, &twitter_item_source);
	}
	gdk_threads_leave();
	if (!exists) {	

		return;
	}

	if (network_is_connected() && (http = microfeed_http_new())) {
		slist = view_get_marked_items();
		for (slist_item = slist; slist_item; slist_item = g_slist_next(slist_item)) {
			if ((url = strchr(slist_item->data, ' '))) {
				unique_id = g_strndup(slist_item->data, url - ((gchar*)slist_item->data));
				printf("%s %s\n", unique_id, url + 1);
				if (g_str_has_prefix(unique_id, jaiku_item_source.name) &&
				    gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_JAIKU_ENABLED, NULL) &&
				    jaiku_successful && jaiku_session) {
					jaiku_successful = jaiku_handle_marked_item(jaiku_session, http, unique_id, url + 1);
				} else if (g_str_has_prefix(unique_id, twitter_item_source.name) &&
				           gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_TWITTER_ENABLED, NULL) &&
					   twitter_successful && twitter_session) {
					twitter_successful = twitter_handle_marked_item(twitter_session, http, unique_id, url + 1);
				}
				g_free(unique_id);
			}
		}
		microfeed_http_free(http);
	}
	
	gdk_threads_enter();
	if (jaiku_session) {
		view_adding_session_end(jaiku_session, jaiku_successful, FALSE);
	}
	if (twitter_session) {
		view_adding_session_end(twitter_session, twitter_successful, FALSE);
	}
	gdk_threads_leave();
}

static void* update_thread(void* args) {
	UpdateThreadData* data;
	gboolean exists;
	ViewAddingSession* session;
	MicrofeedHttp* http;
	JsonObject* feed;
	time_t delta;
	gboolean successful = FALSE;

	data = (UpdateThreadData*)args;

	/* In Linux, nice affects only a current thread, not a whole process. */
	nice(UPDATE_THREAD_NICE_INCREMENT);

	gdk_threads_enter();
	if ((exists = view_exists(data->view))) {
		session = view_begin_adding_session(data->view, data->item_source);
	}
	gdk_threads_leave();
	if (!exists || !session) {
	
		return;
	}

	if (network_is_connected() && (http = microfeed_http_new())) {
		if (data->userpass) {
			microfeed_http_set_basic_authentication(http, data->userpass);
		}
		if (data->initialize_http_callback) {
			data->initialize_http_callback(http, data->user_data);
		}
		if ((feed = microfeed_http_get_json_object(http, data->url))) {
			if (microfeed_http_get_server_time(http)) {
				delta = microfeed_http_get_server_time(http) - microfeed_http_get_reply_start_time(http);
			} else {
				delta = 0;
			}
			g_assert(data->callback != NULL);
			successful = data->callback(session, feed, http, delta, data->user_data);
			json_object_put(feed);
		}
		microfeed_http_free(http);
	}
	
	gdk_threads_enter();
	if (successful) {
		view_adding_session_end(session, TRUE, data->keep_old_items);
	} else {
		view_adding_session_end(session, FALSE, TRUE);
	}
	gdk_threads_leave();

	g_free(data->url);
	g_free(data->userpass);
	g_free(data);

	return NULL;
}
