/*****************************************************************************
 *** Mauku - Jaiku client for Maemo devices 
 ***
 *** Copyright (c) 2007 Henrik Hedberg <hhedberg@innologies.fi>
 ***
 *** Licensed under the Apache License, Version 2.0 (the "License");
 *** you may not use this file except in compliance with the License.
 *** You may obtain a copy of the License at
 ***
 ***     http://www.apache.org/licenses/LICENSE-2.0
 ***
 *** Unless required by applicable law or agreed to in writing, software
 *** distributed under the License is distributed on an "AS IS" BASIS,
 *** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *** See the License for the specific language governing permissions and
 *** limitations under the License.
 ***
 *****************************************************************************/

#ifndef INCLUDED_VIEW_H
#define INCLUDED_VIEW_H

#include "config.h"

#include <gtk/gtk.h>

/* Data types */

typedef enum {
	VIEW_ITEM_FLAG_NEW = 1 << 0,
	VIEW_ITEM_FLAG_OLD = 1 << 1
} ViewItemFlags;

typedef struct _View View;
typedef struct _ViewItem ViewItem;
typedef struct _ViewItemSource ViewItemSource;
typedef struct _ViewItemClass ViewItemClass;
typedef struct _ViewAddingSession ViewAddingSession;
typedef struct _ViewItemSelection ViewItemSelection;

struct _ViewItemSelection {
	gchar* title; // freed after the widget creation
	GCallback callback;
	gpointer data; // not freed by the view
};

struct _ViewItem {
	/* Public, filled by view_add_item caller */
	ViewItemClass* item_class;
	gchar* unique_id;
	gchar* referred_unique_id;
	gchar* url;
	gpointer user_data;
	GList* selections;
	GdkPixbuf* avatar;
	GdkPixbuf* icon;
	gchar* text_begin;
	gchar* text_end; /* or NULL if no automatic timestamp updating */
	time_t timestamp;
	gchar* speech;
	
	/* Private, not to be used outside View functions */
	ViewItemFlags flags;
	gboolean shaded;
	View* view;
	GtkWidget* event_box;
	GtkWidget* hbox;
	GtkWidget* label;
	GtkWidget* menu;
	GtkWidget* marked_image;
	GtkWidget* mark_menu_item;
	guint timeout_id;
	gulong change_font_size_signal_id;
};

struct _ViewItemSource {
	const gchar* name;
	GdkColor bg_color;
	GdkColor fg_color;
};

struct _ViewItemClass {
	ViewItemSource* source;
	GdkColor color_normal;
	GdkColor color_shaded;
	void (*clicked_callback)(View* view, ViewItem* item, gboolean was_avatar);
	void (*destroy_callback)(ViewItem*);
};

/* Constructor and destructor */

View* view_new(gchar* unique_id, gchar* title, UiSensitiveActionType sensitive_actions,
               void (*update_callback)(View* view), void (*destroy_callback)(View* view));
void view_destroy(View* view);

/* Member functions */

ViewAddingSession* view_begin_adding_session(View* view, ViewItemSource* source);
gboolean view_exists(View* view);
gint view_get_item_count(View* view);
gchar* view_get_unique_id(View* view);
gpointer view_get_user_data(View* view, const gchar* key);
gpointer view_get_user_data_by_if(View* view, const GQuark id);
gboolean view_get_visibility(View* view);
GtkWidget* view_get_window(View* view);
void view_set_top_bottom_widget(View* view, GtkWidget* widget, gboolean top);
void view_set_user_data(View* view, const gchar* key, gpointer value, GDestroyNotify destroy_func);
void view_set_user_data_by_if(View* view, const GQuark id, gpointer value, GDestroyNotify destroy_func);
void view_set_visibility(View* view, gboolean is_visible);
void view_start_update(View* view, gboolean user_initiated);

/* Adding session and view item specific functions */

void view_adding_session_add_item(ViewAddingSession* session, ViewItem* item);
void view_adding_session_end(ViewAddingSession* session, gboolean successful, gboolean keep_old_items);
ViewItem* view_adding_session_get_item_and_remove_previous_items(ViewAddingSession* session, const gchar* unique_id);
View* view_adding_session_get_view(ViewAddingSession* session);
gboolean view_adding_session_is_valid(ViewAddingSession* session);
void view_item_update_texts(ViewItem* item, gchar* begin, gchar* end);

/* Static functions */

GSList* view_get_marked_items();
View* view_get_topmost_view();
View* view_get_view_by_unique_id(gchar* unique_id);
guint view_get_view_count();

#endif
