/*****************************************************************************
 *** Mauku - Jaiku client for Maemo devices 
 ***
 *** Twitter backend
 ***
 *** Copyright (c) 2007-2008 Henrik Hedberg <hhedberg@innologies.fi>
 ***
 *** Licensed under the Apache License, Version 2.0 (the "License");
 *** you may not use this file except in compliance with the License.
 *** You may obtain a copy of the License at
 ***
 ***     http://www.apache.org/licenses/LICENSE-2.0
 ***
 *** Unless required by applicable law or agreed to in writing, software
 *** distributed under the License is distributed on an "AS IS" BASIS,
 *** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *** See the License for the specific language governing permissions and
 *** limitations under the License.
 ***
 *****************************************************************************/

#include "config.h"

#include <glib.h>
#include <gtk/gtk.h>
#include <gconf/gconf-client.h>
#include <json.h>
#include <time.h>

#include "mauku.h"

typedef struct {
	gchar* title;
	gchar* nick;
} TwitterData;

static gint compare_item_nicks(gconstpointer a, gconstpointer b);
static gchar* handle_json_item(JsonObject* json, ViewAddingSession* session, Http* http, TwitterMessageType type, time_t delta);
static void on_contact_clicked_callback(View* view, ViewItem* item, gboolean was_avatar);
static void on_tweet_menu_item_activate(GtkMenuItem* menu_item, gpointer user_data);
static void on_item_destroy_callback(ViewItem* item);
static void on_tweet_clicked_callback(View* view, ViewItem* item, gboolean was_avatar);
static ViewItem* parse_json_item(JsonObject* json, gchar* unique_id, Http* http, TwitterMessageType type, time_t delta);
static gboolean update_contacts_view_callback(ViewAddingSession* session, JsonObject* feed, Http* http, time_t delta, gpointer user_data);
static gboolean update_generic_view_callback(ViewAddingSession* session, JsonObject* feed, Http* http, time_t delta, gpointer user_data);
static void update_tweet_view(View* view);
static gboolean update_tweet_view_callback(ViewAddingSession* session, JsonObject* feed, Http* http, time_t delta, gpointer user_data);
static void update_user_view(View* view);
static gboolean update_user_view_callback(ViewAddingSession* session, JsonObject* feed, Http* http, time_t delta, gpointer user_data);

ViewItemSource twitter_item_source = {
	"twitter",
	{ 0, 0x3333, 0xcccc, 0xffff },
	{ 0, 0xffff, 0xffff, 0xffff }
};

static ViewItemClass twitter_item_class = {
	&twitter_item_source,
	{ 0, 0xeeee, 0xeeee, 0xffff },
	{ 0, 0xdddd, 0xdddd, 0xffff },
	on_tweet_clicked_callback,
	on_item_destroy_callback
};

static ViewItemClass contact_item_class = {
	&twitter_item_source,
	{ 0, 0xeeee, 0xeeee, 0xffff },
	{ 0, 0xdddd, 0xdddd, 0xffff },
	on_contact_clicked_callback,  NULL
};

gboolean twitter_handle_marked_item(ViewAddingSession* session, Http* http, gchar* unique_id, gchar* url) {
	gboolean retvalue = FALSE;
	gchar* name = NULL;
	gchar* password = NULL;
	gchar* userpass;
	gchar* real_url;
	JsonObject* feed;
	time_t delta;
	
	real_url = g_strconcat("http://twitter.com/statuses/show/", unique_id + 8, ".json", NULL);
	if ((name = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_TWITTER_NAME, NULL)) && *name &&
	    (password = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_TWITTER_PASSWORD, NULL)) && *password) {
		userpass = g_strdup_printf("%s:%s", name, password);
		if ((feed = http_get_json_object(http, real_url))) {
			http_set_basic_authentication(http, NULL);

			if (http_get_server_time(http)) {
				delta = http_get_server_time(http) - http_get_reply_start_time(http);
			} else {
				delta = 0;
			}
			if (handle_json_item(feed, session, http, TWITTER_ITEM_TYPE_TWEET, delta)) {
				retvalue = TRUE;
			}
			json_object_put(feed);
		} else {
			http_set_basic_authentication(http, NULL);
		}
	}
	g_free(name);
	g_free(password);
	g_free(real_url);

	return retvalue;
}

gboolean twitter_send(const gchar* text) {
	gboolean retvalue = FALSE;
	Http* http;
	gchar* name = NULL;
	gchar* password = NULL;
	gchar* userpass;
	JsonObject* object;
	gchar* s;
	gchar* data;
	
	if (http_make_connected() && (http = http_new())) {
		if ((name = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_TWITTER_NAME, NULL)) && *name &&
		    (password = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_TWITTER_PASSWORD, NULL)) && *password) {
			userpass = g_strconcat(name, ":", password, NULL);
			http_set_basic_authentication(http, userpass);
			g_free(userpass);
			
			s = http_url_encode(http, text);
			data = g_strconcat("status=", s, "&source=mauku", NULL);
			http_free(http, s);
			if ((object = http_post_json_object(http, "http://twitter.com/statuses/update.json", data))) {
				retvalue = TRUE;
			}
			g_free(data);
		}
		http_destroy(http);
	}
	
	g_free(name);
	g_free(password);
	
	return retvalue;
}

void twitter_update_contacts_view(View* view) {
	gchar* name = NULL;
	gchar* password = NULL;
	gchar* userpass;
	gchar* credientials;
	
	if ((name = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_TWITTER_NAME, NULL)) && *name &&
	    (password = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_TWITTER_PASSWORD, NULL)) && password) {
		userpass = g_strconcat(name, ":", password, NULL);
		backend_start_update(view, "http://twitter.com/statuses/friends.json", userpass, &twitter_item_source, TRUE, update_contacts_view_callback, NULL);
		g_free(userpass);
	}
	
	g_free(name);	
	g_free(password);
}

void twitter_update_explore_view(View* view) {
	backend_start_update(view, "http://twitter.com/statuses/public_timeline.json", NULL, &twitter_item_source, FALSE, update_generic_view_callback, NULL);
}

void twitter_update_generic_view(View* view) {
	gchar* name = NULL;
	gchar* password = NULL;
	gchar* userpass;
	
	if ((name = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_TWITTER_NAME, NULL)) && *name &&
	    (password = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_TWITTER_PASSWORD, NULL)) && *password) {
		userpass = g_strdup_printf("%s:%s", name, password);
		backend_start_update(view, (gchar*)view_get_user_data(view, "twitter:url"), userpass, &twitter_item_source, TRUE, update_generic_view_callback, NULL);
		g_free(userpass);
	}
	
	g_free(name);
	g_free(password);
}

static gint compare_item_nicks(gconstpointer a, gconstpointer b) {
	return strcmp(((ViewItem*)a)->unique_id, ((ViewItem*)b)->unique_id);
}

static gchar* handle_json_item(JsonObject* json, ViewAddingSession* session, Http* http, TwitterMessageType type, time_t delta) {
	gboolean valid;
	ViewItem* item;
	gchar* unique_id;
	JsonObject* object;
	gchar* s;
	GString* string;
	
	if ((object = json_object_object_get(json, "id"))) {
		unique_id = g_strconcat(twitter_item_source.name, ":", json_object_get_string(object), NULL);
	} else{
	
		return NULL;
	}
	
	gdk_threads_enter();
	if ((valid = view_adding_session_is_valid(session))) {
		item = view_adding_session_get_item_and_remove_previous_items(session, unique_id);
	}
	gdk_threads_leave();
	if (!valid) {
		g_free(unique_id);

		return "";
	}

	if (!item) {
		if ((item = parse_json_item(json, unique_id, http, type, delta))) {
			gdk_threads_enter();
			if (view_adding_session_is_valid(session)) {
				view_adding_session_add_item(session, item);
			}
			gdk_threads_leave();
		} else {

			return NULL;
		}
	}

	return item->unique_id;
}

static void on_contact_clicked_callback(View* view, ViewItem* item, gboolean was_avatar) {
	gchar* url;
	gchar* nick;
	View* new_view;

	url = g_strconcat("http://twitter.com/statuses/user_timeline/", item->unique_id, ".json", NULL);
	nick = g_strdup(item->unique_id);
	new_view = view_new(url, item->unique_id, UI_SENSITIVE_ACTION_ALL & (~UI_SENSITIVE_ACTION_SEND),
		 update_user_view, NULL);
	view_set_user_data(new_view, "twitter:nick", nick, g_free);
	view_start_update(new_view, TRUE);
	g_free(url);
}

static void on_item_destroy_callback(ViewItem* item) {
	TwitterData* data;
	
	data = (TwitterData*)item->user_data;
	g_free(data->title);
	g_free(data->nick);
	g_free(data);
}

static void on_link_menu_item_activate(GtkMenuItem* menu_item, gpointer user_data) {
	const gchar* url;
	osso_rpc_t retval;

	if (user_data) {
		url = (gchar*)user_data;
	} else {
		url = gtk_label_get_text(GTK_LABEL(gtk_bin_get_child(GTK_BIN(menu_item))));
	}
	osso_rpc_run(osso_context, "com.nokia.osso_browser", "/com/nokia/osso_browser", "com.nokia.osso_browser",
	             "open_new_window", &retval,  DBUS_TYPE_STRING, url, DBUS_TYPE_INVALID);
}

static void on_tweet_menu_item_activate(GtkMenuItem* menu_item, gpointer user_data) {
	gchar* url;
	gchar* nick;
	osso_rpc_t retval;

	nick = (gchar*)user_data;
	url = g_strdup_printf("http://twitter.com/%s", nick);
	osso_rpc_run(osso_context, "com.nokia.osso_browser", "/com/nokia/osso_browser", "com.nokia.osso_browser",
	             "open_new_window", &retval,  DBUS_TYPE_STRING, url, DBUS_TYPE_INVALID);
	g_free(url);
}

static void on_tweet_clicked_callback(View* view, ViewItem* item, gboolean was_avatar) {
	TwitterData* data;
	gchar* url;
	gchar* nick;
	View* new_view;
	
	data = (TwitterData*)item->user_data;
	nick = g_strdup(data->nick);
	if (was_avatar) {
		url = g_strconcat("http://twitter.com/statuses/user_timeline/", data->nick, ".json", NULL);
		new_view = view_new(url, data->nick, UI_SENSITIVE_ACTION_ALL & (~UI_SENSITIVE_ACTION_SEND),
		         update_user_view, NULL);
		view_set_user_data(new_view, "twitter:nick", nick, g_free);
		g_free(url);
	} else {
		new_view = view_new(item->url, data->title, UI_SENSITIVE_ACTION_ALL & (~UI_SENSITIVE_ACTION_SEND),
		         update_tweet_view, NULL);
		view_set_user_data(new_view, "twitter:nick", nick, g_free);
	}
	view_start_update(new_view, TRUE);
}

static ViewItem* parse_json_item(JsonObject* json, gchar* unique_id, Http* http, TwitterMessageType type, time_t delta) {
	ViewItem* item;
	GString* string;
	GString* speech;
	JsonObject* object;
	JsonObject* user;
	gchar* s;
	ViewItemSelection* selection;

	item = g_malloc0(sizeof(ViewItem));
	item->item_class = &twitter_item_class;
	item->unique_id = unique_id;
	item->user_data = g_malloc0(sizeof(TwitterData));

	string = g_string_new("");
	speech = g_string_new("");

	if ((object = json_object_object_get(json, "text"))) {
		s = json_object_get_string(object);
		((TwitterData*)item->user_data)->title = g_strdup(s);
		s = g_markup_escape_text(s, -1);
		g_string_append(string, s);
		if (speech) {
			g_string_append(speech, s);
		}
		g_free(s);
	}
	g_string_append(string, "\n<small>");
	if ((user = json_object_object_get(json, (type == TWITTER_ITEM_TYPE_RECEIVED_DM ? "sender" :
	                                          (type == TWITTER_ITEM_TYPE_SENT_DM ? "recipient" : "user"))))) {
		if ((object = json_object_object_get(user, "screen_name"))) {
			((TwitterData*)item->user_data)->nick = g_strdup(json_object_get_string(object));
			if (speech) {
				g_string_prepend(speech, ": ");
				g_string_prepend(speech, ((TwitterData*)item->user_data)->nick);
				g_string_prepend(speech, "Tweet from ");
			}
			g_string_append(string, (type == TWITTER_ITEM_TYPE_RECEIVED_DM ? "From " :
	                                          (type == TWITTER_ITEM_TYPE_SENT_DM ? "To " : "By ")));

			s = g_markup_escape_text(((TwitterData*)item->user_data)->nick, -1);
			g_string_append(string, s);
			g_free(s);

			selection = g_new0(ViewItemSelection, 1);
			selection->title = g_strdup_printf("Open %s in a browser", ((TwitterData*)item->user_data)->nick);
			selection->callback = G_CALLBACK(on_tweet_menu_item_activate);
			selection->data = ((TwitterData*)item->user_data)->nick;
			item->selections = g_list_append(item->selections, selection);

/*			selection = g_new0(ViewItemSelection, 1);
			selection->title = g_strdup("Open the tweet in a browser");
			selection->callback = G_CALLBACK(on_link_menu_item_activate);
			thread_selection = selection;
*/			/* thread_selection should be this one until ... */
/*			item->selections = g_list_append(item->selections, selection);
*/		}
		if ((object = json_object_object_get(user, "profile_image_url"))) {
			/* gdk_threads_enter(); */ /* Just for sure */
			item->avatar = image_cache_load_image(image_cache, http, json_object_get_string(object));
			/* gdk_threads_leave(); */
		}
	}
	
	if ((object = json_object_object_get(json, "id"))) {
		item->url = g_strconcat("http://twitter.com/statuses/show/", json_object_get_string(object), ".json", NULL);
	}

	g_string_append(string, " in Twitter ");

	if ((object = json_object_object_get(json, "created_at"))) {
		item->timestamp = http_parse_date(http, json_object_get_string(object)) - delta;
	} else {
		item->timestamp = time(NULL) - delta;
	}

	item->selections = backend_append_links_to_selections(item->selections, string->str, G_CALLBACK(on_link_menu_item_activate));
	item->text_begin = g_string_free(string, FALSE);
	item->text_end = g_strdup(".</small>");
	item->speech = g_string_free(speech, FALSE);

	return item;
}

static gboolean update_contacts_view_callback(ViewAddingSession* session, JsonObject* contacts, Http* http, time_t delta, gpointer user_data) {
	ViewItem* item;
	GList* list;
	GList* list_item;
	JsonObject* json;
	JsonObject* object;
	JsonObject* status;
	gchar* url;
	int i;
	GString* string;
	gchar* s;
	gboolean comma;
	ViewItem* existing;
	
	if (!json_object_is_type(contacts, json_type_array)) {
	
		return FALSE;
	}
	http_set_basic_authentication(http, NULL);

	list = NULL;
	for (i = 0; i < json_object_array_length(contacts); i++) {
		if (!(json = json_object_array_get_idx(contacts, i))) {
			break;
		}
		if (!(object = json_object_object_get(json, "screen_name"))) {
			continue;
		}
		string = g_string_new("<big>");
		item = g_malloc0(sizeof(ViewItem));
		item->item_class = &contact_item_class;
		
		item->unique_id = g_strdup(json_object_get_string(object));
		s = g_markup_escape_text(item->unique_id, -1);
		g_string_append(string, s);
		g_free(s);
		g_string_append(string, "</big> ");
		
		if ((object = json_object_object_get(json, "name"))) {
			s = g_markup_escape_text(json_object_get_string(object), -1);
			g_string_append(string, s);
			g_free(s);
		}
		g_string_append(string, "\n<small>");
		if ((object = json_object_object_get(json, "profile_image_url"))) {
			item->avatar = (GdkPixbuf*)json_object_get_string(object);
		}
		if ((status = json_object_object_get(json, "status"))) {
			if ((object = json_object_object_get(status, "text"))) {
				s = g_markup_escape_text(json_object_get_string(object), -1);
				g_string_append(string, s);
				g_free(s);
			}
			if ((object = json_object_object_get(status, "created_at"))) {
				item->timestamp = http_parse_date(http, json_object_get_string(object)) - delta;
			} else {
				item->timestamp = time(NULL) - delta;
			}
		} else {
			item->timestamp = time(NULL) - delta;
		}

		if ((object = json_object_object_get(json, "location"))) {
			g_string_append(string, " in ");
			s = g_markup_escape_text(json_object_get_string(object), -1);
			g_string_append(string, s);
			g_free(s);
		}
		g_string_append(string, " ");

		item->text_begin = g_string_free(string, FALSE);
		item->text_end = g_strdup(".</small>");

		list = g_list_prepend(list, item);
	}
	list = g_list_sort(list, compare_item_nicks);

	for (list_item = g_list_first(list); list_item; list_item = g_list_next(list_item)) {
		item = list_item->data;
		/* gdk_threads_enter(); */ /* Could be after image_cache_load_image, but just for sure here */
		item->avatar = image_cache_load_image(image_cache, http, (gchar*)item->avatar);

		gdk_threads_enter(); /* See two lines above! */
		if (view_adding_session_is_valid(session)) {
			if ((existing = view_adding_session_get_item_and_remove_previous_items(session, item->unique_id))) {
				view_item_update_texts(existing, item->text_begin, item->text_end);
			} else {
				view_adding_session_add_item(session, item);
			}
		}
		gdk_threads_leave();		
	}
	g_list_free(list);
		
	return TRUE;
}

static gboolean update_generic_view_callback(ViewAddingSession* session, JsonObject* feed, Http* http, time_t delta, gpointer user_data) {
	gboolean retvalue = TRUE;
	JsonObject* json;
	int i = 0;
	TwitterMessageType type;

	if (json_object_is_type(feed, json_type_array)) {
		if (!(type = (TwitterMessageType)view_get_user_data(view_adding_session_get_view(session), "twitter:type"))) {
			type = TWITTER_ITEM_TYPE_TWEET;
		}
		http_set_basic_authentication(http, NULL);
		for (i = 0; i < json_object_array_length(feed); i++) {
			if (!(json = json_object_array_get_idx(feed, i))) {
				retvalue = FALSE;
				break;
			}
			if (!handle_json_item(json, session, http, type, delta)) {
				retvalue = FALSE;
				break;
			}
		}
	} else {
		retvalue = FALSE;;
	}
	
	return retvalue;;
}

static void update_tweet_view(View* view) {
	gchar* name = NULL;
	gchar* password = NULL;
	gchar* userpass;
	
	if ((name = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_TWITTER_NAME, NULL)) && *name &&
	    (password = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_TWITTER_PASSWORD, NULL)) && *password) {
		userpass = g_strdup_printf("%s:%s", name, password);
		backend_start_update(view, view_get_unique_id(view), userpass, &twitter_item_source, TRUE, update_tweet_view_callback, NULL);
		g_free(userpass);
	}
	
	g_free(name);
	g_free(password);
}

static gboolean update_tweet_view_callback(ViewAddingSession* session, JsonObject* feed, Http* http, time_t delta, gpointer user_data) {
	gchar* referred_unique_id;
	JsonObject* stream;
	JsonObject* json;
	int i;

	if (!handle_json_item(feed, session, http, TWITTER_ITEM_TYPE_TWEET, delta)) {
		
		return FALSE;
	}
	
	return TRUE;
}

static void update_user_view(View* view) {
	gchar* name = NULL;
	gchar* password = NULL;
	gchar* userpass;
	
	if ((name = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_TWITTER_NAME, NULL)) && *name &&
	    (password = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_TWITTER_PASSWORD, NULL)) && *password) {
		userpass = g_strdup_printf("%s:%s", name, password);
		backend_start_update(view, view_get_unique_id(view), userpass, &twitter_item_source, TRUE, update_generic_view_callback, NULL);
		g_free(userpass);
	}
	
	g_free(name);
	g_free(password);
}
