/*****************************************************************************
 *** Mauku - Jaiku client for Maemo devices 
 ***
 *** GConf settings and settings dialog
 ***
 *** Copyright (c) 2007-2008 Henrik Hedberg <hhedberg@innologies.fi>
 ***
 *** Licensed under the Apache License, Version 2.0 (the "License");
 *** you may not use this file except in compliance with the License.
 *** You may obtain a copy of the License at
 ***
 ***     http://www.apache.org/licenses/LICENSE-2.0
 ***
 *** Unless required by applicable law or agreed to in writing, software
 *** distributed under the License is distributed on an "AS IS" BASIS,
 *** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *** See the License for the specific language governing permissions and
 *** limitations under the License.
 ***
 *****************************************************************************/

#include "config.h"
 
#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <errno.h>
#include <gconf/gconf-client.h>
#include <string.h>
#include <stdlib.h>

#if HILDON == 1

#include <gtk/gtkenums.h>

#else

#include <hildon-widgets/hildon-input-mode-hint.h>

#endif

#include <miaouw/miaouwscrolledwindow.h>

#include "mauku.h"

typedef struct {
	GtkWidget* jaiku_enabled_check_button;
	GtkWidget* jaiku_name_entry;
	GtkWidget* jaiku_password_entry; /* Must be after name_entry */
	GtkWidget* twitter_enabled_check_button;
	GtkWidget* twitter_name_entry;
	GtkWidget* twitter_password_entry; /* Must be after name_entry */
	GtkWidget* seconds_entry;
	GtkWidget* hours_entry;
	GtkWidget* blink_led_check_button;
	GtkWidget* backlight_check_button;
	GtkWidget* raise_window_check_button;
	GtkWidget* speech_check_button;
} SettingsDialogData;

static void append_login_tab_to_notebook(GtkNotebook* notebook, gchar* tab_title, gchar* url,
 gchar* name_gconf_key, gchar* password_gconf_key, gchar* enabled_gconf_key,
 GtkWidget** name_entry_pointer, GtkWidget** password_entry_pointer, GtkWidget** enabled_check_button_pointer, gulong* handler_id_pointer);
static void on_blink_led_notify(GConfClient* client, guint cnxn_id, GConfEntry *entry, gpointer user_data);
static void on_enabled_toggled(GtkToggleButton* toggle_button, gpointer user_data);
static gboolean on_focus_in_event(GtkEntry* entry, GdkEventFocus* event, gulong* handler_id);
static void on_join_button_clicked(GtkButton* button, gpointer user_data);
static void on_response(GtkDialog* dialog, gint response, SettingsDialogData* data);
static void on_update_notify(GConfClient* client, guint cnxn_id, GConfEntry *entry, gpointer user_data);
static gboolean on_update_timeout(gpointer data);

static guint timeout_id = 0;
static gboolean settings_dialog_active = FALSE;
GConfClient* gconf = NULL;

void settings_init() {
	GConfValue* value;
	gchar* s;
	gint version;
	
	gconf = gconf_client_get_default();
	gconf_client_add_dir(gconf, MAUKU_GCONF_DIR, GCONF_CLIENT_PRELOAD_RECURSIVE, NULL);

	if ((version = gconf_client_get_int(gconf, MAUKU_GCONF_KEY_CONFIGURATION_VERSION, NULL)) < 3) {
		gconf_client_set_int(gconf, MAUKU_GCONF_KEY_CONFIGURATION_VERSION, 3, NULL);
		if (version < 1) {
			gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_SHOW_SCROLLBAR, FALSE, NULL);
		}
		if (version < 2) {
			if ((value = gconf_client_get(gconf, "/blink_led", NULL))) {
				gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_BLINK_LED, gconf_value_get_bool(value), NULL);
				gconf_value_free(value);
				gconf_client_unset(gconf, "/blink_led", NULL);
			}
			if ((value = gconf_client_get(gconf, "/backlights", NULL))) {
				gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_BACKLIGHT, gconf_value_get_bool(value), NULL);
				gconf_value_free(value);
				gconf_client_unset(gconf, "/backlights", NULL);
			}
			if ((value = gconf_client_get(gconf, "/raise_window", NULL))) {
				gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_RAISE_WINDOW, gconf_value_get_bool(value), NULL);
				gconf_value_free(value);
				gconf_client_unset(gconf, "/raise_window", NULL);
			}
			if ((value = gconf_client_get(gconf, "/speech", NULL))) {
				gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_SPEECH, gconf_value_get_bool(value), NULL);
				gconf_value_free(value);
				gconf_client_unset(gconf, "/speech", NULL);
			}
		}
		if (version < 3) {
			if ((value = gconf_client_get(gconf, MAUKU_GCONF_DIR "/key", NULL))) {
				gconf_client_set(gconf, MAUKU_GCONF_KEY_JAIKU_KEY, value, NULL);
				gconf_value_free(value);
				gconf_client_unset(gconf, MAUKU_GCONF_DIR "/key", NULL);
			}
			if ((value = gconf_client_get(gconf, MAUKU_GCONF_DIR "/name", NULL))) {
				gconf_client_set(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, value, NULL);
				gconf_value_free(value);
				gconf_client_unset(gconf, MAUKU_GCONF_DIR "/name", NULL);
			}
			if ((value = gconf_client_get(gconf, MAUKU_GCONF_DIR "/password", NULL))) {
				gconf_client_set(gconf, MAUKU_GCONF_KEY_JAIKU_PASSWORD, value, NULL);
				gconf_value_free(value);
				gconf_client_unset(gconf, MAUKU_GCONF_DIR "/password", NULL);
			}
			gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_JAIKU_ENABLED, TRUE, NULL);
		}
	}

	if (!(value = gconf_client_get(gconf, MAUKU_GCONF_KEY_UPDATE, NULL))) {
		gconf_client_set_int(gconf, MAUKU_GCONF_KEY_UPDATE, 300, NULL);
	} else {
		gconf_value_free(value);
	}
	if (!(value = gconf_client_get(gconf, MAUKU_GCONF_KEY_KEEP_OLD_ITEMS, NULL))) {
		gconf_client_set_int(gconf, MAUKU_GCONF_KEY_KEEP_OLD_ITEMS, 12, NULL);
	} else {
		gconf_value_free(value);
	}
	if (!(value = gconf_client_get(gconf, MAUKU_GCONF_KEY_JAIKU_ENABLED, NULL))) {
		gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_JAIKU_ENABLED, FALSE, NULL);
	} else {
		gconf_value_free(value);
	}
	if (!(value = gconf_client_get(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, NULL))) {
		gconf_client_set_string(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, "", NULL);
	} else {
		gconf_value_free(value);
	}
	if (!(value = gconf_client_get(gconf, MAUKU_GCONF_KEY_JAIKU_PASSWORD, NULL))) {
		gconf_client_set_string(gconf, MAUKU_GCONF_KEY_JAIKU_PASSWORD, "", NULL);
	} else {
		gconf_value_free(value);
	}
	if (!(value = gconf_client_get(gconf, MAUKU_GCONF_KEY_TWITTER_ENABLED, NULL))) {
		gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_TWITTER_ENABLED, FALSE, NULL);
	} else {
		gconf_value_free(value);
	}
	if (!(value = gconf_client_get(gconf, MAUKU_GCONF_KEY_TWITTER_NAME, NULL))) {
		gconf_client_set_string(gconf, MAUKU_GCONF_KEY_TWITTER_NAME, "", NULL);
	} else {
		gconf_value_free(value);
	}
	if (!(value = gconf_client_get(gconf, MAUKU_GCONF_KEY_TWITTER_PASSWORD, NULL))) {
		gconf_client_set_string(gconf, MAUKU_GCONF_KEY_TWITTER_PASSWORD, "", NULL);
	} else {
		gconf_value_free(value);
	}
	if (!(value = gconf_client_get(gconf, MAUKU_GCONF_KEY_SHOW_SCROLLBAR, NULL))) {
		gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_SHOW_SCROLLBAR, FALSE, NULL);
	} else {
		gconf_value_free(value);
	}
	if (!(value = gconf_client_get(gconf, MAUKU_GCONF_KEY_SHOW_TOOLBAR, NULL))) {
		gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_SHOW_TOOLBAR, TRUE, NULL);
	} else {
		gconf_value_free(value);
	}
	if (!(value = gconf_client_get(gconf, MAUKU_GCONF_KEY_NEW_ITEM_NOTIFICATION, NULL))) {
		gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_NEW_ITEM_NOTIFICATION, TRUE, NULL);
	} else {
		gconf_value_free(value);
	}
	if (!(value = gconf_client_get(gconf, MAUKU_GCONF_KEY_BLINK_LED, NULL))) {
		gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_BLINK_LED, TRUE, NULL);
	} else {
		gconf_value_free(value);
	}
	if (!(value = gconf_client_get(gconf, MAUKU_GCONF_KEY_BACKLIGHT, NULL))) {
		gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_BACKLIGHT, TRUE, NULL);
	} else {
		gconf_value_free(value);
	}
	if (!(value = gconf_client_get(gconf, MAUKU_GCONF_KEY_RAISE_WINDOW, NULL))) {
		gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_RAISE_WINDOW, FALSE, NULL);
	} else {
		gconf_value_free(value);
	}
	if (!(value = gconf_client_get(gconf, MAUKU_GCONF_KEY_SPEECH, NULL))) {
		gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_SPEECH, TRUE, NULL);
	} else {
		gconf_value_free(value);
	}
	if (!(value = gconf_client_get(gconf, MAUKU_GCONF_KEY_MARKED_ITEMS, NULL))) {
		gconf_client_set_list(gconf, MAUKU_GCONF_KEY_MARKED_ITEMS, GCONF_VALUE_STRING, NULL, NULL);
	} else {
		gconf_value_free(value);
	}
	if (!(s = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, NULL)) || !*s ||
	    !(s = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_PASSWORD, NULL)) || !*s) {
		gconf_client_set_string(gconf, MAUKU_GCONF_KEY_JAIKU_KEY, "", NULL);
	}
	g_free(s);
	
	/* Hack: Caching is not working properly. */
	gconf_client_suggest_sync(gconf, NULL);
	gconf_client_clear_cache(gconf);

	gconf_client_notify_add(gconf, MAUKU_GCONF_KEY_BLINK_LED, on_blink_led_notify, NULL, NULL, NULL);
	gconf_client_notify(gconf, MAUKU_GCONF_KEY_BLINK_LED);

	gconf_client_notify_add(gconf, MAUKU_GCONF_KEY_UPDATE, on_update_notify, NULL, NULL, NULL);
	gconf_client_notify(gconf, MAUKU_GCONF_KEY_UPDATE);
}

void settings_cleanup() {
	g_object_unref(gconf);
}

void settings_show_dialog() {
	static gulong jaiku_handler_id;
	static gulong twitter_handler_id;

	SettingsDialogData* data;
	GtkWidget* dialog;
	GtkWidget* notebook;
	GtkWidget* vbox;
	GtkWidget* hbox;
	GtkWidget* vbox2;
	GtkWidget* label;
	int seconds;
	int hours;
	char buffer[100];
	gchar* s;
	
	if (settings_dialog_active) {
	
		return;
	}
	settings_dialog_active = TRUE;
	
	data = (SettingsDialogData*)g_malloc(sizeof(SettingsDialogData));

	dialog = gtk_dialog_new_with_buttons("Settings", NULL, GTK_DIALOG_MODAL|GTK_DIALOG_DESTROY_WITH_PARENT,
	                                     GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);
	notebook = gtk_notebook_new();
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), notebook, TRUE, TRUE, 0);

	append_login_tab_to_notebook(GTK_NOTEBOOK(notebook), "Jaiku", "http://m.jaiku.com/join",
	                             MAUKU_GCONF_KEY_JAIKU_NAME, MAUKU_GCONF_KEY_JAIKU_PASSWORD, MAUKU_GCONF_KEY_JAIKU_ENABLED,
				     &data->jaiku_name_entry, &data->jaiku_password_entry, &data->jaiku_enabled_check_button,
				     &jaiku_handler_id);
	append_login_tab_to_notebook(GTK_NOTEBOOK(notebook), "Twitter", "https://twitter.com/signup",
	                             MAUKU_GCONF_KEY_TWITTER_NAME, MAUKU_GCONF_KEY_TWITTER_PASSWORD, MAUKU_GCONF_KEY_TWITTER_ENABLED,
				     &data->twitter_name_entry, &data->twitter_password_entry, &data->twitter_enabled_check_button,
				     &twitter_handler_id);

	vbox = gtk_vbox_new(FALSE, 4);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox, gtk_label_new("Updating"));

	hbox = gtk_hbox_new(TRUE, 4);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 4);
	label = gtk_label_new("Update period (in seconds):");
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_container_add(GTK_CONTAINER(hbox), label);
	data->seconds_entry = gtk_entry_new();
#if HILDON == 1
	hildon_gtk_entry_set_input_mode(GTK_ENTRY(data->seconds_entry), HILDON_GTK_INPUT_MODE_NUMERIC);
#else
	g_object_set(G_OBJECT(data->seconds_entry), "input-mode", HILDON_INPUT_MODE_HINT_NUMERIC, NULL);
#endif
	seconds = gconf_client_get_int(gconf, MAUKU_GCONF_KEY_UPDATE, NULL);
	snprintf(buffer, 100, "%d", seconds);
	gtk_entry_set_text(GTK_ENTRY(data->seconds_entry), buffer);
	gtk_container_add(GTK_CONTAINER(hbox), data->seconds_entry);

	hbox = gtk_hbox_new(TRUE, 4);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 4);
	label = gtk_label_new("Keep old items (in hours):");
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_container_add(GTK_CONTAINER(hbox), label);
	data->hours_entry = gtk_entry_new();
#if HILDON == 1
	hildon_gtk_entry_set_input_mode(GTK_ENTRY(data->hours_entry), HILDON_GTK_INPUT_MODE_NUMERIC);
#else
	g_object_set(G_OBJECT(data->hours_entry), "input-mode", HILDON_INPUT_MODE_HINT_NUMERIC, NULL);
#endif
	hours = gconf_client_get_int(gconf, MAUKU_GCONF_KEY_KEEP_OLD_ITEMS, NULL);
	snprintf(buffer, 100, "%d", hours);
	gtk_entry_set_text(GTK_ENTRY(data->hours_entry), buffer);
	gtk_container_add(GTK_CONTAINER(hbox), data->hours_entry);

	hbox = gtk_hbox_new(TRUE, 4);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 4);
	label = gtk_label_new("New item notifications:");
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.0);
	gtk_container_add(GTK_CONTAINER(hbox), label);
	vbox2 = gtk_vbox_new(FALSE, 4);
	gtk_container_add(GTK_CONTAINER(hbox), vbox2);
	data->blink_led_check_button = gtk_check_button_new_with_label("Blink the LED");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data->blink_led_check_button), gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_BLINK_LED, NULL));
	gtk_container_add(GTK_CONTAINER(vbox2), data->blink_led_check_button);
	data->backlight_check_button = gtk_check_button_new_with_label("Turn the backlight on");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data->backlight_check_button), gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_BACKLIGHT, NULL));
	gtk_container_add(GTK_CONTAINER(vbox2), data->backlight_check_button);
	data->raise_window_check_button = gtk_check_button_new_with_label("Raise the window");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data->raise_window_check_button), gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_RAISE_WINDOW, NULL));
	gtk_container_add(GTK_CONTAINER(vbox2), data->raise_window_check_button);
	data->speech_check_button = gtk_check_button_new_with_label("Speak using flite");
	gtk_container_add(GTK_CONTAINER(vbox2), data->speech_check_button);
	if (!text_to_speech_service_is_available()) {
		gtk_widget_set_sensitive(data->speech_check_button, FALSE);
	} else {
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data->speech_check_button), gconf_client_get_bool(gconf, MAUKU_GCONF_KEY_SPEECH, NULL));
	}

	g_signal_connect(G_OBJECT(dialog), "response", G_CALLBACK(on_response), data);	
	gtk_widget_show_all(dialog);
}

static void append_login_tab_to_notebook(GtkNotebook* notebook, gchar* tab_title, gchar* url,
 gchar* name_gconf_key, gchar* password_gconf_key, gchar* enabled_gconf_key,
 GtkWidget** name_entry_pointer, GtkWidget** password_entry_pointer, GtkWidget** enabled_check_button_pointer, gulong* handler_id_pointer) {
	GtkWidget* vbox;
	GtkWidget* hbox;
	GtkWidget* label;
	GtkWidget* button;
	gchar* s;
	static gulong handler_id;

	vbox = gtk_vbox_new(FALSE, 4);
	gtk_notebook_append_page(notebook, vbox, gtk_label_new(tab_title));

	hbox = gtk_hbox_new(TRUE, 4);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 4);
	label = gtk_label_new("Enable account:");
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.0);
	gtk_container_add(GTK_CONTAINER(hbox), label);
	*enabled_check_button_pointer = gtk_check_button_new_with_label(tab_title);
	g_signal_connect(*enabled_check_button_pointer, "toggled", G_CALLBACK(on_enabled_toggled), name_entry_pointer);
	/* The state is set after the entries has been created... */
	gtk_container_add(GTK_CONTAINER(hbox), *enabled_check_button_pointer);

	hbox = gtk_hbox_new(TRUE, 4);	
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 4);
	label = gtk_label_new("Name:");
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_container_add(GTK_CONTAINER(hbox), label);
	*name_entry_pointer = gtk_entry_new();
#if HILDON == 1
	hildon_gtk_entry_set_input_mode(GTK_ENTRY(*name_entry_pointer), HILDON_GTK_INPUT_MODE_FULL);
#else
	g_object_set(G_OBJECT(*name_entry_pointer), "autocap", FALSE, NULL);
#endif
	if ((s = gconf_client_get_string(gconf, name_gconf_key, NULL))) {
		gtk_entry_set_text(GTK_ENTRY(*name_entry_pointer), s);
		g_free(s);
	}	
	gtk_container_add(GTK_CONTAINER(hbox), *name_entry_pointer);

	hbox = gtk_hbox_new(TRUE, 4);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 4);
	label = gtk_label_new("Password:");
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_container_add(GTK_CONTAINER(hbox), label);
	*password_entry_pointer = gtk_entry_new();
#if HILDON == 1
	hildon_gtk_entry_set_input_mode(GTK_ENTRY(*password_entry_pointer), HILDON_GTK_INPUT_MODE_FULL | HILDON_GTK_INPUT_MODE_INVISIBLE);
#else
	g_object_set(G_OBJECT(*password_entry_pointer), "autocap", FALSE, NULL);
#endif
	gtk_entry_set_visibility(GTK_ENTRY(*password_entry_pointer), FALSE);
	if ((s =  gconf_client_get_string(gconf, password_gconf_key, NULL))) {
		gtk_entry_set_text(GTK_ENTRY(*password_entry_pointer), s);
		g_free(s);
	}
	*handler_id_pointer = g_signal_connect(*password_entry_pointer, "focus-in-event", G_CALLBACK(on_focus_in_event), handler_id_pointer);
	gtk_container_add(GTK_CONTAINER(hbox), *password_entry_pointer);
	
	s = g_strdup_printf("Join %s (opens in a browser)", tab_title);
	button = gtk_button_new_with_label(s);
	g_free(s);
	g_signal_connect(button, "clicked", G_CALLBACK(on_join_button_clicked), url);
	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 4);

	/* ... here */
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(*enabled_check_button_pointer), gconf_client_get_bool(gconf, enabled_gconf_key, NULL));
	gtk_toggle_button_toggled(GTK_TOGGLE_BUTTON(*enabled_check_button_pointer));
}

static void on_blink_led_notify(GConfClient* client, guint cnxn_id, GConfEntry *entry, gpointer user_data) {
	if (gconf_value_get_bool(gconf_entry_get_value(entry))) {
		miaouw_window_set_led_pattern_name(MAUKU_LED_PATTERN_NAME);
	} else {
		miaouw_window_set_led_pattern_name(NULL);
	}
}

static void on_enabled_toggled(GtkToggleButton* toggle_button, gpointer user_data) {
	GtkWidget** widgets = (GtkWidget**)user_data;
	
	gtk_widget_set_sensitive(widgets[0], gtk_toggle_button_get_active(toggle_button));
	gtk_widget_set_sensitive(widgets[1], gtk_toggle_button_get_active(toggle_button));
}

static gboolean on_focus_in_event(GtkEntry* entry, GdkEventFocus* event, gulong* handler_id) {
	gtk_entry_set_text(entry, "");
	g_signal_handler_disconnect(entry, *handler_id);
	
	return FALSE;
}

static void on_join_button_clicked(GtkButton* button, gpointer user_data) {
	gchar* url;
	osso_rpc_t retval;

	url = (gchar*)user_data;
	osso_rpc_run(osso_context, "com.nokia.osso_browser", "/com/nokia/osso_browser", "com.nokia.osso_browser",
	             "open_new_window", &retval,  DBUS_TYPE_STRING, url, DBUS_TYPE_INVALID);
}

static void on_response(GtkDialog* dialog, gint response, SettingsDialogData* data) {
	osso_rpc_t retval;
	const gchar* text;
	gint integer;
	gchar* s;
	gboolean clear_key = FALSE;
	gboolean update_overview = FALSE;

	if (response != GTK_RESPONSE_REJECT) {
		gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_JAIKU_ENABLED, gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(data->jaiku_enabled_check_button)), NULL);
		
		text = gtk_entry_get_text(GTK_ENTRY(data->jaiku_name_entry));
		if (!(s = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, NULL)) || strcmp(text, s)) {
			gconf_client_set_string(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, text, NULL);
			clear_key = TRUE;
		}
		g_free(s);

		text = gtk_entry_get_text(GTK_ENTRY(data->jaiku_password_entry));
		if (!(s = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_PASSWORD, NULL)) || strcmp(text, s)) {
			gconf_client_set_string(gconf, MAUKU_GCONF_KEY_JAIKU_PASSWORD, text, NULL);
			clear_key = TRUE;
		}
		g_free(s);
		
		if (clear_key) {
			gconf_client_set_string(gconf, MAUKU_GCONF_KEY_JAIKU_KEY, "", NULL);
			jaiku_start_login();
		}

		gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_TWITTER_ENABLED, gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(data->twitter_enabled_check_button)), NULL);
		
		text = gtk_entry_get_text(GTK_ENTRY(data->twitter_name_entry));
		if (!(s = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_TWITTER_NAME, NULL)) || strcmp(text, s)) {
			gconf_client_set_string(gconf, MAUKU_GCONF_KEY_TWITTER_NAME, text, NULL);
			update_overview = TRUE;
		}
		g_free(s);

		text = gtk_entry_get_text(GTK_ENTRY(data->twitter_password_entry));
		if (!(s = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_TWITTER_PASSWORD, NULL)) || strcmp(text, s)) {
			gconf_client_set_string(gconf, MAUKU_GCONF_KEY_TWITTER_PASSWORD, text, NULL);
			update_overview = TRUE;
		}
		g_free(s);

		text = gtk_entry_get_text(GTK_ENTRY(data->seconds_entry));
		integer = atoi(text);
		if (integer < 0) {
			integer = 0;
		}
		gconf_client_set_int(gconf, MAUKU_GCONF_KEY_UPDATE, integer, NULL);
		
		text = gtk_entry_get_text(GTK_ENTRY(data->hours_entry));
		integer = atoi(text);
		if (integer < 0) {
			integer = 0;
		}
		gconf_client_set_int(gconf, MAUKU_GCONF_KEY_KEEP_OLD_ITEMS, integer, NULL);

		gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_BLINK_LED, gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(data->blink_led_check_button)), NULL);
		gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_BACKLIGHT, gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(data->backlight_check_button)), NULL);
		gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_RAISE_WINDOW, gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(data->raise_window_check_button)), NULL);
		if (GTK_WIDGET_IS_SENSITIVE(data->speech_check_button)) {
			gconf_client_set_bool(gconf, MAUKU_GCONF_KEY_SPEECH, gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(data->speech_check_button)), NULL);
		}

		/* Hack: Caching is not working properly. */
		gconf_client_suggest_sync(gconf, NULL);
		gconf_client_clear_cache(gconf);
	}

	gtk_widget_destroy(GTK_WIDGET(dialog));
	settings_dialog_active = FALSE;
	
	g_free(data);
	
	if (update_overview) {
		ui_start_update_overview();
	}
}

static void on_update_notify(GConfClient* client, guint cnxn_id, GConfEntry *entry, gpointer user_data) {
	gint seconds;
	
	if (timeout_id != 0) {
		g_source_remove(timeout_id);
	}
	if ((seconds = gconf_value_get_int(gconf_entry_get_value(entry)))) {
		timeout_id = g_timeout_add(seconds * 1000, on_update_timeout, NULL);
		/*on_update_timeout(NULL);*/
	}
}

static gboolean on_update_timeout(gpointer data) {
	gdk_threads_enter();
	ui_start_update_overview();
	gdk_threads_leave();
	
	return TRUE;
}
