/*****************************************************************************
 *** Mauku - Jaiku client for Maemo devices 
 ***
 *** Copyright (c) 2007 Henrik Hedberg <hhedberg@innologies.fi>
 ***
 *** Licensed under the Apache License, Version 2.0 (the "License");
 *** you may not use this file except in compliance with the License.
 *** You may obtain a copy of the License at
 ***
 ***     http://www.apache.org/licenses/LICENSE-2.0
 ***
 *** Unless required by applicable law or agreed to in writing, software
 *** distributed under the License is distributed on an "AS IS" BASIS,
 *** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *** See the License for the specific language governing permissions and
 *** limitations under the License.
 ***
 *****************************************************************************/

#include "config.h"

#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE /* strptime/time.h: glibc2 needs this */
#endif

#include <glib.h>
#include <gtk/gtk.h>
#include <gconf/gconf-client.h>
#include <json.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#if HILDON == 1

#include <hildon/hildon.h>

#else

#include <hildon-widgets/hildon-banner.h>
#include <hildon-widgets/hildon-note.h>

#endif

#include "miaouwwindow.h"
#include "miaouwscrolledwindow.h"

#include "mauku.h"

typedef struct {
	gchar* title;
	gchar* nick;
	gchar* channel;
} JaikuData;

static gint compare_item_nicks(gconstpointer a, gconstpointer b);
static gchar* do_login();
static void* login_thread(void* args);
static GString* get_comment_count(JsonObject* comments);
static gboolean get_posting_data(Http* http, gchar* url, gchar** nonce, gchar** location);
static gchar* handle_json_item(JsonObject* json, ViewAddingSession* session, Http* http, gboolean skip_comment, gchar* referred_unique_id, time_t delta);
static void on_contact_clicked_callback(View* view, ViewItem* item, gboolean was_avatar);
static void on_item_destroy_callback(ViewItem* item);
static void on_jaiku_clicked_callback(View* view, ViewItem* item, gboolean was_avatar);
static void on_channel_menu_item_activate(GtkMenuItem* menu_item, gpointer user_data);
static void on_jaiku_menu_item_activate(GtkMenuItem* menu_item, gpointer user_data);
static void on_link_menu_item_activate(GtkMenuItem* menu_item, gpointer user_data);
static ViewItem* parse_json_item(JsonObject* json, gchar* unique_id, Http* http, gchar* referred_unique_id, time_t delta);
static void parse_json_item_comment(JsonObject* json, ViewItem* item, Http* http, GString* string,  GString* speech);
static void parse_json_item_jaiku(JsonObject* json, ViewItem* item, Http* http, GString* string, GString* speech);
static gboolean update_contacts_view_callback(ViewAddingSession* session, JsonObject* feed, Http* http, time_t delta, gpointer user_data);
static void* update_marked_items_thread(void* data);
static gboolean update_overview_callback(ViewAddingSession* session, JsonObject* feed, Http* http, time_t delta, gpointer user_data);
static void update_thread_view(View* view);
static gboolean update_thread_view_callback(ViewAddingSession* session, JsonObject* feed, Http* http, time_t delta, gpointer user_data);
static void update_user_view(View* view);
static gboolean update_user_view_callback(ViewAddingSession* session, JsonObject* feed, Http* http, time_t delta, gpointer user_data);

ViewItemSource jaiku_item_source = {
	"jaiku",
	{ 0, 0x9696, 0xc6c6, 0x3f3f },
	{ 0, 0xffff, 0xffff, 0xffff }
};

static ViewItemClass comment_item_class = {
	&jaiku_item_source,
	{ 0, 0xffff, 0xffff, 0xdddd },
	{ 0, 0xffff, 0xffff, 0xbbbb },
	on_jaiku_clicked_callback,
	on_item_destroy_callback
};

static ViewItemClass jaiku_item_class = {
	&jaiku_item_source,
	{ 0, 0xeeee, 0xeeee, 0xffff },
	{ 0, 0xdddd, 0xdddd, 0xffff },
	on_jaiku_clicked_callback,
	on_item_destroy_callback
};

static ViewItemClass contact_item_class = {
	&jaiku_item_source,
	{ 0, 0xeeee, 0xeeee, 0xffff },
	{ 0, 0xdddd, 0xdddd, 0xffff },
	on_contact_clicked_callback,  NULL
};

gboolean jaiku_handle_marked_item(ViewAddingSession* session, Http* http, gchar* unique_id, gchar* url) {
	gboolean retvalue = FALSE;
	gchar* name = NULL;
	gchar* key = NULL;
	gchar* real_url = NULL;
	JsonObject* feed;
	time_t delta;
	
	gdk_threads_enter();
	if ((name = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, NULL)) && *name &&
	    (key = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_KEY, NULL)) && *key) {
		real_url = g_strconcat(url, "/json?user=", name, "&personal_key=", key, NULL);
	}
	gdk_threads_leave();
	
	if (real_url) {
		if ((feed = http_get_json_object(http, real_url))) {
			if (http_get_server_time(http)) {
				delta = http_get_server_time(http) - http_get_reply_start_time(http);
			} else {
				delta = 0;
			}
			if (handle_json_item(feed, session, http, FALSE, NULL, delta)) {
				retvalue = TRUE;
			}
			json_object_put(feed);
		}
		g_free(real_url);
	} else {
		retvalue = TRUE;
	}

	return retvalue;
}

void jaiku_start_login() {
	gchar* key = NULL;
	gchar* name = NULL;
	gchar* password = NULL;

	if ((!(key = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_KEY, NULL)) || !*key)) {
		if ((name = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, NULL)) && *name &&
		    (password = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_PASSWORD, NULL)) && *password) {
			if (http_is_connected()) {
				if (g_thread_create(login_thread, view_get_topmost_view(), FALSE, NULL) == NULL) {
					/* TODO: Failed... */
				}
			}
		}
	}
	g_free(key);
	g_free(name);
	g_free(password);
}

gboolean jaiku_send_comment(const gchar* url, const gchar* text) {
	gboolean retvalue = FALSE;
	gchar* name = NULL;
	gchar* key = NULL;
	Http* http;
	gchar* real_url;
	GString* string;
	gchar* nonce;
	gchar* location;
	gchar* s;
	size_t len;

	if (http_make_connected() && (http = http_new())) {
		if ((name = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, NULL)) && *name &&
		    (key = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_KEY, NULL)) && *key) {
			real_url = g_strconcat(url, "?user=", name, "&personal_key=", key, NULL);
			if (get_posting_data(http, real_url, &nonce, &location) && nonce) {
				string = g_string_new("user=");
				g_string_append(string, gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, NULL));
				g_string_append(string, "&personal_key=");
				g_string_append(string, gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_KEY, NULL));
				g_string_append(string, "&_nonce=");
				g_string_append(string, nonce);
				g_string_append(string, "&comment=");
				s = http_url_encode(http, text);
				g_string_append(string, s);
				http_free(http, s);

				if (http_post_data(http, url, &len, string->str)) {
					retvalue = TRUE;
				}			

				g_string_free(string, TRUE);
				g_free(nonce);
				g_free(location);
			}
			g_free(real_url);
		}
		http_destroy(http);
	}
	
	return retvalue;
}

gboolean jaiku_send_jaiku(const gchar* channel, const gchar* text, gint icon_id) {
	gboolean retvalue = FALSE;
	gchar* name = NULL;
	gchar* key = NULL;
	Http* http;
	gchar* url;
	GString* string;
	GtkWidget* note;
	gchar* s;
	gchar* nonce;
	gchar* location;
	JsonObject* object;
	
	if (http_make_connected() && (http = http_new())) {
		if ((name = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, NULL)) && *name &&
		    (key = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_KEY, NULL)) && *key) {
			url = g_strconcat("http://jaiku.com/?user=", name, "&personal_key=", key, NULL);
			if (get_posting_data(http, url, &nonce, &location)) {
				string = g_string_new("method=presence.send&user=");
				g_string_append(string, gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, NULL));
				g_string_append(string, "&personal_key=");
				g_string_append(string, gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_KEY, NULL));
				if (icon_id) {
					g_string_append(string, "&icon=");
					g_string_append_printf(string, "%d", icon_id);
				}
				g_string_append(string, "&location=");
				if (location) {
					s = http_url_encode(http, location);
					g_string_append(string, s);
					http_free(http, s);
				} else {
					g_string_append(string, "Out%20there%20with%20Mauku");
				}
				g_string_append(string, "&message=");
				if (channel) {
					g_string_append(string, "#");
					g_string_append(string, channel);
					g_string_append(string, "%20");
				}
				s = http_url_encode(http, text);
				g_string_append(string, s);
				http_free(http, s);

				if ((object = http_post_json_object(http, "http://api.jaiku.com/json", string->str))) {
					retvalue = TRUE;
				}			

				g_string_free(string, TRUE);
				g_free(nonce);
				g_free(location);
			}
			g_free(url);
		}
		http_destroy(http);
	}
	
	return retvalue;
}
	
void jaiku_update_contacts_view(View* view) {
	gchar* name = NULL;
	gchar* key = NULL;
	gchar* url;
	gchar* credientials;
	
	if ((name = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, NULL)) && *name &&
	    (key = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_KEY, NULL)) && *key) {
		url = g_strdup_printf("http://%s.jaiku.com/json?user=%s&personal_key=%s", name, name, key);
		backend_start_update(view, url, NULL, &jaiku_item_source, TRUE, update_contacts_view_callback, NULL);
		g_free(url);
	}
	
	g_free(name);	
	g_free(key);
}

void jaiku_update_explore_view(View* view) {
	backend_start_update(view, "http://jaiku.com/feed/json", NULL, &jaiku_item_source, FALSE, update_overview_callback, NULL);
}


void jaiku_update_overview(View* view) {
	gchar* name = NULL;
	gchar* key = NULL;
	gchar* url;
	
	if ((name = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, NULL)) && *name &&
	    (key = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_KEY, NULL)) && *key) {
		url = g_strdup_printf("http://%s.jaiku.com/contacts/feed/json?user=%s&personal_key=%s", name, name, key);
		backend_start_update(view, url, NULL, &jaiku_item_source, TRUE, update_overview_callback, NULL);
		g_free(url);
	}
	
	g_free(name);
	g_free(key);
}
void jaiku_update_your_jaikus_view(View* view) {
	gchar* name = NULL;
	gchar* key = NULL;
	gchar* url;
	
	if ((name = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, NULL)) && *name &&
	    (key = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_KEY, NULL)) && *key) {
		url = g_strdup_printf("http://%s.jaiku.com/feed/json?user=%s&personal_key=%s", name, name, key);
		backend_start_update(view, url, NULL, &jaiku_item_source, TRUE, update_user_view_callback, NULL);
		g_free(url);
	}
	
	g_free(name);
	g_free(key);
}

static gint compare_item_nicks(gconstpointer a, gconstpointer b) {
	return strcmp(((ViewItem*)a)->unique_id, ((ViewItem*)b)->unique_id);
}

static gchar* do_login() {
	gchar* key = NULL;
	gchar* name = NULL;
	gchar* password = NULL;
	gchar buffer[1024];
	Http* http;
	gchar* reply;
	size_t len;

	if ((name = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, NULL)) && *name &&
	    (password = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_PASSWORD, NULL)) && *password &&
	    (http = http_new())) {
		snprintf(buffer, 1024, "%s/.mauku/cookies", (getenv("HOME") ? getenv("HOME") : "/home/user"));
		http_set_cookie_file(http, buffer);
		snprintf(buffer, 1024, "redirect_to=http://api.jaiku.com/key&log=%s&pwd=%s", name, password);
		if ((reply = http_post_data(http, "http://jaiku.com/login?redirect_to=http://api.jaiku.com/key", &len, buffer)) &&
		    len < 80) {
			key = g_strndup(reply, len);
		}
		http_destroy(http);
	}
	g_free(name);
	g_free(password);
	
	return key;
}

static void* login_thread(void* args) {
	View* view;
	gchar* key;
	GtkWidget* note;
	
	view = (View*)args;
	
	gdk_threads_enter();
	note = hildon_note_new_information(NULL, "Please wait!\nLogging into Jaiku...");
	gtk_widget_show(note);
	gdk_threads_leave();

	key = do_login();

	gdk_threads_enter();
	gtk_widget_destroy(note);
	if (key && *key) {
		gconf_client_set_string(gconf, MAUKU_GCONF_KEY_JAIKU_KEY, key, NULL);

		/* Hack: Caching is not working properly. */
		gconf_client_suggest_sync(gconf, NULL);
		gconf_client_clear_cache(gconf);
		
		ui_start_update_overview();
	} else {
		gconf_client_set_string(gconf, MAUKU_GCONF_KEY_JAIKU_PASSWORD, "", NULL);

		/* Hack: Caching is not working properly. */
		gconf_client_suggest_sync(gconf, NULL);
		gconf_client_clear_cache(gconf);

		note = hildon_note_new_information(NULL, "Login failed.\nPlease, check your name and password.");
		gtk_dialog_run(GTK_DIALOG(note));
		gtk_widget_destroy(note);
		settings_show_dialog();
	}

	gdk_threads_leave();

	return NULL;
}

static GString* get_comment_count(JsonObject* comments) {
	GString* string;
	gint count;
	gchar* s;

	if (json_object_get_type(comments) == json_type_array) {
		if ((count = json_object_array_length(comments)) == 0) {
		
			return NULL;
		}
		s = g_strdup_printf("%d", count);
	} else if (*(s = json_object_get_string(comments)) != '0') {
		s = g_markup_escape_text(s, -1);
	} else {
	
		return NULL;
	}
	string = g_string_new(" (");
	g_string_append(string, s);
	if (s[0] == '1' && s[1] == 0) {
		g_string_append(string, " comment).</small>");
	} else {
		g_string_append(string, " comments).</small>");
	}
	g_free(s);

	return string;
}

static gboolean get_posting_data(Http* http, gchar* url, gchar** nonce, gchar** location) {
	gchar* page;
	gchar* start;
	gchar* end;
	size_t len;
	
	if ((page = http_get_data(http, url, &len)) &&
	    (page = strstr(page, "<form"))) {
	    	if ((start = strstr(page, "_nonce")) &&
		    (start = strstr(start, "value=")) &&
		    (end = strchr(start + 7, '"')) &&
		    start != end) {
			*end = 0;
			*nonce = g_strdup(start + 7);
		} else {
			*nonce = NULL;
		}
	    	if ((start = strstr(page, "location")) &&
		    (start = strstr(start, "value=")) &&
		    (end = strchr(start + 7, '"')) &&
		    start != end) {
			*end = 0;
			*location = g_strdup(start + 7);
		} else {
			*location = NULL;
		}		
	} else {
		
		return FALSE;
	}	
	
	return TRUE;
}

static gchar* handle_json_item(JsonObject* json, ViewAddingSession* session, Http* http, gboolean skip_comment, gchar* referred_unique_id, time_t delta) {
	gboolean valid;
	ViewItem* item;
	gchar* unique_id;
	JsonObject* object;
	gchar* s;
	GString* string;
	
	if ((object = json_object_object_get(json, "id"))) {
		if (referred_unique_id) {
			unique_id = g_strconcat(jaiku_item_source.name, ":c", json_object_get_string(object), NULL);
		} else {
			unique_id = g_strconcat(jaiku_item_source.name, ":", json_object_get_string(object), NULL);
		}
	} else if ((object = json_object_object_get(json, "comment_id"))) {
		unique_id = g_strconcat(jaiku_item_source.name, ":c", json_object_get_string(object), NULL);
	} else {
	
		return NULL;
	}
	
	gdk_threads_enter();
	if ((valid = view_adding_session_is_valid(session))) {
		item = view_adding_session_get_item_and_remove_previous_items(session, unique_id);
	}
	gdk_threads_leave();
	if (!valid) {
		g_free(unique_id);

		return "";
	}

	if (item) {
		g_free(unique_id);
		
		if (!skip_comment &&
		    (object = json_object_object_get(json, "comments")) &&
		    (string = get_comment_count(object))) {
			gdk_threads_enter();
			if (view_adding_session_is_valid(session)) {
				view_item_update_texts(item, item->text_begin, string->str);
			}
			gdk_threads_leave();
			g_string_free(string, TRUE);			
		}
	} else if ((item = parse_json_item(json, unique_id, http, referred_unique_id, delta))) {
		if (!skip_comment &&
		    (object = json_object_object_get(json, "comments")) &&
		    (string = get_comment_count(object))) {
			item->text_end = g_string_free(string, FALSE);			
		} else {
			item->text_end = g_strdup(".</small>");
		}
		gdk_threads_enter();
		if (view_adding_session_is_valid(session)) {
			view_adding_session_add_item(session, item);
		}
		gdk_threads_leave();		
	} else {
	
		return NULL;
	}

	return item->unique_id;
}

static void on_contact_clicked_callback(View* view, ViewItem* item, gboolean was_avatar) {
	gchar* url;
	View* new_view;

	if (*item->unique_id == '#') {
		url = g_strdup_printf("http://jaiku.com/channel/%s/feed/json", item->unique_id + 1);
	} else {
		url = g_strdup_printf("http://%s.jaiku.com/feed/json", item->unique_id);
	}
	new_view = view_new(url, item->unique_id, UI_SENSITIVE_ACTION_ALL & (~UI_SENSITIVE_ACTION_ADD_COMMENT),
	         update_user_view, NULL);
	view_start_update(new_view, TRUE);
	g_free(url);
}

static void on_item_destroy_callback(ViewItem* item) {
	JaikuData* data;
	
	data = (JaikuData*)item->user_data;
	g_free(data->title);
	g_free(data->nick);
	g_free(data->channel);
	g_free(data);
}

static void on_jaiku_clicked_callback(View* view, ViewItem* item, gboolean was_avatar) {
	JaikuData* data;
	gchar* url;
	View* new_view;
	
	data = (JaikuData*)item->user_data;
	if (was_avatar) {
		if (*data->nick == '#') {
			url = g_strdup_printf("http://jaiku.com/channel/%s/feed/json", data->nick + 1);
		} else {
			url = g_strdup_printf("http://%s.jaiku.com/feed/json", data->nick);
		}
		new_view = view_new(url, data->nick, UI_SENSITIVE_ACTION_ALL & (~UI_SENSITIVE_ACTION_ADD_COMMENT),
		         update_user_view, NULL);
		g_free(url);
	} else {
		new_view = view_new(item->url, data->title, UI_SENSITIVE_ACTION_ALL,
		         update_thread_view, NULL);
	}
	view_start_update(new_view, TRUE);
}

static void on_channel_menu_item_activate(GtkMenuItem* menu_item, gpointer user_data) {
	gchar* channel;
	gchar* url;
	View* view;
	
	channel = (gchar*)user_data;
	url = g_strdup_printf("http://jaiku.com/channel/%s/feed/json", channel);
	view = view_new(url, channel, UI_SENSITIVE_ACTION_ALL & ~UI_SENSITIVE_ACTION_ADD_COMMENT,
	         update_user_view, NULL);
	view_start_update(view, TRUE);
	g_free(url);
}
	
static void on_jaiku_menu_item_activate(GtkMenuItem* menu_item, gpointer user_data) {
	gchar* url;
	gchar* nick;
	osso_rpc_t retval;

	nick = (gchar*)user_data;
	if (*nick == '#') {
		url = g_strdup_printf("http://jaiku.com/channel/%s/", nick + 1);
	} else {
		url = g_strdup_printf("http://%s.jaiku.com/", nick);
	}
	osso_rpc_run(osso_context, "com.nokia.osso_browser", "/com/nokia/osso_browser", "com.nokia.osso_browser",
	             "open_new_window", &retval,  DBUS_TYPE_STRING, url, DBUS_TYPE_INVALID);
	g_free(url);
}

static void on_link_menu_item_activate(GtkMenuItem* menu_item, gpointer user_data) {
	const gchar* url;
	osso_rpc_t retval;

	if (user_data) {
		url = (gchar*)user_data;
	} else {
		url = gtk_label_get_text(GTK_LABEL(gtk_bin_get_child(GTK_BIN(menu_item))));
	}
	osso_rpc_run(osso_context, "com.nokia.osso_browser", "/com/nokia/osso_browser", "com.nokia.osso_browser",
	             "open_new_window", &retval,  DBUS_TYPE_STRING, url, DBUS_TYPE_INVALID);
}

static ViewItem* parse_json_item(JsonObject* json, gchar* unique_id, Http* http, gchar* referred_unique_id, time_t delta) {
	ViewItem* item;
	GString* string;
	GString* speech;
	JsonObject* object;
	gchar* s;
	struct tm struct_tm;

	item = g_malloc0(sizeof(ViewItem));
	item->unique_id = unique_id;
	item->user_data = g_malloc0(sizeof(JaikuData));

	string = g_string_new("");
	speech = g_string_new("");

	if (unique_id[6] == 'c') {
		item->item_class = &comment_item_class;
		parse_json_item_comment(json, item, http, string, speech);
	} else {
		if (referred_unique_id) {
			item->item_class = &comment_item_class;
			item->referred_unique_id = g_strdup(referred_unique_id); /* XXX: item->referred_unique_id ??? */
		} else {
			item->item_class = &jaiku_item_class;
		}
		parse_json_item_jaiku(json, item, http, string, speech);
	}

	if (!(object = json_object_object_get(json, "created_at")) ||
	    !(s = strptime(json_object_get_string(object), "%Y-%m-%dT%H:%M:%S GMT", &struct_tm)) || *s) {
		item->timestamp = time(NULL) - delta;
	} else {
		item->timestamp = timegm(&struct_tm) - delta;
	}

	item->selections = backend_append_links_to_selections(item->selections, string->str, G_CALLBACK(on_link_menu_item_activate));
	item->text_begin = g_string_free(string, FALSE);
	item->speech = g_string_free(speech, FALSE);

	return item;
}

static void parse_json_item_comment(JsonObject* json, ViewItem* item, Http* http, GString* string, GString* speech) {
	JsonObject* user;
	JsonObject* object;
	gchar* s;
	gchar* s2;
	ViewItemSelection* selection;
	
	if (!(object = json_object_object_get(json, "comment_id"))) {
		object = json_object_object_get(json, "id");
	}
	item->unique_id = g_strconcat(jaiku_item_source.name, ":c", json_object_get_string(object), NULL);
	if ((object = json_object_object_get(json, "entry_title"))) {
		((JaikuData*)item->user_data)->title = g_strdup(json_object_get_string(object));
	}
	if ((object = json_object_object_get(json, "url")) &&
	    (s = json_object_get_string(object))) {
		if (!(s2 = strchr(s, '#'))) {
			s2 = s + strlen(s);
		}
		item->url = g_strndup(s, s2 - s);
		if ((s = strrchr(s, '/'))) {
			s = g_strndup(s + 1, s2 - s - 1);
			item->referred_unique_id = g_strconcat(jaiku_item_source.name, ":", s, NULL);
			g_free(s);
		}
	}
	if ((object = json_object_object_get(json, "content"))) {
		s = json_object_get_string(object);
		s = g_markup_escape_text(s, -1);
		g_string_append(string, s);
		if (speech) {
			g_string_append(speech, s);
		}
		g_free(s);
	}
	g_string_append(string, "\n<small>");
	if ((user = json_object_object_get(json, "user"))) {
		if ((object = json_object_object_get(user, "nick"))) {
			((JaikuData*)item->user_data)->nick = g_strdup(json_object_get_string(object));
			if (speech) {
				g_string_prepend(speech, ": ");
				g_string_prepend(speech, ((JaikuData*)item->user_data)->nick);
				g_string_prepend(speech, "Comment from ");
			}
		} else {
			((JaikuData*)item->user_data)->nick = g_strdup("unknown");
		}
		if ((object = json_object_object_get(user, "avatar"))) {
			/* gdk_threads_enter(); */ /* Just for sure */
			item->avatar = image_cache_load_image(image_cache, http, json_object_get_string(object));
			/* gdk_threads_leave(); */
		}
	}
	if ((object = json_object_object_get(json, "title"))) {
		s = g_markup_escape_text(json_object_get_string(object), -1);
		g_string_append(string, s);
		g_free(s);
	} else {
		g_string_append(string, "Comment from ");
		s = g_markup_escape_text(((JaikuData*)item->user_data)->nick, -1);
		g_string_append(string, s);
		g_free(s);
	}
	if ((object = json_object_object_get(json, "url")) &&
	    (s = json_object_get_string(object)) &&
	    !strncmp(s, "http://jaiku.com/channel/", 25)) {
		s += 25;
		if (!(s2 = strchr(s, '/'))) {
			s2 = s + strlen(s);
		}
		((JaikuData*)item->user_data)->channel = g_strndup(s, s2 - s);
		selection = g_new0(ViewItemSelection, 1);
		selection->title = g_strdup_printf("Open #%s", ((JaikuData*)item->user_data)->channel);
		selection->callback = G_CALLBACK(on_channel_menu_item_activate);
		selection->data = ((JaikuData*)item->user_data)->channel;
		item->selections = g_list_append(item->selections, selection);
		item->selections = g_list_append(item->selections, NULL);
		g_string_append(string, " to #");
		g_string_append(string, ((JaikuData*)item->user_data)->channel);
	}
	
	selection = g_new0(ViewItemSelection, 1);
	selection->title = g_strdup_printf("Open %s in a browser", ((JaikuData*)item->user_data)->nick);
	selection->callback = G_CALLBACK(on_jaiku_menu_item_activate);
	selection->data = ((JaikuData*)item->user_data)->nick;
	item->selections = g_list_append(item->selections, selection);

	selection = g_new0(ViewItemSelection, 1);
	selection->title = g_strdup("Open the thread in a browser");
	selection->callback = G_CALLBACK(on_link_menu_item_activate);
	selection->data = item->url;
	item->selections = g_list_append(item->selections, selection);

	g_string_append(string, " in Jaiku ");
}

static void parse_json_item_jaiku(JsonObject* json, ViewItem* item, Http* http, GString* string, GString* speech) {
	JsonObject* user;
	JsonObject* object;
	JsonObject* content;
	gchar* s;
	gchar* s2;
	ViewItemSelection* selection;
	ViewItemSelection* thread_selection = NULL;
	size_t len;
	
	if ((object = json_object_object_get(json, "title"))) {
		((JaikuData*)item->user_data)->title = g_strdup(json_object_get_string(object));
	}
	if ((object = json_object_object_get(json,"title")) || (object = json_object_object_get(json,"content"))) {
		s = json_object_get_string(object);
		s = g_markup_escape_text(s, -1);
		g_string_append(string, s);
		if (speech) {
			g_string_append(speech, s);
		}
		g_free(s);
	}
	g_string_append(string, "\n<small>");
	if ((user = json_object_object_get(json, "user"))) {
		if ((object = json_object_object_get(user, "nick"))) {
			((JaikuData*)item->user_data)->nick = g_strdup(json_object_get_string(object));
			if (speech) {
				g_string_prepend(speech, ": ");
				g_string_prepend(speech, ((JaikuData*)item->user_data)->nick);
				g_string_prepend(speech, "Jaiku from ");
			}
			if (*(((JaikuData*)item->user_data)->nick) == '#') {
				g_string_append(string, "In ");
			} else {
				g_string_append(string, "By ");
			}
			s = g_markup_escape_text(((JaikuData*)item->user_data)->nick, -1);
			g_string_append(string, s);
			g_free(s);

			selection = g_new0(ViewItemSelection, 1);
			selection->title = g_strdup_printf("Open %s in a browser", ((JaikuData*)item->user_data)->nick);
			selection->callback = G_CALLBACK(on_jaiku_menu_item_activate);
			selection->data = ((JaikuData*)item->user_data)->nick;
			item->selections = g_list_append(item->selections, selection);

			selection = g_new0(ViewItemSelection, 1);
			selection->title = g_strdup("Open the thread in a browser");
			selection->callback = G_CALLBACK(on_link_menu_item_activate);
			thread_selection = selection;
			/* thread_selection should be this one until ... */
			item->selections = g_list_append(item->selections, selection);
		}
		if ((object = json_object_object_get(user, "avatar"))) {
			/* gdk_threads_enter(); */ /* Just for sure */
			item->avatar = image_cache_load_image(image_cache, http, json_object_get_string(object));
			/* gdk_threads_leave(); */
		}
	}
	if ((object = json_object_object_get(json, "url")) &&
	    (s = json_object_get_string(object))) {
		if (!strncmp(s, "http://jaiku.com/channel/", 25)) {
			if (!(s2 = strchr(s, '#'))) {
				s2 = s + strlen(s);
			}
			item->url = g_strndup(s, s2 - s);
			s += 25;
			if (!(s2 = strchr(s, '/'))) {
				s2 = s + strlen(s);
			}
			((JaikuData*)item->user_data)->channel = g_strndup(s, s2 - s);
			selection = g_new0(ViewItemSelection, 1);
			selection->title = g_strdup_printf("Open #%s", ((JaikuData*)item->user_data)->channel);
			selection->callback = G_CALLBACK(on_channel_menu_item_activate);
			selection->data = ((JaikuData*)item->user_data)->channel;
			item->selections = g_list_prepend(item->selections, NULL);
			item->selections = g_list_prepend(item->selections, selection);
			g_string_append(string, " to #");
			g_string_append(string, ((JaikuData*)item->user_data)->channel);
		} else if ((strncmp(s, "http://", 7) || !(s2 = strchr(s + 7, '.')) || strncmp(s2, ".jaiku.com/presence/", 20))) {
			if (*(((JaikuData*)item->user_data)->nick) == '#') {
				item->url = g_strdup_printf("http://jaiku.com/channel/%s/presence/%s", ((JaikuData*)item->user_data)->nick + 1, item->unique_id + 6);
			} else {
				item->url = g_strdup_printf("http://%s.jaiku.com/presence/%s", ((JaikuData*)item->user_data)->nick, item->unique_id + 6);
			}

			selection = g_new0(ViewItemSelection, 1);
			selection->title = g_strdup(s);
			selection->callback = G_CALLBACK(on_link_menu_item_activate);
			selection->data = NULL;
			item->selections = g_list_prepend(item->selections, NULL);
			item->selections = g_list_prepend(item->selections, selection);
			
			if (!strncmp(s, "http://www.flickr.com/photos/", 29) &&
			    (content = json_object_object_get(json, "content")) &&
			    (s2 = json_object_get_string(content)) && (len = strlen(s2)) &&
			    len > 6 && !strcmp(s2 + (len - 6), "_m.jpg")) {
				s2 = g_strdup(s2);
				s2[len - 5] = 's';
				/* gdk_threads_enter(); */ /* Just for sure */
				item->icon = image_cache_load_image(image_cache, http, s2);
				/* gdk_threads_leave(); */
				g_free(s2);
			}
		} else {
			if (!(s2 = strchr(s, '#'))) {
				s2 = s + strlen(s);
			}
			item->url = g_strndup(s, s2 - s);
		}
		/* ... the thread_selection should be set earlier. */
		if (thread_selection) {
			thread_selection->data = item->url;
		}
	}
	if (!item->icon && (object = json_object_object_get(json, "icon")) &&
	    (s = json_object_get_string(object)) &&
	    strlen(s) > 0) {
		/* gdk_threads_enter(); */ /* Just for sure */
		item->icon = image_cache_load_image(image_cache, http, s);
		/* gdk_threads_leave(); */
	}
	if ((object = json_object_object_get(json, "location"))) {
		g_string_append(string, " in ");
		s = g_markup_escape_text(json_object_get_string(object), -1);
		g_string_append(string, s);
		g_free(s);
	}
	g_string_append(string, " in Jaiku ");
}

static gboolean update_contacts_view_callback(ViewAddingSession* session, JsonObject* feed, Http* http, time_t delta, gpointer user_data) {
	JsonObject* contacts;
	ViewItem* item;
	GList* list;
	GList* list_item;
	JsonObject* json;
	JsonObject* object;
	JsonObject* presence;
	JsonObject* location;
	gchar* url;
	int i;
	GString* string;
	gchar* s;
	gboolean comma;
	ViewItem* existing;
	
	if (!(contacts = json_object_object_get(feed, "contacts"))) {

		return FALSE;
	}

	list = NULL;
	for (i = 0; i < json_object_array_length(contacts); i++) {
		if (!(json = json_object_array_get_idx(contacts, i))) {
			break;
		}
		if (!(object = json_object_object_get(json, "nick"))) {
			continue;
		}
		string = g_string_new("<big>");
		item = g_malloc0(sizeof(ViewItem));
		item->item_class = &contact_item_class;
		
		item->unique_id = g_strdup(json_object_get_string(object));
		s = g_markup_escape_text(item->unique_id, -1);
		g_string_append(string, s);
		g_free(s);
		g_string_append(string, "</big> ");
		
		if ((object = json_object_object_get(json, "first_name"))) {
			s = g_markup_escape_text(json_object_get_string(object), -1);
			g_string_append(string, s);
			g_string_append(string, " ");
			g_free(s);
		}
		if ((object = json_object_object_get(json, "last_name"))) {
			s = g_markup_escape_text(json_object_get_string(object), -1);
			g_string_append(string, s);
			g_string_append(string, " ");
			g_free(s);
		}
		g_string_append(string, "\n<small>");
		if ((object = json_object_object_get(json, "avatar"))) {
			item->avatar = (GdkPixbuf*)json_object_get_string(object);
		}
#if 0		
		if (*(item->unique_id) != '#') {
			name = NULL;
			key = NULL;
			if ((name = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, NULL)) && *name &&
			    (key = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_KEY, NULL)) && *key)) {
				url = g_strdup_printf("http://%s.jaiku.com/presence/json/json?user=%s&personal_key=%s", item->unique_id, name, key);	
				if ((presence = http_get_json_object(http, real_url))) {
					if ((object = json_object_object_get(presence, "line"))) {
						s = g_markup_escape_text(json_object_get_string(object), -1);
						g_string_append(string, s);
						g_free(s);
					}
					if ((location = json_object_object_get(presence, "location"))) {
						g_string_append(string, " in ");
						comma = FALSE;
						if ((object = json_object_object_get(location, "neighbourhood"))) {
							if (*(s = g_markup_escape_text(json_object_get_string(object), -1))) {
								g_string_append(string, s);
								g_string_append(string, " ");
								comma = TRUE;
							}
							g_free(s);
						}
						if ((object = json_object_object_get(location, "city"))) {
							if (*(s = g_markup_escape_text(json_object_get_string(object), -1))) {
								if (comma) {
									g_string_append(string, ", ");
								}
								g_string_append(string, s);
								g_string_append(string, " ");
								comma = TRUE;
							}
							g_free(s);
						}
						if ((object = json_object_object_get(location, "country"))) {
							if (*(s = g_markup_escape_text(json_object_get_string(object), -1))) {
								if (comma) {
									g_string_append(string, ", ");
								}
								g_string_append(string, s);
								g_string_append(string, " ");
							}
							g_free(s);
						}
					}
					json_object_put(presence);
				}
				g_free(url);
			}
			g_free(name);
			g_free(password);
		}
#endif
		g_string_append(string, "</small>");
		item->text_begin = g_string_free(string, FALSE);
		item->text_end = NULL;

		list = g_list_prepend(list, item);
	}
	list = g_list_sort(list, compare_item_nicks);

	for (list_item = g_list_first(list); list_item; list_item = g_list_next(list_item)) {
		item = list_item->data;
		/* gdk_threads_enter(); */ /* Could be after image_cache_load_image, but just for sure here */
		item->avatar = image_cache_load_image(image_cache, http, (gchar*)item->avatar);

		gdk_threads_enter(); /* See two lines above! */
		if (view_adding_session_is_valid(session)) {
			if ((existing = view_adding_session_get_item_and_remove_previous_items(session, item->unique_id))) {
				view_item_update_texts(existing, item->text_begin, item->text_end);
			} else {
				view_adding_session_add_item(session, item);
			}
		}
		gdk_threads_leave();		
	}
	g_list_free(list);
		
	return TRUE;
}

static gboolean update_overview_callback(ViewAddingSession* session, JsonObject* feed, Http* http, time_t delta, gpointer user_data) {
	JsonObject* stream;
	JsonObject* json;
	int i;

	jaiku_update_location(http);
	if ((stream = json_object_object_get(feed, "stream"))) {		
		for (i = 0; i < json_object_array_length(stream); i++) {
			if (!(json = json_object_array_get_idx(stream, i))) {
				
				return FALSE;
			}
			if (!handle_json_item(json, session, http, FALSE, NULL, delta)) {

				return FALSE;
			}
		}
	}
	
	return TRUE;
}

static void update_thread_view(View* view) {
	gchar* name = NULL;
	gchar* key = NULL;
	gchar* url;

	if ((name = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, NULL)) && *name &&
	    (key = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_KEY, NULL)) && *key) {
		url = g_strdup_printf("%s/json?user=%s&personal_key=%s", view_get_unique_id(view), name, key);	
		backend_start_update(view, url, NULL, &jaiku_item_source, FALSE, update_thread_view_callback, NULL);
		g_free(url);
	}
	g_free(name);
	g_free(key);
}

static gboolean update_thread_view_callback(ViewAddingSession* session, JsonObject* feed, Http* http, time_t delta, gpointer user_data) {
	gchar* referred_unique_id;
	JsonObject* stream;
	JsonObject* json;
	int i;

	if (!(referred_unique_id = handle_json_item(feed, session, http, TRUE, NULL, delta))) {
		
		return FALSE;
	}

	if ((stream = json_object_object_get(feed, "comments"))) {
		for (i = 0; i < json_object_array_length(stream); i++) {
			if (!(json = json_object_array_get_idx(stream, i))) {
				
				return FALSE;
			}
			if (!handle_json_item(json, session, http, TRUE, referred_unique_id, delta)) {
				
				return FALSE;
			}
		}
	}
	
	return TRUE;
}

static void update_user_view(View* view) {
	gchar* name = NULL;
	gchar* key = NULL;
	gchar* url;

	if ((name = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_NAME, NULL)) && *name &&
	    (key = gconf_client_get_string(gconf, MAUKU_GCONF_KEY_JAIKU_KEY, NULL)) && *key) {
		url = g_strdup_printf("%s?user=%s&personal_key=%s", view_get_unique_id(view), name, key);	
		backend_start_update(view, url, NULL, &jaiku_item_source, FALSE, update_user_view_callback, NULL);
		g_free(url);
	}
	g_free(name);
	g_free(key);
}

static gboolean update_user_view_callback(ViewAddingSession* session, JsonObject* feed, Http* http, time_t delta, gpointer user_data) {
	JsonObject* stream;
	JsonObject* json;
	int i;

	if ((stream = json_object_object_get(feed, "stream"))) {		
		for (i = 0; i < json_object_array_length(stream); i++) {
			if (!(json = json_object_array_get_idx(stream, i))) {
				
				return FALSE;
			}
			if (!handle_json_item(json, session, http, FALSE, NULL, delta)) {

				return FALSE;
			}
		}
	}
	
	return TRUE;
}
