/*****************************************************************************
 *** Mauku - Jaiku client for Maemo devices 
 ***
 *** Copyright (c) 2007 Henrik Hedberg <hhedberg@innologies.fi>
 ***
 *** Licensed under the Apache License, Version 2.0 (the "License");
 *** you may not use this file except in compliance with the License.
 *** You may obtain a copy of the License at
 ***
 ***     http://www.apache.org/licenses/LICENSE-2.0
 ***
 *** Unless required by applicable law or agreed to in writing, software
 *** distributed under the License is distributed on an "AS IS" BASIS,
 *** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *** See the License for the specific language governing permissions and
 *** limitations under the License.
 ***
 *****************************************************************************/
 
#include "config.h"

#include <curl/curl.h>
#include <json.h>
#include <glib.h>
#include <time.h>
#include <strings.h>
#include <string.h>

#ifdef OSSO_IC

#include <osso-ic.h>
#include <osso-ic-dbus.h>
#include <dbus/dbus.h>

#else

#include <conic.h>

#endif

#if HILDON == 1

#include <hildon/hildon.h>

#else

#include <hildon-widgets/hildon-note.h>

#endif

#include "mauku.h"

struct _Http {
	GString* buffer;
	CURL* curl;
	gchar* cookie_filename;
	gchar* userpass;
	time_t server_time;
	time_t reply_start_time;
};

static size_t curl_header(void* ptr, size_t size, size_t nmemb, Http* http);
static size_t curl_write(void* ptr, size_t size, size_t nmemb, GString* string);
static void iap_connected(const gchar* iap_name);
static void iap_disconnected(const gchar* iap_name);
#ifdef OSSO_IC
static void iap_status_changed(struct iap_event_t* event, void* user_data);
static DBusHandlerResult iap_state_changed(DBusConnection* connection, DBusMessage* message, void* user_data);
#else
static void connection_event(ConIcConnection* connection, ConIcConnectionEvent* event, gpointer user_data);
#endif

gboolean connected = FALSE;
GtkWidget* connection_waiting_widget = NULL;
#ifndef OSSO_IC
ConIcConnection* connection = NULL;
#endif

void http_init() {
	CURLcode code;

#ifdef OSSO_IC
	DBusError error;
	const gchar* filter = "interface=" ICD_DBUS_INTERFACE;

	dbus_error_init(&error);
	dbus_bus_add_match((DBusConnection*)osso_get_sys_dbus_connection(osso_context), filter, &error);
	if (dbus_error_is_set(&error)) {
		fprintf(stderr, "ERROR: dbus_bus_add_match failed: %s\n", error.message);
		return;

	}
	if (!dbus_connection_add_filter((DBusConnection*)osso_get_sys_dbus_connection(osso_context), iap_state_changed, NULL, NULL)) {
		fprintf(stderr, "ERROR: dbus_connection_add_filter failed: %s\n", error.message);
		return;
	}

	osso_iap_cb(iap_status_changed);
	osso_iap_connect(OSSO_IAP_ANY, OSSO_IAP_REQUESTED_CONNECT, NULL);
#else	

	if ((connection = con_ic_connection_new())) {
		g_signal_connect(G_OBJECT(connection), "connection-event", G_CALLBACK(connection_event), NULL);
		g_object_set(G_OBJECT(connection), "automatic-connection-events", TRUE, NULL);
		con_ic_connection_connect(connection, CON_IC_CONNECT_FLAG_NONE);
	}	
#endif

	if ((code = curl_global_init(CURL_GLOBAL_ALL)) != 0) {
		fprintf(stderr, "ERROR: curl_global_init failed: %d\n", code);
	}

}

void http_cleanup() {
	curl_global_cleanup();
#ifdef OSSO_IC
	osso_iap_disconnect(OSSO_IAP_ANY, NULL);
#else	
	con_ic_connection_disconnect(connection);
#endif

}

/* NULL if error. */
Http* http_new() {
	Http* http;
	CURLcode code;

	if (!(http = g_malloc0(sizeof(Http)))) {
	
		return NULL;
	}	 

	if ((http->curl = curl_easy_init()) == NULL) {
		g_free(http);

		return NULL;
	}

	http->buffer = g_string_new("");
	http->cookie_filename = NULL;
	if ((code = curl_easy_setopt(http->curl, CURLOPT_WRITEFUNCTION, curl_write)) ||
	    (code = curl_easy_setopt(http->curl, CURLOPT_WRITEDATA, http->buffer)) ||
	    (code = curl_easy_setopt(http->curl, CURLOPT_HEADERFUNCTION, curl_header)) ||
	    (code = curl_easy_setopt(http->curl, CURLOPT_HEADERDATA, http)) ||
	    (code = curl_easy_setopt(http->curl, CURLOPT_COOKIEFILE, "")) ||
	    (code = curl_easy_setopt(http->curl, CURLOPT_FOLLOWLOCATION, (long)1)) || 
	    (code = curl_easy_setopt(http->curl, CURLOPT_TIMEOUT, (long)300)) ||
	    (code = curl_easy_setopt(http->curl, CURLOPT_NOSIGNAL, (long)1)) ||
	    (code = curl_easy_setopt(http->curl, CURLOPT_USERAGENT, "Mauku/" VERSION))) {
		g_string_free(http->buffer, TRUE);
		curl_easy_cleanup(http->curl);
	    	g_free(http);
		
		return NULL;
	}

	return http;
}

void http_destroy(Http* http) {
	if (http) {
		curl_easy_cleanup(http->curl);
		g_string_free(http->buffer, TRUE);
		g_free(http->cookie_filename);
		g_free(http->userpass);
		g_free(http);
	}
}

void http_free(Http* http, gchar* ptr) {
	curl_free(ptr);
}
	
/* NULL if error. Returned buffer may contain NULL bytes. Returned buffer must NOT be freed; it is reused in the next call. */
char* http_get_data(Http* http, const char* url, size_t* len) {
	CURLcode code;

	*len = 0;
	g_string_truncate(http->buffer, 0);
	if ((code = curl_easy_setopt(http->curl, CURLOPT_POST, 0)) ||
	    (code = curl_easy_setopt(http->curl, CURLOPT_URL, url))) {

		return NULL;
	}
	http->reply_start_time = http->server_time = 0;
	if ((code = curl_easy_perform(http->curl)) != 0) {

		return NULL;
	}
	*len = http->buffer->len;

	return http->buffer->str;
}

/* NULL if error. Returned object MUST be freed (put) after use. */
JsonObject* http_get_json_object(Http* http, const char* url) {
	JsonObject* object;
	char* data;
	size_t len;

	if (!(data = http_get_data(http, url, &len))) {
	
		return NULL;
	}

	object = json_tokener_parse(http->buffer->str);
	if (is_error(object)) {
	
		return NULL;
	}

	return object;
}

time_t http_get_reply_start_time(Http* http) {

	return http->reply_start_time;
}

time_t http_get_server_time(Http* http) {

	return http->server_time;
}

char* http_post_data(Http* http, const char* url, size_t* len, const char* post_data) {
	CURLcode code;
	struct curl_slist* slist = NULL;

	*len = 0;
	g_string_truncate(http->buffer, 0);
	if (!(slist = curl_slist_append(slist, "Content-Type: application/x-www-form-urlencoded; charset=utf-8"))) {

		return NULL;
	}
	if ((code = curl_easy_setopt(http->curl, CURLOPT_POST, 1)) ||
	    (code = curl_easy_setopt(http->curl, CURLOPT_POSTFIELDS, post_data)) ||
	    (code = curl_easy_setopt(http->curl, CURLOPT_URL, url)) ||
	    (code = curl_easy_setopt(http->curl, CURLOPT_HTTPHEADER, slist))) {
		curl_slist_free_all(slist);
	    
		return NULL;
	}
	http->reply_start_time = http->server_time = 0;
	if ((code = curl_easy_perform(http->curl)) != 0) {
		curl_slist_free_all(slist);
		curl_easy_setopt(http->curl, CURLOPT_HTTPHEADER, NULL);

		return NULL;
	}
	curl_slist_free_all(slist);
	curl_easy_setopt(http->curl, CURLOPT_HTTPHEADER, NULL);

	*len = http->buffer->len;

	return http->buffer->str;
}

time_t http_parse_date(Http* http, gchar* datestring) {
	time_t retvalue;
	
	if ((retvalue = curl_getdate(datestring, NULL)) == -1) {
		retvalue = 0;
	}
	
	return retvalue;
}

/* NULL if error. Returned object MUST be freed (put) after use. */
JsonObject* http_post_json_object(Http* http, const char* url, const char* post_data) {
	JsonObject* object;
	char* data;
	size_t len;

	if (!(data = http_post_data(http, url, &len, post_data))) {
	
		return NULL;
	}

	object = json_tokener_parse(http->buffer->str);
	if (is_error(object)) {
	
		return NULL;
	}
	
	return object;
}

gboolean http_set_cookie_file(Http* http, const gchar* filename) {
	if (http->cookie_filename) {
		g_free(http->cookie_filename);
	}
	http->cookie_filename = g_strdup(filename);
	if (curl_easy_setopt(http->curl, CURLOPT_COOKIEFILE, http->cookie_filename) || 
	    curl_easy_setopt(http->curl, CURLOPT_COOKIEJAR, http->cookie_filename)) {

		return FALSE;
	}

	return TRUE;
}

gboolean http_set_basic_authentication(Http* http, const gchar* userpass) {
	if (http->userpass) {
		g_free(http->userpass);
	}
	if (userpass) {
		http->userpass = g_strdup(userpass);
		if (curl_easy_setopt(http->curl, CURLOPT_HTTPAUTH, CURLAUTH_BASIC) ||
		    curl_easy_setopt(http->curl, CURLOPT_USERPWD, http->userpass)) {

			return FALSE;
		}
	} else {
		http->userpass = NULL;
		if (curl_easy_setopt(http->curl, CURLOPT_USERPWD, NULL)) {

			return FALSE;
		}
	}	
	
	return TRUE;
}

/* Return a newly allocated string that must be freed using http_free! */
gchar* http_url_encode(Http* http, const gchar* s) {
#ifdef OSSO_IC
	return curl_escape(s, 0);
#else
	return curl_easy_escape(http->curl, s, 0);
#endif
}

gboolean http_is_connected() {

	return connected;
}

gboolean http_make_connected() {
	
	if (!connected) {
		connection_waiting_widget = hildon_note_new_information(NULL, "Waiting for a network connection to come up...");
		hildon_note_set_button_text(HILDON_NOTE(connection_waiting_widget), "Cancel");
		g_object_ref(connection_waiting_widget);
#ifdef OSSO_IC
		osso_iap_connect(OSSO_IAP_ANY, OSSO_IAP_REQUESTED_CONNECT, NULL);
#else
		con_ic_connection_connect(connection, CON_IC_CONNECT_FLAG_NONE);
#endif
		gtk_dialog_run(GTK_DIALOG(connection_waiting_widget));
		gtk_widget_destroy(connection_waiting_widget);
		g_object_unref(connection_waiting_widget);
		connection_waiting_widget = NULL;
	}
	
	return connected;
}

static size_t curl_header(void* ptr, size_t size, size_t nmemb, Http* http) {
	struct tm struct_tm;
	char* s;
	
	if (!http->reply_start_time) {
		http->reply_start_time = time(NULL);
	}

	if (size * nmemb > 5 && !strncasecmp(ptr, "date:", 5)) {
		s = g_strndup(ptr + 5, size * nmemb - 5);
		http->server_time = curl_getdate(s, NULL);
		g_free(s);
	}
	
	return size * nmemb;
}

static size_t curl_write(void* ptr, size_t size, size_t nmemb, GString* string) {
	g_string_append_len(string, ptr, size * nmemb);
	
	return size * nmemb;
}

static void iap_connected(const gchar* iap_name) {
	gchar* key = NULL;
	
	if (!connected) {
		connected = TRUE;
		ui_start_update_overview();
	}
}

static void iap_disconnected(const gchar* iap_name) {
	connected = FALSE;
}

#ifdef OSSO_IC

static void iap_status_changed(struct iap_event_t* event, void* user_data) {
	gdk_threads_enter();

#ifdef i386
        connected = TRUE;
        ui_start_update_overview();
#else
        if (event->type == OSSO_IAP_DISCONNECTED || event->type == OSSO_IAP_ERROR) {
                iap_disconnected(event->iap_name);
        } else if (event->type == OSSO_IAP_CONNECTED) {
                iap_connected(event->iap_name);
                if (connection_waiting_widget) {
                        gtk_widget_destroy(GTK_WIDGET(connection_waiting_widget));
                }
        }
#endif

	gdk_threads_leave();
}

static DBusHandlerResult iap_state_changed(DBusConnection* connection, DBusMessage* message, void* user_data) {
	DBusHandlerResult ret = DBUS_HANDLER_RESULT_HANDLED;
	DBusError error;
        gchar* name;
        gchar* type;
        gchar* state;

	gdk_threads_enter();

        dbus_error_init (&error);
        if (!dbus_message_is_signal(message, ICD_DBUS_INTERFACE, ICD_STATUS_CHANGED_SIG ) ||
            !dbus_message_get_args (message, &error, DBUS_TYPE_STRING, &name, DBUS_TYPE_STRING, &type,
                                    DBUS_TYPE_STRING, &state, DBUS_TYPE_INVALID)) {
                ret = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
        } else if (!strcmp(state, "DISCONNECTING")) {
                iap_disconnected(name);
        } else if (!strcmp(state, "CONNECTED")) {
                iap_connected(name);
        }

	gdk_threads_leave();

        return ret;
}

#else

static void connection_event(ConIcConnection* connection, ConIcConnectionEvent* event, gpointer user_data) {
	gdk_threads_enter();
	
#ifdef i386
	connected = TRUE;
//	ui_start_update_overview();
#else
	switch (con_ic_connection_event_get_status(event)) {
		case CON_IC_STATUS_CONNECTED:
			iap_connected(con_ic_event_get_iap_id(CON_IC_EVENT(event)));
			break;
		case CON_IC_STATUS_DISCONNECTED:
			iap_disconnected(con_ic_event_get_iap_id(CON_IC_EVENT(event)));
			break;
	}
#endif
	if (connection_waiting_widget) {
		gtk_widget_destroy(GTK_WIDGET(connection_waiting_widget));
		connection_waiting_widget = NULL;
	}
	
	gdk_threads_leave();
}

#endif
